/*
 * File Name: mxView.c
 */

/*
 * This file is part of mxSudoku.
 *
 * mxSudoku is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * mxSudoku is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 Marcel Hendrickx
 * All rights reserved.
 */

//------------------------------------------------------------------------------
// Include Files
//------------------------------------------------------------------------------

// configuration parameters of the project
#include "config.h"

// The implemented interface
#include "mxGrid.h"
#include "mxView.h"
// system include files

// local include files
#include "log.h"
#include "i18n.h"
#include "mxControl.h"


//------------------------------------------------------------------------------
// Type Declarations
//------------------------------------------------------------------------------


//------------------------------------------------------------------------------
// Global Constants
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// Static Variables
//------------------------------------------------------------------------------

// use GtkLabel to force redraw
static GtkWidget *g_label;

// drawing info for 3x4 cells of keypad
static char* keys3x4_ornaments[] = {
	"4224", "4222", "4422",
	"2224", "2222", "2422",
	"2224", "2222", "2422",
	"2244", "2242", "2442"
};

// how will the keys be located on the screen
static char* keys3x4_content[] = {
	"7","8","9",
	"4","5","6",
	"1","2","3",
	"?","C","V"
};

// translate a cell number back to a value
// TODO: make this better configurable if more views are added
static int  g_trans_key_to_num[] = {7,8,9,4,5,6,1,2,3,-2,-1,-1};
static char g_trans_num_to_char[] = "0123456789ABCDEF";

// variables that depend on the size/structure of the keypad
// adapt when other keypad is selected
static int g_keys_cmd_verify = 11;

static GtkGrid *g_board;
static GtkGrid *g_keypad;

// the active cell in grid 2 (keypad)
static int g_active_nr = -1;

static char *g_static_message = NULL;
//==============================================================================
// Local Function Definitions
//==============================================================================

static void force_redraw(void);
static void on_cell_selected(GtkGrid *grid, int cell_nr, gpointer data);
static void on_number_entered(GtkGrid *grid, int cell_nr, int number, gpointer data);
static GtkWidget *screen_layout_with_keypad (p_mxview_game game);

//==============================================================================
// Functions Implementation
//==============================================================================

GtkWidget *mxview_create(p_mxview_game game)
{
	GtkWidget *widget = NULL;
	
    LOGPRINTF("entry");

	// check if game data is supplied and use it to create the correct view for the game
	if (game)
	{
		switch (game->type)
		{
			case mxview_keypad:
			    widget = screen_layout_with_keypad(game);
			break;
				
			default:
				LOGPRINTF("game type is unknown:%d", (int)game->type);
			break;
		}
	}
	
	return widget;
}

void mxview_delete(void)
{
	// TODO: !!!!
}

void mxview_update()
{
	force_redraw();
}

// show a status message at the bottom of the screen
void mxview_status_msg(char *msg)
{
	gtk_label_set_text( GTK_LABEL(g_label), msg );
}

// set the value of a cell on the board widget
void mxview_cell_set_value(int cell_nr, int value, int flags)
{
	//LOGPRINTF("entry: cell: %d, value %d flags %d", cell_nr, value, flags);

	if (g_board)
	{
		if (cell_nr >= 0)
		{
			char str[2] = " "; // empty
			
			// fill cell with selected
			if ((value >= 0) && (value < strlen(g_trans_num_to_char)))
			{
				str[0] = g_trans_num_to_char[value];
			}
			// else empty
			
			gtk_grid_set_cell_text(g_board, cell_nr, str);
			
			// also adapt the fg color depending on the READONLY attrib
			{
				GdkColor grey  = {0, 0x4444, 0x4444, 0x4444};
				GdkColor black = {0, 0x0000, 0x0000, 0x0000};
				GdkColor white = {0, 0xFFFF, 0xFFFF, 0xFFFF};
				if ((flags & MXVIEW_CELL_VALUE_READONLY) == MXVIEW_CELL_VALUE_READONLY)
				{
					// read only
					gtk_grid_set_cell_fg_color(g_board, cell_nr, grey);
				}
				else
				{
					gtk_grid_set_cell_fg_color(g_board, cell_nr, black);
				}
				
				if ((flags & MXVIEW_CELL_VALUE_WRONG) == MXVIEW_CELL_VALUE_WRONG) // value is wrong
				{
					//LOGPRINTF("bg grey");
					gtk_grid_set_cell_bg_color(g_board, cell_nr, grey);
				}
				else
				{
					//LOGPRINTF("bg white");
					gtk_grid_set_cell_bg_color(g_board, cell_nr, white);
				}
			}
		}
		
	}
}

// draw last recorded scribble
void mxview_test_draw_last(void)
{
	gtk_grid_test_draw_last(g_board);
}

void mxview_update_board(p_mxview_game game)
{
	char             **pp_ornaments;
	int              nr_horizontal;
	int				 nr_vertical;

	nr_horizontal = game->board_horizontal;
	nr_vertical   = game->board_vertical;
	pp_ornaments  = game->pp_ornaments;
	
	gtk_widget_hide(GTK_WIDGET(g_board));
	gtk_grid_set_size(GTK_GRID(g_board), nr_horizontal, nr_vertical);
	gtk_grid_set_preferred_cell_size(GTK_GRID(g_board), 0, 0, MXGRID_CELL_SIZE_MODE_SQUARE); // auto size
	gtk_grid_set_ornaments (GTK_GRID(g_board), (nr_horizontal*nr_vertical), pp_ornaments);
	gtk_widget_show(GTK_WIDGET(g_board));
	
}

// set the 'static' status message.
void mxview_set_static_message(char *msg)
{
	LOGPRINTF("Message: %s", msg);
	if (g_static_message != NULL)
	{
		g_free(g_static_message);
	}
	g_static_message = g_strdup_printf(msg);
	
	force_redraw();
}

//==============================================================================
// Local Function Implementation
//==============================================================================

// update the label to trigger an update of the screen!!
// The screen is not always updated like I want it (at least not in qemu.
// I 'discovered' that adding a label at the bottom and updating it forced a
// redraw. The behaviour on the DR seems to be a little different, but it also
// seems to work there.
// TODO: check if this is also need on DR
static void force_redraw(void)
{
	gchar       *msg  = NULL;
		
	if (g_static_message != NULL)
	{
		gtk_label_set_text( GTK_LABEL(g_label), g_static_message );
	}
	else
	{
		msg = g_strdup_printf( _("mxSudoku"));
		gtk_label_set_text( GTK_LABEL(g_label), msg );
		g_free(msg);
	}
}	

// a cell in the grid is selected, react on it
static void on_cell_selected(GtkGrid *grid, int cell_nr, gpointer data)
{
	GdkColor grey  = {0, 0x8888, 0x8888, 0x8888};
	GdkColor white = {0, 0xFFFF, 0xFFFF, 0xFFFF};
	int grid_nr = (int)data;
	int redraw = 1;

	//LOGPRINTF("Grid: %d Cell: %d Active:%d", grid_nr, cell_nr, active_nr);
	
	// The Keys grid (this is a view internal matter, no feedback to control)
	if (grid_nr == 2) // keys
	{
		if (cell_nr >= 0)
		{
			// deselect previous
			if (g_active_nr >= 0)
			{
				gtk_grid_set_cell_bg_color(grid, g_active_nr, white); 
			}
			
			// special case, trigger verify (do not select cell)
			if (cell_nr == g_keys_cmd_verify)
			{
				// signal mxControl to do the verification
				{
					mxcontrol_event event;
					
					event.type = mxcontrol_event_cmd_verify;
					event.cell_nr = 0;
					redraw = mxcontrol_handle_view_event(&event);
				}
			}
			else if ( g_active_nr == cell_nr)
			{
				// deselect if pressed twice
				g_active_nr = -1;
			}
			else
			{
				// select new
				g_active_nr = cell_nr;
				gtk_grid_set_cell_bg_color(grid, g_active_nr, grey);
			}
		}
	}
	// The board grid
	else if (grid_nr == 1)
	{
		// only send event if valid cell is selected and key on pad is selected
		if ((cell_nr >= 0) && (g_active_nr >= 0))
		{
			mxcontrol_event event;
			
			event.type = mxcontrol_event_set_value;
			event.cell_nr = cell_nr;
			event.value = g_trans_key_to_num[g_active_nr];
			// let the called function decide if a forced redraw should be done
			// this will overwrite the status message at the bottom of the screen.
			redraw = mxcontrol_handle_view_event(&event);
		}
	}
	// else error?

	if (redraw == 1)
	{
		// only update if requested, else GtkLabel text will
		// already be overwritten (UGLY construction...)
		// TODO: a flag could be set to check if message is set when calling an 
		//       external (control) function. Only update if no msg set.
		force_redraw();
	}
}

// a value has been added using scribbles
static void on_number_entered(GtkGrid *grid, int cell_nr, int number, gpointer data)
{
	int grid_nr = (int)data;
	int redraw = 1;

	LOGPRINTF("entry cell:%d, number:%d", cell_nr, number);
	
	// The board grid
	if (grid_nr == 1)
	{
		// only send event if valid cell is selected and NO key on pad is selected
		if ((cell_nr >= 0) && (number >= 0) && (g_active_nr == -1))
		{
			mxcontrol_event event;
			
			event.type = mxcontrol_event_set_value;
			event.cell_nr = cell_nr;
			event.value = number;
			// let the called function decide if a forced redraw should be done
			// this will overwrite the status message at the bottom of the screen.
			redraw = mxcontrol_handle_view_event(&event);
		}
	}
	// else error?

	if (redraw == 1)
	{
		// only update if requested, else GtkLabel text will
		// already be overwritten (UGLY construction...)
		// TODO: a flag could be set to check if message is set when calling an 
		//       external (control) function. Only update if no msg set.
		force_redraw();
	}
}

// create main screen layout
static GtkWidget *screen_layout_with_keypad (p_mxview_game game)
{
    GtkWidget        *background=NULL;
    GtkWidget        *table;
    GtkWidget        *widget;
	GtkBox 	         *vbox;
	int i;
	char             **pp_ornaments;
	int              nr_horizontal;
	int				 nr_vertical;
	
    LOGPRINTF("entry");
	
	// handle configuration parameters!
	nr_horizontal = game->board_horizontal;
	nr_vertical   = game->board_vertical;
	pp_ornaments  = game->pp_ornaments;
	
	LOGPRINTF("Ignoring hor:%d ver:%d and symbols:%d", 
			  game->board_horizontal,
			  game->board_vertical,
			  game->keypad_nr_symbols);

    // object hierarchy:
    //           vbox
    //             |
    widget = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(widget);
    vbox = GTK_BOX(widget);
	background = widget;
    //             |
    //             |-- table 3x3
    //                   |
	widget = gtk_table_new(3,3, TRUE);
	gtk_box_pack_start(GTK_BOX(vbox), widget, TRUE, TRUE, 0);
	for (i=0; i<3; i++)
	{
		gtk_table_set_row_spacing(GTK_TABLE(widget), i, 2);
		gtk_table_set_col_spacing(GTK_TABLE(widget), i, 2);
	}
	gtk_widget_show(widget);
	table = widget;
    //                   |                |XXX|
    //                   |-- board (grid) |XXX|
    //                   |                |   |
	widget = gtk_grid_new();
	gtk_grid_set_size(GTK_GRID(widget), game->board_horizontal, game->board_vertical);
	gtk_grid_set_preferred_cell_size(GTK_GRID(widget), 0, 0, MXGRID_CELL_SIZE_MODE_SQUARE); // auto size
	gtk_grid_set_ornaments (GTK_GRID(widget), (nr_horizontal*nr_vertical), pp_ornaments);
	g_signal_connect(G_OBJECT(widget), "grid_cell_selected", G_CALLBACK(on_cell_selected), (gpointer)1L);
	g_signal_connect(G_OBJECT(widget), "grid_number_entered", G_CALLBACK(on_number_entered), (gpointer)1L);
	gtk_table_attach(GTK_TABLE(table), widget, 0, 3, 0, 2,
			GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);
	gtk_widget_show(widget);
	g_board = GTK_GRID(widget);
    //                   |                 |   |
    //                   |-- keypad (grid) |   |
    //                   |                 | X |
	widget = gtk_grid_new();
	// TODO: base keypad on game data (keypad_nr_symbols)
	gtk_grid_set_size(GTK_GRID(widget), 3, 4);
	gtk_grid_set_preferred_cell_size(GTK_GRID(widget), 0, 0, MXGRID_CELL_SIZE_MODE_DEFAULT); // auto size
	gtk_grid_set_ornaments (GTK_GRID(widget), (3*4), keys3x4_ornaments);
	for (i=0; i<(3*4); i++)
	{
		gtk_grid_set_cell_text (GTK_GRID(widget), i, keys3x4_content[i]);
	}
	g_signal_connect(G_OBJECT(widget), "grid_cell_selected", G_CALLBACK(on_cell_selected), (gpointer)2L);
	gtk_table_attach(GTK_TABLE(table), widget, 1, 2, 2, 3,
			GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);
	gtk_widget_show(widget);
	g_keypad = GTK_GRID(widget);
    //             |
    //             |-- label
    //             |
	widget = gtk_label_new(NULL);
	gtk_widget_set_name(widget, "label");
	gtk_box_pack_start(GTK_BOX(vbox), widget, FALSE, FALSE, 0);
	gtk_widget_show(widget);
	g_label = widget;
    
    return background;
}

