/*
 * File Name: mxControl.c
 */

/*
 * This file is part of mxSudoku.
 *
 * mxSudoku is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * mxSudoku is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 Marcel Hendrickx
 * All rights reserved.
 */

//------------------------------------------------------------------------------
// Include Files
//------------------------------------------------------------------------------

// configuration parameters of the project
#include "config.h"

// The implemented interface
#include "mxControl.h"
#include "do.h" // move to other file?

// system include files

// local include files
#include "mxModel.h"
#include "mxView.h"
#include "log.h"
#include "i18n.h"
#include "main.h"
#include "ipc.h"


//------------------------------------------------------------------------------
// Type Declarations
//------------------------------------------------------------------------------

typedef enum _game_mode_t {
	GameMode_Play,
	GameMode_Construct
} game_mode, *p_game_mode;

//------------------------------------------------------------------------------
// Global Constants
//------------------------------------------------------------------------------

// the location to load/store the last played game
#define LASTGAME "/media/mmcblk0p1/Programs/_mxSudoku/last_game"
// the location of the games
#define GAMES    "/media/mmcblk0p1/Programs/_mxSudoku/games/"
//------------------------------------------------------------------------------
// Static Variables
//------------------------------------------------------------------------------

static int g_model=-1;
static mxview_game g_game;
static GtkWidget *g_window=NULL; // window, needed for dialogs

// indicates wether a valid game has been generated or loaded
static int g_playing=0;

// demo mode
static int g_demo_mode=0;

// game mode
static game_mode g_game_mode = GameMode_Play;

//==============================================================================
// Local Function Definitions
//==============================================================================

static void display_game(void);
static void create_game_data();

//==============================================================================
// Functions Implementation
//==============================================================================

// This function handles events that are generated by the view or menu.
int mxcontrol_handle_view_event(p_mxcontrol_event event)
{
	int empty_cells = -1;  // to hold the number of cells not filled yet
	int redraw = 1;        // set to 0 when no forced redraw should be done
	
    //LOGPRINTF("entry");

	// an event has been triggered from the view
	switch (event->type)
	{
		// the event indicates that a new value is entered for a cell
		case mxcontrol_event_set_value:
			{
				mxmodel_cell cell;
				int flags;
				
				// check if cell can be changed
				mxmodel_get_cell_info(g_model, event->cell_nr, &cell);
				if (g_game_mode == GameMode_Play)
				{
					flags = ((cell.flags & MXMODEL_CELL_READONLY) == MXMODEL_CELL_READONLY) ? 1 : 0;
				}
				else // construct
				{
					flags = ((cell.flags & MXMODEL_CELL_MODIFYABLE) == MXMODEL_CELL_MODIFYABLE) ? 0 : 1;
				}
				if (flags == 0)
				{
					// check if we are in 'cheat' mode
					if (event->value == -2)
					{
						// copy the must value!! CHEATING!!
						event->value = cell.must_value;
					}
					// give the cell its new value (model and view)
					if (g_game_mode == GameMode_Play)
					{
						empty_cells = mxmodel_set_cell_value(g_model, event->cell_nr, event->value);
					}
					else // construct mode
					{
						mxmodel_set_cell_must_value(g_model, event->cell_nr, event->value);
					}
					
					// update the view
					mxview_cell_set_value(event->cell_nr, event->value, MXVIEW_CELL_VALUE_NORMAL);
					
					// is demo mode enabled?
					if (g_demo_mode & MXCTRL_DEMO_MODE_SCRIBBLE)
					{
						// also draw last scribbles
						mxview_test_draw_last();
					}
				}
				else
				{
					gchar *msg;
					
					//cell can not be modified
					LOGPRINTF("Cell can not be modified");
					ipc_sys_beep( 250, 1);
					msg = g_strdup_printf( _("Cell can not be modified!"));
					mxview_status_msg(msg);
					g_free(msg);
					redraw = 0;
				}
			}
			break;
		
		// request the current game to be verified
		case mxcontrol_event_cmd_verify:
			{
				do_verify_game();
				redraw = 0;
			}
			break;
			
		// TODO:
		// To make the seperation more visible should all menu command also go via this function?
		// I should probably refactor the content of menu.c also, or move it into here?
		default:
			break;
	}
	
	// check how many cells have not been entered
	empty_cells = mxmodel_get_empty_cells(g_model);
	
	if (empty_cells == 0)
	{
		// auto verify if all cells are filled
		do_verify_game();
		redraw = 0; // verify_game will have shown a message, we don't want to overwrite this
	}
	
	return redraw;
}

// create the model, view and controller for the game
void mxcontrol_create_game(GtkWidget *window, mxmodel_type type)
{
    GtkWidget        *widget;
	int              result;

    LOGPRINTF("entry");

	// store the window
	g_window = window;
	
	// Create the model
	if (type == mxmodel_unknown)
	{
		// get type of last saved file
		type = mxmodel_file_get_type(LASTGAME);
		// if unknown, set to default type
		if (type == mxmodel_unknown)
		{
			type = mxmodel_sudoku3x3;
		}
	}
	g_model = mxmodel_create(type);
	
	// Get info of current model to create view
	create_game_data();
	
	// ... and Create the view
	widget = mxview_create(&g_game);
	gtk_container_add(GTK_CONTAINER(window), widget);

	mxview_set_static_message(_("mxGame"));
	
	// open last game, which will be saved when the program closes
	result = mxmodel_file_load(g_model, LASTGAME, 1);
	if (result > 0)
	{
		// the load might have changed the ornaments
		mxview_update_board(&g_game);
		display_game();
	}
}

void mxcontrol_quit_game()
{
	int result;
	
    LOGPRINTF("entry");
	
	// BIG TODO: !!!free all memory (and other resources)!!!
	if (g_model)
	{
		mxmodel_delete(g_model);
	}
	
	mxview_delete();

	// close last game, when we were playing
	if (g_playing > 0)
	{
		result = mxmodel_file_save(g_model, LASTGAME);
	}
}

void mxcontrol_set_demo_mode(int mode)
{
	g_demo_mode = mode;
}

//==============================================================================
// Show the about/help dialog
void do_about_game(void)
{
    LOGPRINTF("entry");
	
	// show about/help
	main_about_game();
}

// open a game from file
void do_open_game(void)
{
	GtkWidget *file_chooser;
	mxmodel_type type;

	LOGPRINTF("entry: %d", (int)g_window);

	// choose a file
	file_chooser = gtk_file_chooser_dialog_new(
			_("Load Puzzle"),
			GTK_WINDOW(g_window), 
			GTK_FILE_CHOOSER_ACTION_OPEN, 
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
			GTK_STOCK_OK, GTK_RESPONSE_OK, NULL);

	// set the game folder
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(file_chooser), GAMES);

	// and get it
	if(gtk_dialog_run(GTK_DIALOG(file_chooser)) == GTK_RESPONSE_OK) 
	{
		gchar *sudoku_file;
		mxmodel_game game;
		
		sudoku_file = 
			gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(file_chooser));
		
		// determine the type of the file
		type = mxmodel_file_get_type(sudoku_file);
		if (type == mxmodel_unknown)
		{
			// assume old style file
			type = mxmodel_sudoku3x3;
			// TODO: show message and quit when backwards compatability is not needed anymore
		}
		mxmodel_get_info(g_model, &game);
		
		if (type != game.type)
		{
			// first destroy old type info
			
			LOGPRINTF("Resetting model and view old%d, new %d", game.type, type);
			// the type of the game changed, we need to update the model and the view
			mxmodel_delete(g_model);
			g_model = mxmodel_create(type);
			
			// Get info of current model to create view
			create_game_data();
		}
		
		// load the file into the model
		mxmodel_file_load(g_model, sudoku_file, 0);
		
		// and update the view
		mxview_update_board(&g_game);
		
		// display the model
		display_game();
		
		// free resources
		g_free(sudoku_file);

		// update game mode to play after each save (ends construct mode)?
		g_game_mode = GameMode_Play;
		// restore original message
		mxview_set_static_message(_("mxGame"));
	}

	gtk_widget_destroy(file_chooser);
}

// save a game from file
void do_save_as(void)
{
    GtkWidget *file_chooser;

	LOGPRINTF("entry");
	
	if (g_game_mode == GameMode_Construct)
	{
		// check if the constructed game is finished!
		if (mxmodel_check_complete(g_model) <= 0)
		{
			show_popup_msg("!!", _("You must create a full solution first before you can save"));
			// current constructed game not OK yet
			return;
		}
		// and update the view
		mxview_update_board(&g_game);
		
		// display the model
		display_game();
	}

	// choose a filename to store into
	file_chooser = gtk_file_chooser_dialog_new(
			_("Save As..."),
			GTK_WINDOW(g_window),
			GTK_FILE_CHOOSER_ACTION_SAVE,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, NULL);
	
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(file_chooser), GAMES);
	gtk_file_chooser_set_do_overwrite_confirmation(
			GTK_FILE_CHOOSER(file_chooser), TRUE);

	// if a file was chosen
	if(gtk_dialog_run(GTK_DIALOG(file_chooser)) == GTK_RESPONSE_ACCEPT) 
	{
		gchar       *sudoku_file;

		// get the selected filename
		sudoku_file = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(file_chooser));
		
		// and store it
		mxmodel_file_save (g_model, sudoku_file);

		// free resources
		g_free(sudoku_file);

		// update game mode to play after each save (ends construct mode)?
		g_game_mode = GameMode_Play;
		// restore original message
		mxview_set_static_message(_("mxGame"));
	}
	
	gtk_widget_destroy(file_chooser);
}

void do_construct_game_start(void)
{
//	mxmodel_type type = mxmodel_unknown;
	
    LOGPRINTF("entry");
	
	// load (empty) file
	// TODO
	
	// set game mode
	g_game_mode = GameMode_Construct;
	
	// display the model
	display_game();
}

// reset all non-read-only cells to empty
void do_reset_game(void)
{
	int i;
	
    LOGPRINTF("entry");

	// when in construction mode, do nothing
	if (g_game_mode == GameMode_Construct)
	{
		return;
	}
	
	// update the model and view with the must_values only
	for (i=0; i< (g_game.board_horizontal * g_game.board_vertical); i++)
	{
		mxmodel_cell cell;
	    int flags;
		int value;
	
		// get the data
		mxmodel_get_cell_info(g_model, i, &cell);
		flags = ((cell.flags & MXMODEL_CELL_READONLY) == MXMODEL_CELL_READONLY) ? MXVIEW_CELL_VALUE_READONLY : MXVIEW_CELL_VALUE_NORMAL;
		
		// copy only must value in model, clear others
		value = (flags == MXVIEW_CELL_VALUE_READONLY) ? cell.must_value : -1;
		mxmodel_set_cell_value(g_model, i, value);
		
		// update view of the (board) cell
		mxview_cell_set_value(i, value, flags);
	}
	mxview_update(); // force update
}


// generate a new random game
void do_generate_game(void)
{
	int i;
	
    LOGPRINTF("entry");
	
	if (mxmodel_can_generate(g_model) == 1)
	{
		// generate a new random game, difficulty level = 1
		mxmodel_generate(g_model, 1);
	}
	else
	{
		// clear current set and must values
		for (i=0; i< (g_game.board_horizontal * g_game.board_vertical); i++)
		{
			mxmodel_set_cell_value(g_model, i, -1);
			mxmodel_set_cell_must_value(g_model, i, -1);
		}
	
		// update the ornaments
		mxmodel_check_complete(g_model);
			
		// and update the view
		mxview_update_board(&g_game);

		// we enter construction mode
		g_game_mode = GameMode_Construct;
		
	    mxview_set_static_message(_("mxGame (construction mode)"));
	}
	
	// now update the view with the values from the model
	display_game();
}


// Show the solution for the current game
void do_solve_game(void)
{
	int i;
	
    LOGPRINTF("entry");

	// when in construction mode, do nothing
	if (g_game_mode == GameMode_Construct)
	{
		return;
	}
	
	// now update the view with the must_values from the model
	for (i=0; i< (g_game.board_horizontal * g_game.board_vertical); i++)
	{
		mxmodel_cell cell;
	    int flags;
	
		// get the data
		mxmodel_get_cell_info(g_model, i, &cell);
		flags = ((cell.flags & MXMODEL_CELL_READONLY) == MXMODEL_CELL_READONLY) ? MXVIEW_CELL_VALUE_READONLY : MXVIEW_CELL_VALUE_NORMAL;
		
		// copy must to value in model
		if (flags != MXVIEW_CELL_VALUE_READONLY)
		{
			mxmodel_set_cell_value(g_model, i, cell.must_value);
		}
		
		// update view of the (board) cell
		mxview_cell_set_value(i, cell.must_value, flags);
	}
	mxview_update();
}

// Verify if game is correct so far, print a message on status of game
void do_verify_game(void)
{
	int i;
	int count_ok = 0;
	int count_wrong = 0;

	//LOGPRINTF("entry");
	
	// when in construction mode, do nothing
	if (g_game_mode == GameMode_Construct)
	{
		return;
	}
	
	// now update the view with the must_values from the model
	for (i=0; i< (g_game.board_horizontal * g_game.board_vertical); i++)
	{
		mxmodel_cell cell;
	    int flags;
	
		// get the data
		mxmodel_get_cell_info(g_model, i, &cell);
		flags = ((cell.flags & MXMODEL_CELL_READONLY) == MXMODEL_CELL_READONLY) ? 1 : 0;

		//LOGPRINTF("%d: %d %d=%d", i, flags, cell.value, cell.must_value);
		if (flags == 0) // only non-read only cells
		{
			if ((cell.value != -1) && (cell.must_value != -1))
			{
				// both user value and must-value are set
				if (cell.value == cell.must_value)
				{
					count_ok++;
				}
				else
				{
					//LOGPRINTF("WRONG: %d: val:%d must:%d", i, cell.value, cell.must_value);
					count_wrong++;
					// mark the cell
		            mxview_cell_set_value(i, cell.value, MXVIEW_CELL_VALUE_WRONG);
				}
			}
		}
	}

	// any errors detected?
	if (count_wrong == 0)
	{
		gchar       *msg  = NULL;
		int remaining = mxmodel_get_empty_cells(g_model);
		
		// Still some cells not filled?
		if (remaining == 0)
		{
			// all good, no field to fill
			// allow internationalisation
			msg = g_strdup_printf( _("GOOD you SOLVED the puzzle"));
			mxview_status_msg(msg);
			g_free(msg);
		}
		else
		{
			// allow internationalisation
			msg = g_strdup_printf( _("GOOD everything is OK so far (%d remaining)"), remaining);
			mxview_status_msg(msg);
			g_free(msg);
		}
	}
	else
	{
		gchar       *msg  = NULL;
		
		// allow internationalisation
		msg = g_strdup_printf( _("Please check the mistakes!"));
		mxview_status_msg(msg);
		g_free(msg);
	}
}


//==============================================================================
// Local Function Implementation
//==============================================================================

static void display_game(void)
{
	int i, row, col;
	
	// display the model
	i=0;
	for (row = 0; row < g_game.board_vertical; ++row)
	{
		for (col = 0; col < g_game.board_horizontal; ++col) 
		{
			mxmodel_cell cell;
			int flags;
		
			// get the data
			mxmodel_get_cell_info(g_model, i, &cell);
			if (g_game_mode == GameMode_Play)
			{
				// game mode make only non-read-only cells writable
				flags = ((cell.flags & MXMODEL_CELL_READONLY) == MXMODEL_CELL_READONLY) ? MXVIEW_CELL_VALUE_READONLY : MXVIEW_CELL_VALUE_NORMAL;
			}
			else // construct mode!
			{
				// construct mode make all modifyable cells writable
				flags = ((cell.flags & MXMODEL_CELL_MODIFYABLE) == MXMODEL_CELL_MODIFYABLE) ? MXVIEW_CELL_VALUE_NORMAL : MXVIEW_CELL_VALUE_READONLY;
			}
			
			// update view of the (board) cell
			mxview_cell_set_value(i, cell.value, flags);
			i++;
		}
	}
	g_playing = 1; // we are playing
	mxview_update(); // force update of screen
}

static void create_game_data()
{
	mxmodel_game     game;

	// Get info of current model to create view
	mxmodel_get_info(g_model, &game);
	// the board-layout
	g_game.type = mxview_keypad;
	// game dimensions
	g_game.board_horizontal  = game.horizontal;
	g_game.board_vertical    = game.vertical;
	g_game.pp_ornaments      = mxmodel_get_ornaments(g_model);
	if (mxmodel_is_sudoku(g_model))
	{
		// the number of symbols (equal to horizontal/vertical cells)
		g_game.keypad_nr_symbols = game.horizontal;
	}
	else
	{
		// the number of symbols (equal to available digits)
		g_game.keypad_nr_symbols = 9;
	}
}	


