/*
 * xepdmgr_einkfb.c
 *
 * Access to the einkfb funtionality.
 *
 * History:
 *      21/11/2008 DRF Initial version using code of Irex's displayMgr.
 *      26/11/2008 DRF Add the partial updates.
 *      08/12/2008 DRF Honour the Typing and fast update in partial updates.
 *      11/12/2008 DRF Fixed a bug in the open of fb0.
 *
 * Authors:
 *      DRf Dario Rodriguez dario@softhome.net
 *
 * Acknowledgements:
 *      There is code here from irex's displayMgr, (c) irex technologies
 *
 * This program is licensed under the terms of the GNU GPL v3
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include "xepdmgr_einkfb.h"

/* ioctl numbers */
#define FBIO_UPLOAD_WAVEFORM 0x40007606
#define FBIO_ERASE_WHITE     0x40087602
#define FBIO_DISPLAY         0x40087601
#define FBIO_DISPLAYPARTIAL  0xfff

/* ioctl structures parameters */
#define WAVEFORM_4BPP_IMAGE       1
#define WAVEFORM_DRAW             4
#define WAVEFORM_FAST_BLACK_WHITE 6
#define WAVEFORM_TYPING           9

/* Misc. constants */
#define SCREEN_HEIGHT 1024
#define SCREEN_WIDTH  768

/* ioctl structures */
struct display_update_info {
    int waveform;
    int sequence;
};

struct coordinates {
        unsigned short x1;
        unsigned short y1;
        unsigned short x2;
        unsigned short y2;
};

struct partial_display_update_info {
    struct coordinates coord;
    int waveform;
    int sequence;
};

/* eink data type */
typedef struct TypeEink {
        int fd;
} _sEink;

/* exported functions */
sEink *
EinkInit(void)
{
        _sEink *Eink;
        if((Eink=malloc(sizeof(_sEink)))==NULL)
                return(NULL);
        memset(Eink,0,sizeof(_sEink));
        if((Eink->fd=open("/dev/fb0",O_RDWR))==-1) {
                free(Eink),Eink=NULL;
                return(NULL);
        }
        return((sEink *)Eink);
}

void
EinkFini(sEink *EinkParam)
{
        _sEink *Eink=(_sEink *)EinkParam;
        if(Eink==NULL)
                return;
        close(Eink->fd),Eink->fd=-1;
        return;
}

int
EinkWaveformUpload(sEink *EinkParam, char *Waveform)
{
        /* Waveform is 128 kbytes */
        _sEink *Eink=(_sEink *)EinkParam;
        int error;
        if(Eink==NULL || Waveform==NULL)
                return(-1);
        if((error=ioctl(Eink->fd,FBIO_UPLOAD_WAVEFORM,Waveform))!=0)
                return(error);
        return(0);
}

/* The following functions return usecs left to complete request, -1 on error */
int
EinkBlank(sEink *EinkParam)
{
        _sEink *Eink=(_sEink *)EinkParam;
        struct display_update_info Update;
        int error;
        Update.waveform = WAVEFORM_4BPP_IMAGE;
        Update.sequence = 0;
        if((error=ioctl(Eink->fd,FBIO_ERASE_WHITE,&Update))!=0)
                return(error);
        return(100000);
}

int
EinkUpdateFull(sEink *EinkParam, int UseTyping, int Fast)
{
        _sEink *Eink=(_sEink *)EinkParam;
        struct display_update_info Update;
        int error;
        Update.waveform = ((Fast!=0)?WAVEFORM_FAST_BLACK_WHITE:
                               (UseTyping!=0)?WAVEFORM_TYPING:
                               WAVEFORM_4BPP_IMAGE);
        Update.sequence = 0;
        if((error=ioctl(Eink->fd,FBIO_DISPLAY,&Update))!=0)
                return(error);
        return(1000000);
}

int
EinkUpdatePartial(sEink *EinkParam, int x1, int y1, int x2, int y2, int UseTyping, int Fast)
{
        _sEink *Eink=(_sEink *)EinkParam;
        struct partial_display_update_info PartialUpdate;
        int error;
        PartialUpdate.sequence = 0;
        PartialUpdate.waveform = ((Fast!=0)?WAVEFORM_FAST_BLACK_WHITE:
                               (UseTyping!=0)?WAVEFORM_TYPING:
                               WAVEFORM_4BPP_IMAGE);
        /* Make the rectangle coods multiple of four */
        x1&=~3;
        y1&=~3;
        x2+=3,x2&=~3;
        y2+=3,y2&=~3;
        /* Don't redraw the 4 pixels adjacent to the border */
        PartialUpdate.coord.x1=(x1<4)?4:x1;
        PartialUpdate.coord.y1=(y1<4)?4:y1;
        PartialUpdate.coord.x2=(x2>SCREEN_HEIGHT-4)?SCREEN_HEIGHT-4:x2;
        PartialUpdate.coord.y2=(y2>SCREEN_WIDTH-4)?SCREEN_WIDTH-4:y2;
        if((error=ioctl(Eink->fd,FBIO_DISPLAYPARTIAL,&PartialUpdate))!=0)
                return(error);
        return(1000000);
}

