#!/usr/bin/env lua
--
-- H.P. Elbers hansel@hpelbers.org, march 2009
-- This code is hereby placed in the public domain.
--

require("lgob.gtk")

------- HELPERS, Less typing, more fun --------------------------------

-- extend the basic widget class
function gtk.Widget:configure(tbl)
  if tbl then
    for k,v in pairs(tbl) do
      if type(v)=='function' then
        self:connect(k,v)
      elseif type(v)=='table' then
        self:connect(k,v[1],v[2])	-- function with user data
      else
        self:set(k,v)
      end
    end
  end
  return self
end

-- Shorcut for constructors
function gtk.button(tbl)
  return gtk.Button.new():configure(tbl)
end
function gtk.label(tbl)
  return gtk.Label.new():configure(tbl)
end
function gtk.entry(tbl)
  return gtk.Entry.new():configure(tbl)
end

--------- Start of Application ----------------------------------------
local app={}		-- Aplication data

-- protected environment to run expression entered bu user ------------
local user={
  -- functions that work with degrees  
  sin_d  = function(x) return math.sin(math.rad(x))  end,
  asin_d = function(x) return math.deg(math.asin(x)) end,
  cos_d  = function(x) return math.cos(math.rad(x))  end,
  acos_d = function(x) return math.deg(math.acos(x)) end,
  tan_d  = function(x) return math.tan(math.rad(x))  end,
  atan_d = function(x) return math.deg(math.atan(x)) end,

  pi=math.pi,
  e=math.exp(1),

  rad  = math.rad,	-- deg to rad
  deg  = math.deg,	-- rad to deg

  log  = math.log10,
  ln   = math.log,
  exp  = math.exp,
  sqrt = function(x) return x^.5 end
}

-- switch to degrees 
local function setDeg() 
  user.sin  = user.sin_d
  user.asin = user.asin_d
  user.cos  = user.cos_d
  user.acos = user.acos_d
  user.tan  = user.tan_d
  user.atan = user.atan_d
end
-- switch to radians 
local function setRad() 
  user.sin  = math.sin
  user.asin = math.asin
  user.cos  = math.cos
  user.acos = math.acos
  user.tan  = math.tan
  user.atan = math.atan
end

local function checkKey(k) 
  --print('doKey', k)
end
local function doApply(n)
  for i=n,#app do
    local r = app[i]
    local expr = r.expr:get_text()
    local txt, val = 'error', 0/0       -- nan
    func, err = loadstring("return " .. expr)

    if func then        -- expression was valid for Lua

      -- Make the values of previous rows available to this one
      for r=1,i-1 do
        user[app[r].name]=app[r].val
      end
      for r=i,#app do user[app[r].name]=0/0 end -- nan

      -- run the expression enterd by the user in a sandbox
      setfenv(func, user)
      stat, ret = pcall(func)
      if stat then      -- call went ok
        val = ret or 0
        txt = tostring(val)
      else
        txt = 'error'
      end

    else
        txt = 'syntax'
    end

    -- update value in GUI
    r.result:set_text(txt)
    r.val=val
  end   -- of for loop
end

-- find item where cursor is
local function activeEntry()
  local nr, entry= -1, nil
  -- find the item that has the focus
  for i, item in ipairs(app) do
    if item.expr and item.expr:is_focus() then
      entry = item.expr; nr = i
      break
    end
  end
  return entry, nr
end

-- Handle speciall keys
local function doSpecial(k) 
  local entry, nr = activeEntry()
  if entry then
    local pos = entry:get_position()            -- cursor position
    local len = string.len(entry:get_text())

    if k=='UP' then
      if nr>1 then app[nr-1].expr:grab_focus() end
    elseif k=='DOWN' then
      if nr<#app then app[nr+1].expr:grab_focus() end
    elseif k=='LEFT' then
      if pos>0 then
        pos=pos-1 
        entry:set_position(pos)
      end
    elseif k=='RIGHT' then
      if pos<len then
        pos=pos+1 
        entry:set_position(pos)
      end
    elseif k=='CLR' then
      entry:set_text('')
      doApply(nr)
    elseif k=='CLR_ALL' then
      for i=1,#app do
        app[i].expr:set_text('')
      end
      app[1].expr:grab_focus()
      doApply(1)
    elseif k=='ENTER' then
      doApply(nr)
    elseif k=='DEL' or  k=='BS' then
      local sel,b,e=entry:get_selection_bounds()  -- selection?
      if not sel then
        b,e = pos,pos
        if k=='BS' and b>0 then b=b-1
        elseif e<len then e=e+1 end
      end
      entry:delete_text(b,e)
      entry:set_position(b)
    else
      print("UNHANDLED SPECIAL")
    end
  end
end

local function doKey(k) 
  local special = string.gsub(k, '^_', '')
  if not(special==k) then return doSpecial(special) end

  local entry= activeEntry()
  if entry then
    local sel,b,e=entry:get_selection_bounds()  -- selection?
    selText=''
    if (sel) then 
       selText=entry:get_chars(b,e)
       entry:delete_text(b,e)
       entry:set_position(b)
    end
    local pos = entry:get_position()            -- cursor position
    local func = nil
    for _,f in ipairs{'()','sqrt', 'exp','ln','log',
        'cos', 'acos', 'sin', 'asin', 'tan', 'atan',
        'rad', 'deg' } do
      if k==f then
        if f=='()' then f='' end
        func = f .. '(' .. selText .. ')'
        break
      end
    end

    if func then
      pos = entry:insert_text(func, string.len(func), pos)
      if selText == '' then pos=pos-1 end	-- set cursor inside ()
    else
      pos = entry:insert_text(k, string.len(k), pos)
    end

    entry:set_position(pos)
  end
end

local function doApply(n) 
  for i=n,#app do
    local r = app[i]
    local expr = r.expr:get_text()
    local txt, val = 'error', 0/0	-- nan
    func, err = loadstring("return " .. expr)

    if func then	-- expression was valid for Lua

      -- Make the values of previous rows available to this one
      for r=1,i-1 do
        user[app[r].name]=app[r].val
      end
      for r=i,#app do user[app[r].name]=0/0 end	-- nan

      -- run the expression enterd by the user in a sandbox
      setfenv(func, user)
      stat, ret = pcall(func)
      if stat then	-- call went ok
        val = ret or 0
        txt = tostring(val)
      else
        txt = 'error'
      end

    else
        txt = 'syntax'
    end

    -- update value in GUI
    r.result:set_text(txt)
    r.val=val
  end   -- of for loop
end


local function doQuit() 
  gtk.main_quit()
end

local vars	= {'A', 'B', 'C', 'D', 'E', 'F', 'G', 'I'}
local window    = gtk.Window.new()
local vbox      = gtk.VBox.new(false, 5)
--vbox:configure{spacing=50}
local table     = gtk.Table.new(#vars,12,true)
local w_keyboard  = gtk.Table.new(5,8, true)
local w_quit    = gtk.button({label='Quit', clicked=doQuit})
local w_info    = gtk.label({label='this is MultiCalc'})

-- Choice   radians <--> degrees
local hbox = gtk.HBox.new(false, 5)
w_deg = gtk.RadioButton.new()
w_rad = w_deg.new_with_label_from_widget(w_deg, 'rad')
hbox:add(w_deg); hbox:add(w_rad)
w_deg:configure({label='deg', clicked=setDeg})
w_rad:configure({label='rad', clicked=setRad})
setDeg()

window:add(vbox)
vbox:pack_start(w_quit)
vbox:pack_start(hbox)
vbox:pack_start(table)
vbox:pack_start(w_keyboard)
vbox:pack_start(w_info,true)

-- GUI for all Calc rows
for i, v in ipairs(vars) do
  local row = {}; app[i] = row 	-- application data for this row
  row.val = 0
  local label   = gtk.label({label=v..':'})
  row.expr = gtk.entry({
    activate={doApply, i},
    ['key-press-event']={checkKey, i},
    ['focus-out-event']={doApply, i},
  })
  row.name = v
  row.result  = gtk.entry({
    editable=false, 
    ['can-focus']=false,
    text=tostring(row.val)
  })

  local b,w = 0,0
  w=1; table:attach_defaults(label,   b, b+w, i-1,i); b = b+w
  w=6; table:attach_defaults(row.expr, b, b+w, i-1,i); b = b+w
  w=1; table:attach_defaults(gtk.label{label='='},   b, b+w, i-1,i); b = b+w
  w=4; table:attach_defaults(row.result,  b, b+w, i-1,i); b = b+w
end

-- Virtual keyboard
keyboard = {
 {{k='A'},{k='B'},{k='C'},{k='D'},{k='E'},{k='F'},{k='G'},{k='H'}},
 {{k='pi'},{k='log'},{k='exp'},{k='asin'},{k='acos'},{k='atan'},{k='rad'},{k='deg'}},
 {{k='e'},{k='sqrt'},{k='ln'},{k='sin'},{k='cos'},{k='tan'}},
 {{k='('},{k='()'},{k=')'},{k='^'}},
 {{k='7'},{k='8'},{k='9'},{k='*'}},
 {{k='4'},{k='5'},{k='6'},{k='-'}},
 {{k='1'},{k='2'},{k='3'},{k='+'}},
 {{k='0'},{k=' '},{k='.'},{k='/'}},
}

function addKey(key, label, x, y, w, h)
  x = x-1; y=y-1;   		-- Table starts at 0, Lua starts at 1
  w = w or 1; h = h or 1; 	-- default key is 1x1
  local b = gtk.button{
       label=label,
       ['can-focus']=false,
       clicked={doKey, key}
     }
  if label then b:set_label(label) end
  w_keyboard:attach_defaults(b, x, x+w, y, y+h);
  return b
end

for i,row in ipairs(keyboard) do
  for j,key in ipairs(row) do
     addKey(key.k, key.k, j, i)
  end
end

addKey('_CLR',   'CLR',  6,7,1,2)
addKey('_CLR_ALL','CLR\nALL',  7,5,1,2)
addKey('_DEL',   'DEL',  8,3,1,2)
addKey('_BS',    'BS',   5,4,1,2)
addKey('_ENTER', '=',    5,6,1,3)
addKey('_UP',    nil,    7,3,1,2):add(gtk.Arrow.new(gtk.ARROW_UP, gtk.SHADOW_IN))
addKey('_DOWN',  nil,    7,7,1,2):add(gtk.Arrow.new(gtk.ARROW_DOWN, gtk.SHADOW_IN))
addKey('_LEFT',  nil,    6,5,1,2):add(gtk.Arrow.new(gtk.ARROW_LEFT, gtk.SHADOW_IN))
addKey('_RIGHT', nil,    8,5,1,2):add(gtk.Arrow.new(gtk.ARROW_RIGHT, gtk.SHADOW_IN))

app[1].expr:grab_focus()	-- set focus on first row

window:connect("delete-event", gtk.main_quit)
window:show_all()

gtk.main()
