/*
 * File Name: scribbles.c
 */

/*
 * This file is part of mxSudoku.
 *
 * mxSudoku is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * mxSudoku is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 Marcel Hendrickx
 * All rights reserved.
 */

//------------------------------------------------------------------------------
// Include Files
//------------------------------------------------------------------------------

// configuration parameters of the project
#include "config.h"

// The implemented interface
#include "scribbles.h"

// system include files

// local include files
#include "log.h"


//------------------------------------------------------------------------------
// Type Declarations
//------------------------------------------------------------------------------


//------------------------------------------------------------------------------
// Global Constants
//------------------------------------------------------------------------------

// the maximum number of entries in the list
#define SIZE_OF_LIST 100

// margin for no move
#define NO_MOVE 2

// number for slope that is very big

#define BIG_SLOPE 1000.0

//------------------------------------------------------------------------------
// Static Variables
//------------------------------------------------------------------------------

// type to store point information in the list
typedef struct _list_entry
{
	int x;
	int y;
} list_entry, *p_list_entry;

// The list
static list_entry list_data[SIZE_OF_LIST];

// first unused position in list_data
static int list_position = 0;

// The direction of the stroke
static char list_direction[SIZE_OF_LIST];
static char list_direction_condensed[SIZE_OF_LIST];

typedef struct _char_entry {
	char character;
	char *match;
} char_entry, *p_char_entry;

static char_entry number_data[] = {
	 { '1', "2"}
	,{ '2', "9316"}
	,{ '3', "3131"}
	,{ '4', "16"}
	,{ '5', "231"}
	,{ '6', "167"}
	,{ '7', "61"}
	,{ '8', "16761"}  // maybe not the most logical for an 8 ...
	,{ '9', "761"}    // this also does not start were you would normally start
	,{ '9', "43824"}  // alternative 9
	//,{ '0', "6248"}
};

// this list contains the neighbours for every direction
// first is direction, others are neightbours
static char *valid_matches[9] = {
	"142",
	"213",
	"326",
	"471",
	"5",
	"639",
	"748",
	"879",
	"986"
};

//==============================================================================
// Local Function Definitions
//==============================================================================

static void calc_direction();
static void condense();
char match();

//==============================================================================
// Functions Implementation
//==============================================================================

// reset the internal list of positions and start at first position
void scribble_start(int posx, int posy)
{
    LOGPRINTF("entry");

	// reset position
	list_position = 0;
	scribble_add(posx, posy);
}

// add a position to the list
void scribble_add(int posx, int posy)
{
    LOGPRINTF("entry");

	// check if room in list
	if (list_position < SIZE_OF_LIST)
	{
		list_data[list_position].x = posx;
		list_data[list_position].y = posy;
		list_position++;
	}
}

// convert list to number
int scribble_get_number()
{
	char number;
	
    LOGPRINTF("entry");

	// first convert the segments to a string of directions
	calc_direction();
	
	// condense the stream of directions
	condense();
	
	// match the directions with the numbers
	number = match();
	LOGPRINTF("number:%d", (number-'0'));
	
	return number;
}

//==============================================================================
// Local Function Implementation
//==============================================================================

static void calc_direction()
{
	int i;
	char dir;
	int deltax, deltay; // delta in x and y direction of segment
	int length2;        // square of length
	float slope;        // slope of the segment
	
    LOGPRINTF("entry");

	list_direction[0] = '\0';
	
	for (i=1; i<list_position; i++)
	{
		// calculate the delta
		deltax = list_data[i].x - list_data[i-1].x; // right is positive
		deltay = list_data[i-1].y - list_data[i].y; // up is positive
		
		LOGPRINTF("pos%d: %d %d - %d %d = %d %d", i, list_data[i-1].x, list_data[i-1].y, list_data[i].x, list_data[i].y, deltax, deltay);
		// length (we only need measure for length, square of length is easy to calc)
		length2 = deltax*deltax + deltay*deltay;
		
		// slope (x/y)
		if (deltay != 0)
		{
			slope = (float)deltax/(float)deltay;
		}
		else
		{
			slope = BIG_SLOPE;
		}
		LOGPRINTF("len%d: %d %f", i, length2, slope);
		
		// 5: no move
		if (length2 < NO_MOVE) { dir = '5'; }
		// 1: down left
		else if ((deltay < 0) && (slope< 2.0 && slope> 0.5)) { dir = '1'; }
		// 2: down
		else if ((deltay < 0) && (slope< 0.5 && slope>-0.5)) { dir = '2'; }
		// 3: down right
		else if ((deltay < 0) && (slope<-0.5 && slope>-2.0)) { dir = '3'; }
		// 4: left
		else if ((deltax < 0) && (slope> 2.0 || slope<-2.0)) { dir = '4'; }
		// 6: right
		else if ((deltax > 0) && (slope> 2.0 || slope<-2.0)) { dir = '6'; }
		// 7: up left
		else if ((deltay > 0) && (slope<-0.5 && slope>-2.0)) { dir = '7'; }
		// 8: up
		else if ((deltay > 0) && (slope< 0.5 && slope>-0.5)) { dir = '8'; }
		// 9: up right
		else if ((deltay > 0) && (slope> 0.5 && slope< 2.0)) { dir = '9'; }
		// else?
		else { dir = '5'; }

		LOGPRINTF("dir:%c", dir);
		// store
		list_direction[i-1] = dir;
		// end string
		list_direction[i] = '\0';
	}
}

static void condense()
{
	int i, j;
	
	// Use a simple condense function only remove duplicates
	
	LOGPRINTF("entry:%s", list_direction);

	// phase 0: remove begin and end entries to make algo more stable
	// put a '5' in the entries, so they will be removed
	list_direction[0] = '5';
	list_direction[strlen(list_direction)-1] = '5';
	
	// phase I: remove 'no movement' --------------------------------------
	j = 0;
	for (i=0; i<strlen(list_direction); i++)
	{
		// skip no movement
		if (list_direction[i] != '5')
		{
			list_direction_condensed[j] = list_direction[i];
			j++;
			// end string
			list_direction_condensed[j] = '\0';
		}
	}
	LOGPRINTF("condensedI:%s",list_direction_condensed);
	
#if 0
	// phase II: remove single entries
	//           not sure if this is the correct step, I wanted to remove some false
	//           segments at start and end (but came up with this ... :-)
	//           Need to test of removing first and last value gives better results
	strcpy(list_direction, list_direction_condensed);
	j = 0;
	for (i=0; i<strlen(list_direction); i++)
	{
		// only add if same as next
		if (list_direction[i] == list_direction[i+1])
		{
			list_direction_condensed[j] = list_direction[i];
			j++;
			// end string
			list_direction_condensed[j] = '\0';
		}
	}
	LOGPRINTF("condensedII:%s",list_direction_condensed);
#endif
	
	// phase III: remove doubles
	strcpy(list_direction, list_direction_condensed);
	// copy first value
	list_direction_condensed[0] = list_direction[0];
	list_direction_condensed[1] = '\0';
	j = 1;
	for (i=1; i<strlen(list_direction); i++)
	{
		// only add if different from previous
		if (list_direction_condensed[j-1] != list_direction[i])
		{
			list_direction_condensed[j] = list_direction[i];
			j++;
			// end string
			list_direction_condensed[j] = '\0';
		}
	}
	LOGPRINTF("condensed:%s",list_direction_condensed);
}

char match()
{
	int entries;
	int i;
	int best = -1;       // index to best scoring entry
	int best_score = -100;// best score so far
	int score;           // current score
	int pos_list=0;      // position in the direction-list
	char *s;             // current string to match for
	int pos_match=0;     // position in the match string
	int number = -1;     // -1 no valid value found
	
	LOGPRINTF("matching:%s", list_direction_condensed);

	// calc change of match for every number
	entries = sizeof(number_data)/sizeof(*number_data);
	
	for (i=0; i<entries; i++)
	{
		// character to match for
		pos_match = 0;
		s = number_data[i].match;
		score = 0;
		LOGPRINTF("match %s against:%s (%c)", list_direction_condensed, s, number_data[i].character);
		for (pos_list = 0; pos_list < strlen(list_direction_condensed); pos_list++)
		{
		    // check if equal to current
			if (s[pos_match] == list_direction_condensed[pos_list] )
			{
				score += 10;
				LOGPRINTF("match: %c %c current score:%d", s[pos_match], list_direction_condensed[pos_list], score);
				continue; // next position
			}

		    // check if equal to next (only look ahead if we had a match already)
			if ((score > 0) && (s[pos_match+1] != '\0')) // there is a next
			{
				if (s[pos_match+1] == list_direction_condensed[pos_list] )
				{
					score += 10;
					// increase index in s
					pos_match++;
					LOGPRINTF("match: %c %c next score:%d", s[pos_match], list_direction_condensed[pos_list], score);
 	    			continue; // next position
				}
			}
			
			// check if near next (first check next, to give preference to 'going forward'
			if ((score > 0) && (s[pos_match+1] != '\0')) // there is a next
			{
				if (strchr(valid_matches[s[pos_match+1]-'1'], list_direction_condensed[pos_list] ) != NULL)
				{
					score += 10;
					// increase index in s
					pos_match++;
					LOGPRINTF("match: %c %c near next score:%d", s[pos_match], list_direction_condensed[pos_list], score);
					continue; // next position
				}
			}
			
			// check if near current
			if (strchr(valid_matches[s[pos_match]-'1'], list_direction_condensed[pos_list] ) != NULL)
			{
				score += 10;
				LOGPRINTF("match: %c %c near current score:%d", s[pos_match], list_direction_condensed[pos_list], score);
				continue; // next position
			}
			
			// no rule to match against
			{
				score -= 20;
				LOGPRINTF("match: %c %c mismatch score:%d", s[pos_match], list_direction_condensed[pos_list], score);
			}
		} // for (pos_list...
		
		// now compensate score for directions that have not yet been matched
		if (s[pos_match] != '\0')
		{
			pos_match++;
			while (s[pos_match] != '\0')
			{
				score -= 10;
				LOGPRINTF("match: %c missing score:%d", s[pos_match], score);
				pos_match++;
			}
		}
		
		// adjust score for length
		score = score - ((int)strlen(s) * 5);
		
		// check how it scored
		if (score > best_score)
		{
			LOGPRINTF("better %d", score);
			best_score = score;
			best = i;
		}
	}
	
	if (best >= 0)
	{
		number = number_data[best].character;
		LOGPRINTF("best:%d=%c", best, number);
	}
	
	return number;
}
