/*
 * File Name: mxGrid.c
 */

/*
 * This file is part of mxSudoku.
 *
 * mxSudoku is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * mxSudoku is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 Marcel Hendrickx
 * All rights reserved.
 */

//------------------------------------------------------------------------------
// Include Files
//------------------------------------------------------------------------------

// configuration parameters of the project
#include "config.h"

// The implemented interface
#include "mxGrid.h"

// system include files
#include <glib.h>
#include <gtk/gtk.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>

// local include files
#include "log.h"
#include "scribbles.h"


//------------------------------------------------------------------------------
// Type Declarations
//------------------------------------------------------------------------------


// This structure defined the active areas in the widget
// Areas can be nested, which allows sub areas in areas

typedef struct _mxgrid_active_area_t
{
	int x; // the relative position of the area in widget coordinates
	int y;
	int w;
	int h;
	
	int cell_nr; // the cell number to which the area is related
	
	// if the area has any sub areas they are here, coordinates of sub areas
	// are relative to their parent.
	struct _mxgrid_active_area_t *children;
} mxgrid_active_area_t, *p_mxgrid_active_area_t;

#define MX_CELL_TEXT_SIZE 10
#define MX_CELL_ORNAMENT_SIZE 10
typedef struct _mxgrid_cell_data_t
{
	int x; // the relative position of the area in widget coordinates
	int y;
	int w;
	int h;
	
	int cell_nr; // the cell number to which the area is related
	
	char ornaments[MX_CELL_ORNAMENT_SIZE]; // defines border and other ornaments of the cell
	char text[MX_CELL_TEXT_SIZE];      // the text to display in the cell
	GdkColor t_color;  // the text color of the cell
	GdkColor o_color;  // the ornaments color of the cell
	GdkColor b_color;  // the background color of the cell
} mxgrid_cell_data_t, *p_mxgrid_cell_data_t;

// mxGrid Status
// CREATED: memory is reserved, but sizes not yet known, no drawing is possible
// INITIALISED: sizes known, drawing is possible
#define MXGRID_STATUS_CREATED		0
#define MXGRID_STATUS_INITIALISED	1

struct _GtkGrid
{
	// the component used to construct the grid
	GtkDrawingArea       area;
	GdkPixmap            *pixmap;   // The backend pixmap always has the same size as the widget
	
	GdkGC                *own_gc;   // Own gc used for drawing
	
	unsigned long        status;    // contains the current state of the widget
	unsigned long        flags;     // the flags configuring the widget
	
	// The structure of the grid
	int cell_width;		     // 'netto' with of the cells in the grid
	int cell_height;         // 'netto' height of the cells in the grid
	int nr_cells_horizontal; // number of cells on a horizontal line
	int nr_cells_vertical;   // number of cells on a vertical line
	int top_margin;          // number of pixels above grid
	int left_margin;         // number of pixels left from grid

	// content of the cells
	p_mxgrid_cell_data_t data;
	
	// The active areas in the widget
	p_mxgrid_active_area_t active_areas;
};

struct _GtkGridClass
{
	GtkDrawingAreaClass parent_class;
	
	// Signal handlers
	void (*grid_cell_selected) (GtkGrid * grid, int cell);
};

typedef void (*mxGridCallBack)(GtkGrid *grid, p_mxGridEvent_t event, void *data);

enum
{
	// signals emitted when the user selects a cell
	CELL_SELECTED,
	NUMBER_ENTERED,
	LAST_SIGNAL
};

static guint grid_signals[LAST_SIGNAL] = { 0 };

//------------------------------------------------------------------------------
// Global Constants
//------------------------------------------------------------------------------

#define MX_CALLBACK_CELL_SELECT 0

//------------------------------------------------------------------------------
// Static Variables
//------------------------------------------------------------------------------

//==============================================================================
// Local Function Definitions
//==============================================================================

// event handlers
static gboolean on_expose_event (GtkWidget *widget, GdkEventExpose *event, gpointer data);
static gboolean on_configure_event (GtkWidget *widget, GdkEventConfigure *event, gpointer data);
static gboolean on_button_press_event (GtkWidget *widget, GdkEventButton *event, gpointer data);
static gboolean on_button_release_event (GtkWidget *widget, GdkEventButton *event, gpointer data);
static gboolean on_motion_notify_event (GtkWidget *widget, GdkEventMotion *event, gpointer data);
static gboolean on_key_press_event (GtkWidget *widget, GdkEventKey *event, gpointer data);
static gboolean on_key_release_event (GtkWidget *widget, GdkEventKey *event, gpointer data);
static gboolean on_delete_event (GtkWidget *widget, GdkEventAny *event, gpointer data);

// widget local functions
static void gtk_grid_class_init( GtkGridClass *klass);
static void gtk_grid_init(GtkGrid *grid);
static void gtk_grid_reset(GtkGrid *grid);

// cell local functions
static int cell_get_border_pixels_top(char *content);
static int cell_get_border_pixels_right(char *content);
static int cell_get_border_pixels_bottom(char *content);
static int cell_get_border_pixels_left(char *content);
static void cell_calc_size(GtkGrid *grid, char *content, int *x, int *y);
static void cell_draw(GtkGrid *grid, p_mxgrid_cell_data_t data);

// grid local functions
static void grid_get_border_pixels(GtkGrid *grid, int *w, int *h);
static void grid_calc_max_cell_size(GtkGrid *grid, int *w, int *h);
static void grid_calc_margins(GtkGrid *grid, int *w, int *h);
static void gtk_grid_draw(GtkGrid *grid);
static void grid_create_active_areas(GtkGrid *grid);
static int grid_find_active_area(GtkGrid *grid, int x, int y, int *cx, int *cy);
static void grid_create_cell_data(GtkGrid *grid);

//==============================================================================
// Functions Implementation
//==============================================================================

GType gtk_grid_get_type(void)
{
	static GType grid_type = 0;
	
	if (!grid_type)
	{
		// first time, create info and register
		static const GTypeInfo grid_info = {
			sizeof(GtkGridClass),
			NULL,        /* base_init */
			NULL,        /* base_finalize */
			(GClassInitFunc) gtk_grid_class_init,
			NULL,        /* class finalize */
			NULL,        /* class data */
			sizeof(GtkGrid),
			0,           /* n preallocs */
			(GInstanceInitFunc) gtk_grid_init,
		};
		
		grid_type = 
			g_type_register_static (GTK_TYPE_DRAWING_AREA,
									"mxGrid",
									&grid_info,
									(GTypeFlags)(0));
	}
	
	return grid_type;
}

static void gtk_grid_class_init( GtkGridClass *klass)
{
	GtkObjectClass *object_class;
	
	object_class = (GtkObjectClass *) klass;
	
	// create signals for the widget
	grid_signals[CELL_SELECTED] = 
		g_signal_new( "grid_cell_selected",
					  G_OBJECT_CLASS_TYPE(object_class),
					  G_SIGNAL_RUN_LAST,
					  G_STRUCT_OFFSET(GtkGridClass, grid_cell_selected),
					  NULL, NULL,
					  gtk_marshal_VOID__INT,
					  G_TYPE_NONE,
					  1,
					  G_TYPE_INT);

	grid_signals[NUMBER_ENTERED] = 
		g_signal_new( "grid_number_entered",
					  G_OBJECT_CLASS_TYPE(object_class),
					  G_SIGNAL_RUN_LAST,
					  G_STRUCT_OFFSET(GtkGridClass, grid_cell_selected),
					  NULL, NULL,
					  gtk_marshal_VOID__INT_INT,
					  G_TYPE_NONE,
					  2,
					  G_TYPE_INT,
					  G_TYPE_INT);
}

void gtk_grid_init(GtkGrid *grid)
{
	// indicate the events we want to receive
	gtk_widget_add_events(GTK_WIDGET(grid),
						  GDK_BUTTON_PRESS_MASK |
						  GDK_BUTTON_RELEASE_MASK |
						  GDK_POINTER_MOTION_MASK |
						  GDK_KEY_PRESS_MASK |
						  GDK_KEY_RELEASE_MASK);
	
	// connect event handlers
	g_signal_connect (G_OBJECT (grid), "expose_event",
					G_CALLBACK (on_expose_event), grid);

	g_signal_connect (G_OBJECT (grid), "configure_event",
					G_CALLBACK (on_configure_event), grid);

	g_signal_connect (G_OBJECT (grid), "button_press_event",
					G_CALLBACK (on_button_press_event), grid);

	g_signal_connect (G_OBJECT (grid), "button_release_event",
					G_CALLBACK (on_button_release_event), grid);

	g_signal_connect (G_OBJECT (grid), "motion_notify_event",
					G_CALLBACK (on_motion_notify_event), grid);

	g_signal_connect (G_OBJECT (grid), "key_press_event",
					G_CALLBACK (on_key_press_event), grid);

	g_signal_connect (G_OBJECT (grid), "key_release_event",
					G_CALLBACK (on_key_release_event), grid);

	g_signal_connect (G_OBJECT (grid), "delete_event",
					G_CALLBACK (on_delete_event), grid);

	// and show
	gtk_widget_show(GTK_WIDGET(grid));
}

//========================== mx_grid_new =======================================
GtkWidget * gtk_grid_new()
{
	GtkGrid *grid = NULL;
	
    //LOGPRINTF("entry");

	grid = (GtkGrid *) g_object_new(GTK_GRID_TYPE, NULL);
	
	// initialise cell data
	gtk_grid_reset(grid);

	grid->pixmap = 0;
	grid->own_gc = 0;
	
	// set focus to enable keyboard events (to get keyboard events)
	// TODO: why did I do this?
	GTK_WIDGET_SET_FLAGS(grid, GTK_CAN_FOCUS);
	
	return GTK_WIDGET(grid);
}

static void gtk_grid_reset(GtkGrid *grid)
{
	g_return_if_fail(IS_GTK_GRID(grid));
	
	grid->status               = MXGRID_STATUS_CREATED;
	grid->flags                = 0;
	
	grid->cell_width           = 0; // auto determine
	grid->cell_height          = 0;
	grid->nr_cells_horizontal  = 9; // default 9x9
	grid->nr_cells_vertical    = 9;

	// Cell data
	grid->data = NULL;
	
	// Handle the active areas
	grid->active_areas = NULL;
	
	// TODO: check if all fields are initialised in this function
}

//==============================================================================
void gtk_grid_set_preferred_size(GtkGrid *grid, int w, int h, int mode)
{
    //LOGPRINTF("entry");

	g_return_if_fail(IS_GTK_GRID(grid));

	// store the values, will only be used after redraw!
	if (grid)
	{
		grid->cell_width           = w;
		grid->cell_height          = h;
		// Only if both are unequal to zero the preferred values will be used
		if ((w != 0) && (h != 0))
		{
			// preferred mode
			grid->flags |= MXGRID_CONFIG_FIT_PREFERRED;
			grid->flags |= MXGRID_CONFIG_FIT_CENTRE;
		}
		else
		{
			// auto
			grid->flags |= MXGRID_CONFIG_FIT_FULL;
			grid->flags |= MXGRID_CONFIG_FIT_CENTRE;
			if (mode == 1) // use square cells
			{
				grid->flags |= MXGRID_CONFIG_FIT_SQUARE;
			}
		}
	}
}

void gtk_grid_set_size(GtkGrid *grid, int horizontal, int vertical)
{
    //LOGPRINTF("entry");

	g_return_if_fail(IS_GTK_GRID(grid));

	// store the values, will only be used after redraw!
	if (grid)
	{
		grid->nr_cells_horizontal = horizontal;
		grid->nr_cells_vertical   = vertical;
	}
	
	// init the data structure
	grid_create_cell_data(grid);
}

void gtk_grid_set_ornaments(GtkGrid *grid, int size, char **pp_ornaments)
{
	int i;
    LOGPRINTF("entry");

	g_return_if_fail(IS_GTK_GRID(grid));

	// limit number to max number of cells
	if (size > (grid->nr_cells_horizontal * grid->nr_cells_vertical))
	{
		size = grid->nr_cells_horizontal * grid->nr_cells_vertical;
	}
	
	if (grid->data != NULL)
	{
		for (i=0; i<size; i++)
		{
			// TODO: now limited to 4 chars will not be enough for future games
			strncpy(grid->data[i].ornaments, pp_ornaments[i], 4); // max 4 chars
			grid->data[i].ornaments[4] = '\0'; // end string
			//LOGPRINTF("ornament: %d = %s", i, grid->data[i].ornaments);
		}
	}
}

//==============================================================================
// Local Function Implementation
//==============================================================================

//==============================================================================
// called when the widget is shown for the first time or when it was previously 
// covered, but now visible again
static gboolean
on_expose_event (GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
	GtkGrid *grid = GTK_GRID(data);
	
    //LOGPRINTF("entry");
	
	g_return_val_if_fail(IS_GTK_GRID(grid), FALSE);
	
	// Copy damaged region of image from pixmap to output
	gdk_draw_drawable(widget->window,
					  widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
					  grid->pixmap,
					  event->area.x, event->area.y,
					  event->area.x, event->area.y,
					  event->area.width, event->area.height);
	
    return FALSE;
}

//==============================================================================
// need to get the new DC from the widget and store it.
static gboolean
on_configure_event (GtkWidget *widget, GdkEventConfigure *event, gpointer data)
{
	GtkGrid *grid = GTK_GRID(data);
	
    LOGPRINTF("entry");

	if (grid->pixmap)
	{
		g_object_unref(grid->pixmap);
	}
	if (grid->own_gc)
	{
		g_object_unref(grid->own_gc);
	}
	
	// re-create pixmap and gc
	grid->pixmap = gdk_pixmap_new(widget->window,
								  widget->allocation.width,
								  widget->allocation.height,
								  -1);
	grid->own_gc = gdk_gc_new(grid->pixmap);
			
	// TODO: I do not like this if, can I get ride of it?
	if (grid->status == MXGRID_STATUS_CREATED)
	{
		// create the cell data structures if not yet created
		if (grid->data == NULL)
		{
			grid_create_cell_data(grid);
		}
		
		// craete the active areas
		grid_create_active_areas(grid);
		
		// indicate sizes are known
		grid->status = MXGRID_STATUS_INITIALISED;
	}
	
	// first calculate the parameters to use
	// calc size of the cells in the grid if not given
	// TODO: also calc when current settings do not fit!
	if ((grid->flags & MXGRID_CONFIG_FIT_PREFERRED) == 0) // when not preferred, then auto
	{
		grid_calc_max_cell_size(grid, 
								&(grid->cell_width), 
								&(grid->cell_height));
		if ((grid->flags & MXGRID_CONFIG_FIT_SQUARE) == MXGRID_CONFIG_FIT_SQUARE)
		{
			// set to smalles of both
			if (grid->cell_width > grid->cell_height)
			{
				grid->cell_width = grid->cell_height;
			}
			else
			{
				grid->cell_height = grid->cell_width;
			}
		}

	}
	// calc the remaining margins
	{
		int bw, bh;
		
		grid_calc_margins(grid, &bw, &bh);
		grid->left_margin = bw/2;
		grid->top_margin = bh/2;
	}
		
    // clear background
	{
		GdkColor white = {0, 0xffff, 0xffff, 0xffff};
		gdk_gc_set_rgb_fg_color(grid->own_gc, 
								&white);
		gdk_draw_rectangle(grid->pixmap,
						   widget->style->white_gc,
						   TRUE,
						   0, 0, 
						   widget->allocation.width,
						   widget->allocation.height);
	}
	// draw the grid all data is now available
	// this will also update the active areas
	gtk_grid_draw(grid);
	
    return FALSE;
}

//==============================================================================
// TODO: not really used...
static gboolean
on_button_press_event (GtkWidget *widget, GdkEventButton *button_event, gpointer data)
{
	GtkGrid *grid = (GtkGrid *)data;

    // Usually, on the device, only the left button.
    switch (button_event->button)
    {
    case 1: // left_button
        {
            int x, y, cx, cy, nr;
			
            x = (int)button_event->x;
            y = (int)button_event->y;
			nr = grid_find_active_area(grid, x, y, &cx, &cy);
			//LOGPRINTF("Position:%d [%d, %d]", nr, x, y);
			scribble_start(x, y);
		}
        break;
    default:
        LOGPRINTF("unknown button");
        break;
    }

    return TRUE;
}

//==============================================================================
static gboolean
on_button_release_event (GtkWidget *widget, GdkEventButton *button_event, gpointer data)
{
	GtkGrid *grid = (GtkGrid *)data;

    // Usually, on the device, only the left button.
    switch (button_event->button)
    {
    case 1: // left_button
        {
            int x, y;
			mxGridEvent_t  event;
			int number;

			// get the position of the event and find the cell-info
            x = (int)button_event->x;
            y = (int)button_event->y;
			event.cell_nr = grid_find_active_area(grid, x, y, &event.x, &event.y);
			//LOGPRINTF("Position:%d [%d, %d]", event.cell_nr, x, y);

			// Check if any callback installed and call it
			g_signal_emit(grid,
						  grid_signals[CELL_SELECTED],
						  0,
						  event.cell_nr);
			
			// add the last scribble position and get the entered number
			scribble_add(x, y);
			number = scribble_get_number();

			// Check if any callback installed and call it
			g_signal_emit(grid,
						  grid_signals[NUMBER_ENTERED],
						  0,
						  event.cell_nr,
						  (number-'0'));
			
		}
        break;
    default:
        LOGPRINTF("unknown button");
        break;
    }

    return TRUE;
}

//==============================================================================
// TODO: not really used...
static gboolean
on_motion_notify_event (GtkWidget *widget, GdkEventMotion *event, gpointer data)
{
	GtkGrid *grid = (GtkGrid *)data;
	int x, y;
	
	(void)grid;
	
    //LOGPRINTF("entry");

	// get the position of the event and find the cell-info
	x = (int)event->x;
	y = (int)event->y;

	scribble_add(x, y);
	
    return TRUE;
}

//==============================================================================
// TODO: not really used...
static gboolean
on_key_press_event (GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	GtkGrid *grid = (GtkGrid *)data;
	
	(void)grid;
	
    //LOGPRINTF("entry");

    return FALSE;
}

//==============================================================================
// TODO: not really used...
static gboolean
on_key_release_event (GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	GtkGrid *grid = (GtkGrid *)data;
	
	(void)grid;
	
    //LOGPRINTF("entry");

    return FALSE;
}

//==============================================================================
static gboolean
on_delete_event (GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	GtkGrid *grid = (GtkGrid *)data;
	
	(void)grid;
	
	// !!!!!!!!!!TODO!!!!!!!!!!!! ?
	
    LOGPRINTF("entry");

    return FALSE;
}

//TODO: as ornaments only borders are supported at this moment, so code below
//      is very limited and will certainly change when more games are supported

//==============================================================================
// use the content structure to determine the number of pixels on the top 
// of the cell
static int cell_get_border_pixels_top(char *content)
{
	int size = 2; // default 2 pixels
	
    //LOGPRINTF("entry");

	// parse content string
	// current only support digits e.g. "3223" for top, right, bottom, left borders
	if (content != NULL)
	{
		if (strlen(content) > 0)
		{
			// no error checking!
			size = content[0] - '0';
		}
	}
	
	return size;
}

//==============================================================================
// use the content structure to determine the number of pixels on the right
// of the cell
static int cell_get_border_pixels_right(char *content)
{
	int size = 2; // default 2 pixels
	
    //LOGPRINTF("entry");

	// parse content string
	// current only support digits e.g. "3223" for top, right, bottom, left borders
	if (content != NULL)
	{
		if (strlen(content) > 1)
		{
			// no error checking!
			size = content[1] - '0';
		}
	}
	
	return size;
}

//==============================================================================
// use the content structure to determine the number of pixels on the bottom
// of the cell
static int cell_get_border_pixels_bottom(char *content)
{
	int size = 2; // default 2 pixels
	
    //LOGPRINTF("entry");

	// parse content string
	// current only support digits e.g. "3223" for top, right, bottom, left borders
	if (content != NULL)
	{
		if (strlen(content) > 2)
		{
			// no error checking!
			size = content[2] - '0';
		}
	}
	
	return size;
}

//==============================================================================
// use the content structure to determine the number of pixels on the left
// of the cell
static int cell_get_border_pixels_left(char *content)
{
	int size = 2; // default 2 pixels
	
    //LOGPRINTF("entry");

	// parse content string
	// current only support digits e.g. "3223" for top, right, bottom, left borders
	if (content != NULL)
	{
		if (strlen(content) > 3)
		{
			// no error checking!
			size = content[3] - '0';
		}
	}
	
	return size;
}

//==============================================================================
// use the data available in the grid-structure to determine the number of pixels 
// used in the horizontal and vertical borders.
static void grid_get_border_pixels(GtkGrid *grid, int *w, int *h)
{
	int i=0;
	
    //LOGPRINTF("entry");

	// initialise
	*w = 0; *h = 0;
	
	// for horizontal only use first line
	for(i=0; i<grid->nr_cells_horizontal;i++)
	{
		char *content;
		
		// is cell data installed
		if (grid->data != NULL)
		{
			// TODO: check if index exsists
			content = grid->data[i].ornaments;
		}
		else
		{
			content = NULL;
		}
		*w +=  (cell_get_border_pixels_right(content) +
				cell_get_border_pixels_left(content) );
	}

	// for vertical only use first row
	for(i=0; i<grid->nr_cells_vertical;i++)
	{
		char *content;
		
		// is cell data installed
		if (grid->data != NULL)
		{
			int index = i*grid->nr_cells_horizontal;
			// TODO: check if index exsists
			content = grid->data[index].ornaments;
		}
		else
		{
			content = NULL;
		}
		*h +=  (cell_get_border_pixels_top(content) +
				cell_get_border_pixels_bottom(content) );
	}
}

//==============================================================================
// use the data available in the grid-structure to determine the max size of the 
// cells.
static void grid_calc_max_cell_size(GtkGrid *grid, int *w, int *h)
{
	int max_width, max_height;
	int bw, bh;
	
    //LOGPRINTF("entry");

	// determine size of the used pixmap
	gdk_drawable_get_size(grid->pixmap, &max_width, &max_height);
	//LOGPRINTF("max: %d %d", max_width, max_height);
	
	// get size of the defined borders
	grid_get_border_pixels(grid, &bw, &bh);
	//LOGPRINTF("border: %d %d", bw, bh);
	
	// correct max-sizes
	max_width -= bw;
	max_height -= bh;
	// TODO: check for positive values
	
	// TODO: check that nr_cells are filled and > 0
	*w = max_width/grid->nr_cells_horizontal;
	*h = max_height/grid->nr_cells_vertical;
	
	//LOGPRINTF("size: %d %d", *w, *h);

}

//==============================================================================
// use the data available in the grid-structure to determine the size remaining 
// for the margins.
static void grid_calc_margins(GtkGrid *grid, int *w, int *h)
{
	int max_width, max_height;
	int bw, bh;
	
    //LOGPRINTF("entry");

	// determine size of the used pixmap
	gdk_drawable_get_size(grid->pixmap, &max_width, &max_height);
	//LOGPRINTF("max: %d %d", max_width, max_height);
	
	// get size of the defined borders
	grid_get_border_pixels(grid, &bw, &bh);
	//LOGPRINTF("border: %d %d", bw, bh);
	
	// correct max-sizes
	max_width -= bw;
	max_height -= bh;
	// TODO: check for positive values

	// correct for cell-size * number of cells
	max_width -= (grid->cell_width * grid->nr_cells_horizontal);
	max_height -= (grid->cell_height * grid->nr_cells_vertical);
	// TODO: check for positive values
	
	*w = max_width;
	*h = max_height;
	
	//LOGPRINTF("margins: %d %d", *w, *h);
}

static void cell_calc_size(GtkGrid *grid, char *content, int *x, int *y)
{
	int t, b, l, r; // size of borders
	int w, h;       // width/height of cell including borders

    //LOGPRINTF("entry");

	// ============== Borders =====================
	// determine border sizes
	t = cell_get_border_pixels_top(content);
	b = cell_get_border_pixels_bottom(content);
	l = cell_get_border_pixels_left(content);
	r = cell_get_border_pixels_right(content);
	//LOGPRINTF("tblr: %d %d %d %d", t, b, l, r);
	
	// calc cell size
	w = l + grid->cell_width + r;
	h = t + grid->cell_height + b;
	//LOGPRINTF("wh: %d %d", w, h);

	// Now adjust x and y to lower corner
	*x += w;
	*y += h;
}

static void cell_draw(GtkGrid *grid, p_mxgrid_cell_data_t data)
{
	int t, b, l, r; // size of borders
	int x, y, w, h; // size of cell including borders

    //LOGPRINTF("entry");

	g_return_if_fail((IS_GTK_GRID(grid)));
	
	if (!grid->pixmap)
	{
		return;
	}

	if (grid->status != MXGRID_STATUS_INITIALISED)
	{
		// no sizes known, quit drawing
        LOGPRINTF("No draw, wrong status");
		return;
	}
	
	// ============== Borders =====================
	// determine border sizes
	t = cell_get_border_pixels_top(data->ornaments);
	b = cell_get_border_pixels_bottom(data->ornaments);
	l = cell_get_border_pixels_left(data->ornaments);
	r = cell_get_border_pixels_right(data->ornaments);
	//LOGPRINTF("tblr: %d %d %d %d", t, b, l, r);
	
	// calc cell size
	x = data->x;
	y = data->y;
	w = data->w;
	h = data->h;
	//LOGPRINTF("xywh: %d %d %d %d", x, y, w, h);

	// set the draw color, use o_color for borders!
	gdk_gc_set_rgb_fg_color(grid->own_gc, 
							&data->o_color);
	// top border
	if (t > 0)
	{
		gdk_draw_rectangle( grid->pixmap,
							grid->own_gc,
							TRUE, 
							x,
							y,
							w,
							t);
	}
	
	// right border
	if (r > 0)
	{
		gdk_draw_rectangle( grid->pixmap,
							grid->own_gc,
							TRUE, 
							x + w - r,
							y,
							r,
							h);
	}
	
	// bottom border
	if (b > 0)
	{
		gdk_draw_rectangle( grid->pixmap,
							grid->own_gc,
							TRUE, 
							x,
							y + h - b,
							w,
							b);
	}
	
	// left border
	if (l > 0)
	{
		gdk_draw_rectangle( grid->pixmap,
							grid->own_gc,
							TRUE, 
							x,
							y,
							l,
							h);
	}
	
	// =============== other ornaments ======================
	// TODO: i.e. should they be filled with different color (grey) etc.

	// =============== background ======================
	// set the draw color, use b_color
	gdk_gc_set_rgb_fg_color(grid->own_gc, 
							&data->b_color);
	
	// background, so exclude border
	gdk_draw_rectangle( grid->pixmap,
			            grid->own_gc,
						TRUE, 
					    x+l,
                        y+t,
                        w-l-r,
                        h-t-b);
	//LOGPRINTF("bg: %d %d %d %d", x+l, y+t, w-l-r, h-t-b);

	// =============== content/text ======================
	// set the draw color, use t_color
	gdk_gc_set_rgb_fg_color(grid->own_gc, 
							&data->t_color);
	
	if (data->text[0] != '\0')
	{
		PangoLayout *layout;
		PangoRectangle logical_rect;
		PangoRectangle ink_rect;
		int ox, oy; // offset for centering

		layout = pango_layout_new(gdk_pango_context_get());
		pango_layout_set_text(layout, data->text, -1);

		pango_layout_get_pixel_extents(layout,
								       &ink_rect,
								       &logical_rect);
	    //LOGPRINTF("ex: %d %d %d %d", logical_rect.x, logical_rect.y, logical_rect.width, logical_rect.height);
	    //LOGPRINTF("inc: %d %d %d %d", ink_rect.x, ink_rect.y, ink_rect.width, ink_rect.height);

		// now centre the text
		ox = ((w-l-r) - (logical_rect.x + logical_rect.width))/2;
		oy = ((h-t-b) - (logical_rect.y + logical_rect.height))/2;
		gdk_draw_layout( grid->pixmap, 
						 grid->own_gc, 
						 x+l+ox, // centre the text
						 y+t+oy, 
						 layout);

		g_object_unref(layout);	
	}

    //LOGPRINTF("queue draw");
	// make things appear on the screen
	gtk_widget_queue_draw(GTK_WIDGET(grid));
}

void gtk_grid_set_cell_fg_color(GtkGrid *grid, int nr, GdkColor color)
{
    //LOGPRINTF("entry");

	g_return_if_fail(IS_GTK_GRID(grid));

	if ((grid->data) && (nr >= 0) && (nr < grid->nr_cells_horizontal * grid->nr_cells_vertical))
	{
		grid->data[nr].t_color = color;
		
		// redraw cell
		cell_draw(grid, &grid->data[nr]);
	}	
}

void gtk_grid_set_cell_bg_color(GtkGrid *grid, int nr, GdkColor color)
{
    //LOGPRINTF("entry");

	g_return_if_fail(IS_GTK_GRID(grid));

	if ((grid->data) && (nr >= 0) && (nr < grid->nr_cells_horizontal * grid->nr_cells_vertical))
	{
		grid->data[nr].b_color = color;
		
		// redraw cell
		cell_draw(grid, &grid->data[nr]);
	}	
}

void gtk_grid_set_cell_text(GtkGrid *grid, int nr, char *text)
{
	//LOGPRINTF("entry: cell: %d text:%s", nr, text);

	g_return_if_fail(IS_GTK_GRID(grid));

	if ((grid->data) && (nr >= 0) && (nr < grid->nr_cells_horizontal * grid->nr_cells_vertical))
	{
		strncpy(grid->data[nr].text, text, (MX_CELL_TEXT_SIZE-1) );
		grid->data[nr].text[MX_CELL_TEXT_SIZE-1] = '\0';
	
		// redraw cell
		cell_draw(grid, &grid->data[nr]);
	}
}

static void grid_create_cell_data(GtkGrid *grid)
{
	int x, y;
	p_mxgrid_cell_data_t data; // helper var
	
	if (grid->data != NULL);
	{
		free(grid->data);
	}
	grid->data = malloc( sizeof(mxgrid_cell_data_t) 
								* grid->nr_cells_horizontal
								* grid->nr_cells_vertical);
	// TODO: check for errors
	
	data = grid->data;
	
	for (x=0; x < grid->nr_cells_horizontal; x++)
	{
		for (y=0; y < grid->nr_cells_vertical; y++)
		{
			p_mxgrid_cell_data_t current;
			
			current = &data[x + y * grid->nr_cells_horizontal];
			current->cell_nr = x + y * grid->nr_cells_horizontal;
			current->x = 0;
			current->y = 0;
			current->w = 0;
			current->h = 0;
			strcpy(current->ornaments, "2222");
			strcpy(current->text, "");
			//current->text[0] = '0'+x; // test code
			//current->text[1] = '\0';
			current->t_color.pixel = 0;         // text black
			current->t_color.red   = 0x0000;
			current->t_color.green = 0x0000;
			current->t_color.blue  = 0x0000;
			current->o_color.pixel = 0;         // ornament black
			current->o_color.red   = 0x0000;
			current->o_color.green = 0x0000;
			current->o_color.blue  = 0x0000;
			current->b_color.pixel = 0;         // background white
			current->b_color.red   = 0xFFFF;
			current->b_color.green = 0xFFFF;
			current->b_color.blue  = 0xFFFF;
		}
	}
}

static void grid_create_active_areas(GtkGrid *grid)
{
	int x, y;
	p_mxgrid_active_area_t areas; // helper var
	
	if (grid->active_areas != NULL);
	{
		free(grid->active_areas);
	}
	grid->active_areas = malloc( sizeof(mxgrid_active_area_t) 
								* grid->nr_cells_horizontal
								* grid->nr_cells_vertical);
	// TODO: check for errors
	
	areas = grid->active_areas;
	
	for (x=0; x < grid->nr_cells_horizontal; x++)
	{
		for (y=0; y < grid->nr_cells_vertical; y++)
		{
			p_mxgrid_active_area_t current;
			
			current = &areas[x + y * grid->nr_cells_horizontal];
			current->cell_nr = x + y * grid->nr_cells_horizontal;
			current->x = 0;
			current->y = 0;
			current->w = 0;
			current->h = 0;
			current->children = 0L;
		}
	}
}

static int grid_find_active_area(GtkGrid *grid, int x, int y, int *cx, int *cy)
{
	int posx, posy;
	p_mxgrid_active_area_t current=0L;
	p_mxgrid_active_area_t areas = grid->active_areas; // helper var
	int cell_nr = -1;  // init to 'not found'
	
    //LOGPRINTF("entry");

	*cx = -1; *cy = -1; // init to 'not found'

	posx = 0;
	posy = 0;
	if (&areas[0])
	{
		// set the first active area
		current = &areas[0];
	}

	// first x-position 
	// assumes areas are travered in sorted order, smallest x first
	while(current && 
		  ((x < current->x) ||
		   (x >  (current->x + current->w))) )
	{
		// not found yet, next
		posx++;
		if (posx < grid->nr_cells_horizontal)
		{
	    	current = &areas[posx + posy * grid->nr_cells_horizontal];
		}
		else
		{
			// not in available active areas
			current = 0L;
		}
	}
	// if current == 0L, then outside of active area!

	// then y-position 
	while(current && 
		  ((y < current->y) ||
		   (y >  (current->y + current->h))))
	{
		posy++;
		if (posy < grid->nr_cells_vertical)
		{
	    	current = &areas[posx + posy * grid->nr_cells_horizontal];
		}
		else
		{
			current = 0L;
		}
	}

	if (current)
	{
		cell_nr = current->cell_nr;
		*cx = posx;
		*cy = posy;
		//LOGPRINTF("found: %d (%d,%d)", cell_nr, current->x, current->y);
	}

    // TODO: handle children!
	
	return cell_nr;
}

static void cell_add_active_area(GtkGrid *grid, int cx, int cy, int x, int y, int tx, int ty)
{
	if (grid->active_areas != NULL)
	{
		p_mxgrid_active_area_t current=0L;
		p_mxgrid_active_area_t areas = grid->active_areas; // helper var
		
		current = &areas[cx + cy * grid->nr_cells_horizontal];
		current->x = x;
		current->y = y;
		current->w = (tx - x);
		current->h = (ty - y);
		current->cell_nr = (cx + cy * grid->nr_cells_horizontal);
		
        //LOGPRINTF("[%d] (%d, %d) (%d, %d)", current->cell_nr, current->x, current->y, current->w, current->h);
	}
}

static void cell_update_data(GtkGrid *grid, int cx, int cy, int x, int y, int tx, int ty)
{
	if (grid->data != NULL)
	{
		p_mxgrid_cell_data_t current=0L;
		p_mxgrid_cell_data_t data = grid->data; // helper var
		
		current = &data[cx + cy * grid->nr_cells_horizontal];
		current->x = x;
		current->y = y;
		current->w = (tx - x);
		current->h = (ty - y);
		current->cell_nr = (cx + cy * grid->nr_cells_horizontal);
		
        //LOGPRINTF("[%d] (%d, %d) (%d, %d)", current->cell_nr, current->x, current->y, current->w, current->h);
	}
}

static void gtk_grid_draw(GtkGrid *grid)
{
	int cx=0, cy=0; // position of active cell
	int x, y;       // top-left offset of current cell in widget (border)
	int nr;
	
    LOGPRINTF("entry %d %d", grid->left_margin, grid->top_margin);

	g_return_if_fail((IS_GTK_GRID(grid)));
	
	if (!grid->pixmap)
	{
		return;
	}

	if (grid->status != MXGRID_STATUS_INITIALISED)
	{
		// no sizes known, quit drawing
        //LOGPRINTF("No draw, wrong status");
		return;
	}
	
	// =============== First determine position of every cell ===============
	// set initial position
	x = grid->left_margin;
	
	// for all columns
	for(cx=0; cx<grid->nr_cells_horizontal; cx++)
	{
		int tx = x; // only change y position
		
	    // reset y position
		y = grid->top_margin;
		
		// for all rows
		for(cy=0; cy<grid->nr_cells_vertical; cy++)
		{
			char *content;
			int ty; // temp y value
			
		    //LOGPRINTF("for cx:%d cy%d", cx, cy);
			// is cell data installed
			if (grid->data != NULL)
			{
				int index = cy*grid->nr_cells_horizontal+cx;
				// TODO: check if index exsists
				content = grid->data[index].ornaments;
			}
			else
			{
				content = NULL;
			}
			// determine cell size, give current top-left position and get back
			// the bottom-right position (for the next cell to draw)
			tx = x; // use temp value, so only y is adapted
			ty = y; // use temp value to remember start position
	        //LOGPRINTF("draw: %d %d", x, y);
			cell_calc_size(grid, content, &tx, &ty);
			// update the internal data (TODO: fill this data first and use it to draw)
			cell_update_data(grid, cx, cy, x, y, tx, ty);
			// now create the active area for the cell
			cell_add_active_area(grid, cx, cy, x, y, tx, ty);
			// update y to next cell
			y = ty;
		}
		
		// now copy last 'right'-position to x to move to next column
		x = tx;
	}
	// =============== Now DRAW ===============
	// for all columns
	for(nr=0; nr < (grid->nr_cells_horizontal*grid->nr_cells_vertical); nr++)
	{
		cell_draw(grid, &grid->data[nr]);
	}
}
