/*
 * File Name: mxModel.c
 */

/*
 * This file is part of mxSudoku.
 *
 * mxSudoku is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * mxSudoku is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 Marcel Hendrickx
 * All rights reserved.
 */

//------------------------------------------------------------------------------
// Include Files
//------------------------------------------------------------------------------

// configuration parameters of the project
#include "config.h"

// The implemented interface
#include "mxModel.h"
#include "generator.h"
#include "solver.h"

// system include files
#include <stdlib.h>

// local include files
#include "log.h"
#include "main.h"


//------------------------------------------------------------------------------
// Type Declarations
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// Global Constants
//------------------------------------------------------------------------------

// At this moment only one model can be active at any time
#define MXMODEL_MAX_MODELS 1

//------------------------------------------------------------------------------
// Static Variables
//------------------------------------------------------------------------------

// info on the available (active) models
static int nr_models = 0;
static mxmodel_game_data game_data_array[MXMODEL_MAX_MODELS];

//==============================================================================
// Local Function Definitions
//==============================================================================

static int correct_sudoku(int model, int block_size);

//==============================================================================
// Functions Implementation
//==============================================================================

// create a new model for a game, returns instance to model
int mxmodel_create(mxmodel_type type)
{
	int model = -1;
	p_mxmodel_game_data game_data;
	
	LOGPRINTF("entry: type:%d", type);

	// check if more models can be created
	if (nr_models < MXMODEL_MAX_MODELS)
	{
		model = nr_models;
		nr_models++;
		// get pointer to game data
		game_data = &game_data_array[model];
		game_data->game.type = type;
		switch (type)
		{
			case mxmodel_sudoku3x3:
			    game_data->game.horizontal = 9;
			    game_data->game.vertical   = 9;
				game_data->game.empty_squares = 81;
				break;
			default:
			    // release model data
			    nr_models--;
				model = -1;
				break;
		}
		// if type is supported
		if (model >= 0)
		{
			p_mxmodel_cell cells;
			int i;
			
			// reserve memory for cells of the game
			cells = malloc(sizeof(mxmodel_cell) * 
						   game_data->game.horizontal *
						   game_data->game.vertical);
			game_data->cells = cells;
			
			// init cells
			for (i=0; i<game_data->game.horizontal * game_data->game.vertical; i++)
			{
				cells[i].flags = MXMODEL_CELL_READONLY; //no edit without a game...
				cells[i].must_value = -1; //unknown
				cells[i].value = -1; // not assigned
			}
		}
	}
	
	return model;
}

// delete data of the model (release all resources)
int mxmodel_delete(int model)
{
	LOGPRINTF("entry: model:%d", model);

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		return -1; // FAILED
	}
	
	switch (game_data_array[model].game.type)
	{
		case mxmodel_sudoku3x3:
			// release memory
			free(game_data_array[model].cells);
			game_data_array[model].game.type = mxmodel_unknown;
			break;
		default:
		    // do not know what to do?
			break;
	}
	
	return 0;
}

// generate a new game for the instance with the indicated difficulty level
int mxmodel_generate(int model, int level)
{
	LOGPRINTF("entry: model:%d", model);

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		return -1; // FAILED
	}
	
	// and generate the random game data for the model
	switch (game_data_array[model].game.type)
	{
		case mxmodel_sudoku3x3:
		    game_data_array[model].game.empty_squares = 81; //reset squares
			// generate data for soduku (see generate.c)
			mxmodel_gen_soduku_random_game(&game_data_array[model]);
			// and solve it ... (see solve.c)
			mxmodel_get_solution(&game_data_array[model]);
			break;
		default:
		    // do not know what to do?
			break;
	}
	
	return 1;
}

// verifies the game and returns number of incorrect answers
int mxmodel_verify(int model)
{
	LOGPRINTF("entry: model:%d", model);

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		return -1; // FAILED
	}

	// check
	return(correct_sudoku(model, game_data_array[model].game.horizontal));
}

// solve the game, returns 0 on success
int mxmodel_solve(int model)
{
	LOGPRINTF("entry: model:%d", model);

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		return -1; // FAILED
	}
	
	return -1; // Function not used
}

// get info on the game (into supplied structure)
int mxmodel_get_info(int model, p_mxmodel_game game)
{
	LOGPRINTF("entry: model:%d", model);

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		return -1; // FAILED
	}
	
	// copy game data
	game->type       = game_data_array[model].game.type;
	game->horizontal = game_data_array[model].game.horizontal;
	game->vertical   = game_data_array[model].game.vertical;
	
	return 0;
}

// get cell info (into supplied structure)
int mxmodel_get_cell_info(int model, int cell_nr, p_mxmodel_cell cell)
{
	p_mxmodel_cell game_cell;
	
	//LOGPRINTF("entry: model:%d", model);

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		return -1; // FAILED
	}
	
	game_cell = &game_data_array[model].cells[cell_nr];
	cell->flags      = game_cell->flags;
	cell->must_value = game_cell->must_value;
	cell->value      = game_cell->value;

	//LOGPRINTF("leave: flags: %d must:%d value:%d", game_cell->flags, game_cell->must_value, game_cell->value);

	return 1;
}

// can change call (0=no)
int mxmodel_can_change_cell(int model, int cell_nr)
{
	p_mxmodel_cell game_cell;
	
	LOGPRINTF("entry");

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		return -1; // FAILED
	}
	
	game_cell = &game_data_array[model].cells[cell_nr];
	if ((game_cell->flags & MXMODEL_CELL_READONLY) != 0)
	{
		return 0;
	}
	else
	{
		return 1;
	}
}

// set cell value
int mxmodel_set_cell_value(int model, int cell_nr, int value)
{
	p_mxmodel_cell game_cell;
	
	//LOGPRINTF("entry");

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		return -1; // FAILED
	}
	
	game_cell = &game_data_array[model].cells[cell_nr];

	if ((game_cell->flags & MXMODEL_CELL_READONLY) != 0)
	{
		// read only
		return 0;
	}
	else
	{
		// update empty_squares-count
		// was not filled and now filled
		if ((value != -1) && (game_cell->value == -1))
		{
			// cell filled
			game_data_array[model].game.empty_squares--;
		}
		
		// was filled and now not filled
		if ((value == -1) && (game_cell->value != -1))
		{
			// cell cleared
			game_data_array[model].game.empty_squares++;
		}
		
		// store value in model
		game_cell->value = value;
			
		// return the number of empty_squares
		return game_data_array[model].game.empty_squares;
	}
}

// return the number of empty_squares
int mxmodel_get_empty_cells(int model)
{
	//LOGPRINTF("entry");

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		return -1; // FAILED
	}

	return game_data_array[model].game.empty_squares;
}

// load game from file
int mxmodel_load_game(int model, char *filename, int no_warning)
{
	FILE *sudoku_file;
	char number[2];         // sudoku square value as string
	int row, col;
	int value;              // value of the cell
	int flags;              // flags for the cell
	p_mxmodel_cell game_cell; // the cell to update
	int index = 0;            // index in cell-array
	
	LOGPRINTF("entry");

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		return -1; // FAILED
	}
	
	if( (sudoku_file = fopen(filename, "r")) == NULL) 
	{
		if ((no_warning & 1) == 0)
		{
			char *message;
			message = g_strdup_printf("Unable to open %s!", filename);
			show_popup_msg("?", message);
			g_free(message);
		}
		return 0;
	}

	// NUMBER will be treated as a string to place within sudoku square entry
	number[1] = '\0';

	// reset empty cells
	game_data_array[model].game.empty_squares = 
		game_data_array[model].game.vertical *
		game_data_array[model].game.horizontal;
	
	// This code assumes the same format for the files a gtk-sudoku
	// TODO: Create a new file format, preferably with several difficulty
	//       levels in the same file.
	for (row = 0; row < game_data_array[model].game.vertical; ++row)
	{
		for (col = 0; col < game_data_array[model].game.horizontal; ++col) 
		{
			game_cell = &game_data_array[model].cells[index++];
	
			// check for 'unexpected' EOF
			if ( (*number = fgetc(sudoku_file)) == EOF ) 
			{
				if (game_data_array[model].game.horizontal != 9)
				{
					show_popup_msg("!!", "Your file's too short\n"
							"It's probably a 9x9 sudoku.");
				}
				else
				{
					show_popup_msg("!!", "Your file's missing some values.");
				}
				if (fclose(sudoku_file) != 0)
				{
					g_print("  * Unable to close %s.\n", filename);
				}
				return 0;
			}
			
			flags = 0;
			switch (*number) 
			{
				// 9x9 sudoku
				case '1': case '2': case '3':
				case '4': case '5': case '6':
				case '7': case '8': case '9':
				    value = *number - '0'; // '1' = 1, etc
					break;
				// extra symbols 16x16 sudoku
				case '0': case 'A': case 'B':
				case 'C': case 'D': case 'E':
				case 'F':
					if (game_data_array[model].game.horizontal != 16) 
					{
						show_popup_msg("!!", "Incorrect file format");
						return FALSE;
					}
					value = (*number == '0') ? 0 : 10 + *number - 'A'; // '0' = 0, 'A' = 10, etc
					break;
				// 9x9 sudoku, unchangable values
				case 'L': case 'M': case 'N':
				case 'O': case 'P': case 'Q':
				case 'R': case 'S': case 'T':
					value = *number - 'L' + 1;
					flags = MXMODEL_CELL_READONLY;
					break;
				case 'K':
				case 'U': case 'V': case 'W':
				case 'X': case 'Y': case 'Z':
					if (game_data_array[model].game.horizontal != 16) 
					{
						show_popup_msg("!!", "Incorrect file format");
						return FALSE;
					}
					value = (*number == 'K') ? 0 : *number - 'U' + 10;  // 'K' = 0, 'U' = 10, etc
					flags = MXMODEL_CELL_READONLY;
					break;
				// treat all other input as a blank square
				default:
				    value = -1;
					break;
			} // switch
			
			// store values
			//LOGPRINTF("read: [%d] = %d (flags:%d)", index-1, value, flags);
			game_cell->flags = flags;
			game_cell->value = value;
			if (flags & MXMODEL_CELL_READONLY)
			{
				// the value was a given value
				game_cell->must_value = value;
			}
			else
			{
				game_cell->must_value = -1;
			}
			if (value != -1)
			{
				// cell is filled
				game_data_array[model].game.empty_squares--;

			}
		}
	}
	
	// and solve it ... to update the must_values
	mxmodel_get_solution(&game_data_array[model]);

	if (fclose(sudoku_file) != 0)
	{
		g_print("  * Unable to close %s.\n", filename);
	}
	
	return 1;
}

// save game to file
int mxmodel_save_game(int model, char *filename)
{
	register int row;
	register int col;
	FILE        *file;
	int          value;
	p_mxmodel_cell game_cell; // the cell to update
	int index = 0;            // index in cell-array

	LOGPRINTF("entry");

	// check if model is valid
	if ( (model >= nr_models) || 
		 (model < 0) ||
		 (game_data_array[model].game.type == mxmodel_unknown))
	{
		return -1; // FAILED
	}
	

	file = fopen(filename, "w");
	// if opening the specified file was successful
	if(file != NULL) 
	{
	    // This code assumes the same format for the files a gtk-sudoku
		for(row = 0; row < game_data_array[model].game.vertical; ++row)
		{
			for(col = 0; col < game_data_array[model].game.horizontal; ++col)
			{
				game_cell = &game_data_array[model].cells[index++];
				
				if(game_cell->flags & MXMODEL_CELL_READONLY) 
				{
					// readonly cells
					value = game_cell->must_value;
					if(value <= 9)
					{
						// 0123456789
						// KLMNOPQRST
						value += 'K';
					}
					else
					{
						// ABCDEF
						// UVWXYZ
						value -= 10;
						value += 'U';
					}
				    fputc(value, file);
				}
				else 
				{
					value = game_cell->value;
					if(value == -1)
					{
						// no value yet
						fputc('.', file);
					}
					else
					{
						if(value <= 9)
						{
							// 0123456789
							value += '0';
						}
						else
						{
							// ABCDEF
							value -= 10;
							value += 'A';
						}
						fputc(value, file);
					}
				} 
			}
		}
		fclose(file);
	}

	return index;
}

//==============================================================================
// Local Function Implementation
//==============================================================================

// TODO: following 3 functions should be refactored out since the solutions
//       is now known in the must_values.
static int rows_check_out(int model, int block_size, int *values)
{
	register int row;
	register int col;
	int val;
	
	LOGPRINTF("entry");

	// check for every row that every symbol is exactly used once
	for(row = 0; row < block_size; ++row) 
	{
		// clear columns
		for(col = 0; col < block_size; ++col)
		{
			values[col] = 0;
		}
		// check columns
		for(col = 0; col < block_size; ++col) 
		{
			mxmodel_cell cell;
			
			// get the data
			mxmodel_get_cell_info(model, row*block_size+col, &cell);
			val = cell.value;
			if(val == -1)
			{
				return 0;
			}
			values[val] = 1; 
		}
		for(col = 0; col < block_size; ++col)
		{
			if(!values[col])
			{
				return 0;
			}
		}
	}
	return 1;
}

static int cols_check_out(int model, int block_size, int *values)
{
	register int row;
	register int col;
	int val;
	
	LOGPRINTF("entry");

	// check all columns
	for(col = 0; col < block_size; ++col) 
	{
		// clear rows
		for(row = 0; row < block_size; ++row)
		{
			values[row] = 0;
		}
		// check rows 
		for(row = 0; row < block_size; ++row) 
		{
			mxmodel_cell cell;
			
			// get the data
			mxmodel_get_cell_info(model, row*block_size+col, &cell);
			val = cell.value;
			if(val == -1)
			{
				return 0;
			}
			// TODO: check size
			values[val] = 1; 
		}
		for(row = 0; row < block_size; ++row)
		{
			if(!values[row])
			{
				return 0;
			}
		}
	}
	return 1;
}

static int correct_sudoku(int model, int block_size)
{
	int *values = (int *)malloc(block_size * sizeof(int));

	LOGPRINTF("entry");

	if(values == NULL) 
	{
		fputs("  * Out of memory!\n", stderr);
		return 0;
	}

	if(!rows_check_out(model, block_size, values))
	{
		return 0;
	}
	if(!cols_check_out(model, block_size, values))
	{
		return 0;
	}

	free(values);
	return 1;
}


