/*
 * File Name: menu.c
 */

/*
 * This file is part of mxSudoku.
 * based on the helloworld example of iRex
 *
 * hello_world is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * hello_world is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 Mackx.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <stdio.h>
#include <unistd.h>

// ereader include files, between < >

// local include files, between " "
#include "log.h"
#include "i18n.h"
#include "ipc.h"
#include "menu.h"

#include "do.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

// menus for application, must be unique
static const char *MENU_MAIN                = "mxsudoku_menu_main";
                                            
// menu groups, must be unique
static const char *GROUP_FILE               = "mxsudoku_file";
static const char *GROUP_ACTIONS            = "mxsudoku_actions";
                                            
// menu items
//   group "Actions"                          
static const char *ITEM_ACTION_VERIFY       = "verify";
static const char *ITEM_ACTION_RESET        = "reset";
static const char *ITEM_ACTION_SOLVE        = "solve";
static const char *ITEM_ACTION_INFO         = "info";

//   group "File"                        
static const char *ITEM_FILE_NEW            = "new";
static const char *ITEM_FILE_OPEN           = "open";
static const char *ITEM_FILE_SAVE           = "save";
static const char *ITEM_FILE_CLOSE          = "close";
                                
// item states
static const char *STATE_NORMAL             = "normal";
static const char *STATE_DISABLED           = "disabled";
//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------



//============================================================================
// Local Function Definitions
//============================================================================



//============================================================================
// Functions Implementation
//============================================================================

// initialise popup menu
void menu_init ( void ) 
{
    static gboolean firsttime = TRUE;
    const char      *group;

    LOGPRINTF("entry");

    // execute only once
    if ( !firsttime )
    {
        WARNPRINTF("function called twice");
        return;
    }
    firsttime = FALSE;

    // add groups
    group = GROUP_FILE;
    ipc_menu_add_group( group,                  "",    "folder"          );
    ipc_menu_add_item ( ITEM_FILE_NEW,          group, "/media/mmcblk0p1/Programs/_mxSudoku/icons/sudoku_new.png"             );
    ipc_menu_add_item ( ITEM_FILE_OPEN,         group, "open"            );
    ipc_menu_add_item ( ITEM_FILE_SAVE,         group, "save"            );
    ipc_menu_add_item ( ITEM_FILE_CLOSE,        group, "close"           );
	
    //                                                                                                                
    group = GROUP_ACTIONS;
    ipc_menu_add_group( group,                  "",    "folder"          );
    ipc_menu_add_item ( ITEM_ACTION_VERIFY,     group, "/media/mmcblk0p1/Programs/_mxSudoku/icons/sudoku_verify.png"          );
    ipc_menu_add_item ( ITEM_ACTION_RESET,      group, "/media/mmcblk0p1/Programs/_mxSudoku/icons/sudoku_reset.png"           );
    ipc_menu_add_item ( ITEM_ACTION_SOLVE,      group, "/media/mmcblk0p1/Programs/_mxSudoku/icons/sudoku_solve.png"           );
    ipc_menu_add_item ( ITEM_ACTION_INFO,       group, "info"            );

    // add menus
    ipc_menu_add_menu( MENU_MAIN, GROUP_FILE, GROUP_ACTIONS, NULL );

    // set menu texts
    menu_set_text();

	// not all menus are implemented, disable them
	group = GROUP_FILE;
	ipc_menu_set_item_state (ITEM_FILE_SAVE, group, STATE_DISABLED);
	
    // set menu context
    menu_show();    
}


// remove the proper popup menu
void menu_destroy (void)
{
    LOGPRINTF("entry");

    // remove the main menu
    ipc_remove_menu( MENU_MAIN );
}


// show the proper popup menu
void menu_show (void)
{
    LOGPRINTF("entry");

    // show the main menu
    ipc_menu_show_menu( MENU_MAIN );
}


// initialise popup menu names
// also call this function when the LOCALE has changed
void menu_set_text ( void ) 
{
    const char      *group;

    LOGPRINTF("entry");

    // set groups + items
    group = GROUP_FILE;
    ipc_menu_set_group_label( group,                     _("File")           );
    ipc_menu_set_item_label ( ITEM_FILE_NEW,      group, _("New")            );
    ipc_menu_set_item_label ( ITEM_FILE_OPEN,     group, _("Open")           );
    ipc_menu_set_item_label ( ITEM_FILE_SAVE,     group, _("Save")           );
    ipc_menu_set_item_label ( ITEM_FILE_CLOSE,    group, _("Close")          );
    //                                                                                                                 
    group = GROUP_ACTIONS;                                                     
    ipc_menu_set_group_label( group,                     _("Actions")        );
    ipc_menu_set_item_label ( ITEM_ACTION_VERIFY, group, _("Verify")         );
    ipc_menu_set_item_label ( ITEM_ACTION_RESET,  group, _("Reset")          );
    ipc_menu_set_item_label ( ITEM_ACTION_SOLVE,  group, _("Solve")          );
    ipc_menu_set_item_label ( ITEM_ACTION_INFO,   group, _("Info")           );

    // set menus
    ipc_menu_set_menu_label( MENU_MAIN,  _("mxSudoku Menu") );
}


//----------------------------------------------------------------------------
// Callbacks from popupmenu
//----------------------------------------------------------------------------

// user has pressed a menu button
void menu_on_item_activated ( const gchar *item,
                              const gchar *group,
                              const gchar *menu,
                              const gchar *state )
{
    gboolean    ok = TRUE;
    char        *msg = NULL;

    LOGPRINTF("entry: item [%s] group [%s]", item, group);

    if ( strcmp(group, GROUP_ACTIONS) == 0 )
    {
        if ( strcmp(state, STATE_NORMAL) == 0 )
        {
            if (      strcmp(item, ITEM_ACTION_VERIFY) == 0 )
            {
                // do verify
				do_verify_game();
            }
            else if ( strcmp(item, ITEM_ACTION_RESET) == 0 )
            {
                // do reset
				do_reset_game();
            }
            else if ( strcmp(item, ITEM_ACTION_SOLVE) == 0 )
            {
                // do solve
				do_solve_game();
            }
            else if ( strcmp(item, ITEM_ACTION_INFO) == 0 )
            {
                // do info
				do_about_game();
            }
            else
            {
                WARNPRINTF("unexpected menu item [%s] in group [%s]", item, group);
                ok = FALSE;
            }
        }
        else
        {
            WARNPRINTF("ignore state [%s] item [%s][%s]", state, item, group);
        }
    }
    else if ( strcmp(group, GROUP_FILE) == 0 )
    {
        if ( strcmp(state, STATE_NORMAL) == 0 )
        {
            if (      strcmp(item, ITEM_FILE_NEW) == 0 )
            {
                // do new
				do_generate_game();
            }
            else if ( strcmp(item, ITEM_FILE_OPEN) == 0 )
            {
                // do open
				do_open_game();
            }
            else if ( strcmp(item, ITEM_FILE_SAVE) == 0 )
            {
                // do save
				do_save_as();
            }
            else if ( strcmp(item, ITEM_FILE_CLOSE) == 0 )
            {
                // do close
				do_close();
				_exit(0);
				return;
            }
            else
            {
                WARNPRINTF("unexpected menu item [%s] in group [%s]", item, group);
                ok = FALSE;
            }
        }
        else
        {
            WARNPRINTF("ignore state [%s] item [%s][%s]", state, item, group);
        }
    }
    else
    {
        WARNPRINTF("unexpected menu group [%s]", group);
        ok = FALSE;
    }

    if (!ok)
    {
        msg = g_strdup_printf( _("Unhandled menu item.\n"
                                 "menu: %s\n"
                                 "group: %s\n"
                                 "item: %s\n"
                                 "state: %s"),
                               menu,
                               group,
                               item,
                               state );
        //LOGPRINTF(msg);
        g_free(msg);
    }
}
