/*
 * File Name: mxControl.c
 */

/*
 * This file is part of mxSudoku.
 *
 * mxSudoku is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * mxSudoku is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 Marcel Hendrickx
 * All rights reserved.
 */

//------------------------------------------------------------------------------
// Include Files
//------------------------------------------------------------------------------

// configuration parameters of the project
#include "config.h"

// The implemented interface
#include "mxControl.h"
#include "do.h" // move to other file?

// system include files

// local include files
#include "mxModel.h"
#include "mxView.h"
#include "log.h"
#include "i18n.h"
#include "main.h"


//------------------------------------------------------------------------------
// Type Declarations
//------------------------------------------------------------------------------


//------------------------------------------------------------------------------
// Global Constants
//------------------------------------------------------------------------------

// the location to load/store the last played game
#define LASTGAME "/media/mmcblk0p1/Programs/_mxSudoku/last_game"
// the location of the games
#define GAMES    "/media/mmcblk0p1/Programs/_mxSudoku/games/"
//------------------------------------------------------------------------------
// Static Variables
//------------------------------------------------------------------------------

static int g_model=-1;
static mxview_game g_game;
static GtkWidget *g_window=NULL; // window, needed for dialogs
// indicates wether a valid game has been generated or loaded
static int playing=0;

//==============================================================================
// Local Function Definitions
//==============================================================================

//==============================================================================
// Functions Implementation
//==============================================================================

// This function handles events that are generated by the view or menu.
int mxcontrol_handle_view_event(p_mxcontrol_event event)
{
	int empty_cells = -1;  // to hold the number of cells not filled yet
	int redraw = 1;        // set to 0 when no forced redraw should be done
	
    //LOGPRINTF("entry");

	// an event has been triggered from the view
	switch (event->type)
	{
		// the event indicates that a new value is entered for a cell
		case mxcontrol_event_set_value:
			{
				mxmodel_cell cell;
				int flags;
				
				// check if cell can be changed
				mxmodel_get_cell_info(g_model, event->cell_nr, &cell);
				flags = ((cell.flags & MXMODEL_CELL_READONLY) == MXMODEL_CELL_READONLY) ? 1 : 0;
				if (flags == 0)
				{
					// check if we are in 'cheat' mode
					if (event->value == -2)
					{
						// copy the must value!! CHEATING!!
						event->value = cell.must_value;
					}
					// give the cell its new value (model and view)
					empty_cells = mxmodel_set_cell_value(g_model, event->cell_nr, event->value);
					mxview_cell_set_value(event->cell_nr, event->value, 0);
				}
				else
				{
					//TODO: beep? cell can not be modified
				}
			}
			break;
		
		// request the current game to be verified
		case mxcontrol_event_cmd_verify:
			{
				do_verify_game();
				redraw = 0;
			}
			break;
			
		// TODO:
		// To make the seperation more visible should all menu command also go via this function?
		// I should probably refactor the content of menu.c also, or move it into here?
		default:
			break;
	}
	
	// check how many cells have not been entered
	empty_cells = mxmodel_get_empty_cells(g_model);
	
	if (empty_cells == 0)
	{
		// auto verify if all cells are filled
		do_verify_game();
		redraw = 0; // verify_game will have shown a message, we don't want to overwrite this
	}
	
	return redraw;
}

// create the model, view and controller for the game
void mxcontrol_create_game(GtkWidget *window, mxmodel_type type)
{
    GtkWidget        *widget;
	int i;
	int result;

    LOGPRINTF("entry");

	// store the window
	g_window = window;
	
	// Create the model (only mxmodel_sudoku3x3 is supported right now)
	g_model = mxmodel_create(mxmodel_sudoku3x3);
	
	// TODO: use mxmodel_get_info to get game info (hor, vert) and not these fixed values
	// Create the view
	g_game.type =mxview_keypad;
	g_game.board_horizontal = 9;
	g_game.board_vertical = 9;
	g_game.keypad_nr_symbols = 9;
	widget = mxview_create(&g_game);

	gtk_container_add(GTK_CONTAINER(window), widget);
	
	// open last game, which will be saved when the program closes
	result = mxmodel_load_game(g_model, LASTGAME, 1);
	if (result > 0)
	{
		// display the model
		// TODO: the for loop below is copied in several placed in this file so
		//       it is a good candidate to refactor into a seperate function.
		for (i=0; i< (g_game.board_horizontal * g_game.board_vertical); i++)
		{
			mxmodel_cell cell;
			int flags;
		
			// get the data
			mxmodel_get_cell_info(g_model, i, &cell);
			flags = ((cell.flags & MXMODEL_CELL_READONLY) == MXMODEL_CELL_READONLY) ? 1 : 0;

			// update view of the (board) cell
			mxview_cell_set_value(i, cell.value, flags);
		}
		playing = 1; // we are playing
		mxview_update(); // force update of screen
	}
}

void mxcontrol_quit_game()
{
	int result;
	
    LOGPRINTF("entry");
	
	// BIG TODO: !!!free all memory (and other resources)!!!

	// close last game, when we were playing
	if (playing > 0)
	{
		result = mxmodel_save_game(g_model, LASTGAME);
	}
}

//==============================================================================
// Show the about/help dialog
void do_about_game(void)
{
    LOGPRINTF("entry");
	
	// show about/help
	main_about_game();
}

// open a game from file
void do_open_game(void)
{
	GtkWidget *file_chooser;
	int i;

	LOGPRINTF("entry: %d", (int)g_window);

	// choose a file
	file_chooser = gtk_file_chooser_dialog_new(
			"Load Sudoku Puzzle",
			GTK_WINDOW(g_window), 
			GTK_FILE_CHOOSER_ACTION_OPEN, 
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
			GTK_STOCK_OK, GTK_RESPONSE_OK, NULL);

	// set the game folder
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(file_chooser), GAMES);

	// and get it
	if(gtk_dialog_run(GTK_DIALOG(file_chooser)) == GTK_RESPONSE_OK) 
	{
		gchar *sudoku_file;
		sudoku_file = 
			gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(file_chooser));
		
		// load the file into the model
		mxmodel_load_game(g_model, sudoku_file, 0);
		
		// display the model
		for (i=0; i< (g_game.board_horizontal * g_game.board_vertical); i++)
		{
			mxmodel_cell cell;
			int flags;
		
			// get the data
			mxmodel_get_cell_info(g_model, i, &cell);
			flags = ((cell.flags & MXMODEL_CELL_READONLY) == MXMODEL_CELL_READONLY) ? 1 : 0;

			// update view of the (board) cell
			mxview_cell_set_value(i, cell.value, flags);
		}
		playing = 1; // we are playing
		mxview_update(); // force update of screen
		
		// free resources
		g_free(sudoku_file);
	}

	gtk_widget_destroy(file_chooser);
}

// save a game from file
void do_save_as(void)
{
    LOGPRINTF("entry");
	
	// not implemented, is this usefull? Since game being played is already auto-saved?
}

// reset all non-read-only cells to empty
void do_reset_game(void)
{
	int i;
	
    LOGPRINTF("entry");

	// update the model and view with the must_values only
	for (i=0; i< (g_game.board_horizontal * g_game.board_vertical); i++)
	{
		mxmodel_cell cell;
	    int flags;
		int value;
	
		// get the data
		mxmodel_get_cell_info(g_model, i, &cell);
		flags = ((cell.flags & MXMODEL_CELL_READONLY) == MXMODEL_CELL_READONLY) ? 1 : 0;
		
		// copy only must value in model, clear others
		value = (flags == 1) ? cell.must_value : -1;
		mxmodel_set_cell_value(g_model, i, value);
		
		// update view of the (board) cell
		mxview_cell_set_value(i, value, flags);
	}
	mxview_update(); // force update
}


// generate a new random game
void do_generate_game(void)
{
	int i;
	
    LOGPRINTF("entry");

	// generate a new random game, difficulty level = 1
	mxmodel_generate(g_model, 1);

	// now update the view with the values from the model
	for (i=0; i< (g_game.board_horizontal * g_game.board_vertical); i++)
	{
		mxmodel_cell cell;
	    int flags;
	
		// get the data
		mxmodel_get_cell_info(g_model, i, &cell);
		flags = ((cell.flags & MXMODEL_CELL_READONLY) == MXMODEL_CELL_READONLY) ? 1 : 0;

		// update view of the (board) cell
		mxview_cell_set_value(i, cell.value, flags);
	}
	playing = 1;
	mxview_update();
}


// Show the solution for the current game
void do_solve_game(void)
{
	int i;
	
    LOGPRINTF("entry");

	// now update the view with the must_values from the model
	for (i=0; i< (g_game.board_horizontal * g_game.board_vertical); i++)
	{
		mxmodel_cell cell;
	    int flags;
	
		// get the data
		mxmodel_get_cell_info(g_model, i, &cell);
		flags = ((cell.flags & MXMODEL_CELL_READONLY) == MXMODEL_CELL_READONLY) ? 1 : 0;
		
		// copy must to value in model
		if (flags != 1)
		{
			mxmodel_set_cell_value(g_model, i, cell.must_value);
		}
		
		// update view of the (board) cell
		mxview_cell_set_value(i, cell.must_value, flags);
	}
	mxview_update();
}

// Verify if game is correct so far, print a message on status of game
void do_verify_game(void)
{
	int i;
	int count_ok = 0;
	int count_wrong = 0;

	//LOGPRINTF("entry");
	
	// now update the view with the must_values from the model
	for (i=0; i< (g_game.board_horizontal * g_game.board_vertical); i++)
	{
		mxmodel_cell cell;
	    int flags;
	
		// get the data
		mxmodel_get_cell_info(g_model, i, &cell);
		flags = ((cell.flags & MXMODEL_CELL_READONLY) == MXMODEL_CELL_READONLY) ? 1 : 0;

		//LOGPRINTF("%d: %d %d=%d", i, flags, cell.value, cell.must_value);
		if (flags == 0) // only non-read only cells
		{
			if ((cell.value != -1) && (cell.must_value != -1))
			{
				// both user value and must-value are set
				if (cell.value == cell.must_value)
				{
					count_ok++;
				}
				else
				{
					//LOGPRINTF("WRONG: %d: val:%d must:%d", i, cell.value, cell.must_value);
					count_wrong++;
					// mark the cell
		            mxview_cell_set_value(i, cell.value, 2);
				}
			}
		}
	}

	// any errors detected?
	if (count_wrong == 0)
	{
		gchar       *msg  = NULL;
		int remaining = mxmodel_get_empty_cells(g_model);
		
		// Still some cells not filled?
		if (remaining == 0)
		{
			// all good, no field to fill
			// allow internationalisation
			msg = g_strdup_printf( _("GOOD you SOLVED the puzzle"));
			mxview_status_msg(msg);
			g_free(msg);
		}
		else
		{
			// allow internationalisation
			msg = g_strdup_printf( _("GOOD everything is OK so far (%d remaining)"), remaining);
			mxview_status_msg(msg);
			g_free(msg);
		}
	}
	else
	{
		gchar       *msg  = NULL;
		
		// allow internationalisation
		msg = g_strdup_printf( _("Please check the mistakes!"));
		mxview_status_msg(msg);
		g_free(msg);
	}
}


//==============================================================================
// Local Function Implementation
//==============================================================================



