import java.io.FileInputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.awt.image.AffineTransformOp;

import com.sun.image.codec.jpeg.JPEGImageDecoder;
import com.sun.image.codec.jpeg.JPEGCodec;
import com.sun.image.codec.jpeg.ImageFormatException;
import com.sun.image.codec.jpeg.JPEGImageEncoder;

public class KindleSizer {

    private static int KINDLE_MAX_X = 600;
    private static int KINDLE_MAX_Y = 800;

    public static void main(String args[]) {

	if (args.length != 2) {
	    System.out.println("Usage: KindleSizer <input> <output>");
	    System.exit(0);
	}

	String inFile = args[0];
	String outFile = args[1];

	FileInputStream fs = null;

	try {
	    // Open and load source image
	    fs = new FileInputStream(inFile);
	    JPEGImageDecoder decoder = JPEGCodec.createJPEGDecoder(fs);
	    BufferedImage sourceImage = decoder.decodeAsBufferedImage();
	    fs.close();

	    // Now we figure out the correct scaling and positioning for
	    // the image in the destination Kindle sized image.
	    float scale = 0;
	    int x = 0;
	    int y = 0;
	    float width = sourceImage.getWidth();
	    float height = sourceImage.getHeight();

	    if (width >= height) {
		scale = KINDLE_MAX_X / (float)width;
		y = (KINDLE_MAX_Y - (int)(scale * (float)height)) / 2;
	    } else {
		scale = KINDLE_MAX_Y / (float)height;
		x = (KINDLE_MAX_X - (int)(scale * (float)width)) / 2;
	    }


	    System.out.println("Src Width: " + width);
	    System.out.println("Src Height: " + height);
	    System.out.println("Scale: " + scale);
	    System.out.println("x: " + x);
	    System.out.println("y: " + y);

	    AffineTransform af = AffineTransform.getScaleInstance(scale, scale );

	    RenderingHints rh = new RenderingHints(RenderingHints.KEY_ANTIALIASING,
						   RenderingHints.VALUE_ANTIALIAS_ON);

	    AffineTransformOp transform = new AffineTransformOp(af,rh);

	    // We scale the image in it's original color space to let
	    // antialiasing work the best it can.
	    BufferedImage scaledImage = transform.createCompatibleDestImage(sourceImage, sourceImage.getColorModel());
	    transform.filter(sourceImage, scaledImage);

	    // Drawing a color image into a gray bitmap gives us the best
	    // colorspace conversion.
	    BufferedImage grayImage = new BufferedImage(KINDLE_MAX_X, KINDLE_MAX_Y, BufferedImage.TYPE_BYTE_GRAY);  
	    Graphics g = grayImage.getGraphics();  
	    g.fillRect(0, 0, KINDLE_MAX_X, KINDLE_MAX_Y);
	    g.drawImage(scaledImage, x, y, null);  
	    g.dispose();  

	    // Now we encode and store the scaled and gray image
	    FileOutputStream out = new FileOutputStream(outFile);

	    JPEGImageEncoder encoder = JPEGCodec.createJPEGEncoder(out,
								   JPEGCodec.getDefaultJPEGEncodeParam(grayImage));

	    encoder.encode(grayImage);
	    out.close();

	    // Let the user know all is well
	    System.out.println("Saved file " + outFile);
	} catch (FileNotFoundException fnfe ) {
	    System.out.println("File " + inFile + " does not exist!");

	} catch (NumberFormatException nfe){
	    System.out.println("You entered " + args[2] + ". Please enter a decimal expression.");

	} catch (IOException ioe) {
	    System.out.println("IO Exception: " + ioe.getMessage());

	} catch (ImageFormatException ife) {
	    System.out.println("Image Format Excpetion: Could not decode " + inFile);

	} catch (Exception e) {
	    System.out.println(e.getMessage());

	} finally {
	    try {
		if (fs != null) {
		    fs.close();
		}
	    } catch (Exception e) {
		System.err.println(e.getMessage());
	    }
	}
    }
}
