#! /usr/bin/env lua

--[[
	TreeView example. Can be sorted, searched, edited and have its rows
	swapped by drag'n'drop!
	
	Note: If you wan't to just reorder the rows, you can set the property
	"reorderable" of the TreeView. Much more easy and fancy.
--]]

require("lgui")

local window = lgui.Window.new()
local view	 = lgui.TreeView.new()
local model	 = lgui.ListStore.new("gchararray", "gboolean")
local rend1  = lgui.CellRendererText.new()
local rend2  = lgui.CellRendererToggle.new()
local col1	 = lgui.TreeViewColumn.new("Name", rend1, "text", 0, true)
local col2   = lgui.TreeViewColumn.new("Is red?", rend2, "active", 1, true)
local iter	 = lgui.TreeIter.new()
local iter2	 = lgui.TreeIter.new()
local sel	 = view:getSelection()

-- Called when the user confirms an edit.
function rend1Edited(data, path, new)
	lgui.TreeModel.getIterFromString(model, iter, path)
	model:set(iter, 0, new)
end

-- Called when the checkbutton is toggled.
function rend2Toggled(data, path)
	lgui.TreeModel.getIterFromString(model, iter, path)
	local current = lgui.TreeModel.get(model, iter, 1)
	model:set(iter, 1, not current)
end

rend1:set("editable", true)
rend1:connect("edited", rend1Edited)
rend2:connect("toggled", rend2Toggled)

local values = {
	{name = "Pineapple", flag = false},
	{name = "Grape", flag = false},
	{name = "Banana", flag = false},
	{name = "Apple",  flag = true},
	{name = "Orange", flag = false},
}

for i, j in ipairs(values) do
	model:append(iter)
	model:set(iter, 0, j.name, 1, j.flag)
end

function motion(data, context, x, y, time)
	lgui.DragContext.status(context, lgui.ACTION_COPY, time)
	return true
end

function drop(data, context, x, y, time)
	lgui.DragContext.finish(context, true, false, time)
	sel:getSelected(iter)

	local res, path = view:getPathAtPos(x, y)
	path = tostring(tonumber(path) - 1)
	local id, sort = lgui.TreeSortable.getSortColumnId(model)
	
	if res and id == lgui.TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID then
		lgui.TreeModel.getIterFromString(model, iter2, path)
		model:swap(iter, iter2)
	end
	
	return true
end

-- Enable internal drag'n'drop
lgui.DragSource.set(view, lgui.BUTTON1_MASK)
lgui.DragDest.set(view, 0)
lgui.DragSource.setIconStock(view, "gtk-copy")
view:connect("drag-motion", motion)
view:connect("drag-drop", drop)

view:appendColumn(col1, col2)
view:set("model", model)

window:add(view)
window:set("window-position", lgui.WIN_POS_CENTER, "title", "TreeView demo")
window:connect("delete-event", lgui.quit)
window:showAll()

lgui.main()
