#! /usr/bin/env lua

--[[
	Dynamically builded trees that reflects the library contents.
--]]

require("lgui")

Navigator = {}

function Navigator.new()
	local self = {}
	setmetatable(self, {__index = Navigator})
	
	self.window = lgui.Window.new()
	self.scroll1 = lgui.ScrolledWindow.new()
	self.scroll2 = lgui.ScrolledWindow.new()
	
	self.scroll1:set("hscrollbar-policy", lgui.POLICY_AUTOMATIC,
		"vscrollbar-policy", lgui.POLICY_AUTOMATIC)
		
	self.scroll2:set("hscrollbar-policy", lgui.POLICY_AUTOMATIC,
		"vscrollbar-policy", lgui.POLICY_AUTOMATIC)
	
	self.libLabel = lgui.Label.new("Library")
	self.libEntry = lgui.Entry.new()
	self.libEntry:set("text", "lgui")
	self.libButton = lgui.Button.newWithMnemonic("Load")
	
	self.libButton:connect("clicked", 
		function(self)
			self:populate(self.libEntry:get("text"))
		end
	, self
	)
	
	self.libHbox = lgui.HBox.new(false, 0)
	self.libHbox:packStart(false, false, 10, self.libLabel, self.libEntry)
	self.libHbox:packStart(false, false, 0, self.libButton)

	self.classesModel = lgui.TreeStore.new("gchararray")
	self.classesView = lgui.TreeView.newWithModel(self.classesModel)
	self.classesCol = lgui.TreeViewColumn.new("Members", lgui.CellRendererText.new(), "text", 0, true)
	self.iter = lgui.TreeIter.new()
	self.dad = lgui.TreeIter.new()
	
	self.constsModel = lgui.TreeStore.new("gchararray")
	self.constsView = lgui.TreeView.newWithModel(self.constsModel)
	self.constsCol = lgui.TreeViewColumn.new("Constants", lgui.CellRendererText.new(), "text", 0, true)
	
	self.classesView:appendColumn(self.classesCol)
	self.classesCol:set("sort-indicator", false, "sort-order", lgui.SORT_DESCENDING)
	self.constsView:appendColumn(self.constsCol)

	self.scroll1:add(self.classesView)
	self.scroll2:add(self.constsView)
	
	self.hbox = lgui.HBox.new(true, 0)
	self.vbox = lgui.VBox.new(false, 5)

	self.hbox:add(self.scroll1, self.scroll2)
	self.vbox:packStart(false, true, 0, self.libHbox)
	self.vbox:packStart(true, true, 0, self.hbox)
	self.window:add(self.vbox)

	self.window:set("window-position", lgui.WIN_POS_CENTER, "title", "Navigator")
	self.window:set("width-request", 700, "height-request", 450)
	self.window:connect("delete-event", lgui.quit)
	self.window:showAll()
	
	self:populate()
	
	return self
end 

function Navigator:populate(libname)
	local iter, dad = self.iter, self.dad
	
	-- Load the library
	local lib = require(libname or "lgui")
	
	-- Reset the models
	self.classesView:set("model", nil)
	self.classesModel:clear()
	self.constsView:set("model", nil)
	self.constsModel:clear()
	
	-- Disable the sorting
	lgui.TreeSortable.setSortColumnId(self.classesModel, 
		lgui.TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID, lgui.SORT_ASCENDING)
	lgui.TreeSortable.setSortColumnId(self.constsModel, 
		lgui.TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID, lgui.SORT_ASCENDING)
	
	
	for i, j in pairs(lib) do
		-- Classes
		if type(j) == "table" then
			self.classesModel:append(dad)
			self.classesModel:set(dad, 0, i)
		
			-- Methods
			for o, k in pairs(j) do
				self.classesModel:append(iter, dad)
				self.classesModel:set(iter, 0, o)
			end
		-- Constants
		elseif type(j) == "number" then
			self.constsModel:append(iter)
			self.constsModel:set(iter, 0, i)
		-- Functions
		elseif type(j) == "function" then
			self.classesModel:append(iter)
			self.classesModel:set(iter, 0, i)
		end
	end
	
	-- Enable the Sort
	lgui.TreeSortable.setSortColumnId(self.classesModel, 0, lgui.SORT_ASCENDING)
	lgui.TreeSortable.setSortColumnId(self.constsModel, 0, lgui.SORT_ASCENDING)
	
	-- Set the models again
	self.classesView:set("model", self.classesModel)
	self.constsView:set("model", self.constsModel)
end

local nav = Navigator.new()
lgui.main()
