#! /usr/bin/env lua

require("lgui")

---
-- Simple text editor.
Editor = {}
local __BACK, __FORWARD, __CLOSE = 0, 1, 2

---
-- Constructor
--
-- @return A new Editor instance
function Editor.new()
	local self = {}
	setmetatable(self, {__index = Editor})

	self.window = lgui.Window.new()
	self.vbox = lgui.VBox.new(false, 0)
	self.scroll = lgui.ScrolledWindow.new()
	self.scroll:set("hscrollbar-policy", lgui.POLICY_AUTOMATIC,
		"vscrollbar-policy", lgui.POLICY_AUTOMATIC)
	self.statusbar = lgui.Statusbar.new()
	self.context = self.statusbar:getContextId("default")
	self.statusbar:push(self.context, "Untitled document")

	-- Find dialog
	self.findDialog, self.findEntry = self:buildFindDialog()

	-- Text
	self.view = lgui.TextView.new()
	self.buffer = self.view:get("buffer")
	self.buffer:cast(lgui.TextBuffer)

	-- Clipboard
	self.clip = lgui.Clipboard.get(lgui.Atom.intern("CLIPBOARD"))

	-- File dialog
	self.dialog = lgui.FileChooserDialog.new("Select the file", self.window,
		lgui.FILE_CHOOSER_ACTION_OPEN, "gtk-cancel", lgui.RESPONSE_CANCEL,
		"gtk-ok", lgui.RESPONSE_OK)

	-- AccelGroup
	self.accelGroup = lgui.AccelGroup.new()
	self.window:addAccelGroup(self.accelGroup)

	-- ActionGroup
	self.actionGroup = lgui.ActionGroup.new("Default")

	-- Actions
	self.aNewFile = lgui.Action.new("New file", nil, "Create a new file", "gtk-new")
	self.aNewFile:connect("activate", self.newFile, self)
	self.actionGroup:addActionWithAccel(self.aNewFile)
	self.aNewFile:setAccelGroup(self.accelGroup)

	self.aLoadFile = lgui.Action.new("Open file", nil, "Open a file", "gtk-open")
	self.aLoadFile:connect("activate", self.loadFile, self)
	self.actionGroup:addActionWithAccel(self.aLoadFile)
	self.aLoadFile:setAccelGroup(self.accelGroup)

	self.aSaveFile = lgui.Action.new("Save file", nil, "Save the current file", "gtk-save")
	self.aSaveFile:connect("activate", self.saveFile, self)
	self.actionGroup:addActionWithAccel(self.aSaveFile)
	self.aSaveFile:setAccelGroup(self.accelGroup)

	self.aFind = lgui.Action.new("Find", nil, "Searchs the text", "gtk-find")
	self.aFind:connect("activate", self.showFind, self)
	self.actionGroup:addActionWithAccel(self.aFind)
	self.aFind:setAccelGroup(self.accelGroup)

	self.aQuit = lgui.Action.new("Quit", nil, "Quit from the application", "gtk-quit")
	self.aQuit:connect("activate", lgui.quit)
	self.actionGroup:addActionWithAccel(self.aQuit)
	self.aQuit:setAccelGroup(self.accelGroup)

	self.aCut = lgui.Action.new("Cut", nil, "Cut the selection to the clipboard", "gtk-cut")
	self.aCut:connect("activate", self.cut, self)
	self.actionGroup:addActionWithAccel(self.aCut)
	self.aCut:setAccelGroup(self.accelGroup)

	self.aCopy = lgui.Action.new("Copy", nil, "Copy the selection to the clipboard", "gtk-copy")
	self.aCopy:connect("activate", self.copy, self)
	self.actionGroup:addActionWithAccel(self.aCopy)
	self.aCopy:setAccelGroup(self.accelGroup)

	self.aPaste = lgui.Action.new("Paste", nil, "Paste the clipboard info to the buffer", "gtk-paste")
	self.aPaste:connect("activate", self.paste, self)
	self.actionGroup:addActionWithAccel(self.aPaste)
	self.aPaste:setAccelGroup(self.accelGroup)

	self.aDelete = lgui.Action.new("Delete", nil, "Delete the selection", "gtk-delete")
	self.aDelete:connect("activate", self.delete, self)
	self.actionGroup:addActionWithAccel(self.aDelete)
	self.aDelete:setAccelGroup(self.accelGroup)

	self.aAbout = lgui.Action.new("About", nil, "About this application...", "gtk-about")
	self.aAbout:connect("activate", self.showAbout, self)
	self.actionGroup:addActionWithAccel(self.aAbout, "F1")
	self.aAbout:setAccelGroup(self.accelGroup)

	-- Toolbar
	self.toolbar = lgui.Toolbar.new()
	self.toolbar:set("toolbar-style", lgui.TOOLBAR_ICONS)
	self.toolbar:add(self.aNewFile:createToolItem(), lgui.SeparatorToolItem.new(),
		self.aLoadFile:createToolItem(), self.aSaveFile:createToolItem(),
		self.aFind:createToolItem(), lgui.SeparatorToolItem.new(),
		self.aAbout:createToolItem())

	-- Menu
	self.menubar = lgui.MenuBar.new()
	self.file = lgui.Menu.new()
	self.fileItem = lgui.MenuItem.newWithMnemonic("_File")
	lgui.MenuShell.append(self.file, self.aNewFile:createMenuItem(),
		lgui.SeparatorMenuItem.new(), self.aLoadFile:createMenuItem(),
		self.aSaveFile:createMenuItem(), self.aFind:createMenuItem(),
		lgui.SeparatorMenuItem.new(), self.aQuit:createMenuItem())
	self.fileItem:setSubmenu(self.file)

	self.edit = lgui.Menu.new()
	self.editItem = lgui.MenuItem.newWithMnemonic("_Edit")
	self.editItem:connect("activate", self.updateActions, self)

	lgui.MenuShell.append(self.edit, self.aCut:createMenuItem())
	lgui.MenuShell.append(self.edit, self.aCopy:createMenuItem())
	lgui.MenuShell.append(self.edit, self.aPaste:createMenuItem())
	lgui.MenuShell.append(self.edit, self.aDelete:createMenuItem())
	self.editItem:setSubmenu(self.edit)

	self.help = lgui.Menu.new()
	self.helpItem = lgui.MenuItem.newWithMnemonic("_Help")
	lgui.MenuShell.append(self.help, self.aAbout:createMenuItem())
	self.helpItem:setSubmenu(self.help)

	lgui.MenuShell.append(self.menubar, self.fileItem, self.editItem, self.helpItem)

	-- Packing it!
	self.vbox:packStart(false, false, 0, self.menubar)
	self.vbox:packStart(false, false, 0, self.toolbar)
	self.scroll:add(self.view)
	self.vbox:packStart(true, true, 0, self.scroll)
	self.vbox:packStart(false, false, 0, self.statusbar)
	self.window:add(self.vbox)

	-- Set the hooks
	lgui.AboutDialog.setEmailHook(self.openUri, {self, "mailto:"})
	lgui.AboutDialog.setUrlHook(self.openUri, {self})

	-- About dialog

	self.about = lgui.AboutDialog.new()
	self.logo = lgui.Pixbuf.newFromFile("icon.png")
	self.about:set("program-name", "Simple editor", "authors",
		{"Lucas Hermann Negri <kkndrox@gmail.com>"}, "comments", "Example app!",
		"license", "LGPL 3+", "logo", self.logo, "title", "About...",
		"website", "http://oproj.tuxfamily.org", "window-position",
		lgui.WIN_POS_CENTER)

	self.window:set("title", "Simple editor", "width-request", 500,
		"height-request", 520, "window-position", lgui.WIN_POS_CENTER)

	self.window:connect("delete-event", lgui.quit)
	self.view:focus()

	return self
end

---
-- Builds a find dialog.
--
-- @return The dialog, the entry widget
function Editor:buildFindDialog()
	local dialog = lgui.Dialog.new()
	local vbox = dialog:getVBox()
	local entry = lgui.Entry.new()
	vbox:packStart(true, false, 5, lgui.Label.new("Search for:"))
	vbox:packStart(true, true, 5, entry)
	vbox:showAll()

	dialog:set("title", "Find", "window-position", lgui.WIN_POS_CENTER)
	dialog:addButton("gtk-go-back", __BACK, "gtk-go-forward", __FORWARD,
		"gtk-close", __CLOSE)

	return dialog, entry
end

---
-- Shows the about dialog
function Editor:showAbout()
	self.about:run()
	self.about:hide()
end

---
-- Logs a message in the statusbar
--
-- @param msn Message to log
function Editor:log(msg)
	self.statusbar:pop(self.context)
	self.statusbar:push(self.context, msg)
end

---
-- Loads the contents of a file and shows it in the editor
function Editor:loadFile()
	self.dialog:set("action", lgui.FILE_CHOOSER_ACTION_OPEN)
	local res = self.dialog:run()
	self.dialog:hide()

	if res == lgui.RESPONSE_OK then
		local fileName = lgui.FileChooser.getFilename(self.dialog)
		local file = io.open(fileName)
		self.buffer:set("text", file:read("*a"))
		file:close()

		self.openedFile = fileName
		fileName = lobj.Path.getBasename(fileName)
		self:log("Loaded from " .. fileName)
	end
end

---
-- Saves the content of the editor into a file
function Editor:saveFile()
	local destination

	if self.openedFile then
		destination = self.openedFile
	else
		self.dialog:set("action", lgui.FILE_CHOOSER_ACTION_SAVE)
		local res = self.dialog:run()
		self.dialog:hide()

		if res == lgui.RESPONSE_OK then
			destination = lgui.FileChooser.getFilename(self.dialog)
		end
	end

	if destination then
		self:doSave(destination)
	end
end

---
-- Do the real save.
--
-- @param destination File that will receive the buffer text
function Editor:doSave(destination)
	local file = io.open(destination, "w")
	file:write(self.buffer:get("text"))
	file:close()
	self.openedFile = destination
	self:log("Saved to " .. lobj.Path.getBasename(destination))
end

---
-- Creates a new document
function Editor:newFile()
	self.buffer:set("text", "")
	self.openedFile = nil
	self:log("Untitled document")
end

---
-- Copy the selected info to the clipboard
function Editor:copy()
	self.buffer:copyClipboard(self.clip)
end

---
-- Cut the selected info to the clipboard
function Editor:cut()
	self.buffer:cutClipboard(self.clip, true)
end

---
-- Paste the clipboard info to the buffer
function Editor:paste()
	self.buffer:pasteClipboard(self.clip, nil, true)
end

---
-- Delete the selected info from the buffer
function Editor:delete()
	self.buffer:deleteSelection(true, true)
end

---
-- Shows the find dialog
function Editor:showFind()
	local iter, istart, iend = lgui.TextIter.new(), lgui.TextIter.new(), lgui.TextIter.new()
	self.buffer:getStartIter(iter)

	-- Run until closed
	while true do
		local res = self.findDialog:run()

		if res == __BACK or res == __FORWARD then
			local found, text, li = false, self.findEntry:get("text")

			if res == __BACK then
				if mark1 then self.buffer:getIterAtMark(iter, mark1) end
				found = lgui.TextIter.backwardSearch(iter, text, 0, istart, iend)
			else
				if mark2 then self.buffer:getIterAtMark(iter, mark2) end
				found = lgui.TextIter.forwardSearch(iter, text, 0, istart, iend)
			end

			-- Found the text?
			if found then
				self.buffer:selectRange(istart, iend)
				mark1 = self.buffer:createMark("last_start", istart, false)
				mark2 = self.buffer:createMark("last_end", iend, false)
				self.view:scrollToMark(mark1)
				found = false
			end
		else
			break
		end
	end

	self.findDialog:hide()
end

---
-- Opens the default web browser / email client
function Editor:openUri(uri)
	local self, extra = unpack(self)
	local total = (extra or "") .. uri
	
	print("Opening:", total)
	lgui.showUri(total)
end

---
-- Updates the actions.
function Editor:updateActions()
	local selected = self.buffer:getSelectionBounds()
	local paste = self.clip:isTextAvailable()

	self.aCut:set("sensitive", selected)
	self.aDelete:set("sensitive", selected)
	self.aCopy:set("sensitive", selected)
	self.aPaste:set("sensitive", paste)
end

-- * Main *
local inst = Editor.new()
inst.window:showAll()
lgui.main()
