/*
 * File Name: main.c
 */

/*
 * This file is part of hello_world.
 *
 * hello_world is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * hello_world is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <glib.h>
#include <gtk/gtk.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>

// ereader include files, between < >

// local include files, between " "
#include "log.h"
#include "i18n.h"
#include "ipc.h"
#include "main.h"
#include "menu.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

static const char  *rc_filename = DATADIR "/" PACKAGE_NAME ".rc";


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GtkWidget        *g_hello = NULL;        // screen text "hello world"
static GtkWidget        *g_menu  = NULL;        // latest menu selection

static viewtypes_t      g_viewtype = -1;        // currently selected view type


//============================================================================
// Local Function Definitions
//============================================================================

static void     display_view_type       ( void );

static void     main_set_text           ( void );

static gboolean on_startup_complete     (gpointer data);


//============================================================================
// Functions Implementation
//============================================================================

// print usage text and quit
static void usage (const char *argv_0)
{
    static const char *usage_text = 
                        "\n"
                        "usage: %s [options]\n"
                        "\n"
                        "options:\n"
                        "    --help\n"
                        "        Print help text and quit\n";

    printf(usage_text, argv_0);

    exit(1);
}

// get command-line options
static void parse_arguments (int argc, char **argv)
{
    int  i;

    // parse options
    for (i = 1 ; i < argc ; i++)
    {
        if (strcmp(argv[i], "--help") == 0)
        {
            usage(argv[0]);
        }
    }
}


// set locale (language)
void main_set_locale (const char *locale)
{
    LOGPRINTF("entry: locale [%s]", locale);

    g_setenv("LANG", locale, TRUE);
    setlocale(LC_ALL, "");

    main_set_text();
    menu_set_text();
}


// create main screen layout
static GtkWidget *create_screen_layout (void)
{
    GtkWidget   *background = NULL;  // return value
    GtkWidget   *widget;
    GtkBox      *vbox;

    LOGPRINTF("entry");

    // object hierarchy:
    //     background (alignment)
    //       |
    widget = gtk_alignment_new(0.0, 0.0, 1.0, 1.0);
    gtk_alignment_set_padding( GTK_ALIGNMENT(widget), 100, 100, 100, 100);
    gtk_widget_show(widget);
    background = widget;
    //       |
    //       |-- vbox
    //             |
    widget = gtk_vbox_new(FALSE, 100);
    gtk_container_add(GTK_CONTAINER(background), widget);
    gtk_widget_show(widget);
    vbox = GTK_BOX(widget);
    //             |
    //             |-- hello (label)
    //             |
    widget = gtk_label_new(NULL);
    gtk_widget_set_name(widget, "irex-hello-world");
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_show(widget);
    g_hello = widget;
    //             |
    //             |-- menu (label)
    widget = gtk_label_new(NULL);
    gtk_widget_set_name(widget, "irex-hello-menu");
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_show(widget);
    g_menu = widget;
    //             |
    //             |-- volume (label)
    widget = gtk_label_new(NULL);
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_show(widget);
    g_volume = widget;
    //             |
    //             |-- action (label)
    widget = gtk_label_new(NULL);
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_show(widget);
    g_action = widget;
    
    main_set_text();
    
    return background;
}


// set screen texts
static void main_set_text (void)
{
    LOGPRINTF("entry");

    if (g_hello)
    {
        gtk_label_set_text( GTK_LABEL(g_hello), _("Hello world") );
    }
    display_view_type();
}


// show selected view type
void main_set_view_type ( viewtypes_t view_type )
{
    LOGPRINTF("entry: view_type [%d]", view_type);
    g_return_if_fail( view_type >= 0  &&  view_type < N_VIEWTYPES);

    g_viewtype = view_type;
    display_view_type();
    //menu_select_view_type(view_type);
}


// show selected view type
static void display_view_type ( void )
{
    gchar       *msg  = NULL;
    gchar       *type = NULL;

    LOGPRINTF("entry: view_type [%d]", g_viewtype);

    if (g_menu)
    {
        switch (g_viewtype)
        {
            case ICONVIEW:
                type = _("thumbnails");
                break;
            case LISTVIEW:
                type = _("details");
                break;
            default:
                type = _("unknown");
        }

        msg = g_strdup_printf( _("Current view type: %d (%s)"), g_viewtype, type);
        gtk_label_set_text( GTK_LABEL(g_menu), msg );
        g_free(msg);
    }
}


// terminate application
void main_quit (void)
{
    WARNPRINTF("entry");
    
    menu_destroy();
    
    if (g_main_window)
    {
        gtk_widget_destroy(g_main_window);
        g_main_window = NULL;
    }    
    
    if (gtk_main_level() > 0)
    {
        WARNPRINTF("quit main loop");
        gtk_main_quit();
    }
    else
    {
        WARNPRINTF("no main loop to quit, exit directly");
        _exit(0);
    }
}


// terminate signal
static void on_sigterm (int signo)
{
    WARNPRINTF("    -- entry " PACKAGE_NAME ", my pid [%d]", getpid());

    // stop main process, prepare to quit application
    main_quit();

    WARNPRINTF("    -- leave " PACKAGE_NAME);
}


// report startup completed
static gboolean on_startup_complete (gpointer data)
{
    LOGPRINTF("entry");

    ipc_sys_startup_complete();

    return FALSE; // remove timer source
}


// main function
int old_main (int argc, char *argv[])
{
    GtkWidget        *window;
    GtkWidget        *widget;
    struct sigaction on_term;

    // parse command-line arguments
    parse_arguments(argc, argv);

    // catch the SIGTERM signal
    memset(&on_term, 0x00, sizeof(on_term));
    on_term.sa_handler = on_sigterm;
    sigaction(SIGTERM, &on_term, NULL);
#if LOGGING_ON
    sigaction(SIGINT,  &on_term, NULL);
#endif

    // init gtk, list the default rc files
    gtk_init(&argc, &argv);
    gchar** files = gtk_rc_get_default_files();
    while( *files )
    {
        WARNPRINTF("gtk_rc_get_default_files [%s]", *files);
        files++;
    }

    // init modules
    g_mountpoint = NULL;
    ipc_set_services();
    menu_init();

    // open the RC file associated with this program
    WARNPRINTF("gtk_rc_parse [%s]", rc_filename);
    gtk_rc_parse(rc_filename);

    // create the top level window 
    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(window), PACKAGE " " VERSION);
    gtk_window_set_resizable(GTK_WINDOW(window), FALSE);
    gtk_container_set_border_width(GTK_CONTAINER(window), 0);
    g_main_window = window;

    // add screen details
    widget = create_screen_layout();
    gtk_container_add(GTK_CONTAINER(window), widget);
    main_set_view_type(ICONVIEW);

    // make sure that everything, window and label, are visible 
    gtk_widget_show(window);

    if (argc > 0)
    {
        gchar *msg = NULL;
        msg = g_strdup_printf("Started with: %s", argv[1]);
        gtk_label_set_text( GTK_LABEL(g_action), msg);
        LOGPRINTF("%s", msg);
        g_free(msg);
    }
    
    // tell system daemon we are ready to go
    g_idle_add(on_startup_complete, NULL);
    
    // run the main loop
    LOGPRINTF("before gtk_main");
    gtk_main();
    LOGPRINTF("after gtk_main");

    // clean up modules

    return 0;
}


// run error dialog
void run_error_dialog (const gchar *msg)
{
    GtkWidget   *dialog = NULL;

    ERRORPRINTF("entry: msg [%s]", msg);

    dialog = gtk_message_dialog_new( GTK_WINDOW(g_main_window),
                                     GTK_DIALOG_DESTROY_WITH_PARENT,
                                     GTK_MESSAGE_ERROR,
                                     GTK_BUTTONS_OK,
                                     msg );

    gtk_dialog_run( GTK_DIALOG(dialog) );
    gtk_widget_destroy( dialog );
}
