/*
 * File Name: ipc_launch.c
 */

/*
 * This file is part of hello_world.
 *
 * hello_world is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * hello_world is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <stdio.h>
#include <stdlib.h>
#include <gdk/gdkx.h>
#include <sys/types.h>
#include <unistd.h>

// ereader include files, between < >
#include <liberipc/eripc.h>

// local include files, between " "
#include "log.h"
#include "i18n.h"
#include "ipc.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

// IPC application
// TODO: set DBUS_APPL_NAME to your application name, use lower-case and digits only
//       or set to PACKAGENAME when this is lower-case and digits only
#define DBUS_APPL_NAME                  "launch"            // lower-case and digits only
#define DBUS_SERVICE                     "com.irexnet." DBUS_APPL_NAME
#define DBUS_PATH                       "/com/irexnet/" DBUS_APPL_NAME
#define DBUS_INTERFACE                   "com.irexnet." DBUS_APPL_NAME

// IPC system control
#define DBUS_SERVICE_SYSTEM_CONTROL     "com.irexnet.sysd"

//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static eripc_context_t  *eripcContext = NULL;


//============================================================================
// Local Function Definitions
//============================================================================

//============================================================================
// Functions Implementation
//============================================================================


//----------------------------------------------------------------------------
// Generic
//----------------------------------------------------------------------------

// initialise
void ipc_set_services_launch (void)
{
    LOGPRINTF("entry");
    g_assert(eripcContext == NULL);

    // initialise IPC context
    eripcContext = eripc_init(DBUS_APPL_NAME, "1.0", NULL);
    if (eripcContext == NULL) 
    {
        ERRORPRINTF("Failed to initialize eripc context\n");
        exit(1);
    }

}


// un-initialise
void ipc_unset_services_launch (void)
{
}


//----------------------------------------------------------------------------
// System control
//----------------------------------------------------------------------------

// report "application started"
void ipc_sys_startup_complete_launch ( void )
{
    eripc_error_t   result;

    // parameters for ipc message
    const int       my_pid = getpid();
    //const int       my_xid = GDK_WINDOW_XID(g_main_window->window);
    const int       my_xid = 0;
    const gboolean  is_multidoc = TRUE;        // TODO: set TRUE/FALSE depending on your application

    LOGPRINTF("entry");
    g_assert( eripcContext );

    // broadcast signal over session bus
    result = eripc_send_signal_varargs( eripcContext,
                                        ERIPC_BUS_SESSION,
                                        DBUS_PATH,
                                        DBUS_SERVICE_SYSTEM_CONTROL,
                                        "startupComplete",
                                        ERIPC_TYPE_STRING, DBUS_APPL_NAME,
                                        ERIPC_TYPE_INT,    my_pid,
                                        ERIPC_TYPE_BOOL,   is_multidoc,
                                        ERIPC_TYPE_STRING, DBUS_SERVICE,
                                        ERIPC_TYPE_INT,    my_xid,
                                        ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
    } 
}


// start task (application)
gboolean ipc_sys_start_task_launch ( const gchar  *cmd_line,
                              const gchar  *work_dir,
                              const gchar  *label,
                              const gchar  *thumbnail_path )
{
    gboolean            ok = TRUE;    // return value
    eripc_error_t       result;
    int                 old_timeout = 0;
    gboolean            timeout_set = FALSE;

    WARNPRINTF("entry: cmd_line [%s] work_dir [%s] label [%s] thumbnail_path [%s]",
                       cmd_line,     work_dir,     label,     thumbnail_path       );
    g_assert( eripcContext );
    g_assert( cmd_line && *cmd_line );

    // set busy dialog
    //ipc_menu_busy_show( TRUE );

    // set ipc timeout to "long"
    eripc_error_t eripc_get_timeout (eripc_context_t *context, int *timeout);
    result = eripc_get_timeout(eripcContext, &old_timeout);
    if (result == ERIPC_ERROR_SUCCESS)
    {
        result = eripc_set_timeout(eripcContext, 60*1000);
        if (result == ERIPC_ERROR_SUCCESS)
        {
            timeout_set = TRUE;
        }
    }

    // broadcast signal over session bus
    result = eripc_send_varargs_and_wait( eripcContext,
                                          NULL,               // no reply output
                                          ERIPC_BUS_SESSION,
                                          DBUS_SERVICE_SYSTEM_CONTROL,
                                          "startTask",
                                          ERIPC_TYPE_STRING, cmd_line,
                                          ERIPC_TYPE_STRING, work_dir,
                                          ERIPC_TYPE_STRING, label,
                                          ERIPC_TYPE_STRING, thumbnail_path,
                                          ERIPC_TYPE_INVALID );
    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs_and_wait returns [%d]", result);
        ok = FALSE;
    }

    // restore ipc timeout
    if (timeout_set)
    {
        eripc_set_timeout(eripcContext, old_timeout);
    }

    // remove busy dialog
    //ipc_menu_busy_show( FALSE );

    return ok;
}


// stop task (application)
gboolean ipc_sys_stop_task ( const gchar  *cmd_line )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    WARNPRINTF("entry: cmd_line [%s]", cmd_line );
    g_assert( eripcContext );
    g_assert( cmd_line && *cmd_line );

    // broadcast signal over session bus
    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_SYSTEM_CONTROL,
                                 "stopTask",
                                 ERIPC_TYPE_STRING, cmd_line,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    }

    return ok;
}
