// Copyright (C) 2008 Simon Mendoza
// This file is part of gtk-sudoku.
//
// gtk-sudoku is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// gtk-sudoku is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with gtk-sudoku.  If not, see <http://www.gnu.org/licenses/>.

#ifndef SUDOKU_H
#define SUDOKU_H

#include <gtk/gtk.h>
#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdbool.h>

#define SUDOKU_SQUARE(s) ((Square *)(s))

#define FILE_SIZE      100         // for absolute file names

char       save_file[FILE_SIZE];   // file name in use

GtkWidget *window;                 // parent for popup windows
GtkWidget *statusbar;
GMutex    *mutex;                  // provides protection for statusbar
char      *status_bar_msg;         // to be freed within a thread

int        BLOCK_SIZE;             // NxN blocks within sudoku game
int        GRID_SIZE;              // NxN sudoku game
int        SQUARE_SIZE;            // sudoku square in pixels

int        empty_squares;          // keeps track of when game is over
bool       designing_sudoku;       // flag that user is creating a sudoku
bool       game_started;

struct SudokuSquare {
	GtkWidget *eventbox;
	GtkWidget *entry;
	GtkWidget *label;
	guint      editable : 1;
};
typedef struct SudokuSquare Square;

struct {
	Square  **square;              // top left square of sudoku game
	Square   *last_square_clicked;
} sudoku;

/*
 * Opens up a modal dialog window with given TITLE and MESSAGE.
 */
void     popup_msg           (const char *title,
			                  const char *message);
/*
 * Triggered when user clicks a sudoku square. (button_press_event)
 * single click: Hides the last-clicked square's entry and brings the focus 
 *               to the square clicked.
 * middle click: Erases the square's value.
 * right click : If the user single clicks on a square and then right clicks
 *               a different one, then the value of the previous square will
 *               get duplicated on the new square.
 * @WIDGET the eventbox associated with a sudoku square
 * @DATA the sudoku square struct
 */
gboolean on_click            (GtkWidget      *widget, 
						      GdkEventButton *event, 
						      gpointer        data);
/*
 * "Lights up" the sudoku square when mouse hovers over it.(enter_notify_event)
 * @WIDGET the sudoku square's entry
 * @DATA the eventbox associated with the sudoku square
 */
gboolean on_enter            (GtkWidget *widget, 
						      GdkEvent  *event, 
						      gpointer   data);
/*
 * Reverses the on_enter() method and brings the sudoku square back to its
 * normal color when mouse isn't hovering over it. (leave_notify_event)
 * @WIDGET the sudoku square entry
 * @DATA the eventbox associated with the sudoku square
 */
gboolean on_leave            (GtkWidget *widget, 
						      GdkEvent  *event, 
						      gpointer   data);
/*
 * Callback function when program is closed.
 * Frees up memory already allocated for **grid and **sudoku.square arrays.
 * And if gkt_main() has already begun, it then makes a call to gtk_main_quit().
 * @WIDGET not explicitly used, but is the main window's destroy callback
 * @DATA not used
 */
void     on_close            (GtkWidget *widget, 
						      gpointer   data);

void     verify_quit           (GtkWidget *widget,
                                                      gpointer   data);

/*
 * Callback for ENTRY's backspace signal. Aside from erasing the text within
 * ENTRY, the text in the label associated with ENTRY's sudoku square is
 * also erased. Also ups the number of empty squares.
 * @ENTRY is a sudoku square's entry
 * @DATA not used
 */
void     on_backspace        (GtkEntry *entry, 
						      gpointer  data);
/*
 * Called by correct_sudoku() method to verify if each of the rows contain
 * all values (1-9 for regular sudoku,0-9a-f for super sudoku). Returns true
 * if all rows check out, false otherwise.
 * @VALUES is an non-initialized array of size 9/16 (depending on game size)
 */
bool     rows_check_out      (bool *values);
/*
 * Called by correct_sudoku() method to verify if each of the cols contain
 * all values (1-9 for regular sudoku,0-9a-f for super sudoku). Returns true
 * if all cols check out, false otherwise.
 * @VALUES is an non-initialized array of size 9/16 (depending on game size)
 */
bool     cols_check_out      (bool *values);
/*
 * Should be called when the number of empty squares is 0. Checks the status
 * of the sudoku grid. Returns true if the sudoku is correctly solved,
 * otherwise false. May also return false if there is insufficient memory, in
 * which case it also outputs the appropriate message to stderr.
 */
bool     correct_sudoku      (void);
/*
 * Callback to each sudoku square's entry's insert-text signal. Allows only
 * proper values (1-9/0-9a-f) to be displayed. Also updates the value shown 
 * if another value is inserted, e.g. user presses 3, then presses 7.
 * EDITABLE is updated after this function ends.
 * @EDITABLE is the entry associated with the sudoku square
 * @NUMBER is the value that is inserted by user
 * @LENGTH not used
 * @POSITION is an index corresponding to where the value is being inserted;
 * if this value is 1, then it is set back to 0 so that it replaces the current
 * value
 * @DATA is the sudoku square associated with EDITABLE
 */
void     on_insert           (GtkEditable *editable,
					          gchar       *number, 
					          gint         length, 
					          gint        *position,
					          gpointer     data);

void     save_as             (GtkWidget *widget, 
						      gpointer   data);

void     save_game           (GtkWidget *toolbutton, 
						      gpointer   data);

void     open_game           (GtkWidget *toolbutton, 
						      GtkWidget *done_btn);

gboolean load_file           (char *input_file);

gboolean make_sudoku_board   (GtkBox *box);

gpointer clear_statusbar_msg(gpointer time);

void     solve_game         (GtkToolButton *toolbutton, 
						     gpointer       data);

void     reset_game         (GtkToolButton *toolbutton, 
						     gpointer       data);

void     make_toolbar       (GtkBox    *box,
							 GtkWidget *done_btn);
void     make_menubar       (GtkBox *box);

#endif
