#!/bin/sh
#
# A script for creating a PDF file from plain text, using latex.
#
# Needs Latex, dvipdfm, evince, bunzip2, unzip, unrtf, html2text and 
# the standard gnu command line tools (gawk, sed etc... )
#
# Perttu Haimi 2009
#
# The author or authors of this code dedicate any and all copyright 
# interest in this code to the public domain. We make this dedication 
# for the benefit of the public at large and to the detriment of our 
# heirs and successors. We intend this dedication to be an overt act of 
# relinquishment in perpetuity of all present and future rights to this 
# code under copyright law.

ECHO=/bin/echo


file="$1" 
$ECHO -E "$file"

if [ ! -e "$file" ] ;
then
    $ECHO -E "$file does not exist."
    continue
fi

name=""

# convert some formats to plain text
if [ "`basename \"$file\"`" != "`basename \"$file\" .bz2`" ] ;
then
    name=`basename "$file" .bz2` 
    bunzip2 -c "$file" > "/tmp/$name"
    f="/tmp/$name"
elif [ "`basename \"$file\"`" != "`basename \"$file\" .zip`" ] ;
then
    name=`basename "$file" .zip` 
    unzip -qc "$file" > "/tmp/$name"
    f="/tmp/$name"
elif [ "`basename \"$file\"`" != "`basename \"$file\" .rtf`" ] ;
then
    name=`basename "$file" .rtf` 
    unrtf --text "$file" | grep -v "^###" > "/tmp/$name"
    f="/tmp/$name"
elif [ "`basename \"$file\"`" != "`basename \"$file\" .htm`" ] ;
then
    name=`basename "$file" .htm` 
    html2text "$file" > "/tmp/$name"
    f="/tmp/$name"
elif [ "`basename \"$file\"`" != "`basename \"$file\" .html`" ] ;
then
    name=`basename "$file" .html` 
    html2text "$file" > "/tmp/$name"
    f="/tmp/$name"
else
    f="$file"
fi

if [ "`basename \"$f\"`" != "`basename \"$f\" .txt`" ] ;
then
    name=`basename "$f" .txt` 
fi

if [ "x$name" = "x" ] ;
then
    $ECHO -E "Unknown file type: $file"
    continue
fi

$ECHO -E "\documentclass[landscape,twocolumn,12pt]{article}" > "/tmp/$name.tex"
# use for vertical mode
#$ECHO -E "\documentclass[portrait,12pt]{article}" > "/tmp/$name.tex"

$ECHO -E "\usepackage[left=1mm,top=1mm,right=1mm,bottom=3mm,nohead,nofoot]{geometry}" >> "/tmp/$name.tex"
$ECHO -E "\usepackage[dvips]{color}" >> "/tmp/$name.tex"
$ECHO -E "\usepackage[latin1]{inputenc}" >> "/tmp/$name.tex"

#other font possibilities: bookman, chancery, palatino, times
$ECHO -E "\usepackage{newcent}" >> "/tmp/$name.tex"

$ECHO -E "\geometry{papersize={205mm,120mm}}" >> "/tmp/$name.tex"
# use for vertical mode
#$ECHO -E "\geometry{papersize={120mm,205mm}}" >> "/tmp/$name.tex"

$ECHO -E "\definecolor{darkgray}{rgb}{0.25,0.25,0.25}" >> "/tmp/$name.tex"
$ECHO -E "\definecolor{whitish}{rgb}{0.86,0.86,0.8}" >> "/tmp/$name.tex"
$ECHO -E "\definecolor{bluish}{rgb}{0.54,0.83,0.55}" >> "/tmp/$name.tex"

# This special command instructs dvipdfm to format the page correctly
$ECHO -E "\special{pdf: pagesize width 205mm  height 120mm}" >> "/tmp/$name.tex"
# Use for vertical mode
#$ECHO -E "\special{pdf: pagesize width 120mm  height 205mm}" >> "/tmp/$name.tex"

$ECHO -E "\begin{document}" >> "/tmp/$name.tex"

# change the colors to white text on dark background
#$ECHO -E "\pagecolor{darkgray}" >> "/tmp/$name.tex"
#$ECHO -E "\color{white}" >> "/tmp/$name.tex"

#tr -d '\r' < "$f" > "/tmp/tmp.txt" | tr '\366' 'ö' 

# clean up the plain text file for latex
# escape latex special characters
tidy_text_mode.py "$f" | gawk '{print $0 "\n"}' | sed 's/\\/\\backslash/g' | sed 's/&/\\&/g' | sed 's/|/\\verb1\|1/g' | sed 's/\$/\\\$/g' | sed 's/\\backslash/$\\backslash$/g' | sed 's/%/\\%/g' | sed 's/#/\\#/g' | sed 's/_/\\_/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\^/\\\^/g' | sed 's/~/\\verb1~1/g'  >> "/tmp/$name.tex"

$ECHO -E "\end{document}" >> "/tmp/$name.tex"

$ECHO -E "\"$name\"" 
latex -interaction batchmode -output-directory /tmp "/tmp/$name"

dvipdfm -l -o "/tmp/$name.pdf" "/tmp/$name"
# use for vertical mode
#dvipdfm -o "/tmp/$name.pdf" "/tmp/$name"

evince "/tmp/$name.pdf" &

