#!/usr/local/bin/ruby
# Read Feedbooks subscriptions and download them
# (c) 2009 by Thomas Hollstegge <mail@hollstegge.de>

# Please configure this script in config.rb
require "config.rb"

require "rexml/document"
require "ftools"
require "fileutils"
require "net/http"
require "uri"
require "digest/md5"

# Some default values, please do not change them!

@tmpdir = "/tmp/feedbooks"
@download_history = "/mnt/settings/downloadhistory"
@registry_xml = "/mnt/settings/registry/registry.xml"
# Set fallback final dir. CAUTION: Final directory will be read from settings.xml later on,
# so changing this value won't work. If you want your own storage directory, set 
# @finaldir_force in config.rb instead.
@finaldir_default = "/mnt/free/newspapers/Feeds"

def debug(msg)
	puts msg
end

def download(url, filename)
	uri = URI(url)
	Net::HTTP.start(uri.host) do |http|
		request = Net::HTTP::Get.new(uri.request_uri)
		request.basic_auth @feedbooks_username, Digest::MD5.hexdigest(@feedbooks_password)
		response = http.request(request)
# 		response = http.get(uri.path)
		open(filename, "wb") do |file|
			file.write response.body
		end
	end
end

def get_manifest_xml(title, description, filename)
	return <<__XML
<?xml version="1.0" standalone="no"?>
<package>
	<metadata>
		<dc-metadata>
			<Title>#{title}</Title>
			<Description>#{description}</Description>
		</dc-metadata>
		<y-metadata>
			<version>000</version>
			<startpage>#{filename}</startpage>
		</y-metadata>
	</metadata>
</package>
__XML
end

def write_manifest(title, description, document_filename, manifest_filename)
	open(manifest_filename, "wb") do |file|
		file.write(get_manifest_xml(title, description, document_filename))
	end
end

###
# START
###


# First check if we set a forced download directory
if defined? @finaldir_force
	@finaldir = @finaldir_force
else
	# Set fallback directory in case something goes wrong
	@finaldir = @finaldir_default
	
	# Try to read newspapers location from settings.xml
	if File.exists? @registry_xml
		debug "Trying to find newspapers storage from #{@registry_xml}..."
		settings = REXML::Document.new(open(@registry_xml))
		el = settings.elements["/iLiadRegistry/categories/category[@type='NEWSPAPERS']/location"]
		if el != nil
			dir = el.text
			if !dir.eql?("") 
				debug "Setting found: #{dir}"
				@finaldir = dir + "/Feeds"
			end
		end
	end
end
debug "Feeds will be stored in #{@finaldir}"

# Make sure our target directory exists
File.makedirs @finaldir, @tmpdir

# begin
	# Download and read subscription list
	debug "Downloading subscriptions.xml..."
	url = "http://www.feedbooks.com/user/subscriptions.xml"
	download url, "#{@tmpdir}/subscriptions.xml"
	doc = REXML::Document.new(open("#{@tmpdir}/subscriptions.xml"))
	
	debug "Parsing subscriptions.xml..."
	
	subs = doc.elements["/subscriptions"]
	# For each subscription go through the following loop
	subs.each_element do |sub|
		url = sub.elements["url"].text
		title = sub.elements["dc:title"].text
		url =~ /\/(\d+\.\w+)/
		filename = $1
		
		debug "Feed found: '#{title}' (#{url})"
		
		# targetdir: Where fill our feed be stored?
		targetdir = "#{@tmpdir}/#{filename}"
		
		# Create target directory
		debug "Creating #{targetdir}..."
		File.makedirs targetdir
		
		# Download feed file
		debug "Downloading #{url}..."
		download url, "#{targetdir}/#{filename}"
		
		# Create manifest.xml
		debug "Done, creating manifest.xml"
		write_manifest title, "Last update: " + Time.new.to_s, filename, "#{targetdir}/manifest.xml"
		
		# Remove old feed versions
		if File.directory?(@finaldir + "/" + filename)
			debug "Removing old version of the feed..."
			FileUtils.rm_r @finaldir + "/" + filename
		end
		
		# Copy temporary files to the target directory
		debug "Copying files to #{@finaldir}..."
		FileUtils.cp_r targetdir, @finaldir
		
		begin
			# Add file to download history
			debug "Creating download history entry..."
			FileUtils.ln_s @finaldir + "/" + filename, @download_history, :force => true
		rescue Exception
			debug "Could not create symlink: "+ $!
		end
		
		# Remove temporary files
		debug "Removing temporary files..."
		FileUtils.rm_r targetdir
		debug "Done with '#{title}'"
	end
# rescue Exception
# 	puts "Error: " + $!
# end
debug "I'm through!"