/*
 *  linux/drivers/char/irex_er0100_battery.c
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 */

/*
 * Copyright 2006-2008 iRex Technologies B.V.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/major.h>
#include <linux/serial_reg.h>
#include <linux/time.h>
#include <linux/string.h>
#include <linux/types.h>
#include <linux/wait.h>
#include <linux/mm.h>
#include <linux/delay.h>

#include <asm/system.h>
#include <asm/segment.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/fcntl.h>
#include <asm/uaccess.h>

#include "battery.h"

// static void hdq16_irq_handler(int, void*, struct pt_regs*);
static int irex_er0100_ereader_battery_open(struct inode *, struct file *);
static int irex_er0100_ereader_battery_close(struct inode *, struct file *);
static int irex_er0100_ereader_battery_ioctl(struct inode *, struct file *, unsigned int, unsigned long);
static int irex_er0100_ereader_battery_write_cmd(u8 cmd);

static wait_queue_head_t irex_er0100_ereader_battery_waitq;

static u8 battery_pos;
static u8 battery_invalid;
static u16 battery_invalid_count = 0;
static u32 battery_value;
static u32 battery_irq_enable = 0;
/* BATTERY_INVALID is the default battery response when no reliable sample has been taken */
static u32 previous_charge = BATTERY_INVALID;  
static u32 previous_volt = BATTERY_INVALID; 
static u32 previous_time = BATTERY_INVALID; 
static u32 previous_status = BATTERY_INVALID;
static u32 previous_current = BATTERY_INVALID;

static void battery_irq_handler(int irq, void *dev_id, struct pt_regs *regs);

/* file operations */
static struct file_operations irex_er0100_ereader_battery_fops = 
{
	open:		irex_er0100_ereader_battery_open,
	release:	irex_er0100_ereader_battery_close,
	ioctl:		irex_er0100_ereader_battery_ioctl,
};

int irex_er0100_ereader_battery_init(void)
{
	/* register device */
	if (register_chrdev(IREX_ER0100_EREADER_BATTERY_MAJOR, "iRex ER0100 eReader Battery Gas Gauge Driver", &irex_er0100_ereader_battery_fops) < 0) 
	{
		printk(KERN_ERR  "iRex ER0100 eReader Battery Gas Gauge Driver: register_chrdev failed!\n");
		return -EIO;
	}
    printk("iRex ER0100 eReader Battery Gas Gauge Driver initialised\r\n");
	GPDR(BATTERY_GPIO) &= ~GPIO_bit(BATTERY_GPIO);

	/* initialize waiting queue */
	init_waitqueue_head(&irex_er0100_ereader_battery_waitq);

	/* request interrupt line */
	set_GPIO_IRQ_edge(BATTERY_GPIO,GPIO_FALLING_EDGE);
    request_irq(BATTERY_IRQ,&battery_irq_handler,0,"Battery Gas Gauge Response",NULL);

	return(0);
}

void  irex_er0100_ereader_battery_cleanup(void)
{
	free_irq(BATTERY_IRQ,NULL);
	unregister_chrdev(IREX_ER0100_EREADER_BATTERY_MAJOR,"irex ER0100 eReader Battery Gas Gauge Driver");
}

static int irex_er0100_ereader_battery_open(struct inode *inode, struct file *file)
{
	/* nothing for now */
	return 0;
}

static int irex_er0100_ereader_battery_close(struct inode *inode, struct file *file)
{
	/* nothing for now */
	return(0);
}

static void battery_irq_handler(int irq, void *dev_id, struct pt_regs *regs)
{
	u32 sample0, sample1, sample2, sample3, sample4;
	
	if(battery_irq_enable)
	{
		/* This sample should read 0. If not the interrupt latency was to large to get a valid reading */
		sample0= GPLR(BATTERY_GPIO) & GPIO_bit(BATTERY_GPIO);

		udelay(50); /* tDSUB, Interrupt latency will make sure that we are at least 50 us later than the falling edge */
		
		sample1 = GPLR(BATTERY_GPIO) & GPIO_bit(BATTERY_GPIO);
		udelay(5);  /* +55us */
		sample2 = GPLR(BATTERY_GPIO) & GPIO_bit(BATTERY_GPIO);
		udelay(5); /* +60us */
		sample3 = GPLR(BATTERY_GPIO) & GPIO_bit(BATTERY_GPIO);
		udelay(5); /* +65us */
		sample4 = GPLR(BATTERY_GPIO) & GPIO_bit(BATTERY_GPIO);
		if(sample0 == 0 && sample1 == sample2 && sample1 == sample3 && sample1 == sample4)
		{
			if(sample1)
			{
				battery_value |= (1 << battery_pos);
			}
		}
		else
		{
			battery_invalid = 1;
		}
		battery_pos++;
		if(battery_pos == 16)
		{
			wake_up_interruptible(&irex_er0100_ereader_battery_waitq);
		}
	}
}

static int irex_er0100_ereader_battery_write_cmd(u8 cmd)
{
	int i;
	long timeout;

	battery_value = 0;
	battery_pos = 0;
	battery_invalid = 0;

	GPDR(BATTERY_GPIO) |= GPIO_bit(BATTERY_GPIO);
	udelay(200); /* tB - low */

	GPDR(BATTERY_GPIO) &=~ GPIO_bit(BATTERY_GPIO);
	udelay(50); /* tBR - high */
	
	/* send command */
	for(i=0;i<7;i++)
	{
		GPDR(BATTERY_GPIO) |= GPIO_bit(BATTERY_GPIO);
		
		udelay(10); /* tSTRH - low */

		if((cmd & (1<<i))!=0)
		{
			GPDR(BATTERY_GPIO) &=~ GPIO_bit(BATTERY_GPIO);
			/* tDSU */
		}
		udelay(90); /* tDH = 90+10 = 100*/
	
		/* tSSU - high */
		GPDR(BATTERY_GPIO) &=~ GPIO_bit(BATTERY_GPIO);
		udelay(100); /* tCYCH */
	}

	/* read bit */
	GPDR(BATTERY_GPIO) |= GPIO_bit(BATTERY_GPIO);
	/* tSTRH - low */

	udelay(100); /* tDH */
	
	/* tSSU - high */
	GPDR(BATTERY_GPIO) &=~ GPIO_bit(BATTERY_GPIO);

	/* request interrupt line */
/*	set_GPIO_IRQ_edge(BATTERY_GPIO,GPIO_FALLING_EDGE);
    request_irq(BATTERY_IRQ,&battery_irq_handler,0,"Battery Gas Gauge Response",NULL);
*/
	/* Open IRQ for reponse processing */
	battery_irq_enable = 1;

	/* wait for response */
	timeout = interruptible_sleep_on_timeout(&irex_er0100_ereader_battery_waitq,3*HZ/100);
	if(timeout == 0)
	{
		/* Response is invalid because we did not receive 16 start conditions within 20ms */
		battery_invalid = 1;
	}
/*	free_irq(BATTERY_IRQ,NULL);
*/
	battery_irq_enable = 0;
	return 0;
}

static int irex_er0100_ereader_battery_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	int ret = 0;
	unsigned long turbo;

	if (_IOC_TYPE(cmd)!= BATTERY_IOCTL_BASE)
		return -EINVAL;

	switch(cmd)
	{
		case BATTERY_IOCTL_READ_CHARGE:
			irex_er0100_ereader_battery_write_cmd(0x0D);
			if(battery_invalid) 
			{
				battery_value = previous_charge;
			}
			else
			{
				previous_charge = battery_value;
			}
			ret = copy_to_user((void *) arg, &battery_value, sizeof(battery_value));
			break;
		case BATTERY_IOCTL_READ_TIME:
			irex_er0100_ereader_battery_write_cmd(0x12);
			if(battery_invalid) 
			{
				battery_value = previous_time;
			}
			else
			{
				previous_time = battery_value;
			}
			ret = copy_to_user((void *) arg, &battery_value, sizeof(battery_value));
			break;
		case BATTERY_IOCTL_READ_VOLTAGE:
			irex_er0100_ereader_battery_write_cmd(0x9);
			if(battery_invalid) 
			{
				battery_value = previous_volt;
			}
			else
			{
				previous_volt = battery_value;
			}
			ret = copy_to_user((void *) arg, &battery_value, sizeof(battery_value));
			break;
		case BATTERY_IOCTL_READ_STATUS:
			irex_er0100_ereader_battery_write_cmd(0x16);
			if(battery_invalid) 
			{
				battery_value = previous_status;
			}
			else
			{
				previous_status = battery_value;
			}
			ret = copy_to_user((void *) arg, &battery_value, sizeof(battery_value));
			break;
		case BATTERY_IOCTL_READ_CURRENT:
			irex_er0100_ereader_battery_write_cmd(0x0A);
			if(battery_invalid) 
			{
				battery_value = previous_current;
			}
			else
			{
				previous_current = battery_value;
			}
			ret = copy_to_user((void *) arg, &battery_value, sizeof(battery_value));
			break;
		case BATTERY_IOCTL_ENABLE_ETHERNET:
			GPCR(ETHERNET_ON_GPIO) |= GPIO_bit(ETHERNET_ON_GPIO);
			break;
		case BATTERY_IOCTL_DISABLE_ETHERNET:
			GPSR(ETHERNET_ON_GPIO) |= GPIO_bit(ETHERNET_ON_GPIO);
			break;
		case BATTERY_IOCTL_SHUTDOWN:
            /* turn power off */
            GPDR(SHUTDOWN_GPIO) |= GPIO_bit(SHUTDOWN_GPIO);
            GPSR(SHUTDOWN_GPIO) |= GPIO_bit(SHUTDOWN_GPIO);
			break;
		case BATTERY_IOCTL_ENABLE_WACOM:
			GPCR(WACOM_ON_GPIO) |= GPIO_bit(WACOM_ON_GPIO);
			break;
		case BATTERY_IOCTL_DISABLE_WACOM:
			GPSR(WACOM_ON_GPIO) |= GPIO_bit(WACOM_ON_GPIO);
			break;
		case BATTERY_IOCTL_ENABLE_CF:
			GPCR(CF_ON_GPIO) |= GPIO_bit(CF_ON_GPIO);
			break;
		case BATTERY_IOCTL_DISABLE_CF:
			GPSR(CF_ON_GPIO) |= GPIO_bit(CF_ON_GPIO);
			break;
		case BATTERY_IOCTL_ENABLE_MMC:
			GPCR(MMC_ON_GPIO) |= GPIO_bit(MMC_ON_GPIO);
			break;
		case BATTERY_IOCTL_DISABLE_MMC:
			GPSR(MMC_ON_GPIO) |= GPIO_bit(MMC_ON_GPIO);
			break;
		case BATTERY_GET_TURBO_MODE:
			asm( "mrc\tp14, 0, %0, c6, c0, 0" : "=r" (turbo) );
			copy_to_user((void*)arg, &turbo, sizeof(unsigned long));
			break;
		case BATTERY_SET_TURBO_MODE:
			arg=arg&1;
			asm( "mcr\tp14, 0, %0, c6, c0, 0" : : "r" (arg));
			break;
	}
	
	if(battery_invalid)
	{
		battery_invalid_count++;
		if(battery_invalid_count >= 5)
		{
			printk("Battery: Response rejected 5 times. Results INVALID\n");
			previous_charge = BATTERY_INVALID;
			previous_time = BATTERY_INVALID;
			previous_volt = BATTERY_INVALID;
			previous_status = BATTERY_INVALID;
		}
		else
		{
			printk("Battery: Response rejected %d times\n", battery_invalid_count);
		}
	}
	else
	{
		battery_invalid_count = 0;
	}

	return ret;
}

#ifdef MODULE

MODULE_AUTHOR("Matthijs van de Water / Daniel Wiermans");
MODULE_DESCRIPTION("iRex ER0100 Battery driver");
MODULE_LICENSE("PROPRIETARY (c) 2006 iRex Technologies BV");

int init_module(void)
{
    return irex_er0100_ereader_battery_init();
}

void cleanup_module()
{
    irex_er0100_ereader_battery_cleanup();
}

#endif /* MODULE */

