#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__   = 'GPL v3'
__copyright__ = '2026, Comfy.n'
__docformat__ = 'restructuredtext en'

from qt.core import QToolButton, QIcon, QMessageBox
from calibre.gui2.actions import InterfaceAction
from calibre.gui2 import Application, gprefs
from calibre.customize.zipplugin import get_icons

class ThemeToggleAction(InterfaceAction):

    name = 'Toggle Light & Dark'
    action_spec = ('Toggle Light \uD83D\uDE75 Dark', None, 'Toggle between light and dark Calibre user interface colors', '')
    action_shortcut_name = 'Toggle Light & Dark'
    action_menu_clone_qaction = 'Toggle Light & Dark'

    def genesis(self):
        self.rebuild_icon()
        self.qaction.triggered.connect(self.toggle_theme)

        app = Application.instance()
        if app:
            app.palette_changed.connect(self.rebuild_icon)

    def rebuild_icon(self):
        pm = Application.instance().palette_manager
        icon_path = 'images/iconplugin_dark.png' if pm.use_dark_palette else 'images/iconplugin_light.png'
        icon = get_icons(self.plugin_path, icon_path, self.name)
        self.qaction.setIcon(icon)

    def toggle_theme(self):
        app = Application.instance()
        # Check for 'calibre style' interface
        if not getattr(app, 'using_calibre_style', True):
            QMessageBox.information(
                self.gui, 'Toggle Light & Dark',
                'Theme toggle is only available when using "calibre style" interface.\n\n'
                'To use this feature:\n'
                '1. Go to Preferences → Interface → Look & Feel\n'
                '2. Change "User interface style" to "calibre"\n'
                '3. Restart calibre\n\n'
                'Note: Switching between user interface style types (system ↔ calibre) requires a restart '
                'because it involves fundamental changes to how the application handles styling.'
            )
            return

        current = gprefs.get('color_palette', 'light')
        # If set to system default, warn user and do not toggle
        if current == 'system':
            QMessageBox.information(
                self.gui, 'Toggle Light & Dark',
                'Theme toggle only works when the user interface colors (color palette) are set to "Light" or "Dark".\n\n'
                'To use this feature:\n'
                '1. Go to Preferences → Interface → Adjust colors\n'
                '2. Change "Color palette" to "Light" or "Dark"\n'
                '3. Restart calibre if needed.'
            )
            return

        # Toggle between light and dark
        if current == 'light':
            new_theme = 'dark'
        else:
            new_theme = 'light'

        with gprefs:
            gprefs['color_palette'] = new_theme

        # Try to refresh palette if method exists (Calibre 7.26+)
        if hasattr(app, 'palette_manager') and hasattr(app.palette_manager, 'refresh_palette'):
            app.palette_manager.refresh_palette()


