#!/usr/bin/env python
__license__   = 'GPL v3'
__copyright__ = '2025, Comfy.n'

import os

from qt.core import (
    QIcon,
    QColor,
    QFont,
    QMenu,
    QPainter,
    QPainterPath,
    QPixmap,
    QRectF,
    QToolButton,
    QFileSystemWatcher,
    QTimer,
    Qt,
)

from calibre.gui2.actions import InterfaceAction
from calibre.gui2 import Application, info_dialog
from calibre.utils.config import prefs, config_dir

from .config import KEY_SHOW_BADGE, KEY_SHOW_CONFIRMATION, plugin_prefs


class ToggleInputFormatAction(InterfaceAction):
    name = 'Input Order Swap'
    action_spec = ('Input Order Swap', None, 'Swap the top two formats in your preferred input order.', None)
    popup_type = QToolButton.ToolButtonPopupMode.MenuButtonPopup

    def genesis(self):
        self._create_menu()
        self.rebuild_icon()
        self.qaction.triggered.connect(self.swap_first_two)

        # Register this action in Preferences -> Keyboard shortcuts.
        try:
            self.gui.keyboard.register_shortcut(
                'plugin:input_order_swap:swap_top_two',
                'Input Order Swap: swap top two',
                description='Swap the top two formats in your preferred input order.',
                action=self.qaction,
                group='Input Order Swap',
                persist_shortcut=True,
            )
        except Exception:
            pass

        # Listen for preference changes made outside this plugin (e.g. Behavior prefs dialog)
        # by watching the global prefs file. Also keep a light polling fallback in case
        # of atomic-replace writes that some platforms/filesystems do.
        self._prefs_path = os.path.join(config_dir, 'global.py.json')
        self._last_prefs_mtime = None

        self._prefs_watcher = QFileSystemWatcher(self.gui)
        self._prefs_watcher.fileChanged.connect(self._on_prefs_file_changed, type=Qt.ConnectionType.QueuedConnection)
        if os.path.exists(self._prefs_path):
            self._prefs_watcher.addPath(self._prefs_path)
            try:
                self._last_prefs_mtime = os.path.getmtime(self._prefs_path)
            except Exception:
                self._last_prefs_mtime = None

        self._prefs_timer = QTimer(self.gui)
        self._prefs_timer.setInterval(1500)
        self._prefs_timer.timeout.connect(self._poll_prefs_file, type=Qt.ConnectionType.QueuedConnection)
        self._prefs_timer.start()

        app = Application.instance()
        if app:
            try:
                app.palette_changed.connect(self.rebuild_icon, type=Qt.ConnectionType.QueuedConnection)
            except TypeError:
                app.palette_changed.connect(self.rebuild_icon)

    def _create_menu(self):
        m = QMenu(self.gui)
        m.addAction('Swap top two now').triggered.connect(self.swap_first_two)
        m.addSeparator()
        m.addAction('Customize…').triggered.connect(self.customize_plugin)
        m.addAction('Open Preferences -> Behavior').triggered.connect(self.open_behavior_prefs)
        self.qaction.setMenu(m)

    def customize_plugin(self):
        # Standard way to open the plugin customization dialog.
        try:
            self.interface_action_base_plugin.do_user_config(self.gui)
        except Exception:
            try:
                self.interface_action_base_plugin.do_user_config(self.gui)
            except Exception:
                pass

    def open_behavior_prefs(self):
        try:
            self.gui.iactions['Preferences'].do_config(
                initial_plugin=('Interface', 'Behavior'),
                close_after_initial=True,
            )
        except Exception:
            pass

    def _with_badge(self, icon, text):
        # Overlay a small badge so TIF/TOF are visually distinct even
        # when both show the same mimetype icon.
        try:
            base = icon.pixmap(32, 32)
            if base.isNull():
                return icon
            pm = QPixmap(base.size())
            pm.fill(Qt.GlobalColor.transparent)
            p = QPainter(pm)
            p.setRenderHint(QPainter.RenderHint.Antialiasing, True)
            p.drawPixmap(0, 0, base)

            rect = QRectF(pm.width() - 18, pm.height() - 14, 16, 12)
            path = QPainterPath()
            path.addRoundedRect(rect, 3, 3)
            p.fillPath(path, QColor(0, 0, 0, 170))
            p.setPen(QColor(255, 255, 255))
            f = QFont(p.font())
            f.setBold(True)
            f.setPointSize(max(7, f.pointSize() - 2))
            p.setFont(f)
            p.drawText(rect, Qt.AlignmentFlag.AlignCenter, text)
            p.end()
            return QIcon(pm)
        except Exception:
            return icon

    def rebuild_icon(self):
        top = ''
        try:
            order = prefs['input_format_order'] or []
            if order:
                top = str(order[0]).upper()
        except Exception:
            top = ''

        # Dynamic icon: prefer calibre's mimetype icon for the current top format.
        # No bundled fallback icon by design.
        icon = QIcon()
        if top:
            candidate = f'mimetypes/{top.lower()}.png'
            i = QIcon.ic(candidate)
            if i is not None and not i.isNull():
                icon = i

        try:
            show_badge = bool(plugin_prefs.get(KEY_SHOW_BADGE, True))
        except Exception:
            show_badge = True

        self.qaction.setIcon(self._with_badge(icon, 'IN') if show_badge else icon)

    def _on_prefs_file_changed(self, path):
        # QFileSystemWatcher can drop the path if the file is atomically replaced.
        if path and os.path.exists(path) and path not in self._prefs_watcher.files():
            try:
                self._prefs_watcher.addPath(path)
            except Exception:
                pass
        self._refresh_prefs_and_icon(force=True)

    def _poll_prefs_file(self):
        if not getattr(self, '_prefs_path', None):
            return
        try:
            mtime = os.path.getmtime(self._prefs_path)
        except Exception:
            return
        if self._last_prefs_mtime is None or mtime != self._last_prefs_mtime:
            self._last_prefs_mtime = mtime
            self._refresh_prefs_and_icon(force=True)

    def _refresh_prefs_and_icon(self, force=False):
        # prefs is a ConfigProxy that caches values; refresh to pick up external edits.
        try:
            prefs.refresh()
        except Exception:
            # If refresh fails, we can still try to rebuild from cached values.
            pass
        self.rebuild_icon()

    def swap_first_two(self):
        current = list(prefs['input_format_order'] or [])
        if len(current) < 2:
            info_dialog(self.gui, 'Input Order Swap',
                        'Not enough formats to swap (need at least two).', show=True)
            return

        current[0], current[1] = current[1], current[0]
        # calibre.utils.config.prefs is a ConfigProxy; it commits on assignment.
        prefs['input_format_order'] = current

        # Refresh the toolbar icon to match the new top format.
        self.rebuild_icon()

        try:
            show_confirmation = bool(plugin_prefs.get(KEY_SHOW_CONFIRMATION, True))
        except Exception:
            show_confirmation = True

        if show_confirmation:
            info_dialog(
                self.gui,
                'Input Order Swap',
                'Preferred input format order updated.\n\n'
                f'New top two: {current[0]} -> {current[1]}',
                show=True,
            )
