#!/usr/bin/env python

import re
from PyQt5.QtGui import QSyntaxHighlighter, QTextCharFormat, QColor
from calibre_plugins.opf_helper.utils import is_dark_theme

class XMLHighlighter(QSyntaxHighlighter):
    """Syntax highlighter for XML content with theme-aware colors"""
    
    def __init__(self, parent=None):
        super(XMLHighlighter, self).__init__(parent)
        
        self.highlighting_rules = []
        self.is_dark = is_dark_theme()
            
        # Define colors based on theme
        if self.is_dark:
            # Bright colors for dark mode
            tag_color = "#88CCFF"      # Bright light blue
            attr_color = "#50E3C2"     # Bright teal
            value_color = "#FF9D88"    # Bright salmon
            comment_color = "#90EE90"  # Light green
        else:
            # Original colors for light mode
            tag_color = "#000080"      # Navy blue
            attr_color = "#008080"     # Teal
            value_color = "#8B0000"    # Dark red
            comment_color = "#006400"  # Dark green
        
        # XML element format
        tag_format = QTextCharFormat()
        tag_format.setForeground(QColor(tag_color))
        self.highlighting_rules.append((r'<[!?]?[a-zA-Z0-9_:-]+|/?>', tag_format))
        
        # XML attribute format
        attribute_format = QTextCharFormat()
        attribute_format.setForeground(QColor(attr_color))
        self.highlighting_rules.append((r'\s[a-zA-Z0-9_:-]+(?=\s*=)', attribute_format))
        
        # XML value format
        value_format = QTextCharFormat()
        value_format.setForeground(QColor(value_color))
        self.highlighting_rules.append((r'"[^"]*"', value_format))
        
        # Comment format
        comment_format = QTextCharFormat()
        comment_format.setForeground(QColor(comment_color))
        self.highlighting_rules.append((r'<!--[\s\S]*?-->', comment_format))

    def highlightBlock(self, text):
        """Apply highlighting rules to the text block"""
        for pattern, format in self.highlighting_rules:
            for match in re.finditer(pattern, text):
                self.setFormat(match.start(), match.end() - match.start(), format)