#!/usr/bin/env python
"""
OPF Comparison Dialog - Shows current OPF vs. standards-corrected version
"""

import re
import difflib
from xml.etree import ElementTree as ET
from xml.dom import minidom

try:
    # First try importing from Calibre's Qt wrapper (recommended for plugins)
    from calibre.gui2.qt.core import (Qt, QTimer, QUrl)
    from calibre.gui2.qt.widgets import (QDialog, QVBoxLayout, QHBoxLayout, QTextEdit,
                                       QPushButton, QLabel, QSplitter, QGroupBox,
                                       QDialogButtonBox, QFrame, QScrollArea, QLineEdit)
    from calibre.gui2.qt.gui import (QFont, QSyntaxHighlighter, QTextCharFormat, QColor,
                                   QTextCursor, QTextDocument)
    from calibre.gui2.qt.webengine import QIcon
except ImportError:
    # Fall back to PyQt5 if running outside Calibre or with older Calibre versions
    from PyQt5.QtWidgets import (QDialog, QVBoxLayout, QHBoxLayout, QTextEdit,
                               QPushButton, QLabel, QSplitter, QGroupBox,
                               QDialogButtonBox, QFrame, QScrollArea, QLineEdit)
    from PyQt5.QtGui import (QFont, QSyntaxHighlighter, QTextCharFormat, QColor,
                           QTextCursor, QTextDocument, QIcon)
    from PyQt5.QtCore import Qt, QTimer, QUrl

from calibre.constants import DEBUG
from calibre.utils.logging import default_log as debug_print
from calibre.utils.localization import _
from calibre.gui2 import gprefs

class OPFComparisonDialog(QDialog):
    """Dialog for comparing current OPF with standards-corrected version"""

    def __init__(self, parent, current_opf_content, book_title="Unknown Book"):
        super().__init__(parent)
        self.current_opf = current_opf_content
        self.book_title = book_title
        self.corrected_opf = self.generate_corrected_opf(current_opf_content)

        self.setWindowTitle(f"OPF Standards Comparison - {book_title}")
        self.setMinimumWidth(1200)
        self.setMinimumHeight(700)

        self.setup_ui()
        self.generate_comparison()

        # Restore geometry
        geom = gprefs.get('opf_comparison_dialog_geometry', None)
        if geom:
            self.restoreGeometry(geom)
        else:
            self.resize(1200, 700)

    def setup_ui(self):
        """Set up the user interface"""
        layout = QVBoxLayout(self)
        layout.setSpacing(6)
        layout.setContentsMargins(10, 10, 10, 10)

        # Apply scrollbar styling consistent with the main OPF Helper dialog
        self.setStyleSheet(self._scrollbar_stylesheet())

    # Header with instructions only


        desc_label = QLabel("This shows your current OPF file alongside a version corrected "
                           "according to OPF standards. The corrected version is a suggestion "
                           "and does not modify your original file.")
        desc_label.setWordWrap(True)
        layout.addWidget(desc_label)

        # Main vertical splitter (top compare panes + bottom summary)
        v_splitter = QSplitter(Qt.Vertical)
        v_splitter.setChildrenCollapsible(False)
        layout.addWidget(v_splitter, 1)

        # Create splitter for side-by-side view
        splitter = QSplitter(Qt.Horizontal)
        splitter.setChildrenCollapsible(False)
        v_splitter.addWidget(splitter)

        # Left side - Current OPF
        current_group = QGroupBox("Current OPF")
        current_layout = QVBoxLayout(current_group)

        self.current_text = QTextEdit()
        self.current_text.setReadOnly(True)
        self.current_text.setFont(QFont("Courier New", 10))
        try:
            self.current_text.setLineWrapMode(QTextEdit.WidgetWidth)
        except Exception:
            pass
        current_layout.addWidget(self.current_text)

        splitter.addWidget(current_group)

        # Right side - Corrected OPF
        corrected_group = QGroupBox("Standards-Corrected OPF")
        corrected_layout = QVBoxLayout(corrected_group)

        self.corrected_text = QTextEdit()
        self.corrected_text.setReadOnly(True)
        self.corrected_text.setFont(QFont("Courier New", 10))
        try:
            self.corrected_text.setLineWrapMode(QTextEdit.WidgetWidth)
        except Exception:
            pass
        corrected_layout.addWidget(self.corrected_text)

        splitter.addWidget(corrected_group)

        # Set splitter proportions
        splitter.setSizes([600, 600])

        # Bottom section with summary - minimal padding
        summary_group = QGroupBox("Summary of Changes")
        summary_layout = QVBoxLayout(summary_group)
        summary_layout.setContentsMargins(4, 2, 4, 2)  # Minimal margins
        summary_layout.setSpacing(2)

        self.summary_text = QTextEdit()
        self.summary_text.setReadOnly(True)
        # Let the summary fill its panel; the vertical splitter controls overall height.
        self.summary_text.setMinimumHeight(90)
        self.summary_text.setFont(QFont("Courier New", 9))
        try:
            self.summary_text.setLineWrapMode(QTextEdit.WidgetWidth)
        except Exception:
            pass
        summary_text_style = "QTextEdit { padding: 2px; }"
        self.summary_text.setStyleSheet(summary_text_style)
        summary_layout.addWidget(self.summary_text)

        v_splitter.addWidget(summary_group)

        # Initial vertical split: favor the compare panes
        try:
            v_splitter.setSizes([520, 180])
            v_splitter.setStretchFactor(0, 3)
            v_splitter.setStretchFactor(1, 1)
        except Exception:
            pass

        # Buttons - compact layout
        button_layout = QHBoxLayout()
        button_layout.setSpacing(6)
        button_layout.setContentsMargins(0, 4, 0, 0)

        # Add copy buttons on the left
        copy_current_btn = QPushButton("Copy Current OPF")
        copy_current_btn.clicked.connect(lambda: self.copy_to_clipboard(self.current_opf))
        button_layout.addWidget(copy_current_btn)

        copy_corrected_btn = QPushButton("Copy Corrected OPF")
        copy_corrected_btn.clicked.connect(lambda: self.copy_to_clipboard(self.corrected_opf))
        button_layout.addWidget(copy_corrected_btn)

        colors_btn = QPushButton("Highlight Colors…")
        colors_btn.setToolTip("Configure added/removed/unchanged highlight colors")
        colors_btn.clicked.connect(self.configure_highlight_colors)
        button_layout.addWidget(colors_btn)

        button_layout.addStretch()  # Push close button to the right

        # Close button
        close_btn = QPushButton("Close")
        close_btn.clicked.connect(self.reject)
        button_layout.addWidget(close_btn)

        layout.addLayout(button_layout)

        # Apply syntax highlighting (will be overridden by HTML highlighting)
        self.current_highlighter = OPFXMLHighlighter(self.current_text.document())
        self.corrected_highlighter = OPFXMLHighlighter(self.corrected_text.document())

    def _scrollbar_stylesheet(self):
        # Matches the styling used in the main OPF Helper dialog (ShowOPFPlugin tab widget).
        return (
            "QScrollBar::handle {"
            "   border: 1px solid #5B6985;"
            "}"
            "QScrollBar:vertical {"
            "   background: transparent;"
            "   width: 12px;"
            "   margin: 0px 0px 0px 0px;"
            "   padding: 6px 0px 6px 0px;"
            "}"
            "QScrollBar::handle:vertical {"
            "   background: rgba(140, 172, 204, 0.25);"
            "   min-height: 22px;"
            "   border-radius: 4px;"
            "   margin: 4px 0px;"
            "}"
            "QScrollBar::handle:vertical:hover {"
            "   background: rgba(140, 172, 204, 0.45);"
            "}"
            "QScrollBar:horizontal {"
            "   background: transparent;"
            "   height: 12px;"
            "   margin: 0px 0px 0px 0px;"
            "   padding: 0px 6px 0px 6px;"
            "}"
            "QScrollBar::handle:horizontal {"
            "   background: rgba(140, 172, 204, 0.25);"
            "   min-width: 22px;"
            "   border-radius: 4px;"
            "   margin: 0px 4px;"
            "}"
            "QScrollBar::handle:horizontal:hover {"
            "   background: rgba(140, 172, 204, 0.45);"
            "}"
            "QScrollBar::add-page:vertical, QScrollBar::sub-page:vertical,"
            "QScrollBar::add-page:horizontal, QScrollBar::sub-page:horizontal {"
            "   background: none;"
            "}"
        )

    def configure_highlight_colors(self):
        d = HighlightColorsDialog(self)
        exec_fn = getattr(d, 'exec', None) or getattr(d, 'exec_', None)
        if exec_fn is not None and exec_fn() == QDialog.Accepted:
            # Re-render highlights using updated colors
            self.highlight_differences()

    def generate_corrected_opf(self, opf_content):
        """Generate a standards-corrected version of the OPF content"""
        try:
            # Parse the XML
            root = ET.fromstring(opf_content)

            # Apply standard corrections
            self.apply_opf_corrections(root)

            # Convert back to string with proper formatting
            rough_string = ET.tostring(root, encoding='unicode')

            # Pretty print the XML
            reparsed = minidom.parseString(rough_string)
            corrected_xml = reparsed.toprettyxml(indent="  ", encoding=None)

            # Clean up the pretty-printed XML (remove extra newlines)
            lines = corrected_xml.split('\n')
            cleaned_lines = []
            prev_empty = False

            for line in lines:
                is_empty = line.strip() == ''
                if not (is_empty and prev_empty):
                    cleaned_lines.append(line)
                prev_empty = is_empty

            return '\n'.join(cleaned_lines)

        except Exception as e:
            debug_print(f"OPFHelper: Error generating corrected OPF: {str(e)}")
            return opf_content  # Return original if correction fails

    def apply_opf_corrections(self, root):
        """Apply OPF standard corrections to the XML tree"""
        # Get namespace
        ns = {'opf': 'http://www.idpf.org/2007/opf',
              'dc': 'http://purl.org/dc/elements/1.1/'}

        # 1. Ensure proper element ordering in package
        self.reorder_package_elements(root, ns)

        # 2. Ensure proper metadata ordering
        metadata = root.find('.//{http://www.idpf.org/2007/opf}metadata')
        if metadata is not None:
            self.reorder_metadata_elements(metadata, ns)

        # 3. Ensure proper manifest ordering
        manifest = root.find('.//{http://www.idpf.org/2007/opf}manifest')
        if manifest is not None:
            self.reorder_manifest_items(manifest, ns)

        # 4. Ensure proper spine ordering
        spine = root.find('.//{http://www.idpf.org/2007/opf}spine')
        if spine is not None:
            self.reorder_spine_items(spine, ns)

        # 5. Add missing required attributes
        self.add_required_attributes(root, ns)

        # 6. Normalize whitespace and formatting
        self.normalize_formatting(root)

    def reorder_package_elements(self, root, ns):
        """Reorder top-level elements in package according to OPF standard"""
        # Standard order: metadata, manifest, spine, guide (optional), tours (optional)
        desired_order = ['metadata', 'manifest', 'spine', 'guide', 'tours']

        # Remove all children temporarily
        children = []
        for child in list(root):
            root.remove(child)
            children.append((child.tag.split('}')[-1], child))  # Store tag name without namespace

        # Re-add in correct order
        for desired_tag in desired_order:
            for tag_name, child in children:
                if tag_name == desired_tag:
                    root.append(child)
                    children.remove((tag_name, child))
                    break

        # Add any remaining elements at the end
        for _, child in children:
            root.append(child)

    def reorder_metadata_elements(self, metadata, ns):
        """Reorder metadata elements according to Dublin Core and OPF standards"""
        # Dublin Core elements first, then OPF-specific metadata
        dc_order = ['title', 'creator', 'subject', 'description', 'publisher',
                   'contributor', 'date', 'type', 'format', 'identifier',
                   'source', 'language', 'relation', 'coverage', 'rights']

        # Remove all children temporarily
        children = []
        for child in list(metadata):
            tag_name = child.tag.split('}')[-1]  # Remove namespace
            metadata.remove(child)
            children.append((tag_name, child))

        # Re-add Dublin Core elements first in standard order
        for dc_tag in dc_order:
            for tag_name, child in children[:]:  # Copy list to avoid modification issues
                if tag_name == dc_tag:
                    metadata.append(child)
                    children.remove((tag_name, child))
                    break

        # Add OPF-specific metadata elements
        opf_specific = ['meta']
        for opf_tag in opf_specific:
            for tag_name, child in children[:]:
                if tag_name == opf_tag:
                    metadata.append(child)
                    children.remove((tag_name, child))

        # Add any remaining elements
        for _, child in children:
            metadata.append(child)

    def reorder_manifest_items(self, manifest, ns):
        """Reorder manifest items alphabetically by href"""
        if len(manifest) <= 1:
            return

        # Get all item elements
        items = []
        for child in list(manifest):
            if child.tag.endswith('item'):
                manifest.remove(child)
                items.append(child)

        # Sort by href attribute
        items.sort(key=lambda x: x.get('href', '').lower())

        # Re-add in sorted order
        for item in items:
            manifest.append(item)

    def reorder_spine_items(self, spine, ns):
        """Ensure spine itemsref elements are in proper order"""
        # For now, just ensure they're grouped together
        # More complex ordering would require reading the manifest
        pass

    def add_required_attributes(self, root, ns):
        """Add any missing required attributes"""
        # Ensure package has version attribute
        if 'version' not in root.attrib:
            root.set('version', '2.0')  # Default to 2.0 if not specified

        # Ensure package has unique-identifier attribute
        if 'unique-identifier' not in root.attrib:
            # Try to find an identifier in metadata
            metadata = root.find('.//{http://www.idpf.org/2007/opf}metadata')
            if metadata is not None:
                identifiers = metadata.findall('.//{http://purl.org/dc/elements/1.1/}identifier')
                if identifiers:
                    # Use the id of the first identifier
                    ident_id = identifiers[0].get('{http://www.idpf.org/2007/opf}id')
                    if ident_id:
                        root.set('unique-identifier', ident_id)

    def normalize_formatting(self, root):
        """Normalize whitespace and formatting"""
        # Remove excessive whitespace from text content
        for elem in root.iter():
            if elem.text:
                elem.text = elem.text.strip()
            if elem.tail:
                elem.tail = elem.tail.strip()

    def generate_comparison(self):
        """Generate the comparison view with highlighting"""
        # Set the text content
        self.current_text.setPlainText(self.current_opf)
        self.corrected_text.setPlainText(self.corrected_opf)

        # Apply difference highlighting
        self.highlight_differences()

        # Generate summary of changes
        self.generate_change_summary()

    def highlight_differences(self):
        """Highlight differences between current and corrected OPF"""
        current_lines = self.current_opf.split('\n')
        corrected_lines = self.corrected_opf.split('\n')

        # Create a differ
        differ = difflib.Differ()
        diff = list(differ.compare(current_lines, corrected_lines))

        # Generate HTML with highlighting
        current_html = self.generate_highlighted_html(diff, is_current=True)
        corrected_html = self.generate_highlighted_html(diff, is_current=False)

        # Set HTML content
        self.current_text.setHtml(current_html)
        self.corrected_text.setHtml(corrected_html)

    def generate_highlighted_html(self, diff_lines, is_current):
        """Generate HTML with highlighted differences using customizable theme-adaptive colors"""
        html_lines = []

        # Load color preferences
        from calibre_plugins.opf_helper.config import prefs

        # Detect dark mode and use appropriate colors
        try:
            from calibre.gui2 import is_dark_theme
            dark_mode = is_dark_theme()
        except Exception:
            dark_mode = False

        # Use customizable colors from preferences
        if dark_mode:
            # Dark theme colors
            added_bg = prefs.get('comparison_dark_added_bg', '#264F26')
            added_text = prefs.get('comparison_dark_added_text', '#90EE90')
            removed_bg = prefs.get('comparison_dark_removed_bg', '#4F2626')
            removed_text = prefs.get('comparison_dark_removed_text', '#FF9999')
            unchanged_bg = "transparent"
            unchanged_text = prefs.get('comparison_dark_unchanged_text', '#E0E0E0')
        else:
            # Light theme colors
            added_bg = prefs.get('comparison_light_added_bg', '#D4EDDA')
            added_text = prefs.get('comparison_light_added_text', '#155724')
            removed_bg = prefs.get('comparison_light_removed_bg', '#F8D7DA')
            removed_text = prefs.get('comparison_light_removed_text', '#721C24')
            unchanged_bg = "transparent"
            unchanged_text = prefs.get('comparison_light_unchanged_text', '#333333')
        def _line_div(text, bg, fg, marker, bold=False):
            weight = 'font-weight: bold;' if bold else ''
            # Use <div> per line (Qt rich text handles this better than relying on display:block on <span>)
            return (
                f'<div style="margin:0; padding:0 0 0 6px; '
                f'border-left: 4px solid {marker}; '
                f'background-color: {bg}; color: {fg}; {weight} '
                f'white-space: pre-wrap; word-wrap: break-word;">{text}</div>'
            )

        def _blank_line():
            # Keep line alignment across panes when one side has an inserted/deleted line
            return _line_div('&nbsp;', 'transparent', unchanged_text, 'transparent', bold=False)

        unchanged_marker = 'transparent'
        added_marker = added_text
        removed_marker = removed_text

        for line in diff_lines:
            if line.startswith('  '):  # Unchanged line
                escaped_line = self.escape_html(line[2:])
                html_lines.append(_line_div(escaped_line, unchanged_bg, unchanged_text, unchanged_marker, bold=False))
            elif line.startswith('- ') and is_current:  # Line removed from current
                escaped_line = self.escape_html(line[2:])
                html_lines.append(_line_div(escaped_line, removed_bg, removed_text, removed_marker, bold=True))
            elif line.startswith('- ') and not is_current:
                # Removed from current => keep alignment with a blank line on corrected side
                html_lines.append(_blank_line())
            elif line.startswith('+ ') and not is_current:  # Line added to corrected
                escaped_line = self.escape_html(line[2:])
                html_lines.append(_line_div(escaped_line, added_bg, added_text, added_marker, bold=True))
            elif line.startswith('+ ') and is_current:
                # Added to corrected => keep alignment with a blank line on current side
                html_lines.append(_blank_line())
            elif line.startswith('? '):
                continue
            else:
                escaped_line = self.escape_html(line)
                html_lines.append(_line_div(escaped_line, unchanged_bg, unchanged_text, unchanged_marker, bold=False))

        return '<div style="font-family: Courier New, monospace; font-size: 10pt; margin: 0; padding: 0; line-height: 1.3;">' + ''.join(html_lines) + '</div>'

    def escape_html(self, text):
        """Escape HTML special characters"""
        return (text.replace('&', '&amp;')
                .replace('<', '&lt;')
                .replace('>', '&gt;')
                .replace('"', '&quot;')
                .replace("'", '&#39;'))

    def generate_change_summary(self):
        """Generate a summary of the changes made"""
        current_lines = self.current_opf.split('\n')
        corrected_lines = self.corrected_opf.split('\n')

        # Use difflib to find differences
        diff = list(difflib.unified_diff(
            current_lines,
            corrected_lines,
            fromfile='Current OPF',
            tofile='Corrected OPF',
            lineterm='',
            n=3  # Context lines
        ))

        if not diff:
            summary = "No changes were needed - your OPF already conforms to standards!"
        else:
            # Count different types of changes
            additions = sum(1 for line in diff if line.startswith('+'))
            deletions = sum(1 for line in diff if line.startswith('-'))

            summary = f"Changes applied to conform to OPF standards:\n"
            summary += f"• Lines added: {additions}\n"
            summary += f"• Lines removed: {deletions}\n\n"
            summary += "Detailed changes:\n" + '\n'.join(diff[:50])  # Limit to first 50 lines

            if len(diff) > 50:
                summary += f"\n... and {len(diff) - 50} more changes"

        self.summary_text.setPlainText(summary)

    def copy_to_clipboard(self, text):
        """Copy text to clipboard"""
        from calibre.gui2 import QApplication
        clipboard = QApplication.clipboard()
        clipboard.setText(text)

        # Show brief notification
        try:
            from calibre.gui2 import info_dialog
            info_dialog(self, 'Copied', 'Content copied to clipboard', show=True)
        except Exception:
            pass

    def closeEvent(self, e):
        """Save dialog geometry on close"""
        gprefs['opf_comparison_dialog_geometry'] = bytearray(self.saveGeometry())
        super().closeEvent(e)


class HighlightColorsDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        from calibre_plugins.opf_helper.config import prefs

        self.setWindowTitle('OPF Comparison Highlight Colors')
        self.setMinimumWidth(520)

        layout = QVBoxLayout(self)

        info = QLabel('Enter colors as #RRGGBB (or RRGGBB) or a CSS/Qt named color (e.g. rebeccapurple). Changes apply immediately after closing this dialog.')
        info.setWordWrap(True)
        layout.addWidget(info)

        dark_group = QGroupBox('Dark theme')
        dark_layout = QVBoxLayout(dark_group)
        layout.addWidget(dark_group)

        def row(label, widget1, label2=None, widget2=None):
            r = QHBoxLayout()
            r.addWidget(QLabel(label))
            r.addWidget(widget1)
            if label2 is not None and widget2 is not None:
                r.addWidget(QLabel(label2))
                r.addWidget(widget2)
            r.addStretch()
            return r

        def make_swatch():
            sw = QFrame()
            sw.setFixedSize(18, 18)
            sw.setFrameShape(QFrame.StyledPanel)
            sw.setFrameShadow(QFrame.Plain)
            return sw

        def normalize_for_css(v):
            v = (v or '').strip()
            if re.match(r'^[0-9a-fA-F]{6}$', v):
                return '#' + v
            return v

        def is_valid_color(v):
            v = (v or '').strip()
            if not v:
                return False
            if re.match(r'^#[0-9a-fA-F]{6}$', v) or re.match(r'^[0-9a-fA-F]{6}$', v):
                return True
            try:
                return QColor(v).isValid()
            except Exception:
                return False

        def update_swatch(line_edit, swatch):
            txt = (line_edit.text() or '').strip()
            if is_valid_color(txt):
                css = normalize_for_css(txt)
                swatch.setStyleSheet(f"QFrame {{ background-color: {css}; border: 1px solid #5B6985; }}")
            else:
                swatch.setStyleSheet("QFrame { background-color: transparent; border: 1px solid #CC3333; }")

        def make_color_input(initial):
            le = QLineEdit(initial)
            le.setMaximumWidth(140)
            sw = make_swatch()
            update_swatch(le, sw)
            le.textChanged.connect(lambda _t, le=le, sw=sw: update_swatch(le, sw))
            return le, sw

        self.dark_added_bg, self.dark_added_bg_sw = make_color_input(prefs.get('comparison_dark_added_bg', '#005A00'))
        self.dark_added_text, self.dark_added_text_sw = make_color_input(prefs.get('comparison_dark_added_text', '#FFFFFF'))
        r = QHBoxLayout()
        r.addWidget(QLabel('Added: bg'))
        r.addWidget(self.dark_added_bg)
        r.addWidget(self.dark_added_bg_sw)
        r.addWidget(QLabel('text'))
        r.addWidget(self.dark_added_text)
        r.addWidget(self.dark_added_text_sw)
        r.addStretch()
        dark_layout.addLayout(r)

        self.dark_removed_bg, self.dark_removed_bg_sw = make_color_input(prefs.get('comparison_dark_removed_bg', '#5A0000'))
        self.dark_removed_text, self.dark_removed_text_sw = make_color_input(prefs.get('comparison_dark_removed_text', '#FFFFFF'))
        r = QHBoxLayout()
        r.addWidget(QLabel('Removed: bg'))
        r.addWidget(self.dark_removed_bg)
        r.addWidget(self.dark_removed_bg_sw)
        r.addWidget(QLabel('text'))
        r.addWidget(self.dark_removed_text)
        r.addWidget(self.dark_removed_text_sw)
        r.addStretch()
        dark_layout.addLayout(r)

        self.dark_unchanged_text, self.dark_unchanged_text_sw = make_color_input(prefs.get('comparison_dark_unchanged_text', '#D0D0D0'))
        r = QHBoxLayout()
        r.addWidget(QLabel('Unchanged: text'))
        r.addWidget(self.dark_unchanged_text)
        r.addWidget(self.dark_unchanged_text_sw)
        r.addStretch()
        dark_layout.addLayout(r)

        light_group = QGroupBox('Light theme')
        light_layout = QVBoxLayout(light_group)
        layout.addWidget(light_group)

        self.light_added_bg, self.light_added_bg_sw = make_color_input(prefs.get('comparison_light_added_bg', '#D4EDDA'))
        self.light_added_text, self.light_added_text_sw = make_color_input(prefs.get('comparison_light_added_text', '#155724'))
        r = QHBoxLayout()
        r.addWidget(QLabel('Added: bg'))
        r.addWidget(self.light_added_bg)
        r.addWidget(self.light_added_bg_sw)
        r.addWidget(QLabel('text'))
        r.addWidget(self.light_added_text)
        r.addWidget(self.light_added_text_sw)
        r.addStretch()
        light_layout.addLayout(r)

        self.light_removed_bg, self.light_removed_bg_sw = make_color_input(prefs.get('comparison_light_removed_bg', '#F8D7DA'))
        self.light_removed_text, self.light_removed_text_sw = make_color_input(prefs.get('comparison_light_removed_text', '#721C24'))
        r = QHBoxLayout()
        r.addWidget(QLabel('Removed: bg'))
        r.addWidget(self.light_removed_bg)
        r.addWidget(self.light_removed_bg_sw)
        r.addWidget(QLabel('text'))
        r.addWidget(self.light_removed_text)
        r.addWidget(self.light_removed_text_sw)
        r.addStretch()
        light_layout.addLayout(r)

        self.light_unchanged_text, self.light_unchanged_text_sw = make_color_input(prefs.get('comparison_light_unchanged_text', '#333333'))
        r = QHBoxLayout()
        r.addWidget(QLabel('Unchanged: text'))
        r.addWidget(self.light_unchanged_text)
        r.addWidget(self.light_unchanged_text_sw)
        r.addStretch()
        light_layout.addLayout(r)

        reset_row = QHBoxLayout()
        reset_btn = QPushButton('Reset dark defaults')
        reset_btn.clicked.connect(self._reset_dark_defaults)
        reset_row.addWidget(reset_btn)
        reset_btn2 = QPushButton('Reset light defaults')
        reset_btn2.clicked.connect(self._reset_light_defaults)
        reset_row.addWidget(reset_btn2)
        reset_row.addStretch()
        layout.addLayout(reset_row)

        buttons = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        buttons.accepted.connect(self.accept)
        buttons.rejected.connect(self.reject)
        layout.addWidget(buttons)

    def _reset_dark_defaults(self):
        self.dark_added_bg.setText('#005A00')
        self.dark_added_text.setText('#FFFFFF')
        self.dark_removed_bg.setText('#5A0000')
        self.dark_removed_text.setText('#FFFFFF')
        self.dark_unchanged_text.setText('#D0D0D0')

    def _reset_light_defaults(self):
        self.light_added_bg.setText('#D4EDDA')
        self.light_added_text.setText('#155724')
        self.light_removed_bg.setText('#F8D7DA')
        self.light_removed_text.setText('#721C24')
        self.light_unchanged_text.setText('#333333')

    def accept(self):
        from calibre_plugins.opf_helper.config import prefs

        def norm(label, value):
            v = (value or '').strip()
            if not v:
                raise ValueError(f"{label} cannot be empty")

            # Accept hex in either form
            if re.match(r'^#[0-9a-fA-F]{6}$', v):
                return v.upper()
            if re.match(r'^[0-9a-fA-F]{6}$', v):
                return ('#' + v).upper()

            # Accept named colors (and other QColor-parseable CSS strings)
            try:
                if QColor(v).isValid():
                    return v
            except Exception:
                pass

            raise ValueError(f"{label} must be #RRGGBB, RRGGBB, or a named color")

        try:
            prefs['comparison_dark_added_bg'] = norm('Dark added background', self.dark_added_bg.text())
            prefs['comparison_dark_added_text'] = norm('Dark added text', self.dark_added_text.text())
            prefs['comparison_dark_removed_bg'] = norm('Dark removed background', self.dark_removed_bg.text())
            prefs['comparison_dark_removed_text'] = norm('Dark removed text', self.dark_removed_text.text())
            prefs['comparison_dark_unchanged_text'] = norm('Dark unchanged text', self.dark_unchanged_text.text())

            prefs['comparison_light_added_bg'] = norm('Light added background', self.light_added_bg.text())
            prefs['comparison_light_added_text'] = norm('Light added text', self.light_added_text.text())
            prefs['comparison_light_removed_bg'] = norm('Light removed background', self.light_removed_bg.text())
            prefs['comparison_light_removed_text'] = norm('Light removed text', self.light_removed_text.text())
            prefs['comparison_light_unchanged_text'] = norm('Light unchanged text', self.light_unchanged_text.text())
        except Exception as e:
            try:
                from calibre.gui2 import error_dialog
                error_dialog(self, 'Invalid color', str(e), show=True)
            except Exception:
                # If error_dialog isn't available for some reason, just keep the dialog open
                pass
            return

        super().accept()


class OPFXMLHighlighter(QSyntaxHighlighter):
    """XML syntax highlighter for OPF content"""

    def __init__(self, parent=None):
        super().__init__(parent)

        # Check if we're in dark mode
        try:
            from calibre.gui2 import is_dark_theme
            self.is_dark = is_dark_theme()
        except:
            self.is_dark = False

        # Define colors based on theme
        if self.is_dark:
            tag_color = "#88CCFF"     # Light blue
            attr_color = "#FFB366"    # Light orange
            value_color = "#90EE90"   # Light green
            comment_color = "#999999"  # Gray
        else:
            tag_color = "#000080"     # Navy blue
            attr_color = "#A0522D"    # Brown
            value_color = "#006400"   # Dark green
            comment_color = "#808080"  # Gray

        # XML element format
        tag_format = QTextCharFormat()
        tag_format.setForeground(QColor(tag_color))
        self.highlighting_rules = [(r'<[!?]?[a-zA-Z0-9_:-]+|/?>', tag_format)]

        # XML attribute format
        attribute_format = QTextCharFormat()
        attribute_format.setForeground(QColor(attr_color))
        self.highlighting_rules.append((r'\s[a-zA-Z0-9_:-]+(?=\s*=)', attribute_format))

        # XML value format
        value_format = QTextCharFormat()
        value_format.setForeground(QColor(value_color))
        self.highlighting_rules.append((r'"[^"]*"', value_format))

        # Comment format
        comment_format = QTextCharFormat()
        comment_format.setForeground(QColor(comment_color))
        self.highlighting_rules.append((r'<!--[\s\S]*?-->', comment_format))

        # Compile regex patterns for better performance
        import re
        self.rules = [(re.compile(pattern), fmt) for pattern, fmt in self.highlighting_rules]

    def highlightBlock(self, text):
        """Apply syntax highlighting to the given block of text."""
        for pattern, format in self.rules:
            for match in pattern.finditer(text):
                start, length = match.start(), match.end() - match.start()
                self.setFormat(start, length, format)