#!/usr/bin/env python
import os
import sys
from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QTabWidget, QLabel,
    QScrollArea, QFrame, QSizePolicy, QPushButton
)
from PyQt5.QtCore import Qt, QUrl
from PyQt5.QtGui import QDesktopServices

from calibre.utils.logging import default_log as debug_print
from calibre.utils.config import config_dir

class EducationPanel(QWidget):
    """Panel for displaying educational content about OPF and EPUB formats"""

    def __init__(self, parent=None):
        super().__init__(parent)
        self.parent = parent
        self.setup_ui()

    def setup_ui(self):
        """Set up the user interface"""
        layout = QVBoxLayout(self)
        layout.setContentsMargins(0, 0, 0, 0)

        # Create tab widget for different educational content
        self.tabs = QTabWidget()

        # Apply minimal distinctive sub-tab styling to differentiate from main tabs
        self.tabs.setStyleSheet("""
            QTabWidget::pane {
                border: 1px solid palette(dark);
                background-color: palette(base);
            }
            QTabBar::tab {
                background-color: palette(window);
                border: 1px solid palette(dark);
                border-bottom: none;
                border-top-left-radius: 3px;
                border-top-right-radius: 3px;
                padding: 4px 8px;
                margin-right: 0px;
                font-size: 9pt;
                font-weight: normal;
                min-width: 140px;
            }
            QTabBar::tab:selected {
                background-color: palette(window);
                color: palette(text);
                font-weight: bold;
                border: 1px solid palette(dark);
                border-bottom: none;
            }
            QTabBar::tab:!selected {
                color: palette(text);
                background-color: palette(window);
            }
        """)

        layout.addWidget(self.tabs)

        # Add historical context tab (text content with scrolling)
        self.history_widget = self._create_scrollable_text_widget(self._get_history_content())
        self.tabs.addTab(self.history_widget, "Historical Context")

        # Add specifications tab (links to official docs)
        self.specs_widget = self._create_scrollable_text_widget(self._get_specs_content())
        self.tabs.addTab(self.specs_widget, "Specifications")

        # Add OPF evolution tab with text content and PDF button
        evolution_widget = self._create_evolution_tab()
        self.tabs.addTab(evolution_widget, "OPF Evolution")

    # Add missing load_diagram method
    def load_diagram(self):
        """This method is called when a book is loaded, but we don't need to do anything"""
        # This is a stub method to prevent the error
        debug_print("OPF Helper: Education panel diagram loading skipped (using PDF instead)")
        return True

    def _create_evolution_tab(self):
        """Create the OPF Evolution tab with text and PDF button"""
        container = QWidget()
        layout = QVBoxLayout(container)

        # Add introduction text
        intro = QLabel("""<h2>OPF and EPUB Format Evolution</h2>
        <p>This tab provides information about the evolution of the Open Package Format (OPF)
        and its role in the EPUB standard.</p>

        <p>The Open eBook Publication Structure (OEBPS) began in 1999 and evolved into the EPUB
        format we know today. A key milestone was the creation of EPUB 2.0 in 2007, which
        introduced the "triumvirate of specifications":</p>
        <ul>
            <li><b>Open Package Format (OPF)</b> - Package document format</li>
            <li><b>Open Publication Structure (OPS)</b> - Content document format</li>
            <li><b>OEBPS Container Format (OCF)</b> - Container file format</li>
        </ul>""")
        intro.setWordWrap(True)
        intro.setTextFormat(Qt.RichText)
        layout.addWidget(intro)

        # Add button to open the PDF
        open_pdf_button = QPushButton("View Evolution Diagram (PDF)")
        open_pdf_button.setMinimumHeight(40)
        # Use common_utils version instead
        from calibre_plugins.opf_helper.common_utils import open_documentation
        open_pdf_button.clicked.connect(lambda: open_documentation(self))
        layout.addWidget(open_pdf_button)

        # Additional explanation text
        more_info = QLabel("""<p>The detailed diagram shows the complete evolution from OEBPS 1.0
        through to modern EPUB versions, highlighting key milestones and relationships between
        the specifications.</p>""")
        more_info.setWordWrap(True)
        more_info.setTextFormat(Qt.RichText)
        layout.addWidget(more_info)

        layout.addStretch()
        return container

    def _create_scrollable_text_widget(self, html_content):
        """Create a scrollable widget containing html formatted text"""
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setFrameStyle(QFrame.NoFrame)

        content = QLabel(html_content)
        content.setWordWrap(True)
        content.setTextFormat(Qt.RichText)
        content.setOpenExternalLinks(True)
        content.setTextInteractionFlags(Qt.TextBrowserInteraction)

        scroll.setWidget(content)
        return scroll

    def _get_history_content(self):
        """Return HTML formatted text about OPF/EPUB history"""
        return """
        <h2>EPUB and OPF Format History</h2>

        <h3>OEBPS Origins (1999-2002)</h3>
        <p>The Open eBook Publication Structure (OEBPS) was first approved in 1999 by the Open eBook Forum,
        which later became the International Digital Publishing Forum (IDPF). This was the foundational
        specification for what would eventually become EPUB.</p>

        <p>Key milestones in this period:</p>
        <ul>
            <li><b>OEBPS 1.0</b> (1999): Initial specification</li>
            <li><b>OEBPS 1.1</b> (2001): Revision with improvements</li>
            <li><b>OEBPS 1.2</b> (2002): Further refinements</li>
        </ul>

        <h3>EPUB 2.0 Development (2005-2007)</h3>
        <p>In late 2005, work began on a container format for OEBPS, which was approved as the
        OEBPS Container Format (OCF) in 2006. Parallel work began on OEBPS 2.0, which was eventually
        renamed to EPUB.</p>

        <p><b>EPUB 2.0</b> (October 2007) consisted of three core specifications:</p>
        <ol>
            <li><b>Open Package Format (OPF)</b>: Defines the package document that ties together all
            components of an EPUB publication</li>
            <li><b>Open Publication Structure (OPS)</b>: Defines the content format</li>
            <li><b>OEBPS Container Format (OCF)</b>: Defines how all components are packaged together</li>
        </ol>

        <h3>EPUB 2.0.1 and Later</h3>
        <p><b>EPUB 2.0.1</b> (September 2010) was a maintenance update clarifying and correcting
        issues in the 2.0 specification.</p>

        <p>The OPF file (content.opf) is the backbone of an EPUB publication, containing:</p>
        <ul>
            <li>Metadata about the publication (title, author, etc.)</li>
            <li>Manifest listing all publication resources</li>
            <li>Spine defining the reading order</li>
            <li>Guide (optional) with references to key content</li>
        </ul>

        <h3>Role of OPF in EPUB</h3>
        <p>The OPF (Open Package Format) document is the central component in an EPUB file. It:</p>
        <ul>
            <li>Provides a way to express metadata about the publication</li>
            <li>Creates an inventory of all resources included in the publication</li>
            <li>Defines a standard reading order</li>
            <li>Enables navigation and reference to key structural components</li>
        </ul>

        <p>The OPF specification continues to be central to EPUB, with refinements and
        extensions in EPUB 3.0 and later versions.</p>
        """

    def _get_specs_content(self):
        """Return HTML formatted text with specification links"""
        return """
        <h2>EPUB and OPF Specifications</h2>

        <h3>EPUB 2.0 Specifications</h3>
        <ul>
            <li><a href="http://idpf.org/epub/20/spec/OPF_2.0.1_draft.htm">Open Package Format 2.0.1</a></li>
            <li><a href="http://idpf.org/epub/20/spec/OPS_2.0.1_draft.htm">Open Publication Structure 2.0.1</a></li>
            <li><a href="http://idpf.org/epub/20/spec/OCF_2.0.1_draft.htm">Open Container Format 2.0.1</a></li>
        </ul>

        <h3>EPUB 3.0 Specifications</h3>
        <ul>
            <li><a href="https://www.w3.org/TR/epub-33/#sec-package-doc">EPUB 3.3 Package Document</a> (latest W3C version)</li>
            <li><a href="https://www.w3.org/TR/epub-33/#sec-container">EPUB 3.3 Container</a></li>
            <li><a href="https://www.w3.org/TR/epub-33/#sec-nav">EPUB 3.3 Navigation</a></li>
        </ul>

        <h3>Related Specifications</h3>
        <ul>
            <li><a href="https://www.dublincore.org/specifications/dublin-core/dces/">Dublin Core Metadata Element Set</a>: The basis for EPUB metadata</li>
            <li><a href="https://www.w3.org/TR/xml/">XML 1.0</a>: The foundation format for OPF documents</li>
        </ul>

        <h3>Tutorials and Resources</h3>
        <ul>
            <li><a href="https://wiki.mobileread.com/wiki/OPF">MobileRead Wiki: OPF</a>: Community documentation</li>
            <li><a href="https://idpf.github.io/epub-cmt/30/spec/epub30-contentdocs.html">EPUB 3 Content Documents</a>: Specification for content documents</li>
        </ul>

        <h3>Historical Specifications</h3>
        <ul>
            <li><a href="http://web.archive.org/web/20101124201142/http://www.openebook.org/oebps/oebps1.2/index.htm">OEBPS 1.2</a> (Web Archive)</li>
            <li><a href="http://web.archive.org/web/20101124112439/http://www.openebook.org/oebps/oebps1.0/index.htm">OEBPS 1.0</a> (Web Archive)</li>
        </ul>
        """

    def open_pdf_diagram(self):
        """Open the PDF diagram in the system's default PDF viewer"""
        try:
            # First try the main config directory
            plugin_config_dir = os.path.join(config_dir, 'plugins', 'OPF_Helper')
            pdf_path = os.path.join(plugin_config_dir, 'opf_evolution_diagram.pdf')

            # If not found, try the docs subdirectory
            if not os.path.exists(pdf_path):
                pdf_path = os.path.join(plugin_config_dir, 'opf_helper_docs', 'opf_evolution_diagram.pdf')

            if os.path.exists(pdf_path):
                debug_print(f"Opening PDF diagram: {pdf_path}")
                QDesktopServices.openUrl(QUrl.fromLocalFile(pdf_path))
                return True
            else:
                debug_print(f"PDF diagram not found at {pdf_path}")
                from calibre.gui2 import error_dialog
                error_dialog(self.parent, 'Diagram Not Found',
                           'The diagram PDF file was not found. Try reinstalling this plugin.',
                           show=True)
                return False
        except Exception as e:
            debug_print(f"Error opening PDF diagram: {str(e)}")
            import traceback
            traceback.print_exc()
            from calibre.gui2 import error_dialog
            error_dialog(self.parent, 'Error Opening Diagram',
                       f'Failed to open the diagram: {str(e)}',
                       show=True)
            return False