#!/usr/bin/env python

from PyQt5.QtCore import Qt
from PyQt5.QtWidgets import QWidget, QLabel, QVBoxLayout, QSizePolicy
from PyQt5.QtGui import QPixmap

class CoverPanel(QWidget):
    """A widget for displaying book covers with proper scaling and 'No Cover' fallback"""
    
    def __init__(self, parent=None):
        QWidget.__init__(self, parent)
        
        self._layout = QVBoxLayout(self)
        self._layout.setContentsMargins(5, 5, 5, 5)
        
        # Create cover label with proper styling
        self.cover_label = QLabel()
        self.cover_label.setAlignment(Qt.AlignCenter)
        self.cover_label.setMinimumSize(200, 300)
        self.cover_label.setSizePolicy(QSizePolicy.Fixed, QSizePolicy.Preferred)
        self.cover_label.setStyleSheet("""
            QLabel {
                background-color: palette(base);
                border: 1px solid palette(mid);
                border-radius: 5px;
                padding: 5px;
            }
        """)
        self._layout.addWidget(self.cover_label)
        
        # Add a "No Cover" message that's only visible when there's no cover
        self.no_cover_label = QLabel("No Cover Available")
        self.no_cover_label.setAlignment(Qt.AlignCenter)
        self.no_cover_label.setStyleSheet("color: gray; font-style: italic;")
        self.no_cover_label.hide()  # Hidden by default
        self._layout.addWidget(self.no_cover_label)
        
        # Set fixed width for cover panel
        self.setFixedWidth(250)

    def show_cover(self, cover_data=None):
        """Display a cover image or 'No Cover' message if none available"""
        if not cover_data:
            self.cover_label.clear()
            self.no_cover_label.show()
            return

        try:
            pixmap = QPixmap()
            pixmap.loadFromData(cover_data)
            if not pixmap.isNull():
                # Scale pixmap while maintaining aspect ratio
                scaled_pixmap = pixmap.scaled(
                    self.cover_label.width() - 10,  # Subtract padding
                    self.cover_label.height() - 10,
                    Qt.KeepAspectRatio,
                    Qt.SmoothTransformation
                )
                self.cover_label.setPixmap(scaled_pixmap)
                self.cover_label.show()
                self.no_cover_label.hide()
            else:
                self.cover_label.clear()
                self.no_cover_label.show()
        except Exception as e:
            print(f"Error displaying cover: {e}")
            self.cover_label.clear()
            self.no_cover_label.show()

    def resizeEvent(self, event):
        """Handle resize events to maintain cover scaling"""
        super().resizeEvent(event)
        if self.cover_label.pixmap() and not self.cover_label.pixmap().isNull():
            # Re-scale the current pixmap to fit the new size
            orig_pixmap = self.cover_label.pixmap()
            scaled_pixmap = orig_pixmap.scaled(
                self.cover_label.width() - 10,
                self.cover_label.height() - 10,
                Qt.KeepAspectRatio,
                Qt.SmoothTransformation
            )
            self.cover_label.setPixmap(scaled_pixmap)