#!/usr/bin/env python
from PyQt5.QtCore import QUrl
from PyQt5.QtGui import QDesktopServices
import os
from calibre.utils.config import config_dir
from calibre.constants import iswindows
from calibre.gui2 import error_dialog
try:
    from calibre_plugins.opf_helper import DEBUG_OPF_HELPER
    def debug_print(*args, **kwargs):
        if DEBUG_OPF_HELPER:
            from calibre.utils.logging import default_log
            default_log(*args, **kwargs)
except Exception:
    def debug_print(*args, **kwargs):
        pass
import zipfile

# Use the common icon handling from common_icons
from calibre_plugins.opf_helper.common_icons import get_icon

plugin_name = "OPF_Helper"

def get_local_resources_dir(subfolder=None):
    '''Returns path to local resources directory, optionally with subfolder'''
    resources_dir = os.path.join(config_dir, 'plugins', plugin_name)
    if subfolder:
        resources_dir = os.path.join(resources_dir, subfolder)
    if iswindows:
        resources_dir = os.path.normpath(resources_dir)
    return resources_dir

def extract_documentation_from_zip():
    '''Extract the PDF documentation from the plugin zip to config dir - now optional'''
    try:
        # Setup destination paths
        plugin_config_dir = os.path.join(config_dir, 'plugins', plugin_name)
        docs_dir = os.path.join(plugin_config_dir, 'opf_helper_docs')
        pdf_path = os.path.join(docs_dir, 'opf_evolution_diagram.pdf')

        debug_print(f"OPF Helper: Config dir is: {config_dir}")
        debug_print(f"OPF Helper: Plugin config dir will be: {plugin_config_dir}")
        debug_print(f"OPF Helper: Docs dir will be: {docs_dir}")

        # Create docs directory
        os.makedirs(docs_dir, exist_ok=True)

        # Get plugin path
        plugin_dir = os.path.dirname(os.path.abspath(__file__))
        debug_print(f"OPF Helper: Plugin dir is: {plugin_dir}")

        # Try direct copy from plugin installation directory first
        pdf_source = os.path.join(plugin_dir, 'opf_helper_docs', 'opf_evolution_diagram.pdf')
        debug_print(f"OPF Helper: Looking for PDF at: {pdf_source}")

        if os.path.exists(pdf_source):
            debug_print(f"OPF Helper: Found PDF at {pdf_source}")
            import shutil
            shutil.copy2(pdf_source, pdf_path)
            debug_print(f"OPF Helper: Copied PDF to {pdf_path}")
            return True

        # Try extracting from plugin ZIP if running zipped
        # Find the plugin ZIP file (parent dir, or __file__ if zipped)
        zip_candidates = [
            os.path.join(os.path.dirname(plugin_dir), 'OPF Helper.zip'),
            os.path.join(plugin_dir, 'OPF Helper.zip'),
        ]
        for zip_path in zip_candidates:
            debug_print(f"OPF Helper: Checking for plugin ZIP at: {zip_path}")
            if os.path.exists(zip_path):
                try:
                    with zipfile.ZipFile(zip_path, 'r') as zf:
                        pdf_in_zip = 'opf_helper_docs/opf_evolution_diagram.pdf'
                        if pdf_in_zip in zf.namelist():
                            with zf.open(pdf_in_zip) as src, open(pdf_path, 'wb') as dst:
                                dst.write(src.read())
                            debug_print(f"OPF Helper: Extracted PDF from ZIP to {pdf_path}")
                            return True
                except Exception as e:
                    debug_print(f"OPF Helper: Failed to extract PDF from ZIP: {str(e)}")

        debug_print("OPF Helper: PDF not found in any location")
        # It's not a fatal error if the PDF is missing
        return False

    except Exception as e:
        debug_print(f"OPF Helper ERROR: PDF extraction failed - {str(e)}")
        import traceback
        debug_print(traceback.format_exc())
        return False

def ensure_resources():
    '''Ensure documentation resources are in place, but make it optional'''
    try:
        # Extract documentation
        if not extract_documentation_from_zip():
            debug_print("OPF Helper: Failed to copy PDF to config dir")
            return False

        debug_print("OPF Helper: Resources installed successfully")
        return True
    except Exception as e:
        debug_print(f"OPF Helper: Error installing resources - {str(e)}")
        import traceback
        traceback.print_exc()
        return False

def open_documentation(parent=None):
    '''Open the PDF documentation in system viewer'''
    try:
        # Check all possible paths where the PDF might be located
        pdf_paths = []
        plugin_dir = os.path.dirname(os.path.abspath(__file__))
        plugin_config_dir = os.path.join(config_dir, 'plugins', plugin_name)

        # Try common locations
        pdf_paths = [
            # Main plugin directory
            os.path.join(plugin_dir, 'opf_helper_docs', 'opf_evolution_diagram.pdf'),
            # Config directory paths
            os.path.join(plugin_config_dir, 'opf_evolution_diagram.pdf'),
            os.path.join(plugin_config_dir, 'opf_helper_docs', 'opf_evolution_diagram.pdf'),
        ]

        # Find the first path that exists
        pdf_path = None
        for path in pdf_paths:
            debug_print(f"OPF Helper: Checking PDF at {path}")
            if os.path.exists(path):
                debug_print(f"OPF Helper: Found PDF at {path}")
                pdf_path = path
                break

        if pdf_path:
            QDesktopServices.openUrl(QUrl.fromLocalFile(pdf_path))
            return True
        else:
            if parent:
                # If the PDF isn't available, offer a web alternative
                from calibre.gui2 import question_dialog
                if question_dialog(parent, 'Documentation Not Found',
                           'The PDF documentation was not found. Would you like to view the online OPF documentation instead?'):
                    QDesktopServices.openUrl(QUrl('https://wiki.mobileread.com/wiki/OPF'))
                    return True
            return False
    except Exception as e:
        debug_print(f"Error opening documentation: {str(e)}")
        import traceback
        traceback.print_exc()
        if parent:
            error_dialog(parent, 'Error Opening Documentation',
                       f'Failed to open documentation: {str(e)}',
                       show=True)
        return False

def create_menu_action_unique(ia, parent_menu, menu_text, image=None, tooltip=None,
                            shortcut=None, triggered=None, is_checked=None):
    '''Create a menu action with unique name for shortcut purposes'''
    if menu_text is not None:
        ac = ia.create_menu_action(parent_menu, menu_text, menu_text,
                                 icon=None, shortcut=shortcut,
                                 description=tooltip, triggered=triggered)
    if image:
        ac.setIcon(get_icon(image))
    if is_checked is not None:
        ac.setCheckable(True)
        if is_checked:
            ac.setChecked(True)
    return ac