#!/usr/bin/env python
from __future__ import (unicode_literals, division, absolute_import, print_function)

__license__   = 'GPL v3'
__copyright__ = '2024'
__docformat__ = 'restructuredtext en'


import os
import sys
import traceback
import shutil
from calibre.customize import InterfaceActionBase
from calibre.constants import config_dir

# Debug control
DEBUG_OPF_HELPER = False  # Set True for verbose logs
def debug_print(*args, **kwargs):
    if DEBUG_OPF_HELPER:
        from calibre.utils.logging import default_log
        default_log(*args, **kwargs)

# Expose ValidationPanel for importing
from .ValidationPanel import ValidationPanel
__all__ = ['ValidationPanel', 'OPFHelperPlugin']

PLUGIN_NAME = 'OPF Helper'
PLUGIN_VERSION = (1, 0, 2)
PLUGIN_DESCRIPTION = 'A toolkit for inspecting, validating, and managing OPF files in your EPUB library, with advanced search and export feature'

class OPFHelperPlugin(InterfaceActionBase):
    name                = PLUGIN_NAME
    description         = PLUGIN_DESCRIPTION
    supported_platforms = ['windows', 'osx', 'linux']
    author             = 'Comfy.n'
    version            = PLUGIN_VERSION
    minimum_calibre_version = (6, 25, 0)
    actual_plugin       = 'calibre_plugins.opf_helper.ShowOPFPlugin:ShowOPFAction'

    def initialize(self):
        '''Initialize the plugin'''


        try:
            # Let parent initialize first
            super().initialize()

            # Get plugin path
            plugin_path = os.path.dirname(os.path.abspath(__file__))


            # Import required modules
            from . import schema_utils


            # Install schemas first - required
            if not schema_utils.verify_schemas():
                debug_print('OPFHelper: Installing schemas...')
                if not schema_utils.install_schemas():
                    debug_print('OPFHelper ERROR: Schema installation failed')
                    raise Exception("Failed to install schema files")
                debug_print('OPFHelper: Schema installation successful')
            else:
                debug_print('OPFHelper: Schemas verified successfully')

            # Extract documentation - Optional
            try:
                from .common_utils import ensure_resources
                debug_print('OPFHelper: Starting documentation resource installation')
                if not ensure_resources():
                    debug_print('OPFHelper WARNING: Documentation resource installation failed - continuing anyway')
                    # Documentation is optional - don't raise an exception
                else:
                    debug_print('OPFHelper: Documentation resources installed successfully')
            except Exception as e:
                debug_print(f'OPFHelper WARNING: Documentation error - {str(e)}')
                debug_print(traceback.format_exc())
                # Documentation is optional - don't raise an exception


        except Exception as e:
            debug_print('OPFHelper ERROR: Initialization failed -', str(e))
            debug_print('OPFHelper ERROR stack trace:')
            debug_print(traceback.format_exc())
            raise

    def is_customizable(self):
        return True

    def config_widget(self):
        # Import here to avoid circular import
        from calibre_plugins.opf_helper.config import ConfigWidget
        return ConfigWidget()

    def save_settings(self, config_widget):
        config_widget.save_settings()
