#!/usr/bin/env python

from PyQt5.QtWidgets import (QWidget, QVBoxLayout, QTextEdit, QPushButton, QLabel,
                          QHBoxLayout, QSpacerItem, QSizePolicy, QApplication)
from PyQt5.QtGui import QFont, QColor, QTextCursor, QTextCharFormat
from PyQt5.QtCore import Qt, QTimer
from calibre.gui2 import QIcon, I
from calibre.utils.config import JSONConfig

class ValidationPanel(QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)
        self._validate_callback = None

        # Load preferences
        self.prefs = JSONConfig('plugins/OPF_Helper')

        # Check if we're in dark mode
        try:
            from calibre.gui2 import is_dark_theme
            self.is_dark = is_dark_theme()
        except:
            self.is_dark = False

        # Define colors based on theme
        if self.is_dark:
            self.success_color = QColor("#90EE90")  # Light green
            self.error_color = QColor("#FF9D88")    # Light red
            self.warning_color = QColor("#FFB366")  # Light orange
            self.info_color = QColor("#88CCFF")     # Light blue
        else:
            self.success_color = QColor("#006400")  # Dark green
            self.error_color = QColor("#8B0000")    # Dark red
            self.warning_color = QColor("#A0522D")  # Brown
            self.info_color = QColor("#000080")     # Navy blue

        self.setup_ui()

    def setup_ui(self):
        layout = QVBoxLayout()
        self.setLayout(layout)

        # Add info label at the top
        info_label = QLabel("OPF validation results will appear here." +
                          (" (Auto-validation enabled)" if self.prefs.get('auto_validate', True) else ""))
        info_label.setWordWrap(True)
        layout.addWidget(info_label)

        # Results text area
        self.results = QTextEdit()
        self.results.setReadOnly(True)
        font = QFont("Courier New", 10)
        self.results.setFont(font)
        layout.addWidget(self.results)

        # Bottom toolbar
        toolbar = QHBoxLayout()

        # Validation button
        self.validate_button = QPushButton("Validate Now")
        self.validate_button.setIcon(QIcon(I('ok.png')))
        self.validate_button.clicked.connect(self._on_validate_clicked)
        toolbar.addWidget(self.validate_button)

        # Add spacer
        toolbar.addItem(QSpacerItem(20, 10, QSizePolicy.Expanding, QSizePolicy.Minimum))

        # Copy button
        self.copy_button = QPushButton("Copy Results")
        self.copy_button.setIcon(QIcon(I('edit-copy.png')))
        self.copy_button.clicked.connect(self.copy_to_clipboard)
        toolbar.addWidget(self.copy_button)

        layout.addLayout(toolbar)

    def set_validate_callback(self, callback):
        """Set the validation callback function"""
        self._validate_callback = callback
        # Note: Auto-validation is now handled in load_current_book, not here

    def _on_validate_clicked(self):
        """Internal handler for validate button click"""
        if self._validate_callback:
            self._validate_callback()

    def set_results(self, text):
        """Update validation results with color highlighting"""
        self.results.clear()
        cursor = self.results.textCursor()

        for line in text.split('\n'):
            fmt = QTextCharFormat()

            # Color based on line content
            if line.startswith('✓'):
                fmt.setForeground(self.success_color)
            elif line.startswith('❌'):
                fmt.setForeground(self.error_color)
            elif line.startswith('⚠️'):
                fmt.setForeground(self.warning_color)
            elif line.strip().endswith(':'):
                fmt.setForeground(self.info_color)

            # Add text with format
            cursor.insertText(line, fmt)
            cursor.insertText('\n')

        # Move cursor to start
        cursor.setPosition(0)
        self.results.setTextCursor(cursor)

    def copy_to_clipboard(self):
        """Copy validation results to clipboard"""
        QApplication.clipboard().setText(self.results.toPlainText())