#!/usr/bin/env python
from io import StringIO
import os
import re
import traceback
import zipfile

from calibre import prints
from calibre.constants import DEBUG
from xml.etree import ElementTree as ET
from lxml import etree
from calibre.constants import numeric_version, DEBUG
from calibre.gui2.actions import InterfaceAction
from calibre.gui2 import error_dialog, info_dialog, I, gprefs
from calibre.constants import cache_dir
try:
	from calibre_plugins.opf_helper import DEBUG_OPF_HELPER
	def debug_print(*args, **kwargs):
		if DEBUG_OPF_HELPER:
			from calibre.utils.logging import default_log
			default_log(*args, **kwargs)
except Exception:
	def debug_print(*args, **kwargs):
		pass
from calibre.utils.localization import _

from calibre_plugins.opf_helper.common_icons import set_plugin_icon_resources, get_icon
from calibre_plugins.opf_helper.ValidationPanel import ValidationPanel
from calibre_plugins.opf_helper.cover_display import CoverPanel
from calibre_plugins.opf_helper.widgets import ElidedLabel
from calibre_plugins.opf_helper.schema_utils import (verify_schemas, install_schemas,
												   load_schema, basic_opf_validation,
												   get_schema_parser, SchemaResolver)
from calibre_plugins.opf_helper.config import prefs
from calibre_plugins.opf_helper.opf_comparison_dialog import OPFComparisonDialog

try:
	from calibre.gui2 import open_url
	from calibre.gui2.actions import menu_action_unique_name
	from functools import partial
except ImportError:
	open_url = None

try:
	# First try importing from Calibre's Qt wrapper (recommended for plugins)
	from calibre.gui2.qt.core import (Qt, QTimer, QUrl, QT_VERSION_STR)
	from calibre.gui2.qt.widgets import (QDialog, QVBoxLayout, QTextEdit, QPushButton,
									  QApplication, QLabel, QMessageBox, QTreeWidget,
									  QTreeWidgetItem, QSplitter, QLineEdit, QHBoxLayout,
									  QListWidget, QDialogButtonBox, QSpacerItem, QSizePolicy,
									  QWidget, QFrame, QTabWidget, QGroupBox, QGridLayout,
									  QMenu, QAction, QTabBar, QComboBox, QProgressDialog)
	from calibre.gui2.qt.gui import (QFont, QSyntaxHighlighter, QTextCharFormat, QColor,
								   QTextCursor, QTextDocument, QPainter, QPen, QPixmap,
								   QRect, QSize)
	from calibre.gui2.qt.webengine import QIcon, QDesktopServices
except ImportError:
	# Fall back to PyQt5 if running outside Calibre or with older Calibre versions
	from PyQt5.QtWidgets import (QDialog, QVBoxLayout, QTextEdit, QPushButton,
							   QApplication, QLabel, QMessageBox, QTreeWidget,
							   QTreeWidgetItem, QSplitter, QLineEdit, QHBoxLayout,
							   QListWidget, QDialogButtonBox, QSpacerItem, QSizePolicy,
							   QWidget, QFrame, QTabWidget, QGroupBox, QGridLayout,
							   QMenu, QAction, QTabBar, QComboBox, QProgressDialog)
	from PyQt5.QtGui import (QFont, QSyntaxHighlighter, QTextCharFormat, QColor,
						   QTextCursor, QTextDocument, QPainter, QPen, QPixmap,
						   QIcon)
	from PyQt5.QtCore import Qt, QT_VERSION_STR, QTimer, QUrl, QRect, QSize
	from PyQt5.QtGui import QDesktopServices  # Add fallback import for QDesktopServices

PLUGIN_ICONS = ['images/icon-for-dark-theme.png', 'images/icon-for-light-theme.png', 'images/xml_error.png']


def create_menu_action_unique(ia, parent_menu, menu_text, image=None, tooltip=None,
						   shortcut=None, triggered=None, is_checked=None, unique_name=None):
	"""Create a menu action with the specified criteria and action, using a unique name"""
	if menu_text:
		ac = QAction(menu_text, parent_menu)
		# Store shortcut but don't show it in menu text
		ac.setProperty('shortcut_display', shortcut)
	if triggered is not None:
		ac.triggered.connect(triggered)
	if is_checked is not None:
		ac.setCheckable(True)
		if is_checked:
			ac.setChecked(True)
	if image:
		ac.setIcon(get_icon(image))
	if shortcut:
		if len(shortcut) == 2:
			ac.setShortcuts(shortcut)
		else:
			ac.setShortcut(shortcut)
	if tooltip:
		ac.setToolTip(tooltip)
	parent_menu.addAction(ac)
	return ac

def create_open_cover_with_menu(parent_menu):
	m = QMenu(_('Open cover with...'))
	ac = QAction(_('Choose program...'), parent_menu)
	ac.triggered.connect(parent_menu.choose_open_with)
	m.addAction(ac)
	return m

def copy_to_clipboard(pixmap):
	if not pixmap.isNull():
		QApplication.clipboard().setPixmap(pixmap)

def contextMenuEvent(self, ev):
	cm = QMenu(self)
	copy = cm.addAction(QIcon(get_icon('edit-copy.png')), _('Copy cover'))
	copy.triggered.connect(lambda: copy_to_clipboard(self.pixmap))
	cm.addMenu(create_open_cover_with_menu(self))
	cm.exec(ev.globalPos())

def choose_open_with(self):
	from calibre.gui2.open_with import choose_program
	entry = choose_program('cover_image', self)
	if entry is not None:
		self.open_with(entry)

def open_with(self, entry):
	book_id = self.data.get('id', None)
	if book_id is not None:
		self.open_cover_with.emit(book_id, entry)

if False:
	# This is here to keep my python error checker from complaining about
	# the builtin functions that will be defined by the plugin loading system
	get_icons = get_resources = None

# The minimum Calibre version required (6.0.0)
MINIMUM_CALIBRE_VERSION = (6, 0, 0)

class XMLHighlighter(QSyntaxHighlighter):
	def __init__(self, parent=None):
		super().__init__(parent)

		# Check if we're in dark mode
		try:
			from calibre.gui2 import is_dark_theme
			self.is_dark = is_dark_theme()
		except:
			self.is_dark = False

		# Define colors based on theme
		if self.is_dark:
			tag_color = "#88CCFF"     # Light blue
			attr_color = "#FFB366"    # Light orange
			value_color = "#90EE90"   # Light green
			comment_color = "#999999"  # Gray
		else:
			tag_color = "#000080"     # Navy blue
			attr_color = "#A0522D"    # Brown
			value_color = "#006400"   # Dark green
			comment_color = "#808080"  # Gray

		# XML element format
		tag_format = QTextCharFormat()
		tag_format.setForeground(QColor(tag_color))
		self.highlighting_rules = [(r'<[!?]?[a-zA-Z0-9_:-]+|/?>', tag_format)]

		# XML attribute format
		attribute_format = QTextCharFormat()
		attribute_format.setForeground(QColor(attr_color))
		self.highlighting_rules.append((r'\s[a-zA-Z0-9_:-]+(?=\s*=)', attribute_format))

		# XML value format
		value_format = QTextCharFormat()
		value_format.setForeground(QColor(value_color))
		self.highlighting_rules.append((r'"[^"]*"', value_format))

		# Comment format
		comment_format = QTextCharFormat()
		comment_format.setForeground(QColor(comment_color))
		self.highlighting_rules.append((r'<!--[\s\S]*?-->', comment_format))

		# Compile regex patterns for better performance
		import re
		self.rules = [(re.compile(pattern), fmt) for pattern, fmt in self.highlighting_rules]

	def highlightBlock(self, text):
		"""Apply syntax highlighting to the given block of text."""
		for pattern, format in self.rules:
			for match in pattern.finditer(text):
				start, length = match.start(), match.end() - match.start()
				self.setFormat(start, length, format)

class OPFChooserDialog(QDialog):
	def __init__(self, parent, opf_files):
		QDialog.__init__(self, parent)
		self.setWindowTitle('Choose OPF File')
		self.setMinimumWidth(400)

		layout = QVBoxLayout()
		self.setLayout(layout)

		# Add description
		desc = QLabel("Multiple OPF files found. Please choose one to view:")
		layout.addWidget(desc)

		# Create list widget
		self.list_widget = QListWidget()
		self.list_widget.addItems(opf_files)
		self.list_widget.setCurrentRow(0)
		layout.addWidget(self.list_widget)

		# Add standard buttons
		button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
		button_box.accepted.connect(self.accept)
		button_box.rejected.connect(self.reject)
		layout.addWidget(button_box)

	def get_selected_opf(self):
		"""Return the selected OPF file path"""
		current_item = self.list_widget.currentItem()
		if current_item:
			return current_item.text()
		return None

class StatsPanel(QWidget):
	def __init__(self, parent=None):
		super().__init__(parent)

		# Check if we're in dark mode
		try:
			from calibre.gui2 import is_dark_theme
			self.is_dark = is_dark_theme()
		except:
			self.is_dark = False

		# Define metadata field colors based on theme
		# Use a less vivid, theme-adaptive orange for 'language' (EPUB 3.0)
		from calibre.gui2 import QApplication
		palette = QApplication.palette()
		if self.is_dark:
			self.metadata_colors = {
				'title': "#fbff17",
				'creator': "#50E3C2",
				'publisher': "#FF9D88",
				'subject': "#90EE90",
				'language': palette.color(palette.Highlight).name(),  # theme-adaptive
				'identifier': "#FF99CC",
				'date': "#B19CD9",
				'description': "#88CCFF"
			}
		else:
			self.metadata_colors = {
				'title': "#8B4513",
				'creator': "#1A3333",
				'publisher': "#8B0000",
				'subject': "#004225",
				'language': palette.color(palette.Highlight).name(),  # theme-adaptive
				'identifier': "#2A0066",
				'date': "#483D8B",
				'description': "#00008B"
			}

		self.setup_ui()

	def setup_ui(self):
		layout = QVBoxLayout()
		self.setLayout(layout)

		# Add stats sections
		general_group = QGroupBox("General Statistics")
		general_layout = QGridLayout()
		general_group.setLayout(general_layout)
		layout.addWidget(general_group)

		self.total_elements = QLabel("Total Elements: 0")
		self.max_depth = QLabel("Maximum Depth: 0")
		self.namespaces = QLabel("Namespaces: 0")

		general_layout.addWidget(self.total_elements, 0, 0)
		general_layout.addWidget(self.max_depth, 1, 0)
		general_layout.addWidget(self.namespaces, 2, 0)

		# Metadata section
		metadata_group = QGroupBox("Metadata Fields")
		metadata_layout = QVBoxLayout()
		metadata_group.setLayout(metadata_layout)
		layout.addWidget(metadata_group)

		self.metadata_tree = QTreeWidget()
		self.metadata_tree.setHeaderLabels(["Field", "Value"])
		self.metadata_tree.setAlternatingRowColors(True)

		# Set up header style
		header = self.metadata_tree.header()
		header.setDefaultAlignment(Qt.AlignLeft)
		header.setStretchLastSection(True)

		metadata_layout.addWidget(self.metadata_tree)

		# Referenced files section
		files_group = QGroupBox("Referenced Files")
		files_layout = QVBoxLayout()
		files_group.setLayout(files_layout)
		layout.addWidget(files_group)

		self.files_list = QTreeWidget()
		self.files_list.setHeaderLabels(["Type", "Path"])
		self.files_list.setAlternatingRowColors(True)
		files_layout.addWidget(self.files_list)

		# Add stretcher at bottom
		layout.addStretch()

	def update_stats(self, root):
		"""Update statistics display for given XML root element"""
		# Reset all stats
		self.metadata_tree.clear()
		self.files_list.clear()

		if root is None:
			return

		# Calculate general stats
		total_elements = 0
		max_depth = 0
		namespaces = set()

		def count_stats(element, depth=0):
			nonlocal total_elements, max_depth
			total_elements += 1
			max_depth = max(max_depth, depth)

			# Track namespaces
			if element.tag and isinstance(element.tag, str) and element.tag.startswith("{"):
				try:
					ns = element.tag[1:].split("}")[0]
					namespaces.add(ns)
				except (IndexError, AttributeError):
					# Skip elements with malformed tags
					pass

			# Recurse through children
			for child in element:
				count_stats(child, depth + 1)

		count_stats(root)

		# Update labels
		self.total_elements.setText(f"Total Elements: {total_elements}")
		self.max_depth.setText(f"Maximum Depth: {max_depth}")
		self.namespaces.setText(f"Namespaces: {len(namespaces)}")

		# Extract and display metadata
		metadata = root.find(".//{http://www.idpf.org/2007/opf}metadata") or root.find(".//metadata")
		if metadata is not None:
			dc_ns = "{http://purl.org/dc/elements/1.1/}"
			# Common DC metadata fields to look for
			dc_fields = ["title", "creator", "publisher", "date", "identifier", "language", "subject"]

			for field in dc_fields:
				# Try with DC namespace first
				elements = metadata.findall(f".//{dc_ns}{field}") or metadata.findall(f".//{field}")
				if elements:
					for elem in elements:
						item = QTreeWidgetItem(self.metadata_tree)
						# Remove namespace from tag if present
						tag = elem.tag
						if isinstance(tag, str) and "}" in tag:
							tag = tag.split("}")[1]
						item.setText(0, tag)
						item.setText(1, elem.text if elem.text else "")

						# Apply color only to the value column if it's a known metadata field
						if tag.lower() in self.metadata_colors:
							color = QColor(self.metadata_colors[tag.lower()])
							item.setForeground(1, color)  # Only color the value column

						# Add any attributes as child items
						if elem.attrib:
							for key, value in elem.attrib.items():
								if isinstance(key, str) and key.startswith("{"):
									key = key.split("}")[1]
								attr_item = QTreeWidgetItem(item)
								attr_item.setText(0, f"@{key}")
								attr_item.setText(1, value)

						# Expand metadata items by default
						item.setExpanded(True)

		# Find referenced files
		manifest = root.find(".//{http://www.idpf.org/2007/opf}manifest") or root.find(".//manifest")
		if manifest is not None:
			media_map = {}  # Group files by media-type
			for item in manifest.findall(".//item"):
				media_type = item.get("media-type", "unknown")
				href = item.get("href", "")
				if media_type not in media_map:
					media_map[media_type] = []
				media_map[media_type].append(href)

			# Add to tree grouped by type
			for media_type, hrefs in media_map.items():
				type_item = QTreeWidgetItem(self.files_list)
				type_item.setText(0, media_type)
				type_item.setText(1, f"{len(hrefs)} file(s)")

				for href in sorted(hrefs):
					file_item = QTreeWidgetItem(type_item)
					file_item.setText(1, href)

		# Expand both trees
		self.metadata_tree.expandAll()
		self.files_list.expandAll()

class OPFContentDialog(QDialog):
	def __init__(self, gui, book_ids, db, icon=None):
		QDialog.__init__(self, gui)

		# Store gui reference
		self.gui = gui

		# Dictionary to store selected OPF file for each book ID
		self.book_opf_selections = {}

		# Store all available OPFs for current book
		self.current_book_opfs = []
		self.current_opf_path = None

		# Initialize xml_content to prevent attribute errors
		self.xml_content = ""

		# Counter for XML parsing issues
		self.xml_parsing_error_count = 0

		# Load preferences
		from calibre.utils.config import JSONConfig
		self.prefs = JSONConfig('plugins/OPF_Helper')
		self.current_font_size = self.prefs.get('font_size', 10)
		self.font_size_increment = self.prefs.get('font_size_increment', 2)

		# Load configuration settings
		self.show_cover = self.prefs.get('show_cover', True)
		self.show_book_id = self.prefs.get('show_book_id', True)

		# Default tab order configuration - ensure Validation tab is included
		self.tab_order = ['OPF Tree', 'Statistics', 'Validation', 'XML', 'Resources', 'About']

		# Load saved tab order from preferences if available
		self.prefs.defaults['tab_order'] = self.tab_order
		self.tab_order = self.prefs.get('tab_order', self.tab_order)

		# Make sure Validation tab is in the order if missing
		if 'Validation' not in self.tab_order:
			self.tab_order.insert(-1, 'Validation')  # Insert before XML tab

		# Ensure About tab is always included
		if 'About' not in self.tab_order:
			self.tab_order.append('About')

		# Set dialog icon
		if icon and not icon.isNull():
			self.setWindowIcon(icon)

		# Store other attributes before registering shortcuts
		self.db = db
		# Use all visible book ids in the current library view for navigation (Calibre 8.x+ compatible)
		# rowCount() in Calibre 8.x requires a parent argument (QModelIndex())
		model = self.gui.library_view.model()
		try:
			row_count = model.rowCount()
		except TypeError:
			# For Calibre 8.x, must pass parent argument
			from PyQt5.QtCore import QModelIndex
			row_count = model.rowCount(QModelIndex())
		self.book_ids = [model.id(row) for row in range(row_count)]
		debug_print(f'OPFHelper: Constructor - populated book_ids with {len(self.book_ids)} books from library view')
		debug_print(f'OPFHelper: Constructor - first 5 book_ids: {self.book_ids[:5] if len(self.book_ids) > 0 else []}')

		# Find the current book index in the full library list
		if book_ids and len(book_ids) > 0:
			current_book_id = book_ids[0]  # The selected book
			try:
				self.current_book_index = self.book_ids.index(current_book_id)
				debug_print(f'OPFHelper: Constructor - set current_book_index to {self.current_book_index} for book_id {current_book_id}')
			except ValueError:
				# Selected book not in library view, start at beginning
				self.current_book_index = 0
				debug_print(f'OPFHelper: Constructor - selected book {current_book_id} not found in library view, starting at index 0')
		else:
			self.current_book_index = 0

		# Set up keyboard shortcuts - these will be overridden by any user customizations
		self.register_default_shortcuts()


		# Create all widgets first (cover, tree, stats, validation, education)
		self.cover_panel = CoverPanel(self)

		self.text_edit = QTextEdit()
		self.text_edit.setReadOnly(True)
		font = QFont("Courier New", self.current_font_size)
		self.text_edit.setFont(font)

		self.tree_widget = QTreeWidget()
		self.tree_widget.setHeaderLabel("XML Structure")
		self.tree_widget.setMinimumWidth(200)
		self.tree_widget.itemClicked.connect(self.on_tree_item_clicked)

		self.stats_panel = StatsPanel()
		self.validation_panel = ValidationPanel(self)
		self.validation_panel.set_validate_callback(self.validate_opf)
		from .education_panel import EducationPanel
		self.education_panel = EducationPanel(self)

		# Create About tab
		self.about_tab = self.setup_about_tab()

		# Now store tab references with more descriptive names
		self.tabs = {
			'OPF Tree': (self.tree_widget, "OPF Tree"),
			'Statistics': (self.stats_panel, "Statistics"),
			'Validation': (self.validation_panel, "Validation"),
			'Resources': (self.education_panel, "Resources"),
			'XML': (self.text_edit, "XML"),
			'About': (self.about_tab, "About")
		}

		# Create custom tab bar subclass for bold+italic selected tabs
		class BoldSelectedTabBar(QTabBar):
			def tabTextColor(self, index):
				if self.currentIndex() == index:
					font = self.font()
					font.setWeight(QFont.Weight.Bold)
					font.setItalic(True)
					old_text = self.tabText(index)
					self.setTabText(index, old_text)  # This triggers font update
					self.setFont(font)
				else:
					font = self.font()
					font.setWeight(QFont.Weight.Normal)
					font.setItalic(False)
					old_text = self.tabText(index)
					self.setTabText(index, old_text)  # This triggers font update
					self.setFont(font)
				return super().tabTextColor(index)

		# Check if we're in dark mode
		try:
			from calibre.gui2 import is_dark_theme
			self.is_dark = is_dark_theme()
		except:
			self.is_dark = False

		# Define element colors based on theme
		if self.is_dark:
			# Bright colors for dark mode
			self.element_colors = {
				'title': "#fbff17",      # Bright yellow
				'creator': "#50E3C2",    # Bright teal
				'publisher': "#FF9D88",   # Bright salmon
				'subject': "#90EE90",    # Light green
				'language': "#FFB366",    # Bright orange
				'identifier': "#FF99CC",  # Bright pink
				'date': "#B19CD9",       # Light purple
				'description': "#88CCFF"  # Bright light blue
			}
		else:
			# Adjusted darker colors for light mode
			self.element_colors = {
				'title': "#8B4513",      # Saddle Brown (darker)
				'creator': "#1A3333",    # Darker slate gray
				'publisher': "#8B0000",   # Dark Red (unchanged)
				'subject': "#004225",    # Darker forest green
				'language': "#A0522D",   # Sienna (unchanged)
				'identifier': "#2A0066",  # Darker indigo
				'date': "#483D8B",       # Dark Slate Blue (unchanged)
				'description': "#00008B"  # Dark Blue (unchanged)
			}

		# Add references dictionary
		self.references = {
			'EPUB 3.0': 'https://www.w3.org/TR/epub-33/',
			'EPUB 2.0': 'http://idpf.org/epub/20/spec/OPF_2.0.1_draft.htm',
			'OPF 3.0 Specification': 'https://www.w3.org/TR/epub-33/#sec-package-doc',
			'OPF 2.0 Specification': 'http://idpf.org/epub/20/spec/OPF_2.0.1_draft.htm',
			'Dublin Core Metadata': 'https://www.dublincore.org/specifications/dublin-core/dces/',
			'MobileRead OPF Wiki': 'https://wiki.mobileread.com/wiki/OPF'
		}

		# Add help text dictionary
		self.opf_help = {
			'package': 'Root element of the OPF file. Contains metadata, manifest, and spine sections.',
			'metadata': 'Contains all book metadata like title, author, etc. Uses Dublin Core elements.',
			'manifest': 'Lists all files (content documents, images, etc.) that are part of the publication.',
			'spine': 'Defines the reading order of content documents.',
			'guide': 'Optional element providing links to key structural components (EPUB 2.0).',
			'title': 'The title of the publication.',
			'creator': 'The primary author or creator of the publication.',
			'contributor': 'Other contributors to the content (e.g., illustrator, editor).',
			'publisher': 'The publisher of the publication.',
			'date': 'Publication date, typically in YYYY[-MM[-DD]] format.',
			'language': 'The language of the content (RFC 3066 language codes).',
			'identifier': 'Unique identifier for the publication (e.g., ISBN, UUID).',
			'subject': 'Subject categories or keywords describing the content.',
			'description': 'A description or summary of the publication.',
			'rights': 'Copyright and licensing information.',
			'item': 'Manifest entry representing a publication resource (content, images, etc).',
			'itemref': 'Spine entry referencing a manifest item, defining reading order.',
			'reference': 'Guide entry pointing to a key structural component.',
			# Attributes
			'id': 'Unique identifier within the OPF file',
			'href': 'Path to the referenced file',
			'media-type': 'MIME type of the referenced file',
			'unique-identifier': 'References the primary identifier in metadata',
			'version': 'EPUB specification version (2.0 or 3.0)',
			'toc': 'References the navigation control file (NCX/Nav)',
			'properties': 'Special features of the referenced item (EPUB 3.0)',
			'fallback': 'Alternative version if primary cannot be rendered',
			'linear': 'Whether item is part of primary reading order (yes/no)'
		}

		# Check Calibre version
		if numeric_version < MINIMUM_CALIBRE_VERSION:
			error_dialog(gui, 'Version Error',
						'This plugin requires Calibre 6.0.0 or later.',
						show=True)
			self.close()
			return

		self.gui = gui
		self.db = db
		# book_ids and current_book_index are already set above

		# Initialize schema parsers
		self.schema_parsers = {}
		self.initialize_schemas()

		# Initialize UI first
		layout = QVBoxLayout()
		layout.setSpacing(5)
		layout.setContentsMargins(5, 5, 5, 5)
		self.setLayout(layout)

		# Top toolbar with book navigation
		nav_layout = QHBoxLayout()
		nav_layout.setSpacing(2)  # Reduce spacing
		nav_layout.setContentsMargins(0, 0, 0, 0)  # Remove margins

		# Create navigation buttons with compact design
		self.first_book_button = QPushButton(get_icon('images/arrow-double-left.png'), '', self)
		self.first_book_button.setFixedWidth(32)
		self.first_book_button.setToolTip("First Book")
		self.first_book_button.clicked.connect(self.show_first_book)
		nav_layout.addWidget(self.first_book_button)

		self.prev_book_button = QPushButton(get_icon('images/arrow-previous.png'), '', self)
		self.prev_book_button.setFixedWidth(32)
		self.prev_book_button.setToolTip("Previous Book")
		self.prev_book_button.clicked.connect(lambda: (debug_print('OPFHelper: Previous button clicked'), self.show_previous_book()))
		nav_layout.addWidget(self.prev_book_button)

		# Add title label that elides text in the middle
		self.book_label = ElidedLabel()
		self.book_label.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
		self.book_label.setAlignment(Qt.AlignCenter)
		font = self.book_label.font()
		font.setPointSize(9)
		self.book_label.setFont(font)
		nav_layout.addWidget(self.book_label, 1)  # Give expanding space

		self.next_book_button = QPushButton(get_icon('images/arrow-next.png'), '', self)
		self.next_book_button.setFixedWidth(32)
		self.next_book_button.setToolTip("Next Book")
		self.next_book_button.clicked.connect(lambda: (debug_print('OPFHelper: Next button clicked'), self.show_next_book()))
		nav_layout.addWidget(self.next_book_button)

		self.last_book_button = QPushButton(get_icon('images/arrow-double-right.png'), '', self)
		self.last_book_button.setFixedWidth(32)
		self.last_book_button.setToolTip("Last Book")
		self.last_book_button.clicked.connect(self.show_last_book)
		nav_layout.addWidget(self.last_book_button)

		# Add navigation layout to main layout
		layout.addLayout(nav_layout)

		# Add OPF selector layout - will be shown only when multiple OPFs exist
		opf_selector_layout = QHBoxLayout()
		opf_selector_layout.setSpacing(5)

		# Add label for OPF selector
		opf_label = QLabel("OPF File:")
		opf_selector_layout.addWidget(opf_label)

		# Create OPF selector combo box
		self.opf_selector = QComboBox()
		self.opf_selector.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
		self.opf_selector.setToolTip("Select which OPF file to view")
		self.opf_selector.currentIndexChanged.connect(self.on_opf_selection_changed)
		opf_selector_layout.addWidget(self.opf_selector)

		# Add count label
		self.opf_count_label = QLabel("")
		opf_selector_layout.addWidget(self.opf_count_label)

		# Create container for OPF selector that can be shown/hidden
		self.opf_selector_container = QWidget()
		self.opf_selector_container.setLayout(opf_selector_layout)
		self.opf_selector_container.setVisible(False)  # Initially hidden
		layout.addWidget(self.opf_selector_container)

		# Add book info label right after navbar
		self.book_info_label = QLabel()
		self.book_info_label.setWordWrap(True)
		self.book_info_label.setTextFormat(Qt.RichText)
		self.book_info_label.setAlignment(Qt.AlignCenter)
		self.book_info_label.setStyleSheet("""
			QLabel {
				padding: 5px;
				background-color: palette(window);
				border: 1px solid palette(mid);
				border-radius: 3px;
				margin-top: 2px;
				margin-bottom: 2px;
			}
		""")
		layout.addWidget(self.book_info_label)

		# Create toolbar layout
		toolbar_layout = QHBoxLayout()
		toolbar_layout.setSpacing(5)  # Consistent spacing between all buttons

		# Add version label with version-specific colors
		self.version_label = QLabel()
		self.version_label.setStyleSheet("""
			QLabel {
				padding: 2px 8px;
				border-radius: 4px;
				color: white;
				font-weight: bold;
			}
		""")
		toolbar_layout.addWidget(self.version_label)

		# Add all action buttons
		self.add_action_buttons(toolbar_layout)

		layout.addLayout(toolbar_layout)

		# Search section
		search_layout = QHBoxLayout()
		search_layout.setSpacing(5)

		# Add a label for instructions
		info_label = QLabel("Selected EPUB:")  # Changed from "OPF Content for selected EPUB:"
		search_layout.addWidget(info_label)

		# Add search box with more descriptive placeholder
		self.search_box = QLineEdit()
		self.search_box.setPlaceholderText("Find in OPF (XML tab)...")
		self.search_box.textChanged.connect(self.on_search_text_changed)
		self.search_box.returnPressed.connect(self.find_next)
		search_layout.addWidget(self.search_box)

		# Add search options
		self.case_sensitive = QAction("Case sensitive", self)
		self.case_sensitive.setCheckable(True)
		self.case_sensitive.setChecked(prefs.get('search_case_sensitive', False))
		self.case_sensitive.triggered.connect(self.reset_search)
		self.case_sensitive.triggered.connect(self.save_search_options)

		self.whole_words = QAction("Whole words", self)
		self.whole_words.setCheckable(True)
		self.whole_words.setChecked(prefs.get('search_whole_words', False))
		self.whole_words.triggered.connect(self.reset_search)
		self.whole_words.triggered.connect(self.save_search_options)

		self.regex_search = QAction("Regex search", self)
		self.regex_search.setCheckable(True)
		self.regex_search.setChecked(prefs.get('search_regex', False))
		self.regex_search.triggered.connect(self.reset_search)
		self.regex_search.triggered.connect(self.save_search_options)

		# Add search options button
		options_button = QPushButton()
		options_button.setIcon(QIcon.ic('config.png'))
		options_button.setMaximumWidth(30)
		options_button.setToolTip("Search options")
		options_menu = QMenu(self)
		options_menu.addAction(self.case_sensitive)
		options_menu.addAction(self.whole_words)
		options_menu.addAction(self.regex_search)
		options_button.setMenu(options_menu)
		search_layout.addWidget(options_button)

		# Add counter label to show match info
		self.match_label = QLabel("")
		self.match_label.setAlignment(Qt.AlignCenter)
		self.match_label.setMinimumWidth(80)
		search_layout.addWidget(self.match_label)

		# Add search navigation buttons with proper theme-aware icons
		self.prev_button = QPushButton()
		# Use theme-aware icon directly
		self.prev_button.setIcon(QIcon.ic('arrow-up.png'))
		self.prev_button.setMaximumWidth(30)
		self.prev_button.setToolTip("Find previous match")
		self.prev_button.clicked.connect(self.find_previous)
		self.prev_button.setEnabled(False)
		search_layout.addWidget(self.prev_button)

		self.next_button = QPushButton()
		# Use theme-aware icon directly
		self.next_button.setIcon(QIcon.ic('arrow-down.png'))
		self.next_button.setMaximumWidth(30)
		self.next_button.setToolTip("Find next match")
		self.next_button.clicked.connect(self.find_next)
		self.next_button.setEnabled(False)
		search_layout.addWidget(self.next_button)

		layout.addLayout(search_layout)

		# Create main horizontal splitter
		splitter = QSplitter(Qt.Horizontal)
		layout.addWidget(splitter)

		# Create tab widget with custom tab bar
		tab_widget = QTabWidget()
		tab_widget.setTabBar(BoldSelectedTabBar())
		tab_widget.setMovable(True)  # Enable manual tab dragging
		tab_widget.tabBar().tabMoved.connect(self.update_tab_order)  # Handle tab moves

		# Apply CCR-style tab styling
		tab_widget.setStyleSheet("""
			QTabWidget::pane {
				border-top: none;
				   margin-left: 8px;
				   margin-right: 8px;
			}
			QTabBar::tab {
				min-width: 124px;
				width: 3px;
				padding-top: 6px;
				padding-bottom: 6px;
				font-size: 9pt;
				background: transparent;
				border-top: 1px solid palette(mid);
				border-left: 1px solid palette(mid);
				border-right: 1px solid palette(mid);
				border-top-left-radius: 8px;
				border-top-right-radius: 8px;
				/*margin-right: 2px;*/
			}
			QTabBar::tab:selected {
				font-weight: bold;
				font-style: normal;
				border-top: 2px solid palette(link);
				border-left: 2px solid palette(link);
				border-right: 2px solid palette(link);
				color: palette(link);
			}
			QTabBar::tab:!selected {
				color: palette(text);
				margin-top: 2px;
			}
			QScrollBar::handle {
			   border: 1px solid #5B6985; /* subtle border */
			}
			QScrollBar:vertical {
				background: transparent;
				width: 12px;
				margin: 0px 0px 0px 0px;
				/* Reduce reserved top/bottom space so the handle can reach closer to the edges */
				padding: 6px 0px 6px 0px;
			}
			QScrollBar::handle:vertical {
				background: rgba(140, 172, 204, 0.25); /* subtle blue, low opacity */
				min-height: 22px; /* smaller handle height per checkpoint */
				border-radius: 4px;
				/* inset the handle more so it doesn't touch the add/sub slot border */
				margin: 4px 0px;
			}
			QScrollBar::handle:vertical:hover {
				background: rgba(140, 172, 204, 0.45);
			}
			QScrollBar:horizontal {
				background: transparent;
				height: 12px;
				margin: 0px 0px 0px 0px;
				/* Reduce reserved left/right space so the handle can reach closer to the edges */
				padding: 0px 6px 0px 6px;
			}
			QScrollBar::handle:horizontal {
				background: rgba(140, 172, 204, 0.25);
				min-width: 22px; /* smaller handle width per checkpoint */
				border-radius: 4px;
				margin: 0px 4px;
			}
			QScrollBar::handle:horizontal:hover {
				background: rgba(140, 172, 204, 0.45);
			}
			QScrollBar::add-page:vertical, QScrollBar::sub-page:vertical,
			QScrollBar::add-page:horizontal, QScrollBar::sub-page:horizontal {
				background: none;
			}
		""")



		# Add tab widget and cover panel to splitter
		splitter.addWidget(tab_widget)
		if self.show_cover:
			splitter.addWidget(self.cover_panel)
			# Set initial splitter sizes - favor main content
			splitter.setSizes([750, 250])  # 750px for main content, 250px for cover
		else:
			# Hide cover panel completely
			self.cover_panel.hide()

		# Ensure Resources tab exists in order
		if 'Resources' not in self.tab_order:
			xml_index = self.tab_order.index('XML') if 'XML' in self.tab_order else len(self.tab_order)
			self.tab_order.insert(xml_index, 'Resources')

		# Add tabs in configured order, but ensure About tab is always last
		# Remove About from tab_order if present (it will be added last)
		if 'About' in self.tab_order:
			self.tab_order.remove('About')

		# Add all other tabs in configured order
		for tab_name in self.tab_order:
			if tab_name in self.tabs:
				widget, label = self.tabs[tab_name]
				tab_widget.addTab(widget, label)

		# Always add About tab last
		if 'About' in self.tabs:
			widget, label = self.tabs['About']
			tab_widget.addTab(widget, label)

		# Connect validation button after widget creation
		try:
			self.validation_panel.validate_button.clicked.connect(self.validate_opf)
		except Exception:
			pass

		# Try to wire up toolbar validate button if present
		toolbar_validate_button = None
		for item in toolbar_layout.children():
			if isinstance(item, QPushButton) and item.text() == "Validate OPF":
				toolbar_validate_button = item
				break
		if toolbar_validate_button:
			toolbar_validate_button.clicked.connect(self.validate_opf)

		# Store reference to tab widget for context menu
		self.tab_widget = tab_widget

		# Set default tab based on preference
		default_tab = prefs.get('default_tab', 'OPF Tree')
		for i in range(self.tab_widget.count()):
			if self.tab_widget.tabText(i) == default_tab:
				self.tab_widget.setCurrentIndex(i)
				break

		# Enable tab bar context menu
		self.tab_widget.tabBar().setContextMenuPolicy(Qt.CustomContextMenu)
		self.tab_widget.tabBar().customContextMenuRequested.connect(self.show_tab_context_menu)

		# Apply syntax highlighting
		self.highlighter = XMLHighlighter(self.text_edit.document())

		# Initialize search variables
		self.current_search = ""
		self.search_positions = []
		self.current_match = -1

		# Create references section with adjusted link color based on theme
		ref_group = QGroupBox("Specifications && References")
		ref_layout = QHBoxLayout()
		ref_group.setLayout(ref_layout)

		link_color = "#0066CC" if not self.is_dark else "#8AC6F2"  # Darker blue for light mode
		hover_color = "#003366" if not self.is_dark else "#5EA0D9"  # Even darker on hover

		for name, url in self.references.items():
			link = QPushButton(name)
			link.setStyleSheet(f"""
			QPushButton {{
				border: none;
				color: {link_color};
				text-decoration: underline;
				background: transparent;
				text-align: left;
				padding: 0px;
			}}
			QPushButton:hover {{
				color: {hover_color};
			}}
		""")
			link.setCursor(Qt.PointingHandCursor)
			link.clicked.connect(lambda checked, u=url: QDesktopServices.openUrl(QUrl(u)))
			ref_layout.addWidget(link)

		ref_layout.addStretch()
		layout.addWidget(ref_group)

		# Add close button at the bottom using QDialogButtonBox like CCR
		button_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Close)
		button_box.rejected.connect(self.reject)
		layout.addWidget(button_box)

		# Set focus to search box
		self.search_box.setFocus()

		# Setup context menu for tree items
		self.setup_tree_context_menu()

		# Restore/save geometry using gprefs (keep minimum constraints)
		try:
			geom = gprefs.get('opf_helper_dialog_geometry', None)
			if geom:
				self.restoreGeometry(geom)
			else:
				self.resize(1200, 650)
		except Exception:
			# Fallback to defaults
			self.resize(1200, 650)

		self.setMinimumWidth(900)
		self.setMinimumHeight(500)

		# Now load the first book
		self.load_current_book()

	def setup_about_tab(self):
		"""Create the About tab with plugin information"""
		tab = QWidget()
		layout = QHBoxLayout(tab)  # Horizontal layout for columns

		# ===== COLUMN 1: Plugin Description =====
		desc_column = QWidget()
		desc_layout = QVBoxLayout(desc_column)

		# Header
		desc_title = QLabel(_('<b>About OPF Helper</b>'))
		desc_title.setAlignment(Qt.AlignHCenter)
		desc_title.setStyleSheet('font-size: 13pt;')
		desc_layout.addWidget(desc_title)

		# Plugin description
		desc_text = _("OPF Helper is a Calibre plugin that provides comprehensive tools for inspecting, validating, and analyzing OPF (Open Packaging Format) files in EPUB books. It offers a tree view of XML structure, statistics, validation against OPF schemas, educational resources, and EPUB version detection tools.")
		desc_label = QLabel(desc_text)
		desc_label.setWordWrap(True)
		desc_label.setStyleSheet('font-size: 11pt;')
		desc_layout.addWidget(desc_label)

		# Features list
		features_title = QLabel(_('<b>Key Features:</b>'))
		features_title.setStyleSheet('font-size: 12pt; margin-top: 10px;')
		desc_layout.addWidget(features_title)

		features_text = _("""
• Interactive OPF XML tree view with syntax highlighting<br>
• Comprehensive statistics and metadata display<br>
• OPF schema validation for EPUB 2.0 and 3.0<br>
• Educational resources and OPF specification links<br>
• EPUB version detection and library scanning<br>
• Multiple OPF file support<br>
• KEPUB support<br>
• Advanced search and navigation tools
""")
		features_label = QLabel(features_text)
		features_label.setWordWrap(True)
		features_label.setStyleSheet('font-size: 10pt;')
		desc_layout.addWidget(features_label)

		desc_layout.addStretch()
		layout.addWidget(desc_column)

		# ===== COLUMN 2: Links and Support =====
		links_column = QWidget()
		links_layout = QVBoxLayout(links_column)

		# Header
		links_title = QLabel(_('<b>Support & Resources</b>'))
		links_title.setAlignment(Qt.AlignHCenter)
		links_title.setStyleSheet('font-size: 13pt;')
		links_layout.addWidget(links_title)

		# Helper to wire QLabel links to calibre.open_url
		def _wire_label(label):
			try:
				label.setOpenExternalLinks(False)
			except Exception:
				pass
			try:
				label.linkActivated.connect(lambda u: __import__('calibre').gui2.open_url(u))
			except Exception:
				try:
					label.linkActivated.connect(lambda u: QDesktopServices.openUrl(QUrl(u)))
				except Exception:
					pass

		# MobileRead forum link
		mr_container = QWidget()
		mr_layout = QHBoxLayout(mr_container)
		mr_img = QLabel()
		mr_icon = get_icon('images/mobileread')  # Try to get icon from common_icons
		if mr_icon and not mr_icon.isNull():
			mr_img.setPixmap(mr_icon.pixmap(24, 24))
		else:
			mr_img.setText("🔗")
		mr_layout.addWidget(mr_img)
		mr_link = QLabel('<a href="https://www.mobileread.com/forums/showthread.php?t=371086">MobileRead Forum Thread</a>')
		_wire_label(mr_link)
		mr_link.setTextInteractionFlags(Qt.TextBrowserInteraction)
		mr_link.setStyleSheet('font-size: 11pt;')
		mr_layout.addWidget(mr_link)
		mr_layout.addStretch()
		links_layout.addWidget(mr_container)

		# Donate link
		donate_container = QWidget()
		donate_layout = QHBoxLayout(donate_container)
		donate_img = QLabel()
		donate_icon = get_icon('images/donate')  # Try to get icon from common_icons
		if donate_icon and not donate_icon.isNull():
			donate_img.setPixmap(donate_icon.pixmap(24, 24))
		else:
			donate_img.setText("❤️")
		donate_layout.addWidget(donate_img)
		donate_link = QLabel('<a href="https://ko-fi.com/comfy_n">Support Development</a>')
		_wire_label(donate_link)
		donate_link.setTextInteractionFlags(Qt.TextBrowserInteraction)
		donate_link.setStyleSheet('font-size: 11pt;')
		donate_layout.addWidget(donate_link)
		donate_layout.addStretch()
		links_layout.addWidget(donate_container)

		# OPF Specification links
		spec_title = QLabel(_('<b>OPF Specifications:</b>'))
		spec_title.setStyleSheet('font-size: 12pt; margin-top: 15px;')
		links_layout.addWidget(spec_title)

		spec_links = [
			('EPUB 3.0 Specification', 'https://www.w3.org/TR/epub-33/'),
			('EPUB 2.0 Specification', 'http://idpf.org/epub/20/spec/OPF_2.0.1_draft.htm'),
			('Dublin Core Metadata', 'https://www.dublincore.org/specifications/dublin-core/dces/'),
			('MobileRead OPF Wiki', 'https://wiki.mobileread.com/wiki/OPF')
		]

		for title, url in spec_links:
			link = QLabel(f'<a href="{url}">{title}</a>')
			_wire_label(link)
			link.setTextInteractionFlags(Qt.TextBrowserInteraction)
			link.setStyleSheet('font-size: 10pt;')
			links_layout.addWidget(link)

		links_layout.addStretch()
		layout.addWidget(links_column)

		return tab

	def update_tab_order(self, from_index, to_index):
		"""Update the tab order preference when tabs are moved. Debug output included."""
		try:
			debug_print(f"update_tab_order called: from {from_index} to {to_index}")
		except Exception as e:
			print(f"update_tab_order debug print failed: {e}")
		# If you want to persist the new order, implement logic here
		# For now, just print the new order for debugging
		if hasattr(self, 'tab_widget'):
			new_order = [self.tab_widget.tabText(i) for i in range(self.tab_widget.count())]
			try:
				debug_print(f"New tab order: {new_order}")
			except Exception as e:
				print(f"update_tab_order debug print failed: {e}")

	def show_tab_context_menu(self, pos):
		"""Show a simple context menu for tabs (select tab)."""
		try:
			tab_bar = self.tab_widget.tabBar()
			global_pos = tab_bar.mapToGlobal(pos)
			menu = QMenu(self)
			for i in range(self.tab_widget.count()):
				name = self.tab_widget.tabText(i)
				a = QAction(name, self)
				a.setData(i)
				# mark current tab
				a.setCheckable(True)
				a.setChecked(i == self.tab_widget.currentIndex())
				# capture index in default arg
				a.triggered.connect(lambda checked, idx=i: self.tab_widget.setCurrentIndex(idx))
				menu.addAction(a)
			menu.exec_(global_pos)
		except Exception as e:
			debug_print(f"show_tab_context_menu error: {e}")

	def next_tab(self):
		"""Switch to the next tab"""
		try:
			idx = self.tab_widget.currentIndex()
			count = self.tab_widget.count()
			self.tab_widget.setCurrentIndex((idx + 1) % count)
		except Exception:
			pass

	def previous_tab(self):
		"""Switch to the previous tab"""
		try:
			idx = self.tab_widget.currentIndex()
			count = self.tab_widget.count()
			self.tab_widget.setCurrentIndex((idx - 1) % count)
		except Exception:
			pass

	def register_default_shortcuts(self):
		"""Register default keyboard shortcuts"""
		nav_shortcuts = {
			'Alt+Left': self.show_previous_book,
			'Alt+Right': self.show_next_book,
			'Alt+Home': self.show_first_book,
			'Alt+End': self.show_last_book,
			'Ctrl+Alt+E': self.edit_metadata,
			'Ctrl+Alt+V': self.validate_opf,
			'Ctrl+Alt+C': lambda: self.copy_to_clipboard(),
			'Ctrl+Alt+T': self.edit_toc,  # Add shortcut for ToC editor
			'Ctrl++': self.zoom_in,
			'Ctrl+=': self.zoom_in,  # Add alternative zoom in shortcut
			'Ctrl+-': self.zoom_out,
			'Ctrl+Tab': self.next_tab,
			'Ctrl+Shift+Tab': self.previous_tab,
		}

		for shortcut, slot in nav_shortcuts.items():
			action = QAction(self)
			action.setShortcut(shortcut)
			action.triggered.connect(slot)
			self.addAction(action)

		# Register with Calibre's keyboard manager
		from calibre.gui2.keyboard import Manager as KeyboardManager
		km = KeyboardManager(self.gui.keyboard)
		shortcuts = [
			('OPF_Helper_previous_book', _('OPF Helper'), _('Previous Book'), ['Alt+Left']),
			('OPF_Helper_next_book', _('OPF Helper'), _('Next Book'), ['Alt+Right']),
			('OPF_Helper_first_book', _('OPF Helper'), _('First Book'), ['Alt+Home']),
			('OPF_Helper_last_book', _('OPF Helper'), _('Last Book'), ['Alt+End']),
			('OPF_Helper_edit_metadata', _('OPF Helper'), _('Edit Book Metadata'), ['Ctrl+Alt+E']),
			('OPF_Helper_validate_opf', _('OPF Helper'), _('Validate OPF'), ['Ctrl+Alt+V']),
			('OPF_Helper_copy_xml', _('OPF Helper'), _('Copy OPF XML'), ['Ctrl+Alt+C']),
			('OPF_Helper_edit_toc', _('OPF Helper'), _('Edit Table of Contents'), ['Ctrl+Alt+T']),
			('OPF_Helper_zoom_in', _('OPF Helper'), _('Zoom In'), ['Ctrl++', 'Ctrl+=']),
			('OPF_Helper_zoom_out', _('OPF Helper'), _('Zoom Out'), ['Ctrl+-']),
			('OPF_Helper_next_tab', _('OPF Helper'), _('Next Tab'), ['Ctrl+Tab']),
			('OPF_Helper_previous_tab', _('OPF Helper'), _('Previous Tab'), ['Ctrl+Shift+Tab'])
		]

		for name, group, text, default_keys in shortcuts:
			km.register_shortcut(name, group, text, default_keys)

	def find_matches(self, search_text):
		"""Find all occurrences of search text"""
		if not search_text:
			return []

		# For regex search, use Python's regex engine
		if self.regex_search.isChecked():
			try:
				if self.case_sensitive.isChecked():
					pattern = re.compile(search_text)
				else:
					pattern = re.compile(search_text, re.IGNORECASE)

				content = self.text_edit.toPlainText()
				positions = []
				for match in pattern.finditer(content):
					positions.append((match.start(), match.end() - match.start()))
				return positions
			except re.error:
				# Invalid regex, fall back to normal search
				debug_print(f"OPFHelper: Invalid regex pattern: {search_text}")
				return []
		else:
			# Manual search approach to avoid QTextDocument.find() compatibility issues
			content = self.text_edit.toPlainText()
			positions = []

			# Convert to lowercase for case-insensitive search if needed
			if not self.case_sensitive.isChecked():
				content = content.lower()
				search_text = search_text.lower()

			# Find all occurrences
			start_pos = 0
			while True:
				if self.whole_words.isChecked():
					# For whole word search, check word boundaries
					pos = -1
					# Find all potential matches
					while True:
						pos = content.find(search_text, start_pos)
						if pos == -1:
							break

						# Check if this is a whole word
						is_whole_word = True

						# Check character before match
						if pos > 0 and content[pos-1].isalnum():
							is_whole_word = False

						# Check character after match
						end_pos = pos + len(search_text)
						if end_pos < len(content) and content[end_pos].isalnum():
							is_whole_word = False

						if is_whole_word:
							positions.append((pos, len(search_text)))

						# Move to after this occurrence
						start_pos = pos + len(search_text)
				else:
					# Simple search
					pos = content.find(search_text, start_pos)
					if pos == -1:
						break

					positions.append((pos, len(search_text)))
					start_pos = pos + len(search_text)

				if not self.whole_words.isChecked():
					if pos == -1:
						break

			return positions

	def on_search_text_changed(self):
		"""Handle search text changes"""
		search_text = self.search_box.text()
		if not search_text:
			# Clear search when text is empty
			self.current_search = ""
			self.search_positions = []
			self.current_match = -1
			self.prev_button.setEnabled(False)
			self.next_button.setEnabled(False)
			self.match_label.setText("")
			return

		try:
			# Process search if text has changed or options changed
			if search_text != self.current_search:
				self.current_search = search_text
				self.search_positions = self.find_matches(search_text)
				self.current_match = -1

				# Enable/disable navigation buttons
				has_matches = len(self.search_positions) > 0
				self.prev_button.setEnabled(has_matches)
				self.next_button.setEnabled(has_matches)

				# Update match counter
				if has_matches:
					self.match_label.setText(f"1 of {len(self.search_positions)}")
					self.find_next()
				else:
					self.match_label.setText("No matches")
		except Exception as e:
			debug_print(f"OPFHelper ERROR: Search error: {e}")
			self.match_label.setText("Error")

	def find_next(self):
		"""Find and highlight next match"""
		if not self.search_positions:
			return

		self.current_match = (self.current_match + 1) % len(self.search_positions)
		self.highlight_match(self.search_positions[self.current_match])
		# Update match counter
		self.match_label.setText(f"{self.current_match + 1} of {len(self.search_positions)}")

	def find_previous(self):
		"""Find and highlight previous match"""
		if not self.search_positions:
			return

		self.current_match = (self.current_match - 1) % len(self.search_positions)
		self.highlight_match(self.search_positions[self.current_match])
		# Update match counter
		self.match_label.setText(f"{self.current_match + 1} of {len(self.search_positions)}")

	def reset_search(self):
		"""Reset search when options change"""
		if self.search_box.text():
			self.on_search_text_changed()

	def save_search_options(self):
		"""Save the current search option states to preferences"""
		prefs['search_case_sensitive'] = self.case_sensitive.isChecked()
		prefs['search_whole_words'] = self.whole_words.isChecked()
		prefs['search_regex'] = self.regex_search.isChecked()

	def populate_tree(self):
		"""Parse XML and populate the tree widget and stats"""
		self.tree_widget.clear()
		try:
			# Use StringIO for parsing XML string but handle encoding declaration properly
			from io import BytesIO

			# Convert to bytes if it's not already
			xml_bytes = self.xml_content.encode('utf-8') if isinstance(self.xml_content, str) else self.xml_content

			# Parse using BytesIO to avoid encoding declaration issues
			root = ET.parse(BytesIO(xml_bytes)).getroot()

			if root is not None:
				root_item = self.add_element_to_tree(root, self.tree_widget)
				self.stats_panel.update_stats(root)

				# Find and expand only the metadata section
				root_item.setExpanded(True)  # Expand root first
				for i in range(root_item.childCount()):
					child = root_item.child(i)
					if child.text(0).startswith('metadata'):
						child.setExpanded(True)
						# Also expand its immediate children to show all metadata fields
						for j in range(child.childCount()):
							child.child(j).setExpanded(True)
		except ET.ParseError as e:
			# Handle XML parsing errors specifically
			self.xml_parsing_error_count += 1
			error_msg = f'XML parsing error: {str(e)}'
			debug_print(f'OPFHelper ERROR: {error_msg} (Total XML parsing errors: {self.xml_parsing_error_count})')

			# Try to show the problematic area around the error
			try:
				lines = self.xml_content.split('\n')
				if hasattr(e, 'position') and e.position:
					line_num, col_num = e.position
					if line_num <= len(lines):
						line = lines[line_num - 1]
						# Show context around the error
						start = max(0, col_num - 20)
						end = min(len(line), col_num + 20)
						context = line[start:end]
						error_msg += f'\n\nContext around error (line {line_num}, column {col_num}):\n{context}'
						if col_num <= len(line):
							error_msg += '\n' + ' ' * (col_num - start - 1) + '^'
			except Exception as context_error:
				debug_print(f'OPFHelper ERROR: Failed to extract error context: {str(context_error)}')

			# Show detailed error dialog
			error_dialog(self, 'XML Parsing Error',
						f'Failed to parse OPF XML content.\n\n{error_msg}\n\n'
						'This usually indicates an undefined XML entity or malformed XML.',
						show=True)

			# Still try to show the XML content in the text view even if tree fails
			self.text_edit.setPlainText(self.xml_content)

		except Exception as e:
			# Handle other parsing errors
			error_msg = f'Error populating tree: {str(e)}'
			debug_print(f'OPFHelper ERROR: {error_msg}')
			error_dialog(self, 'Tree Population Error',
						error_msg,
						show=True)

	def add_element_to_tree(self, element, parent):
		"""Recursively add XML elements to the tree"""
		namespace = ""
		tag = element.tag
		if tag is None:  # Handle comments and processing instructions
			return None

		if isinstance(tag, str) and tag.startswith("{"):
			try:
				namespace = tag[1:tag.index("}")]
				tag = tag[tag.index("}")+1:]
			except (ValueError, AttributeError):
				# If } not found, keep original tag
				pass

		# Determine element type
		element_type = "element"
		if len(element) > 0:
			element_type = "parent"
		elif element.text and element.text.strip():
			element_type = "text"
		elif not element.attrib and not element.text:
			element_type = "empty"

		display_text = tag
		if namespace:
			display_text = f"{tag} [{namespace}]"

		# Add attributes to display text if present
		if element.attrib:
			attr_list = []
			for key, value in element.attrib.items():
				# Handle namespaced attributes
				attr_name = key
				if isinstance(key, str) and key.startswith("{"):
					try:
						attr_name = key[key.index("}")+1:]
					except (ValueError, AttributeError):
						# If } not found, keep original key
						pass
				attr_list.append(f"{attr_name}=\"{value}\"")
			if attr_list:
				display_text += " (" + ", ".join(attr_list) + ")"

		# Create tree item
		if isinstance(parent, QTreeWidget):
			item = QTreeWidgetItem(parent)
		else:
			item = QTreeWidgetItem(parent)

		item.setText(0, display_text)

		# Add tooltips based on element tag and attributes
		base_tag = tag
		if isinstance(base_tag, str) and base_tag in self.opf_help:
			item.setToolTip(0, self.opf_help[base_tag])

		# Store element data and type
		item.setData(0, Qt.UserRole, element)
		item.setData(0, Qt.UserRole + 1, element_type)

		# Apply custom colors based on element tag
		if isinstance(base_tag, str):
			tag_lower = base_tag.lower()
			if tag_lower in self.element_colors:
				item.setForeground(0, QColor(self.element_colors[tag_lower]))

		# Add child elements
		for child in element:
			child_item = self.add_element_to_tree(child, item)
			if child_item:
				item.addChild(child_item)

		# Add text content if present
		if element.text and element.text.strip():
			text_item = QTreeWidgetItem(item)
			text_item.setText(0, element.text.strip())
			text_item.setData(0, Qt.UserRole + 1, "text")
			text_item.setData(0, Qt.UserRole + 2, element.text.strip())

			# Color the text value if parent is a metadata field
			if tag_lower in self.element_colors:
				color = QColor(self.element_colors[tag_lower])
				text_item.setForeground(0, color)  # Apply color to the text value

		return item

	def on_tree_item_clicked(self, item, column):
		"""When a tree item is clicked, find and highlight the corresponding XML"""
		element = item.data(0, Qt.UserRole)
		if element is not None:
			try:
				content = self.text_edit.toPlainText()

				# Extract element tag name
				element_tag = element.tag
				if element_tag.startswith("{"):
					_, element_tag = element_tag.split("}")

				# Build search pattern with attributes
				tag_pattern = f"<{element_tag}"
				if element.attrib:
					# Add each attribute to search pattern
					for key, value in element.attrib.items():
						if key.startswith("{"):
							_, key = key.split("}")
						tag_pattern += f' {key}="{value}"'

				# Try to find the exact element first
				pos = content.find(tag_pattern)
				if pos >= 0:
					# Find the end of this element
					tag_end = content.find(">", pos)
					if tag_end >= 0:
						if content[tag_end-1] == "/":
							# Self-closing tag
							end_pos = tag_end + 1
						else:
							# Find closing tag
							end_tag = f"</{element_tag}>"
							temp_pos = content.find(end_tag, tag_end)
							if temp_pos >= 0:
								end_pos = temp_pos + len(end_tag)
							else:
								end_pos = tag_end + 1

						# Create cursor and select text
						cursor = self.text_edit.textCursor()
						cursor.setPosition(pos)
						cursor.setPosition(end_pos, QTextCursor.KeepAnchor)

						# Update text editor
						self.text_edit.setTextCursor(cursor)
						self.text_edit.ensureCursorVisible()

						# Switch to XML tab and focus
						for i in range(self.tab_widget.count()):
							if self.tab_widget.tabText(i) == "XML":
								self.tab_widget.setCurrentIndex(i)
								self.text_edit.setFocus()
								break

			except Exception as e:
				debug_print(f"OPFHelper ERROR: Failed to highlight element: {str(e)}")
				debug_print(traceback.format_exc())

	def closeEvent(self, e):
		# Save dialog geometry to gprefs like CCR
		try:
			gprefs['opf_helper_dialog_geometry'] = bytearray(self.saveGeometry())
		except Exception:
			pass
		super().closeEvent(e)

	# New methods for context menu
	def setup_tree_context_menu(self):
		"""Set up context menu for tree items"""
		self.tree_widget.setContextMenuPolicy(Qt.CustomContextMenu)
		self.tree_widget.customContextMenuRequested.connect(self.show_tree_context_menu)

	def show_tree_context_menu(self, position):
		"""Display context menu for tree items"""
		item = self.tree_widget.itemAt(position)
		if not item:
			return

		element = item.data(0, Qt.UserRole)
		element_type = item.data(0, Qt.UserRole + 1)
		text_content = item.data(0, Qt.UserRole + 2)  # For text nodes

		from PyQt5.QtWidgets import QMenu, QAction

		menu = QMenu()

		# Text value copying (if this is a text node or has text content)
		if text_content:
			copy_value = QAction("Copy Value", self)
			copy_value.triggered.connect(lambda: self.copy_to_clipboard(text_content))
			menu.addAction(copy_value)
			menu.addSeparator()

		if element is not None:
			# Element actions
			copy_element = QAction("Copy Element XML", self)
			copy_element.triggered.connect(lambda: self.copy_element_xml(element))
			menu.addAction(copy_element)

			copy_tag = QAction("Copy Tag Name", self)
			copy_tag.triggered.connect(lambda: self.copy_element_tag(element))
			menu.addAction(copy_tag)

			# For elements with text content
			if element.text and element.text.strip():
				copy_text = QAction("Copy Text Content", self)
				copy_text.triggered.connect(lambda: self.copy_element_text(element))
				menu.addAction(copy_text)

			# For elements with attributes
			if element.attrib:
				menu.addSeparator()
				copy_attrs = QAction("Copy All Attributes", self)
				copy_attrs.triggered.connect(lambda: self.copy_element_attributes(element))
				menu.addAction(copy_attrs)

				# Add individual attribute copy actions if there aren't too many
				if len(element.attrib) < 8:
					for key, value in element.attrib.items():
						attr_name = key
						if key.startswith("{"):
							# Handle namespaced attributes
							_, attr_name = key[1:].split("}")
						copy_attr = QAction(f"Copy Attribute: {attr_name}", self)
						copy_attr.triggered.connect(lambda checked, k=key, v=value:
												 self.copy_to_clipboard(f'{k}="{v}"'))
						menu.addAction(copy_attr)

			# Element type info
			menu.addSeparator()
			type_label = QAction(f"Type: {element_type}", self)
			type_label.setEnabled(False)
			menu.addAction(type_label)

			# Namespace info if present
			if element.tag.startswith("{"):
				ns, _ = element.tag[1:].split("}")
				ns_label = QAction(f"Namespace: {ns}", self)
				ns_label.setEnabled(False)
				menu.addAction(ns_label)

			# Navigation actions for container elements
			if element_type == "container":
				menu.addSeparator()
				expand_all = QAction("Expand All Children", self)
				expand_all.triggered.connect(lambda: self.expand_children(item))
				menu.addAction(expand_all)

				collapse_all = QAction("Collapse All Children", self)
				collapse_all.triggered.connect(lambda: self.collapse_children(item))
				menu.addAction(collapse_all)

		# Execute the menu
		menu.exec_(self.tree_widget.viewport().mapToGlobal(position))

	def copy_element_xml(self, element):
		"""Copy the XML representation of an element to clipboard"""
		try:
			xml_str = ET.tostring(element, encoding='unicode')
			self.copy_to_clipboard(xml_str)
		except Exception as e:
			print(f"Error copying element XML: {e}")

	def copy_element_tag(self, element):
		"""Copy the tag name of an element to clipboard"""
		tag = element.tag
		if tag.startswith("{"):
			# Handle namespaced tags
			_, tag = tag[1:].split("}")
		self.copy_to_clipboard(tag)

	def copy_element_attributes(self, element):
		"""Copy all attributes of an element to clipboard"""
		if not element.attrib:
			return

		attr_strings = []
		for key, value in element.attrib.items():
			if key.startswith("{"):
				# Handle namespaced attributes
				_, attr = key[1:].split("}")
				attr_strings.append(f'{attr}="{value}"')
			else:
				attr_strings.append(f'{key}="{value}"')
		self.copy_to_clipboard(", ".join(attr_strings))

	def copy_element_text(self, element):
		"""Copy text content of an element to clipboard"""
		if element.text and element.text.strip():
			self.copy_to_clipboard(element.text.strip())

	def expand_children(self, item):
		"""Recursively expand an item and all its children"""
		item.setExpanded(True)
		for i in range(item.childCount()):
			self.expand_children(item.child(i))

	def collapse_children(self, item):
		"""Recursively collapse all children of an item"""
		for i in range(item.childCount()):
			child = item.child(i)
			child.setExpanded(False)
			self.collapse_children(child)

	def copy_to_clipboard(self, text=None):
		"""Copy text to clipboard with passive notification

		If text is None, copies the entire content of the text edit.
		"""
		clipboard = QApplication.clipboard()
		if text is None:
			text = self.text_edit.toPlainText()
		clipboard.setText(text)

		# Create passive notification
		notification = QFrame(self)
		notification.setFrameStyle(QFrame.Panel | QFrame.Raised)
		notification.setLineWidth(2)

		layout = QHBoxLayout(notification)
		layout.setContentsMargins(6, 6, 6, 6)
		icon = QLabel()
		icon.setPixmap(QIcon(I('ok.png')).pixmap(16, 16))
		layout.addWidget(icon)

		label = QLabel("Content copied to clipboard")
		label.setStyleSheet("color: rgb(0, 140, 0);")
		layout.addWidget(label)

		# Position the notification
		pos = self.mapToGlobal(self.rect().topRight())
		notification.move(pos.x() - notification.sizeHint().width() - 10, pos.y() + 10)

		# Show notification
		notification.show()

		# Auto-hide after 2 seconds
		QTimer.singleShot(2000, notification.deleteLater)

	# Add methods for zoom functionality that are referenced by the buttons
	def zoom_in(self):
		"""Increase the font size of trees and XML view"""
		self.current_font_size = min(24, self.current_font_size + self.font_size_increment)

		# Update XML view font
		xml_font = self.text_edit.font()
		xml_font.setPointSize(self.current_font_size)
		self.text_edit.setFont(xml_font)

		# Update tree view fonts
		tree_font = self.tree_widget.font()
		tree_font.setPointSize(self.current_font_size)
		self.tree_widget.setFont(tree_font)

		# Update stats panel fonts
		stats_font = self.stats_panel.font()
		stats_font.setPointSize(self.current_font_size)
		self.stats_panel.setFont(stats_font)
		self.stats_panel.metadata_tree.setFont(stats_font)
		self.stats_panel.files_list.setFont(stats_font)

		# Update validation panel fonts - using correct attribute name 'results'
		validation_font = self.validation_panel.font()
		validation_font.setPointSize(self.current_font_size)
		self.validation_panel.setFont(validation_font)
		self.validation_panel.results.setFont(validation_font)

		# Save the new size
		self.prefs['font_size'] = self.current_font_size

	def zoom_out(self):
		"""Decrease the font size of trees and XML view"""
		self.current_font_size = max(6, self.current_font_size - self.font_size_increment)

		# Update XML view font
		xml_font = self.text_edit.font()
		xml_font.setPointSize(self.current_font_size)
		self.text_edit.setFont(xml_font)

		# Update tree view fonts
		tree_font = self.tree_widget.font()
		tree_font.setPointSize(self.current_font_size)
		self.tree_widget.setFont(tree_font)

		# Update stats panel fonts
		stats_font = self.stats_panel.font()
		stats_font.setPointSize(self.current_font_size)
		self.stats_panel.setFont(stats_font)
		self.stats_panel.metadata_tree.setFont(stats_font)
		self.stats_panel.files_list.setFont(stats_font)

		# Update validation panel fonts - using correct attribute name 'results'
		validation_font = self.validation_panel.font()
		validation_font.setPointSize(self.current_font_size)
		self.validation_panel.setFont(validation_font)
		self.validation_panel.results.setFont(validation_font)

		# Save the new size
		self.prefs['font_size'] = self.current_font_size

	def export_opf(self):
		"""Export OPF content to a file"""
		try:
			from PyQt5.QtWidgets import QFileDialog

			# Get current book info
			book_id = self.book_ids[self.current_book_index]
			title = self.db.field_for("title", book_id)

			# Create default filename from book title
			safe_title = "".join(x for x in title if x.isalnum() or x in (" ._-"))
			default_filename = f"{safe_title}_opf.xml"

			# Show save file dialog
			filename, _ = QFileDialog.getSaveFileName(
				self,
				"Save OPF Content",
				default_filename,
				"XML Files (*.xml);;OPF Files (*.opf);;All Files (*.*)"
			)

			if filename:  # If user didn't cancel
				with open(filename, 'w', encoding='utf-8') as f:
					f.write(self.xml_content)

				# Show success notification
				notification = QFrame(self)
				notification.setFrameStyle(QFrame.Panel | QFrame.Raised)
				notification.setLineWidth(2)

				layout = QHBoxLayout(notification)
				layout.setContentsMargins(6, 6, 6, 6)
				icon = QLabel()
				icon.setPixmap(QIcon(I('ok.png')).pixmap(16, 16))
				layout.addWidget(icon)

				label = QLabel(f"OPF content saved to: {os.path.basename(filename)}")
				label.setStyleSheet("color: rgb(0, 140, 0);")
				layout.addWidget(label)

				# Position notification
				pos = self.mapToGlobal(self.rect().topRight())
				notification.move(pos.x() - notification.sizeHint().width() - 10, pos.y() + 10)

				# Show and auto-hide notification
				notification.show()
				QTimer.singleShot(2000, notification.deleteLater)

		except Exception as e:
			error_dialog(self.gui, 'Export Error',
					   f'Failed to export OPF content: {str(e)}',
					   show=True)


	def edit_metadata(self):
		"""Open the metadata editor for the current book"""
		try:
			book_id = self.book_ids[self.current_book_index]
			debug_print(f'OPFHelper: Opening metadata editor for book ID {book_id}')

			# Get library view and ensure we're showing it
			self.gui.show_library_view()
			library_view = self.gui.library_view

			# Store current search and selection state
			current_search = str(self.gui.search.current_text)
			current_selection = library_view.get_state()

			# First clear any search restrictions/virtual libraries that might hide our book
			if library_view.model().db.data.get_base_restriction_name():
				self.gui.apply_virtual_library()
			if library_view.model().db.data.get_search_restriction_name():
				self.gui.apply_named_search_restriction()

			# Select our book
			library_view.select_rows([book_id], using_ids=True)

			# Ensure Calibre window is active
			self.gui.activateWindow()
			self.gui.raise_()

			# Edit metadata for current book
			debug_print('OPFHelper: Opening metadata editor')
			self.gui.iactions['Edit Metadata'].edit_metadata(False)

			# Restore previous state
			debug_print('OPFHelper: Restoring previous search and selection')
			self.gui.search.set_search_string(current_search)
			if current_selection:
				library_view.state = current_selection  # Use the state property setter instead
		except Exception as e:
			debug_print(f'OPFHelper ERROR: Failed to edit metadata: {str(e)}')
			error_dialog(self, 'Error',
					  f'Failed to edit metadata: {str(e)}',
					  show=True)

	def load_current_book(self):
		"""Load the current book's OPF content and cover"""
		debug_print(f'OPFHelper: load_current_book called - current_book_index: {self.current_book_index}')
		try:
			book_id = self.book_ids[self.current_book_index]
			debug_print(f'OPFHelper: load_current_book - loading book_id: {book_id}')

			# Get the book's title and path using new_api consistently
			db = self.gui.current_db.new_api
			title = db.field_for('title', book_id)
			# Try KEPUB first, then EPUB
			epub_path = db.format_abspath(book_id, 'KEPUB')
			if not epub_path or not os.path.isfile(epub_path):
				epub_path = db.format_abspath(book_id, 'EPUB')

			debug_print(f'OPFHelper: load_current_book - title: {title}, epub_path exists: {os.path.isfile(epub_path) if epub_path else False}')

			if not epub_path or not os.path.isfile(epub_path):
				self.gui.status_bar.showMessage("EPUB or KEPUB format not available for this book", 3000)
				return

			try:
				# Check if book contains multiple OPF files
				with zipfile.ZipFile(epub_path, 'r') as zf:
					# Get all OPF files in the EPUB
					opf_files = [f for f in zf.namelist() if f.endswith('.opf')]

					if not opf_files:
						error_dialog(self.gui, 'Error',
								  'No OPF file found in this EPUB.',
								  show=True)
						return

					# Store all OPF files for this book
					self.current_book_opfs = opf_files

					# Use saved selection if available
					if book_id in self.book_opf_selections:
						selected_opf = self.book_opf_selections[book_id]
						if selected_opf in opf_files:
							self.current_opf_path = selected_opf
						else:
							# If saved selection is no longer valid, use first OPF
							self.current_opf_path = opf_files[0]
					else:
						self.current_opf_path = opf_files[0]

					# Show OPF selector if multiple OPF files exist
					if (len(opf_files) > 1):
						self.opf_selector_container.setVisible(True)
						self.opf_selector.clear()
						self.opf_selector.addItems(opf_files)
						self.opf_selector.setCurrentText(self.current_opf_path)
						self.opf_count_label.setText(f"({len(opf_files)} files)")
					else:
						self.opf_selector_container.setVisible(False)

					# Read the OPF content
					opf_data = zf.read(self.current_opf_path).decode('utf-8', errors='replace')
					self.xml_content = opf_data

					# Update UI
					self.text_edit.setPlainText(opf_data)
					self.populate_tree()

					# Set window title
					self.setWindowTitle(f"OPF Helper: {title}")

					# Update book label
					author = db.field_for('authors', book_id)
					if author:
						# Handle different types of author data (list, tuple, string)
						if isinstance(author, (list, tuple)) and len(author) > 0:
							author = author[0]
						elif isinstance(author, str):
							pass  # Already a string
						else:
							author = str(author)  # Fallback to string conversion
					else:
						author = "Unknown"

					# Format label - always show just title by author (no ID)
					label_text = f"{title} by {author}"
					self.book_label.setText(label_text)

					# Book info and navigation
					self.update_book_info_label()
					# Always enable navigation buttons for free navigation
					self.prev_book_button.setEnabled(True)
					self.next_book_button.setEnabled(True)

					# Try to detect EPUB version
					if 'version="3.0"' in opf_data:
						version_color = "#FFA500" if self.is_dark else "#FF8C00"  # Bright orange for dark, darker for light
						self.version_label.setText('EPUB 3.0')
						self.version_label.setStyleSheet(f"""
							QLabel {{
								padding: 2px 8px;
								border-radius: 4px;
								color: {version_color};
								font-weight: bold;
								background-color: transparent;
								border: 1px solid {version_color};
							}}
						""")
					elif 'version="2.0"' in opf_data:
						self.version_label.setText('EPUB 2.0')
						self.version_label.setStyleSheet("""
							QLabel {
								padding: 2px 8px;
								border-radius: 4px;
								color: palette(link);
								font-weight: bold;
								background-color: transparent;
								border: 1px solid palette(link);
							}
						""")
					else:
						self.version_label.setText('EPUB ?')
						self.version_label.setStyleSheet("""
							QLabel {
								padding: 2px 8px;
								border-radius: 4px;
								color: palette(link);
								font-weight: bold;
								background-color: transparent;
								border: 1px solid palette(link);
							}
						""")

					# Load cover
					self.load_cover(self.current_opf_path)

					# Auto-validate if enabled
					if prefs.get('auto_validate', True):
						QTimer.singleShot(100, self.validate_opf)

			except zipfile.BadZipFile:
				error_dialog(self.gui, 'Error',
						  'The EPUB file appears to be corrupted.',
						  show=True)
			except Exception as e:
				error_dialog(self.gui, 'Error',
						  f'Failed to read OPF content.\nError: {str(e)}',
						  show=True)
		except Exception as e:
			error_dialog(self.gui, 'Error',
					   f'Failed to load book information.\nError: {str(e)}',
					   show=True)

	def show_next_book(self):
		"""Show the next book in the library view, skipping books without EPUB/KEPUB format"""
		debug_print(f'OPFHelper: show_next_book called - current_index: {self.current_book_index}, book_ids length: {len(self.book_ids) if self.book_ids else 0}')
		if not self.book_ids:
			debug_print('OPFHelper: show_next_book - no book_ids available')
			return

		start_index = self.current_book_index
		books_checked = 0

		while books_checked < len(self.book_ids):
			old_index = self.current_book_index
			self.current_book_index = (self.current_book_index + 1) % len(self.book_ids)
			books_checked += 1

			book_id = self.book_ids[self.current_book_index]
			debug_print(f'OPFHelper: show_next_book - checking book_id: {book_id}')

			# Check if book has EPUB or KEPUB format
			if self.db.has_format(book_id, 'KEPUB') or self.db.has_format(book_id, 'EPUB'):
				debug_print(f'OPFHelper: show_next_book - found compatible format, selecting book_id: {book_id}')
				debug_print(f'OPFHelper: show_next_book - calling select_rows with: {[book_id]}')
				self.gui.library_view.select_rows([book_id], using_ids=True)
				debug_print('OPFHelper: show_next_book - select_rows completed')
				self.load_current_book()
				return
			else:
				debug_print(f'OPFHelper: show_next_book - book_id {book_id} has no compatible format, skipping')

		# If we get here, no compatible books found
		debug_print('OPFHelper: show_next_book - no compatible books found in library')
		self.gui.status_bar.showMessage("No books with EPUB or KEPUB format found in library", 5000)

	def show_previous_book(self):
		"""Show the previous book in the library view, skipping books without EPUB/KEPUB format"""
		debug_print(f'OPFHelper: show_previous_book called - current_index: {self.current_book_index}, book_ids length: {len(self.book_ids) if self.book_ids else 0}')
		if not self.book_ids:
			debug_print('OPFHelper: show_previous_book - no book_ids available')
			return

		start_index = self.current_book_index
		books_checked = 0

		while books_checked < len(self.book_ids):
			old_index = self.current_book_index
			self.current_book_index = (self.current_book_index - 1) % len(self.book_ids)
			books_checked += 1

			book_id = self.book_ids[self.current_book_index]
			debug_print(f'OPFHelper: show_previous_book - checking book_id: {book_id}')

			# Check if book has EPUB or KEPUB format
			if self.db.has_format(book_id, 'KEPUB') or self.db.has_format(book_id, 'EPUB'):
				debug_print(f'OPFHelper: show_previous_book - found compatible format, selecting book_id: {book_id}')
				debug_print(f'OPFHelper: show_previous_book - calling select_rows with: {[book_id]}')
				self.gui.library_view.select_rows([book_id], using_ids=True)
				debug_print('OPFHelper: show_previous_book - select_rows completed')
				self.load_current_book()
				return
			else:
				debug_print(f'OPFHelper: show_previous_book - book_id {book_id} has no compatible format, skipping')

		# If we get here, no compatible books found
		debug_print('OPFHelper: show_previous_book - no compatible books found in library')
		self.gui.status_bar.showMessage("No books with EPUB or KEPUB format found in library", 5000)

	def show_first_book(self):
		"""Show the first compatible book (EPUB/KEPUB) in the library view"""
		if not self.book_ids:
			return

		# Find the first book with EPUB or KEPUB format
		for i, book_id in enumerate(self.book_ids):
			if self.db.has_format(book_id, 'KEPUB') or self.db.has_format(book_id, 'EPUB'):
				self.current_book_index = i
				self.gui.library_view.select_rows([book_id], using_ids=True)
				self.load_current_book()
				return

		# If no compatible books found
		self.gui.status_bar.showMessage("No books with EPUB or KEPUB format found in library", 5000)

	def show_last_book(self):
		"""Show the last compatible book (EPUB/KEPUB) in the library view"""
		if not self.book_ids:
			return

		# Find the last book with EPUB or KEPUB format
		for i in range(len(self.book_ids) - 1, -1, -1):
			book_id = self.book_ids[i]
			if self.db.has_format(book_id, 'KEPUB') or self.db.has_format(book_id, 'EPUB'):
				self.current_book_index = i
				self.gui.library_view.select_rows([book_id], using_ids=True)
				self.load_current_book()
				return

		# If no compatible books found
		self.gui.status_bar.showMessage("No books with EPUB or KEPUB format found in library", 5000)

	class SchemaResolver(etree.Resolver):
		def __init__(self, schema_dir):
			self.schema_dir = schema_dir
			debug_print(f"OPFHelper: Schema resolver initialized with dir: {schema_dir}")

		def resolve(self, system_url, public_id, context):
			"""Resolve schema imports by looking in the local schema directory"""
			debug_print(f"OPFHelper: Resolving schema: {system_url}, {public_id}")
			try:
				if (system_url.endswith('.xsd')):
					schema_file = os.path.basename(system_url)
					schema_path = os.path.join(self.schema_dir, schema_file)
					if (os.path.exists(schema_path)):
						debug_print(f"OPFHelper: Found schema at {schema_path}")
						return self.resolve_filename(schema_path, context)
					else:
						debug_print(f"OPFHelper: Schema not found at {schema_path}")
				return None
			except Exception as e:
				debug_print(f"OPFHelper ERROR: Schema resolution failed: {str(e)}")
				debug_print(traceback.format_exc())
				return None

	def initialize_schemas(self):
		"""Initialize XML Schema parsers with proper namespace handling"""
		try:
			from .schema_utils import verify_schemas, install_schemas, load_schema

			# Initialize schema parsers dictionary
			self.schema_parsers = {}

			# First verify schemas are valid
			if not verify_schemas():
				debug_print("OPFHelper ERROR: Schema verification failed")
				return

			# Load schemas for each version
			for version in ['2.0', '3.0']:
				schema = load_schema(version)
				if schema:
					self.schema_parsers[version] = schema
					debug_print(f"OPFHelper: Loaded schema for version {version}")
				else:
					debug_print(f"OPFHelper ERROR: Failed to load schema for version {version}")

		except Exception as e:
			debug_print(f"OPFHelper ERROR: Failed to initialize schemas: {str(e)}")
			debug_print(traceback.format_exc())
			self.schema_parsers = {}

	def validate_opf(self):
		"""Validate the current OPF content against appropriate schema"""
		if not self.xml_content:
			return error_dialog(self, "No OPF Content", "There is no OPF content to validate.")

		try:
			version = self.version_label.text().replace('EPUB ', '')

			# Parse XML content with custom parser
			parser = get_schema_parser()
			doc = etree.fromstring(self.xml_content.encode('utf-8'), parser)

			results = []
			validation_errors = []

			# Basic XML validation first
			results.append("Basic XML Validation:")
			parser_errors = doc.getroottree().parser.error_log
			if parser_errors:
				results.append("❌ XML parsing failed:")
				for error in parser_errors:
					error_text = str(error.message)  # Convert message to string
					results.append(f"❌ {error_text}")
					validation_errors.append(error)
			else:
				results.append("✓ XML is well-formed")
			results.append("")

			# Schema validation
			results.append(f"OPF Schema Validation (Version {version}):")
			schema = load_schema(version)

			if schema is not None:
				try:
					schema.assertValid(doc)
					results.append("✓ Valid against schema")
				except etree.DocumentInvalid as e:
					results.append("❌ Schema validation failed:")
					for error in e.error_log:
						error_text = str(error.message)  # Convert message to string
						results.append(f"❌ {error_text}")
						validation_errors.extend(e.error_log)
			else:
				results.append("⚠️ No schema available for version")

			# Run additional OPF-specific validation
			is_valid, basic_results = basic_opf_validation(doc.getroottree(), version)
			results.append("")
			results.extend(basic_results)

			# Show results in validation panel
			self.validation_panel.results.setPlainText('\n'.join(results))

		except Exception as e:
			# Show error in validation panel
			error_text = f"❌ Validation failed: {str(e)}"
			self.validation_panel.results.setPlainText(error_text)

	def edit_book(self):
		"""Open the current book in Calibre's Editor, focusing on the OPF file."""
		try:
			book_id = self.book_ids[self.current_book_index]
			title = self.db.field_for("title", book_id)
			# Try KEPUB first, then EPUB
			epub_path = self.db.format_abspath(book_id, 'KEPUB')
			if not epub_path or not os.path.isfile(epub_path):
				epub_path = self.db.format_abspath(book_id, 'EPUB')
			if not epub_path or not os.path.isfile(epub_path):
				error_dialog(self.gui, 'Invalid Book', f'Could not access EPUB or KEPUB file for "{title}"', show=True)
				return

			# Debug what actions are available
			debug_print(f'OPFHelper: Available actions: {list(self.gui.iactions.keys())}')

			# Try to launch ebook-edit directly with the OPF file focused
			try:
				from calibre.gui2.tweak_book import tprefs
				tprefs.refresh()  # In case they were changed

				# Prepare notify data for the editor
				notify = f'{book_id}:EPUB:{self.db.library_id}:{self.db.library_path}'

				# Launch ebook-edit with the OPF file as the file to open
				kwargs = dict(path=epub_path, notify=notify)

				# If we have a current OPF path, pass it as an additional argument
				if hasattr(self, 'current_opf_path') and self.current_opf_path:
					# The ebook-edit command accepts additional file names to open
					# We need to modify the launch to include the OPF file
					debug_print(f'OPFHelper: Launching ebook-edit with OPF focus: {self.current_opf_path}')
					# Unfortunately, the job_manager.launch_gui_app doesn't support additional args easily
					# Let's try a different approach - use the Tweak ePub action but modify it

				# Use "Tweak ePub" action which is the built-in editor in Calibre
				editor_action = self.gui.iactions.get('Tweak ePub')

				if editor_action:
					debug_print('OPFHelper: Found "Tweak ePub" action')
					# First select the book in the library view
					self.gui.library_view.select_rows([book_id], using_ids=True)

					# Call the appropriate method for Tweak ePub action
					if hasattr(editor_action, 'tweak_epub'):
						debug_print('OPFHelper: Using tweak_epub method')
						editor_action.tweak_epub(book_id)
					elif hasattr(editor_action, 'tweak_book'):
						debug_print('OPFHelper: Using tweak_book method')
						# tweak_book() takes no parameters - it relies on the current selection
						editor_action.tweak_book()
					else:
						debug_print('OPFHelper: Using menuless_qaction')
						# Use the menu action directly which works on the selected book
						editor_action.menuless_qaction.trigger()
				else:
					# Fallback to other potential editor actions
					debug_print('OPFHelper: "Tweak ePub" action not found, trying alternative methods')
					for action_name in ['Edit Book', 'edit_book', 'editor']:
						action = self.gui.iactions.get(action_name)
						if action:
							debug_print(f'OPFHelper: Found editor action with name: {action_name}')
							if hasattr(action, 'edit_book_files'):
								action.edit_book_files([book_id])
								return
							elif hasattr(action, 'edit_book'):
								action.edit_book(book_id)
								return
							else:
								# Select the book and trigger action
								self.gui.library_view.select_rows([book_id], using_ids=True)
								action.menuless_qaction.trigger()
								return

					# Last resort - open in metadata editor
					debug_print('OPFHelper: No suitable edit action found, opening metadata editor')
					self.edit_metadata()
					error_dialog(self.gui, 'Editor Not Available',
								'Could not find book editor action. Opening metadata editor instead.',
								show=True)

			except Exception as e:
				debug_print(f'OPFHelper ERROR: Failed to launch ebook-edit directly: {str(e)}')
				# Fall back to the original method
				editor_action = self.gui.iactions.get('Tweak ePub')
				if editor_action:
					self.gui.library_view.select_rows([book_id], using_ids=True)
					if hasattr(editor_action, 'tweak_epub'):
						editor_action.tweak_epub(book_id)
					else:
						editor_action.menuless_qaction.trigger()
				else:
					error_dialog(self, 'Error', f'Failed to open book in editor: {str(e)}', show=True)

		except Exception as e:
			debug_print(f'OPFHelper ERROR: Failed to open book in editor: {str(e)}')
			error_dialog(self, 'Error', f'Failed to open book in editor: {str(e)}', show=True)

	def view_book(self):
		"""Open the current book in Calibre's E-book Viewer"""
		try:
			book_id = self.book_ids[self.current_book_index]
			debug_print(f'OPFHelper: Opening book ID {book_id} in viewer')

			# Get the book's title and path
			title = self.db.field_for("title", book_id)
			# Try KEPUB first, then EPUB
			epub_path = self.db.format_abspath(book_id, 'KEPUB')
			fmt = 'KEPUB'
			if not epub_path or not os.path.isfile(epub_path):
				epub_path = self.db.format_abspath(book_id, 'EPUB')
				fmt = 'EPUB'
			if not epub_path or not os.path.isfile(epub_path):
				error_dialog(self.gui, 'Invalid Book', f'Could not access EPUB or KEPUB file for "{title}"', show=True)
				return

			# Launch the viewer - using correct action name lookup
			viewer_action = self.gui.iactions.get('View')
			if viewer_action:
				viewer_action.view_format_by_id(book_id, fmt)
			else:
				error_dialog(self, 'Error', 'E-book viewer action not available in this Calibre installation', show=True)
		except Exception as e:
			debug_print(f'OPFHelper ERROR: Failed to open book in viewer: {str(e)}')
			error_dialog(self, 'Error',
					  f'Failed to open book in viewer: {str(e)}',
					  show=True)

	def edit_toc(self):
		"""Open the ToC editor for the current book"""
		try:
			book_id = self.book_ids[self.current_book_index]
			debug_print(f'OPFHelper: Opening ToC editor for book ID {book_id}')

			# Get the book's title and path
			title = self.db.field_for("title", book_id)
			# Try KEPUB first, then EPUB
			epub_path = self.db.format_abspath(book_id, 'KEPUB')
			if not epub_path or not os.path.isfile(epub_path):
				epub_path = self.db.format_abspath(book_id, 'EPUB')
			if not epub_path or not os.path.isfile(epub_path):
				error_dialog(self.gui, 'Invalid Book', f'Could not access EPUB or KEPUB file for "{title}"', show=True)
				return

			# Try to find the ToC Editor in available actions
			toc_editor = None

			# Look for the Edit ToC action under different possible names
			for action_name in ['Edit ToC', 'edit_toc', 'toc']:
				action = self.gui.iactions.get(action_name)
				if action:
					debug_print(f'OPFHelper: Found ToC editor with name: {action_name}')
					toc_editor = action
					break

			if toc_editor:
				# First select the book in the library view
				self.gui.library_view.select_rows([book_id], using_ids=True)

				# Ensure Calibre window is active
				self.gui.activateWindow()
				self.gui.raise_()

				# Try different possible methods to invoke the ToC editor
				if hasattr(toc_editor, 'edit_toc'):
					debug_print('OPFHelper: Using edit_toc method')
					toc_editor.edit_toc(book_id)
				elif hasattr(toc_editor, 'edit_book_toc'):
					debug_print('OPFHelper: Using edit_book_toc method')
					toc_editor.edit_book_toc(book_id)
				else:
					debug_print('OPFHelper: Using menuless_qaction')
					# Use the menu action directly which works on the selected book
					toc_editor.menuless_qaction.trigger()
			else:
				# If we can't find a dedicated ToC editor action, try using the Edit Book functionality
				# which includes a ToC editor
				debug_print('OPFHelper: Dedicated ToC editor not found, trying to use Edit Book')
				editor_action = self.gui.iactions.get('Tweak ePub')

				if editor_action:
					debug_print('OPFHelper: Opening book in editor, where ToC can be edited')
					self.gui.library_view.select_rows([book_id], using_ids=True)

					# Call the appropriate method
					if hasattr(editor_action, 'tweak_epub'):
						editor_action.tweak_epub(book_id)
					elif hasattr(editor_action, 'tweak_book'):
						editor_action.tweak_book()
					else:
						editor_action.menuless_qaction.trigger()

					# Inform user to look for ToC editing tools in the editor
					info_dialog(self.gui, 'ToC Editor',
							  'The book has been opened in the editor. To edit the Table of Contents, '
							  'use the "Tools > Table of Contents > Edit Table of Contents" menu option '
							  'in the editor window.',
							  show=True)
				else:
					error_dialog(self.gui, 'ToC Editor Not Available',
							   'Could not find Table of Contents editor in this Calibre installation.',
							   show=True)
		except Exception as e:
			debug_print(f'OPFHelper ERROR: Failed to open ToC editor: {str(e)}')
			error_dialog(self, 'Error', f'Failed to open ToC editor: {str(e)}', show=True)

	def unpack_book(self):
		"""Unpack the current book into individual components for editing"""
		try:
			book_id = self.book_ids[self.current_book_index]
			debug_print(f'OPFHelper: Unpacking book ID {book_id}')

			# Use Calibre's built-in Unpack Book action
			unpack_action = self.gui.iactions.get('Unpack Book')
			if unpack_action:
				unpack_action.do_tweak(book_id)
			else:
				error_dialog(self, 'Error', 'Unpack Book action not available in this Calibre installation', show=True)
		except Exception as e:
			debug_print(f'OPFHelper ERROR: Failed to unpack book: {str(e)}')
			error_dialog(self, 'Error',
					  f'Failed to unpack book: {str(e)}',
					  show=True)

	def refresh_opf_content(self):
		"""Refresh the OPF content from the current book"""
		try:
			debug_print('OPFHelper: Refreshing OPF content')
			self.load_current_book()
			# Show a brief notification that refresh was successful
			info_dialog(self, 'Refreshed', 'OPF content has been refreshed from the current book.', show=True)
		except Exception as e:
			debug_print(f'OPFHelper ERROR: Failed to refresh OPF content: {str(e)}')
			error_dialog(self, 'Error', f'Failed to refresh OPF content: {str(e)}', show=True)

	def update_book_info_label(self):
		"""Update the book info label with current book's details"""
		try:
			book_id = self.book_ids[self.current_book_index]

			# Get book details
			title = self.db.field_for("title", book_id)
			authors = self.db.field_for("authors", book_id)
			authors_text = " & ".join(authors) if authors else "Unknown"
			pubdate = self.db.field_for("pubdate", book_id)
			publisher = self.db.field_for("publisher", book_id)

			# Build info text with HTML formatting - conditionally show ID
			if self.show_book_id:
				info_text = f'<span style="font-weight:500">  🆔 {book_id}  📖 {title} by {authors_text}</span>'
			else:
				info_text = f'<span style="font-weight:500">  📖 {title} by {authors_text}</span>'

			if pubdate:
				pub_year = pubdate.year if hasattr(pubdate, 'year') else ''
				if pub_year:
					info_text += f' 🏛️ Published in {pub_year}'

			if publisher:
				info_text += f' by {publisher}'

			# Import human_readable function from calibre
			from calibre.ebooks.metadata.book.base import human_readable

			formats = self.db.formats(book_id)

			if formats:
				format_parts = []
				for f in formats:
					f_path = self.db.format_abspath(book_id, f)
					if os.path.isfile(f_path):
						size_bytes = os.path.getsize(f_path)
						# Use custom formatting for very small files
						if size_bytes < 102400:  # Less than 100KB
							size_str = f"{size_bytes / 1024:.1f} KB"  # Show in KB instead
						else:
							# Use the calibre function for larger files
							size_str = human_readable(size_bytes)
						format_parts.append(f'{f}: {size_str}')
				if format_parts:
					info_text += '   💾  ' + ' | '.join(format_parts)
			self.book_info_label.setText(info_text)
		except Exception as e:
			debug_print(f"Error updating book info: {e}")
			self.book_info_label.setText("")

	def add_action_buttons(self, toolbar_layout):
		"""Add all action buttons to the toolbar with consistent spacing"""
		# Add zoom buttons to toolbar
		zoom_in_button = QPushButton(get_icon('images/zoom_in.png'), '', self)
		zoom_in_button.setFixedWidth(32)
		zoom_in_button.setToolTip("Increase font size in XML view")
		zoom_in_button.clicked.connect(self.zoom_in)
		toolbar_layout.addWidget(zoom_in_button)

		zoom_out_button = QPushButton(get_icon('images/zoom_out.png'), '', self)
		zoom_out_button.setFixedWidth(32)
		zoom_out_button.setToolTip("Decrease font size in XML view")
		zoom_out_button.clicked.connect(self.zoom_out)
		toolbar_layout.addWidget(zoom_out_button)

		# Add edit book button to toolbar
		edit_book_button = QPushButton("Edit Book")
		edit_book_button.setIcon(QIcon.ic('edit_book.png'))
		edit_book_button.setToolTip("Open book in Calibre's Editor")
		edit_book_button.clicked.connect(self.edit_book)
		toolbar_layout.addWidget(edit_book_button)

		# Add view book button to toolbar
		view_book_button = QPushButton("View Book")
		view_book_button.setIcon(QIcon.ic('view.png'))
		view_book_button.setToolTip("Open book in Calibre's E-book Viewer")
		view_book_button.clicked.connect(self.view_book)
		toolbar_layout.addWidget(view_book_button)

		# Add edit ToC button to toolbar
		edit_toc_button = QPushButton("Edit ToC")
		edit_toc_button.setIcon(QIcon.ic('toc.png'))
		edit_toc_button.setToolTip("Edit Table of Contents of the book")
		edit_toc_button.clicked.connect(self.edit_toc)
		toolbar_layout.addWidget(edit_toc_button)

		# Add unpack book button to toolbar
		unpack_book_button = QPushButton("Unpack Book")
		unpack_book_button.setIcon(QIcon.ic('unpack-book.png'))
		unpack_book_button.setToolTip("Unpack book into individual components for editing")
		unpack_book_button.clicked.connect(self.unpack_book)
		toolbar_layout.addWidget(unpack_book_button)

		# Add refresh button to toolbar
		refresh_button = QPushButton("Refresh")
		refresh_button.setIcon(QIcon.ic('view-refresh.png'))
		refresh_button.setToolTip("Refresh OPF content from the current book")
		refresh_button.clicked.connect(self.refresh_opf_content)
		toolbar_layout.addWidget(refresh_button)

		export_button = QPushButton(get_icon('images/export_icon.png'), "Export OPF", self)
		export_button.setToolTip("Save OPF content to file")
		export_button.clicked.connect(self.export_opf)
		toolbar_layout.addWidget(export_button)



		edit_metadata_button = QPushButton("Open MDE")
		edit_metadata_button.setIcon(QIcon.ic('edit_input.png'))
		edit_metadata_button.setToolTip("Open Edit Metadata dialog for the current book")
		edit_metadata_button.clicked.connect(self.edit_metadata)
		toolbar_layout.addWidget(edit_metadata_button)

		copy_button = QPushButton("Copy to Clipboard")
		copy_button.setIcon(QIcon.ic('edit-copy.png'))
		copy_button.setToolTip("Copy XML content to clipboard (from XML tab)")
		copy_button.clicked.connect(lambda: self.copy_to_clipboard())
		toolbar_layout.addWidget(copy_button)

	def on_opf_selection_changed(self, index):
		"""Handle when the user selects a different OPF file from the dropdown"""
		if index < 0 or index >= len(self.current_book_opfs):
			return

		# Get the selected OPF file path
		selected_opf = self.current_book_opfs[index]
		book_id = self.book_ids[self.current_book_index]

		# Save the selection for this book
		self.book_opf_selections[book_id] = selected_opf

		# Load the selected OPF file
		self.load_opf_content(selected_opf)

	def load_opf_content(self, opf_path):
		"""Load OPF content from specified path within the current EPUB"""
		try:
			book_id = self.book_ids[self.current_book_index]
			title = self.db.field_for("title", book_id)
			epub_path = self.db.format_abspath(book_id, 'EPUB')

			with zipfile.ZipFile(epub_path, 'r') as zf:
				# Read the OPF file
				content = zf.read(opf_path).decode('utf-8')

				# Extract EPUB version
				version = "Unknown"
				version_match = re.search(r'<package[^>]*?\s+version\s*=\s*["\']([^"\']+)["\']', content, re.IGNORECASE)
				if not version_match:
					version_match = re.search(r'package[^>]*?\s+version\s*=\s*["\']([^"\']+)["\']', content, re.IGNORECASE)
				if not version_match:
					version_match = re.search(r'version\s*=\s*["\']([^"\']+)["\']', content)

				if version_match:
					version = f"EPUB {version_match.group(1)}"
					# Set color based on version
					if "3.0" in version:
						version_color = "#FFA500" if self.is_dark else "#FF8C00"
						self.version_label.setStyleSheet(f"""
							QLabel {{
								padding: 2px 8px;
								border-radius: 4px;
								color: {version_color};
								font-weight: bold;
								background-color: transparent;
								border: 1px solid {version_color};
							}}
						""")
					else:
						self.version_label.setStyleSheet("""
							QLabel {
								padding: 2px 8px;
								border-radius: 4px;
								color: palette(link);
								font-weight: bold;
								background-color: transparent;
								border: 1px solid palette(link);
							}
						""")
				self.version_label.setText(version)

				# Update UI
				self.setWindowTitle(f'OPF Helper - {title} ({opf_path})')
				self.text_edit.setPlainText(content)
				self.xml_content = content
				self.current_opf_path = opf_path
				self.populate_tree()

				# Also update the cover based on this OPF
				self.load_cover(opf_path)

		except Exception as e:
			debug_print(f"OPFHelper ERROR: Failed to load OPF content: {str(e)}")
			error_dialog(self.gui, 'Error',
					   f'Failed to read OPF content from:\n{opf_path}\n\nError: {str(e)}',
					   show=True)

	def load_cover(self, opf_path):
		"""Load cover image for the current book based on the selected OPF"""
		try:
			book_id = self.book_ids[self.current_book_index]
			# Try KEPUB first, then EPUB
			epub_path = self.db.format_abspath(book_id, 'KEPUB')
			if not epub_path or not os.path.isfile(epub_path):
				epub_path = self.db.format_abspath(book_id, 'EPUB')

			cover_found = False

			with zipfile.ZipFile(epub_path, 'r') as zf:
				# Method 1: Try parsing OPF for cover image reference
				try:
					content = zf.read(opf_path).decode('utf-8')
					root = ET.fromstring(content)

					# First look for meta cover tag (EPUB3 standard)
					meta_cover = root.find(".//{http://www.idpf.org/2007/opf}meta[@name='cover']")
					if meta_cover is not None:
						cover_id = meta_cover.get('content')
						if cover_id:
							# Find the item with this ID
							cover_item = root.find(f".//{http://www.idpf.org/2007/opf}item[@id='{cover_id}']")
							if cover_item is not None:
								cover_href = cover_item.get('href')
								if cover_href:
									# Resolve relative path to OPF directory
									opf_dir = os.path.dirname(opf_path)
									cover_path = os.path.normpath(os.path.join(opf_dir, cover_href)).replace('\\', '/')
									try:
										cover_data = zf.read(cover_path)
										self.cover_panel.show_cover(cover_data)
										cover_found = True
										debug_print(f"OPFHelper: Found cover via meta content: {cover_path}")
									except Exception as e:
										debug_print(f"OPFHelper: Error reading cover from meta reference: {str(e)}")

					# If not found, look for cover in manifest with cover properties or id containing 'cover'
					if not cover_found:
						for item in root.findall(".//{http://www.idpf.org/2007/opf}item"):
							item_id = item.get('id', '').lower()
							item_props = item.get('properties', '').lower()
							item_href = item.get('href', '')

							if ('cover' in item_props) or ('cover' in item_id):
								# Found cover reference, try to load it
								try:
									# Resolve relative path to OPF directory
									opf_dir = os.path.dirname(opf_path)
									cover_path = os.path.normpath(os.path.join(opf_dir, item_href)).replace('\\', '/')
									debug_print(f"OPFHelper: Trying cover path: {cover_path}")
									cover_data = zf.read(cover_path)
									self.cover_panel.show_cover(cover_data)
									cover_found = True
									debug_print(f"OPFHelper: Found cover from item: {item_href}")
									break
								except Exception as e:
									debug_print(f"OPFHelper: Failed to read cover from item reference: {str(e)}")
				except Exception as e:
					debug_print(f"OPFHelper: Failed to parse OPF for cover: {str(e)}")

				# Method 2: Try common cover filenames relative to current OPF if not found yet
				if not cover_found:
					debug_print(f"OPFHelper: Trying common cover filenames relative to: {opf_path}")
					opf_dir = os.path.dirname(opf_path)

					# Check common cover filenames relative to the OPF directory
					cover_relative_paths = [
						'cover.jpg', 'cover.jpeg', 'cover.png',
						'images/cover.jpg', 'images/cover.jpeg', 'images/cover.png',
						'../images/cover.jpg', '../images/cover.jpeg', '../images/cover.png',
					]

					for rel_path in cover_relative_paths:
						try:
							if opf_dir:
								cover_path = os.path.normpath(os.path.join(opf_dir, rel_path)).replace('\\', '/')
							else:
								cover_path = rel_path

							if cover_path in zf.namelist():
								debug_print(f"OPFHelper: Found cover at: {cover_path}")
								cover_data = zf.read(cover_path)
								self.cover_panel.show_cover(cover_data)
								cover_found = True
								break
						except Exception as e:
							debug_print(f"OPFHelper: Error with cover path {rel_path}: {str(e)}")

				# Method 3: Try common absolute cover paths if not found yet
				if not cover_found:
					cover_files = [
						'cover.jpg', 'cover.jpeg', 'cover.png',
						'OEBPS/cover.jpg', 'OEBPS/cover.jpeg', 'OEBPS/cover.png',
						'OEBPS/images/cover.jpg', 'OEBPS/images/cover.jpeg', 'OEBPS/images/cover.png',
						'META-INF/cover.jpg', 'META-INF/cover.jpeg', 'META-INF/cover.png'
					]

					for cover_file in cover_files:
						try:
							if cover_file in zf.namelist():
								debug_print(f"OPFHelper: Found cover at absolute path: {cover_file}")
								cover_data = zf.read(cover_file)
								self.cover_panel.show_cover(cover_data)
								cover_found = True
								break
						except:
							continue

			# Method 4: Try getting cover from Calibre's metadata if still not found
			if not cover_found:
				try:
					from calibre.ebooks.metadata.meta import get_metadata
					with open(epub_path, 'rb') as f:
						mi = get_metadata(f, 'epub')
						if mi.cover_data and mi.cover_data[1]:
							debug_print("OPFHelper: Found cover in EPUB metadata")
							self.cover_panel.show_cover(mi.cover_data[1])
							cover_found = True
				except Exception as e:
					debug_print(f"OPFHelper: Failed to extract cover from EPUB metadata: {str(e)}")

			# Method 5: Final fallback - try Calibre's database cover
			if not cover_found:
				try:
					cover = self.db.cover(book_id, index_is_id=True)
					if cover:
						debug_print("OPFHelper: Using cover from Calibre database")
						self.cover_panel.show_cover(cover)
						cover_found = True
				except Exception as e:
					debug_print(f"OPFHelper: Failed to get cover from Calibre database: {str(e)}")

			if not cover_found:
				debug_print("OPFHelper: No cover found for book")
				self.cover_panel.show_cover(None)  # Clear cover display

		except Exception as e:
			debug_print(f"OPFHelper ERROR: Failed to load cover: {str(e)}")
			traceback.print_exc()
			self.cover_panel.show_cover(None)

	def highlight_match(self, position_info):
		"""Highlight a specific match in the text"""
		try:
			position, length = position_info

			# Create cursor and set selection using positions directly
			cursor = QTextCursor(self.text_edit.document())
			cursor.setPosition(position)
			cursor.movePosition(QTextCursor.MoveOperation.NextCharacter, QTextCursor.MoveMode.KeepAnchor, length)

			# Apply cursor and ensure visible
			self.text_edit.setTextCursor(cursor)
			self.text_edit.ensureCursorVisible()

			return True
		except Exception as e:
			debug_print(f"OPFHelper ERROR: Highlight error: {e}")
			traceback.print_exc()
			return False

# Simple function to check if we're in dark mode
def is_dark_theme():
	return QApplication.instance().is_dark_theme

class ShowOPFAction(InterfaceAction):
	name = 'OPF Helper'
	# Create our top-level menu/toolbar action (text, icon_path, tooltip, keyboard shortcut)
	action_spec = ('OPF Helper', None, 'Inspect and analyze OPF content of EPUB books', None)  # Changed '' to None

	def genesis(self):
		debug_print('OPFHelper plugin: starting genesis()')

		# --- Theme-aware icon logic (minimal, non-intrusive) ---
		def _opfhelper_update_toolbar_icon():
			try:
				from calibre.gui2 import is_dark_theme
				icon_path = 'images/icon-for-dark-theme.png' if is_dark_theme() else 'images/icon-for-light-theme.png'
				if hasattr(self, 'qaction') and self.qaction is not None:
					self.qaction.setIcon(get_icon(icon_path))
			except Exception:
				pass

		def _opfhelper_theme_setup():
			# Try to connect to Calibre's theme_changed signal, fallback to timer
			try:
				from calibre.gui2 import gui
				g = gui()
				if hasattr(g, 'theme_changed'):
					g.theme_changed.connect(_opfhelper_update_toolbar_icon)
					return
			except Exception:
				pass
			# Fallback: poll for theme changes every second
			try:
				from calibre.gui2 import is_dark_theme
				self._opfhelper_last_theme = is_dark_theme()
				self._opfhelper_theme_timer = QTimer(self.gui if hasattr(self, 'gui') else None)
				def _opfhelper_check_theme():
					try:
						current = is_dark_theme()
						if current != self._opfhelper_last_theme:
							self._opfhelper_last_theme = current
							_opfhelper_update_toolbar_icon()
					except Exception:
						pass
				self._opfhelper_theme_timer.timeout.connect(_opfhelper_check_theme)
				self._opfhelper_theme_timer.start(1000)
			except Exception:
				pass

		_opfhelper_update_toolbar_icon()
		_opfhelper_theme_setup()

		# Create menu
		self.menu = QMenu()
		self.qaction.setMenu(self.menu)

		# Connect to menu first to prevent conflict with menu appearance
		self.menu.aboutToShow.connect(self.about_to_show_menu)

		# Build initial menu and register keyboard shortcuts
		self.rebuild_menus()

		# Register OPF Helper actions with Calibre's keyboard manager (unassigned by default)
		self._setup_opf_actions()

		# Connect the action trigger to main action last
		self.qaction.triggered.connect(self.show_opf)
		debug_print('OPFHelper plugin: genesis() complete')

	def _setup_opf_actions(self):
		"""Create QActions and register them with Calibre's keyboard manager.
		Users can assign their own keys in Preferences > Keyboard.
		"""
		try:
			kb = getattr(self.gui, 'keyboard', None)
			if kb is None:
				return

			# Show OPF Content
			self.opf_show_content = QAction(_('Show OPF Content'), self.gui)
			self.opf_show_content.triggered.connect(self.show_opf)
			self.opf_show_content.calibre_shortcut_unique_name = 'opf_helper.show_opf'
			self.gui.addAction(self.opf_show_content)
			# Register or replace existing shortcut entry
			if 'opf_helper.show_opf' in kb.shortcuts:
				kb.replace_action('opf_helper.show_opf', self.opf_show_content)
			else:
				kb.register_shortcut(
					'opf_helper.show_opf',
					_('OPF Helper: Show OPF Content'),
					default_keys=(), description=_('Show OPF content for selected books'), action=self.opf_show_content,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# Find Books with Multiple OPF Files
			self.opf_find_multiple = QAction(_('Find Books with Multiple OPF Files'), self.gui)
			self.opf_find_multiple.triggered.connect(self.check_for_multiple_opf_files)
			self.opf_find_multiple.calibre_shortcut_unique_name = 'opf_helper.find_multiple'
			self.gui.addAction(self.opf_find_multiple)
			if 'opf_helper.find_multiple' in kb.shortcuts:
				kb.replace_action('opf_helper.find_multiple', self.opf_find_multiple)
			else:
				kb.register_shortcut(
					'opf_helper.find_multiple',
					_('OPF Helper: Find Books with Multiple OPF Files'),
					default_keys=(), description=_('Scan library to find books with multiple OPF files'), action=self.opf_find_multiple,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# Find Books with XML Parsing Issues
			self.opf_find_xml_issues = QAction(_('Find Books with XML Parsing Issues'), self.gui)
			self.opf_find_xml_issues.triggered.connect(self.check_for_xml_parsing_issues)
			self.opf_find_xml_issues.calibre_shortcut_unique_name = 'opf_helper.find_xml_issues'
			self.gui.addAction(self.opf_find_xml_issues)
			if 'opf_helper.find_xml_issues' in kb.shortcuts:
				kb.replace_action('opf_helper.find_xml_issues', self.opf_find_xml_issues)
			else:
				kb.register_shortcut(
					'opf_helper.find_xml_issues',
					_('OPF Helper: Find Books with XML Parsing Issues'),
					default_keys=(), description=_('Scan library to find books with XML parsing issues in OPF files'), action=self.opf_find_xml_issues,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# Export Selected Books OPFs
			self.opf_export_selected = QAction(_('Export Selected Books OPFs'), self.gui)
			self.opf_export_selected.triggered.connect(self.export_selected_opfs)
			self.opf_export_selected.calibre_shortcut_unique_name = 'opf_helper.export_selected'
			self.gui.addAction(self.opf_export_selected)
			if 'opf_helper.export_selected' in kb.shortcuts:
				kb.replace_action('opf_helper.export_selected', self.opf_export_selected)
			else:
				kb.register_shortcut(
					'opf_helper.export_selected',
					_('OPF Helper: Export Selected Books OPFs'),
					default_keys=(), description=_('Export OPF files from selected books to a directory'), action=self.opf_export_selected,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# Find EPUB 3.0 Books
			self.opf_find_epub3 = QAction(_('Find EPUB 3.0 Books'), self.gui)
			self.opf_find_epub3.triggered.connect(self.find_epub_3_books)
			self.opf_find_epub3.calibre_shortcut_unique_name = 'opf_helper.find_epub_3'
			self.gui.addAction(self.opf_find_epub3)
			if 'opf_helper.find_epub_3' in kb.shortcuts:
				kb.replace_action('opf_helper.find_epub_3', self.opf_find_epub3)
			else:
				kb.register_shortcut(
					'opf_helper.find_epub_3',
					_('OPF Helper: Find EPUB 3.0 Books'),
					default_keys=(), description=_('Find all EPUB 3.0 books'), action=self.opf_find_epub3,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# Find EPUB 2.0 Books
			self.opf_find_epub2 = QAction(_('Find EPUB 2.0 Books'), self.gui)
			self.opf_find_epub2.triggered.connect(self.find_epub_2_books)
			self.opf_find_epub2.calibre_shortcut_unique_name = 'opf_helper.find_epub_2'
			self.gui.addAction(self.opf_find_epub2)
			if 'opf_helper.find_epub_2' in kb.shortcuts:
				kb.replace_action('opf_helper.find_epub_2', self.opf_find_epub2)
			else:
				kb.register_shortcut(
					'opf_helper.find_epub_2',
					_('OPF Helper: Find EPUB 2.0 Books'),
					default_keys=(), description=_('Find all EPUB 2.0 books'), action=self.opf_find_epub2,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# Find Non-EPUB 3.0 Books
			self.opf_find_not_epub3 = QAction(_('Find Non-EPUB 3.0 Books'), self.gui)
			self.opf_find_not_epub3.triggered.connect(self.find_epub_not_3_books)
			self.opf_find_not_epub3.calibre_shortcut_unique_name = 'opf_helper.find_not_epub_3'
			self.gui.addAction(self.opf_find_not_epub3)
			if 'opf_helper.find_not_epub_3' in kb.shortcuts:
				kb.replace_action('opf_helper.find_not_epub_3', self.opf_find_not_epub3)
			else:
				kb.register_shortcut(
					'opf_helper.find_not_epub_3',
					_('OPF Helper: Find Non-EPUB 3.0 Books'),
					default_keys=(), description=_('Find EPUB/KEPUB books that are not EPUB 3.0'), action=self.opf_find_not_epub3,
					group=_('OPF Helper'), persist_shortcut=True
				)

			# Apply any existing user-defined shortcuts to these actions
			kb.finalize()

			# --- Add application-level QShortcuts so Alt accelerators work when focus is in filter boxes or tables ---
			try:
				from PyQt5.QtWidgets import QShortcut
				from PyQt5.QtGui import QKeySequence
				# Keep references so they don't get GC'd
				self._app_shortcuts = []
				# Alt+S -> Show OPF Content
				sc_s = QShortcut(QKeySequence('Alt+S'), self.gui)
				sc_s.setContext(Qt.ApplicationShortcut)
				sc_s.activated.connect(self.show_opf)
				self._app_shortcuts.append(sc_s)

				# Alt+M -> Find Books with Multiple OPF Files
				sc_m = QShortcut(QKeySequence('Alt+M'), self.gui)
				sc_m.setContext(Qt.ApplicationShortcut)
				sc_m.activated.connect(self.check_for_multiple_opf_files)
				self._app_shortcuts.append(sc_m)

				# Alt+X -> Find Books with XML Parsing Issues
				sc_x = QShortcut(QKeySequence('Alt+X'), self.gui)
				sc_x.setContext(Qt.ApplicationShortcut)
				sc_x.activated.connect(self.check_for_xml_parsing_issues)
				self._app_shortcuts.append(sc_x)

				# Alt+E -> Export Selected Books OPFs
				sc_e = QShortcut(QKeySequence('Alt+E'), self.gui)
				sc_e.setContext(Qt.ApplicationShortcut)
				sc_e.activated.connect(self.export_selected_opfs)
				self._app_shortcuts.append(sc_e)

				# Alt+3 -> Find EPUB 3.0 Books
				sc_3 = QShortcut(QKeySequence('Alt+3'), self.gui)
				sc_3.setContext(Qt.ApplicationShortcut)
				sc_3.activated.connect(self.find_epub_3_books)
				self._app_shortcuts.append(sc_3)

				# Alt+2 -> Find EPUB 2.0 Books
				sc_2 = QShortcut(QKeySequence('Alt+2'), self.gui)
				sc_2.setContext(Qt.ApplicationShortcut)
				sc_2.activated.connect(self.find_epub_2_books)
				self._app_shortcuts.append(sc_2)

				# Alt+N -> Find Non-EPUB 3.0 Books
				sc_n = QShortcut(QKeySequence('Alt+N'), self.gui)
				sc_n.setContext(Qt.ApplicationShortcut)
				sc_n.activated.connect(self.find_epub_not_3_books)
				self._app_shortcuts.append(sc_n)

			except Exception:
				pass
		except Exception:
			# Non-fatal if keyboard manager not ready
			pass

	def show_help(self):
		"""Open OPF help page in browser"""
		QDesktopServices.openUrl(QUrl('https://wiki.mobileread.com/wiki/OPF'))

	def show_configuration(self):
		self.interface_action_base_plugin.do_user_config(self.gui)

	def check_for_multiple_opf_files(self):
		"""Check library for books with multiple OPF files, mark them, and filter to show marked books"""
		db = self.gui.current_db.new_api

		# Show progress dialog
		progress = QProgressDialog("Checking for multiple OPF files...", "Cancel", 0, 0, self.gui)
		progress.setWindowTitle("Multiple OPF Checker")
		progress.setMinimumDuration(0)
		progress.setValue(0)
		progress.setWindowModality(Qt.WindowModal)

		# Get all book IDs
		book_ids = db.all_book_ids()
		progress.setMaximum(len(book_ids))

		matching_books = []
		for i, book_id in enumerate(book_ids):
			if progress.wasCanceled():
				break

			progress.setValue(i)
			progress.setLabelText(f"Checking book {i+1} of {len(book_ids)}...")

			# Check for EPUB or KEPUB format
			formats = db.formats(book_id)
			fmt = None
			if formats:
				if 'EPUB' in formats:
					fmt = 'EPUB'
				elif 'KEPUB' in formats:
					fmt = 'KEPUB'
			if fmt:
				epub_path = db.format_abspath(book_id, fmt)
				if epub_path:
					try:
						with zipfile.ZipFile(epub_path, 'r') as zf:
							opf_files = [f for f in zf.namelist() if f.endswith('.opf')]
							if len(opf_files) > 1:
								matching_books.append(book_id)
					except Exception as e:
						debug_print(f"Error checking {epub_path}: {str(e)}")

		progress.close()

		# Mark matching books and filter view
		if matching_books:
			if hasattr(self.gui, 'iactions') and 'Mark Books' in self.gui.iactions:
				self.gui.iactions['Mark Books'].add_ids(matching_books)
				self.gui.search.set_search_string('marked:true')
				count = len(matching_books)
				info_dialog(self.gui, 'Books Found and Marked',
						   f'{count} book{"s" if count != 1 else ""} with multiple OPF files found and marked. '
						   'Library filtered to show marked books only.',
						   show=True)
			else:
				error_dialog(self.gui, 'Error', 'Could not access the Mark Books action.', show=True)
		else:
			self.gui.status_bar.showMessage("No books with multiple OPF files were found.", 4000)

	def check_for_xml_parsing_issues(self):
		"""Check library for books with XML parsing issues in OPF files"""
		from io import BytesIO
		import xml.etree.ElementTree as ET

		db = self.gui.current_db.new_api

		# Show progress dialog
		progress = QProgressDialog("Checking for XML parsing issues...", "Cancel", 0, 0, self.gui)
		progress.setWindowTitle("XML Parsing Issues Checker")
		progress.setMinimumDuration(0)
		progress.setValue(0)
		progress.setWindowModality(Qt.WindowModal)

		# Get all book IDs
		book_ids = db.all_book_ids()
		progress.setMaximum(len(book_ids))

		results = []
		for i, book_id in enumerate(book_ids):
			if progress.wasCanceled():
				break

			progress.setValue(i)
			progress.setLabelText(f"Checking book {i+1} of {len(book_ids)}...")

			# Check for EPUB or KEPUB format
			formats = db.formats(book_id)
			if formats and 'EPUB' in formats:
				# Get book path
				epub_path = db.format_abspath(book_id, 'EPUB')
				if epub_path:
					try:
						with zipfile.ZipFile(epub_path, 'r') as zf:
							opf_files = [f for f in zf.namelist() if f.endswith('.opf')]
							if opf_files:
								# Check the first OPF file for parsing issues
								opf_path = opf_files[0]
								try:
									opf_data = zf.read(opf_path).decode('utf-8', errors='replace')
									xml_bytes = opf_data.encode('utf-8')
									ET.parse(BytesIO(xml_bytes))
									# If we get here, parsing was successful
								except ET.ParseError as e:
									title = db.field_for('title', book_id)
									# Extract error details
									error_details = str(e)
									results.append((book_id, title, opf_path, error_details))
								except Exception as e:
									debug_print(f"Error checking OPF parsing for {epub_path}: {str(e)}")
					except Exception as e:
						debug_print(f"Error checking {epub_path}: {str(e)}")

		progress.close()

		# Show results
		if results:
			d = XMLParsingIssuesDialog(self.gui, results)
			d.exec_()
		else:
			info_dialog(self.gui, "XML Parsing Issues Checker",
						"No books with XML parsing issues were found.", show=True)

	def change_epub_scope(self, new_scope):
		"""Change the EPUB version finder scope setting"""
		prefs['epub_version_scope'] = new_scope
		# Update menu checkmarks
		self.scope_library_menu.setChecked(new_scope == 'Library')
		self.scope_selection_menu.setChecked(new_scope == 'Selection')

	def find_epub_3_books(self):
		"""Find and select all books with EPUB 3.0 format"""
		self._find_books_by_epub_version(version_filter='3.0', description="EPUB 3.0")

	def find_epub_2_books(self):
		"""Find and select all books with EPUB 2.0 format"""
		self._find_books_by_epub_version(version_filter='2.0', description="EPUB 2.0")

	def find_epub_not_3_books(self):
		"""Find and select all EPUB/KEPUB books that are not EPUB 3.0"""
		self._find_books_by_epub_version(version_filter='not_3.0', description="non-EPUB 3.0")

	def _find_books_by_epub_version(self, version_filter, description):
		"""Helper method to find books by EPUB version, mark them, and filter to show marked books"""
		# Get scope setting
		scope = prefs.get('epub_version_scope', 'Library')

		db = self.gui.current_db.new_api

		# Get book IDs based on scope
		if scope == 'Selection':
			# Only check selected books
			rows = self.gui.library_view.selectionModel().selectedRows()
			if not rows:
				self.gui.status_bar.showMessage("No books selected", 3000)
				return
			book_ids = []
			model = self.gui.library_view.model()
			for row in rows:
				book_id = model.id(row.row())
				if book_id is not None:
					book_ids.append(book_id)
		else:
			# Check all books in library
			book_ids = db.all_book_ids()

		# Show progress dialog
		progress = QProgressDialog(f"Finding {description} books...", "Cancel", 0, 0, self.gui)
		progress.setWindowTitle(f"EPUB Version Scanner - {description}")
		progress.setMinimumDuration(0)
		progress.setValue(0)
		progress.setWindowModality(Qt.WindowModal)
		progress.setMaximum(len(book_ids))

		matching_books = []
		for i, book_id in enumerate(book_ids):
			if progress.wasCanceled():
				break

			progress.setValue(i)
			progress.setLabelText(f"Scanning book {i+1} of {len(book_ids)}...")

			# Check for EPUB or KEPUB format
			formats = db.formats(book_id)
			fmt = None
			if formats:
				if 'EPUB' in formats:
					fmt = 'EPUB'
				elif 'KEPUB' in formats:
					fmt = 'KEPUB'
			if fmt:
				epub_path = db.format_abspath(book_id, fmt)
				if epub_path:
					try:
						with zipfile.ZipFile(epub_path, 'r') as zf:
							opf_files = [f for f in zf.namelist() if f.endswith('.opf')]
							if opf_files:
								# Read the first OPF file to check version
								opf_path = opf_files[0]
								opf_data = zf.read(opf_path).decode('utf-8', errors='replace')

								# Check version based on filter
								is_match = False
								if version_filter == '3.0':
									is_match = 'version="3.0"' in opf_data
								elif version_filter == '2.0':
									is_match = 'version="2.0"' in opf_data
								elif version_filter == 'not_3.0':
									is_match = 'version="3.0"' not in opf_data and ('version="2.0"' in opf_data or fmt == 'KEPUB')

								if is_match:
									matching_books.append(book_id)

					except Exception as e:
						debug_print(f"Error checking version for {epub_path}: {str(e)}")

		progress.close()

		# Mark matching books and show them
		if matching_books:
			# Mark the books using Calibre's mark functionality
			if hasattr(self.gui, 'iactions') and 'Mark Books' in self.gui.iactions:
				self.gui.iactions['Mark Books'].add_ids(matching_books)
				# Apply "marked:true" search to show only marked books
				self.gui.search.set_search_string('marked:true')
				count = len(matching_books)
				info_dialog(self.gui, 'Books Found and Marked',
						   f'{count} {description} book{"s" if count != 1 else ""} found and marked. '
						   f'Library filtered to show marked books only.',
						   show=True)
			else:
				error_dialog(self.gui, 'Error', 'Could not access the Mark Books action.', show=True)
		else:
			self.gui.status_bar.showMessage(f"No {description} books found in {scope.lower()}", 5000)
	def export_selected_opfs(self):
		"""Export OPF files from selected books to a directory"""
		# Get selected books
		rows = self.gui.library_view.selectionModel().selectedRows()
		if not rows or len(rows) == 0:
			self.gui.status_bar.showMessage("No books selected", 3000)
			return

		db = self.gui.current_db.new_api

		# Get selected book IDs
		book_ids = []
		model = self.gui.library_view.model()
		for row in rows:
			book_id = model.id(row.row())
			if book_id is not None:
				book_ids.append(book_id)

		if not book_ids:
			error_dialog(self.gui, 'Error', 'Could not get book IDs', show=True)
			return

		# Filter to only books with EPUB or KEPUB format
		valid_books = []
		for book_id in book_ids:
			formats = db.formats(book_id)
			if formats and ('EPUB' in formats or 'KEPUB' in formats):
				title = db.field_for('title', book_id)
				authors = db.field_for('authors', book_id)
				author_text = authors[0] if authors else "Unknown"
				valid_books.append((book_id, title, author_text))

		if not valid_books:
			self.gui.status_bar.showMessage("None of the selected books have EPUB or KEPUB format", 5000)
			return

		# Ask user for export directory
		from PyQt5.QtWidgets import QFileDialog
		export_dir = QFileDialog.getExistingDirectory(
			self.gui,
			"Select Export Directory",
			""
		)

		if not export_dir:
			return  # User cancelled

		# Show progress dialog
		from PyQt5.QtWidgets import QProgressDialog
		progress = QProgressDialog("Exporting OPF files...", "Cancel", 0, len(valid_books), self.gui)
		progress.setWindowTitle("Exporting OPF Files")
		progress.setMinimumDuration(0)
		progress.setValue(0)
		progress.setWindowModality(Qt.WindowModal)

		exported_count = 0
		errors = []

		for i, (book_id, title, author) in enumerate(valid_books):
			if progress.wasCanceled():
				break

			progress.setValue(i)
			progress.setLabelText(f"Exporting: {title}")

			try:
				# Get book path
				formats = db.formats(book_id)
				fmt = 'EPUB' if 'EPUB' in formats else 'KEPUB'
				epub_path = db.format_abspath(book_id, fmt)

				if epub_path and os.path.isfile(epub_path):
					with zipfile.ZipFile(epub_path, 'r') as zf:
						opf_files = [f for f in zf.namelist() if f.endswith('.opf')]

						if opf_files:
							# Use first OPF file
							opf_path = opf_files[0]
							opf_content = zf.read(opf_path).decode('utf-8', errors='replace')

							# Create safe filename
							safe_title = "".join(x for x in title if x.isalnum() or x in " ._-").strip()
							safe_author = "".join(x for x in author if x.isalnum() or x in " ._-").strip()
							filename = f"{safe_title} - {safe_author}.opf"
							filepath = os.path.join(export_dir, filename)

							# Handle duplicate filenames
							counter = 1
							base_filepath = filepath
							while os.path.exists(filepath):
								name, ext = os.path.splitext(base_filepath)
								filepath = f"{name}_{counter}{ext}"
								counter += 1

							# Write OPF file
							with open(filepath, 'w', encoding='utf-8') as f:
								f.write(opf_content)

							exported_count += 1

			except Exception as e:
				errors.append(f"{title}: {str(e)}")

		progress.close()

		# Show results
		if exported_count > 0:
			message = f"Successfully exported {exported_count} OPF file{'s' if exported_count != 1 else ''} to {export_dir}"
			if errors:
				message += f"\n\nErrors occurred for {len(errors)} book{'s' if len(errors) != 1 else ''}."
			info_dialog(self.gui, 'Export Complete', message, show=True)
		else:
			error_dialog(self.gui, 'Export Failed', 'No OPF files were exported.', show=True)

		if errors:
			# Show errors in a separate dialog if any
			error_text = "The following books had errors during export:\n\n" + "\n".join(errors)
			error_dialog(self.gui, 'Export Errors', error_text, show=True)

	def about_to_show_menu(self):
		# Update menu items state based on current context
		rows = self.gui.library_view.selectionModel().selectedRows()
		has_selection = bool(rows and len(rows) > 0)

		# Enable/disable menu items based on selection
		for action in self.menu.actions():
			if action.text() == _('Show OPF Content'):
				action.setEnabled(has_selection)
			elif action.text() == _('Export Selected Books OPFs'):
				action.setEnabled(has_selection)

	def show_opf_comparison(self):
		"""Show OPF comparison dialog for selected books"""
		debug_print('OPFHelper: show_opf_comparison() called')
		gui = self.gui
		rows = gui.library_view.selectionModel().selectedRows()
		if not rows or len(rows) == 0:
			gui.status_bar.showMessage("No books selected", 3000)
			return

		try:
			# Get the current database API handle
			db = gui.current_db.new_api

			# Get selected book IDs using the model
			book_ids = []
			model = gui.library_view.model()
			for row in rows:
				book_id = model.id(row.row())
				if book_id is not None:
					book_ids.append(book_id)

			if not book_ids:
				error_dialog(gui, 'Error', 'Could not get book IDs', show=True)
				return

			# Use the first selected book
			book_id = book_ids[0]

			# Filter to only books with EPUB or KEPUB format
			if not (db.has_format(book_id, 'EPUB') or db.has_format(book_id, 'KEPUB')):
				gui.status_bar.showMessage("Selected book does not have EPUB or KEPUB format", 5000)
				return

			# Get book title
			title = db.field_for('title', book_id)

			# Get OPF content
			fmt = 'KEPUB' if db.has_format(book_id, 'KEPUB') else 'EPUB'
			epub_path = db.format_abspath(book_id, fmt)

			if not epub_path or not os.path.isfile(epub_path):
				gui.status_bar.showMessage("EPUB or KEPUB format not available for this book", 3000)
				return

			try:
				with zipfile.ZipFile(epub_path, 'r') as zf:
					opf_files = [f for f in zf.namelist() if f.endswith('.opf')]
					if opf_files:
						opf_path = opf_files[0]  # Use first OPF file
						opf_content = zf.read(opf_path).decode('utf-8', errors='replace')

						# Open comparison dialog
						d = OPFComparisonDialog(gui, opf_content, title)
						d.exec_()
					else:
						gui.status_bar.showMessage("No OPF file found in the selected book", 3000)
			except Exception as e:
				error_dialog(gui, 'Error', f'Failed to read OPF content: {str(e)}', show=True)

		except Exception as e:
			error_dialog(gui, 'Error', str(e), show=True)

	def show_opf(self):
		"""Show OPF content for selected books"""
		debug_print('OPFHelper: show_opf() called')
		gui = self.gui
		rows = gui.library_view.selectionModel().selectedRows()
		if not rows or len(rows) == 0:
			gui.status_bar.showMessage("No books selected", 3000)
			return

		try:
			# Get the current database API handle
			db = gui.current_db.new_api

			# Get selected book IDs using the model
			book_ids = []
			model = gui.library_view.model()
			for row in rows:
				book_id = model.id(row.row())
				if book_id is not None:
					book_ids.append(book_id)

			if not book_ids:
				error_dialog(gui, 'Error', 'Could not get book IDs', show=True)
				return

			# Filter to only books with EPUB or KEPUB format
			valid_book_ids = [bid for bid in book_ids if db.has_format(bid, 'EPUB') or db.has_format(bid, 'KEPUB')]

			if not valid_book_ids:
				gui.status_bar.showMessage("None of the selected books have EPUB or KEPUB format", 5000)
				return

			# Open OPF Content dialog for selected books
			d = OPFContentDialog(gui, valid_book_ids, db, self.qaction.icon())
			d.exec_()
		except Exception as e:
			error_dialog(gui, 'Error', str(e), show=True)

	def rebuild_menus(self):
		"""Build and populate the plugin's menu"""
		debug_print('OPFHelper: rebuilding menus')
		try:
			self.menu.clear()

			# Main action: Show OPF Content
			create_menu_action_unique(self, self.menu, _('Show OPF Content'),
								   image='images/icon-for-light-theme.png',
								   tooltip=_('Show OPF content of selected books'),
								   triggered=self.show_opf)

			# Multiple OPF Checker action
			create_menu_action_unique(self, self.menu, _('Find Books with Multiple OPF Files'),
								   image='images/multiple.png',
								   tooltip=_('Scan library to find books with multiple OPF files'),
								   triggered=self.check_for_multiple_opf_files)

			# XML Parsing Issues Checker action
			create_menu_action_unique(self, self.menu, _('Find Books with XML Parsing Issues'),
								   image='images/xml_error.png',
								   tooltip=_('Scan library to find books with XML parsing issues in OPF files'),
								   triggered=self.check_for_xml_parsing_issues)

			# Export Selected OPFs action
			create_menu_action_unique(self, self.menu, _('Export Selected Books OPFs'),
								   image='images/export_icon.png',
								   tooltip=_('Export OPF files from selected books to a directory'),
								   triggered=self.export_selected_opfs)

			# OPF Standards Comparison action
			create_menu_action_unique(self, self.menu, _('OPF Standards Comparison'),
								   image='images/diff_icon.png',
								   tooltip=_('Compare current OPF with standards-corrected version'),
								   triggered=self.show_opf_comparison)

			# EPUB Version finder actions
			self.menu.addSeparator()
			epub_menu = self.menu.addMenu(_('Find by EPUB Version'))
			epub_menu.setIcon(get_icon('images/search_icon.png'))

			# Add scope submenu
			scope_menu = epub_menu.addMenu(_('Scope'))
			self.scope_library_menu = create_menu_action_unique(self, scope_menu, _('Library'),
															   tooltip=_('Search entire library'),
															   triggered=lambda: self.change_epub_scope('Library'),
															   is_checked = bool(prefs.get('epub_version_scope', 'Library') == 'Library'))
			self.scope_selection_menu = create_menu_action_unique(self, scope_menu, _('Selected book(s)'),
															     tooltip=_('Search only selected books'),
															     triggered=lambda: self.change_epub_scope('Selection'),
															     is_checked = bool(prefs.get('epub_version_scope', 'Library') == 'Selection'))
			epub_menu.addSeparator()

			create_menu_action_unique(self, epub_menu, _('Find EPUB 3.0 Books'),
								   image='images/search_icon.png',
								   tooltip=_('Find and mark all EPUB 3.0 books'),
								   triggered=self.find_epub_3_books)
			create_menu_action_unique(self, epub_menu, _('Find EPUB 2.0 Books'),
								   image='images/search_icon.png',
								   tooltip=_('Find and mark all EPUB 2.0 books'),
								   triggered=self.find_epub_2_books)
			create_menu_action_unique(self, epub_menu, _('Find Non-EPUB 3.0 Books'),
								   image='images/search_icon.png',
								   tooltip=_('Find and mark all EPUB/KEPUB books that are not EPUB 3.0'),
								   triggered=self.find_epub_not_3_books)

			self.menu.addSeparator()

			# Help actions
			create_menu_action_unique(self, self.menu, _('OPF Specification Help'),
								   image='help.png',
								   tooltip=_('Open the OPF specification documentation'),
								   triggered=self.show_help)

			# Configuration action if plugin has configuration
			if hasattr(self.interface_action_base_plugin, 'config_widget'):
				create_menu_action_unique(self, self.menu, _('Customize Plugin...'),
									  image='config.png',
									  tooltip=_('Customize plugin behavior'),
									  triggered=self.show_configuration)

		except Exception as e:
			debug_print(f'OPFHelper ERROR rebuilding menus: {str(e)}')
			traceback.print_exc()

class ElidedLabel(QLabel):
	"""Label that elides text in the middle when too long"""
	def __init__(self, parent=None):
		super().__init__(parent)
		self.full_text = ""

	def setText(self, text):
		self.full_text = text
		self.setToolTip(text)  # Show full text on hover
		# Elide in the middle
		metrics = self.fontMetrics()
		if metrics.horizontalAdvance(text) > self.width():
			available_width = self.width() - metrics.horizontalAdvance("...")
			left_width = available_width // 2
			right_width = available_width - left_width
			left_text = metrics.elidedText(text, Qt.ElideRight, left_width)
			right_text = metrics.elidedText(text[::-1], Qt.ElideLeft, right_width)[::-1]
			display_text = left_text[:-3] + "..." + right_text[3:]
			super().setText(display_text)
		else:
			super().setText(text)

	def resizeEvent(self, event):
		super().resizeEvent(event)
		if self.full_text:
			self.setText(self.full_text)  # Recalculate eliding on resize

class CoverPanel(QWidget):
	def __init__(self, parent=None):
		super().__init__(parent)
		self.pixmap = QPixmap()
		self.current_pixmap_size = QSize()
		self.setMinimumSize(200, 200)  # Keep minimum size reasonable
		self.setMaximumWidth(400)      # Limit maximum width
		self.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Expanding)  # Changed from Expanding to Minimum
		self.data = {}
		self.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
		self.customContextMenuRequested.connect(self.show_context_menu)

	def show_cover(self, cover_data):
		if cover_data:
			self.pixmap.loadFromData(cover_data)
			# Scale pixmap while maintaining aspect ratio
			scaled_size = self.pixmap.size()
			scaled_size.scale(400, 800, Qt.AspectRatioMode.KeepAspectRatio)  # Max dimensions
			self.current_pixmap_size = scaled_size
		else:
			self.pixmap = QPixmap()
			self.current_pixmap_size = QSize()
		self.update()

	def show_context_menu(self, pos):
		menu = QMenu(self)
		if not self.pixmap.isNull():
			copy_action = menu.addAction(QIcon(get_icon('edit-copy.png')), _('Copy cover'))
			copy_action.triggered.connect(self.copy_to_clipboard)
		menu.exec(self.mapToGlobal(pos))

	def copy_to_clipboard(self):
		if not self.pixmap.isNull():
			QApplication.clipboard().setPixmap(self.pixmap)

	def sizeHint(self):
		if self.pixmap.isNull():
			return QSize(200, 300)
		return self.current_pixmap_size

	def paintEvent(self, event):
		if self.pixmap.isNull():
			return

		canvas_size = self.rect()
		target = self.calculate_target_rect(canvas_size)

		p = QPainter(self)
		p.setRenderHints(QPainter.RenderHint.Antialiasing | QPainter.RenderHint.SmoothPixmapTransform)

		try:
			dpr = self.devicePixelRatioF()
		except AttributeError:
			dpr = self.devicePixelRatio()

		spmap = self.pixmap.scaled(target.size() * dpr, Qt.AspectRatioMode.KeepAspectRatio, Qt.TransformationMode.SmoothTransformation)
		spmap.setDevicePixelRatio(dpr)
		p.drawPixmap(target, spmap)

		# Add size overlay with dimensions
		sztgt = target.adjusted(0, 0, 0, -4)
		f = p.font()
		f.setBold(True)
		p.setFont(f)
		sz = f'\xa0{self.pixmap.width()} x {self.pixmap.height()}\xa0'
		flags = Qt.AlignmentFlag.AlignBottom|Qt.AlignmentFlag.AlignRight|Qt.TextFlag.TextSingleLine
		szrect = p.boundingRect(sztgt, flags, sz)
		p.fillRect(szrect.adjusted(0, 0, 0, 4), QColor(0, 0, 0, 200))
		p.setPen(QPen(QColor(255,255,255)))
		p.drawText(sztgt, flags, sz)
		p.end()

	def calculate_target_rect(self, canvas_size):
		"""Calculate the target rectangle for drawing the scaled pixmap"""
		if self.pixmap.isNull():
			return QRect()

		# Get available space
		available_width = min(canvas_size.width(), 400)  # Limit width
		available_height = canvas_size.height()

		# Calculate scaled dimensions maintaining aspect ratio
		scaled_size = self.pixmap.size()
		scaled_size.scale(available_width, available_height, Qt.AspectRatioMode.KeepAspectRatio)

		# Center in available space
		x = (canvas_size.width() - scaled_size.width()) // 2
		y = (canvas_size.height() - scaled_size.height()) // 2

		return QRect(x, y, scaled_size.width(), scaled_size.height())

# Add the MultipleOPFDialog class
class MultipleOPFDialog(QDialog):
	def __init__(self, gui, results):
		QDialog.__init__(self, gui)
		self.gui = gui
		self.setWindowTitle('Multiple OPF Files Report')
		self.setMinimumWidth(700)
		self.setMinimumHeight(400)

		layout = QVBoxLayout()
		self.setLayout(layout)

		# Add description
		desc_text = f"Found {len(results)} books with multiple OPF files:"
		desc = QLabel(desc_text)
		desc.setStyleSheet("font-weight: bold;")
		layout.addWidget(desc)

		# Create text display
		self.text_edit = QTextEdit()
		self.text_edit.setReadOnly(True)
		layout.addWidget(self.text_edit)

		# Format and set results
		result_text = ""
		self.book_ids = []
		for book_id, title, opf_files in results:
			self.book_ids.append(book_id)
			result_text += f"Book: {title} (ID: {book_id})\n"
			result_text += "OPF Files:\n"
			for opf_file in opf_files:
				result_text += f"  • {opf_file}\n"
			result_text += "\n"

		self.text_edit.setPlainText(result_text)

		# Add buttons
		button_layout = QHBoxLayout()

		copy_button = QPushButton("Copy to Clipboard")
		copy_button.setIcon(QIcon.ic('edit-copy.png'))
		copy_button.clicked.connect(self.copy_to_clipboard)
		button_layout.addWidget(copy_button)

		# Add mark button for marking books in calibre with proper icon
		self.mark_button = QPushButton("Mark Affected Books")
		self.mark_button.setIcon(QIcon.ic('marked.png'))
		self.mark_button.setToolTip("Mark these books in Calibre's book list")
		self.mark_button.clicked.connect(self.mark_books)
		button_layout.addWidget(self.mark_button)

		# Add standard dialog buttons
		button_box = QDialogButtonBox(QDialogButtonBox.Ok)
		button_box.accepted.connect(self.accept)
		button_layout.addWidget(button_box)

		layout.addLayout(button_layout)

	def copy_to_clipboard(self):
		text = self.text_edit.toPlainText()
		QApplication.clipboard().setText(text)
		info_dialog(self, 'Copied', 'Report copied to clipboard', show=True)

	def mark_books(self):
		"""Mark the affected books in Calibre's book list"""
		if hasattr(self.gui, 'iactions') and 'Mark Books' in self.gui.iactions:
			self.gui.iactions['Mark Books'].add_ids(self.book_ids)
			count = len(self.book_ids)
			info_dialog(self, 'Books Marked',
						f'{count} book{"s" if count != 1 else ""} with multiple OPF files '
						f'{"have" if count != 1 else "has"} been marked in your library.',
						show=True)
		else:
			error_dialog(self, 'Error', 'Could not access the Mark Books action.', show=True)

# Add the XMLParsingIssuesDialog class
class XMLParsingIssuesDialog(QDialog):
	def __init__(self, gui, results):
		QDialog.__init__(self, gui)
		self.gui = gui
		self.setWindowTitle('XML Parsing Issues Report')
		self.setMinimumWidth(700)
		self.setMinimumHeight(400)

		layout = QVBoxLayout()
		self.setLayout(layout)

		# Add description
		desc_text = f"Found {len(results)} books with XML parsing issues in OPF files:"
		desc = QLabel(desc_text)
		desc.setStyleSheet("font-weight: bold;")
		layout.addWidget(desc)

		# Create text display
		self.text_edit = QTextEdit()
		self.text_edit.setReadOnly(True)
		layout.addWidget(self.text_edit)

		# Format and set results
		result_text = ""
		self.book_ids = []
		for book_id, title, opf_path, error_details in results:
			self.book_ids.append(book_id)
			result_text += f"Book: {title} (ID: {book_id})\n"
			result_text += f"OPF File: {opf_path}\n"
			result_text += f"Error: {error_details}\n\n"

		self.text_edit.setPlainText(result_text)

		# Add buttons
		button_layout = QHBoxLayout()

		copy_button = QPushButton("Copy to Clipboard")
		copy_button.setIcon(QIcon.ic('edit-copy.png'))
		copy_button.clicked.connect(self.copy_to_clipboard)
		button_layout.addWidget(copy_button)

		# Add mark button for marking books in calibre with proper icon
		self.mark_button = QPushButton("Mark Affected Books")
		self.mark_button.setIcon(QIcon.ic('marked.png'))
		self.mark_button.setToolTip("Mark these books in Calibre's book list")
		self.mark_button.clicked.connect(self.mark_books)
		button_layout.addWidget(self.mark_button)

		# Add standard dialog buttons
		button_box = QDialogButtonBox(QDialogButtonBox.Ok)
		button_box.accepted.connect(self.accept)
		button_layout.addWidget(button_box)

		layout.addLayout(button_layout)

	def copy_to_clipboard(self):
		text = self.text_edit.toPlainText()
		QApplication.clipboard().setText(text)
		info_dialog(self, 'Copied', 'Report copied to clipboard', show=True)

	def mark_books(self):
		"""Mark the affected books in Calibre's book list"""
		if hasattr(self.gui, 'iactions') and 'Mark Books' in self.gui.iactions:
			self.gui.iactions['Mark Books'].add_ids(self.book_ids)
			count = len(self.book_ids)
			info_dialog(self, 'Books Marked',
						f'{count} book{"s" if count != 1 else ""} with XML parsing issues '
						f'{"have" if count != 1 else "has"} been marked in your library.',
						show=True)
		else:
			error_dialog(self, 'Error', 'Could not access the Mark Books action.', show=True)
