#!/usr/bin/env python
__license__   = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'
# -*- coding: utf-8 -*-

from qt.core import QApplication
"""
AI Config plugin - toolbar button with theme-aware icon and action selector
"""

from functools import partial
from qt.core import QMenu, QActionGroup, QToolButton
from calibre.gui2.actions import InterfaceAction
from calibre.gui2 import Application
from calibre.utils.localization import _
from calibre.utils.config import JSONConfig
from calibre.ai import AICapabilities
from calibre.ai.prefs import plugin_for_purpose, pref_for_provider, prefs
from . import common_icons

# Persistent config for selected action
plugin_prefs = JSONConfig('plugins/ai_config')

# Action keys and labels (update callbacks to perform the real work)
ACTIONS = [
    # Only the two settings entries: selecting one sets the default action
    {
        'key': 'ai_settings_viewer',
        'label': _('E-book viewer'),
        'icon': 'images/iconplugin2',
        'callback': 'open_ai_settings_viewer',
    },
    {
        'key': 'ai_settings_library',
        'label': _('Main'),
        'icon': 'images/iconplugin',
        'callback': 'open_ai_settings_library',
    },
]


class AIConfigAction(InterfaceAction):
    """Toolbar action. Use action_spec so Calibre will register the action and
    prompt for toolbar placement on first install (same pattern as KoboTouch).
    """
    name = 'AI Config'
    # (text, icon, tooltip, keyboard shortcut)
    action_spec = (_('AI Config'), None, _('Quick access to AI settings'), None)
    # Ensure Calibre renders the popup arrow on the toolbar button
    action_add_menu = True
    popup_type = QToolButton.ToolButtonPopupMode.MenuButtonPopup
    action_type = 'current'
    action_menu_clone_qaction = True

    def genesis(self):
        # Make available to common_icons so it can look up local resources
        common_icons.set_plugin_icon_resources(self.name, getattr(self, 'resources', {}))

        # Load persisted selection
        self.selected_action = plugin_prefs.get('selected_action', ACTIONS[0]['key'])

        # Ensure qaction exists (InterfaceAction provides it via action_spec)
        # Set initial icon/tooltip
        self.rebuild_icon()
        self.qaction.triggered.connect(self.trigger_selected_action)

        # Build selector menu and attach to the QAction so the toolbar button has a menu
        self.menu = self._build_menu()
        self.menu.aboutToShow.connect(self.update_current_provider)
        self.update_current_provider()
        try:
            # QAction has setMenu in newer Qt versions used by calibre
            self.qaction.setMenu(self.menu)
        except Exception:
            # Fallback: nothing — menu will still be accessible via context menu
            pass

        # Register hidden actions for direct shortcuts to each dialog (for accessibility)
        try:
            from qt.core import QAction
            self._viewer_action = QAction(_('Open AI Viewer Settings'), self.gui)
            self._viewer_action.triggered.connect(self.open_ai_settings_viewer)
            self.gui.addAction(self._viewer_action)
            self.gui.keyboard.register_shortcut('ai_config_viewer', _('Open AI Viewer Settings'), default_keys=(), action=self._viewer_action, description=_('Open AI Viewer Settings'), group=self.action_spec[0])

            self._main_action = QAction(_('Open AI Main Settings'), self.gui)
            self._main_action.triggered.connect(self.open_ai_settings_library)
            self.gui.addAction(self._main_action)
            self.gui.keyboard.register_shortcut('ai_config_main', _('Open AI Main Settings'), default_keys=(), action=self._main_action, description=_('Open AI Main Settings'), group=self.action_spec[0])
        except Exception:
            pass

        # No-op: Refresh is now built into _build_menu

    def rebuild_icon(self):
        # Use theme-aware get_icon; fall back gracefully
        icon_name = self._get_selected()['icon']
        try:
            icon = common_icons.get_icon(icon_name)
            self.qaction.setIcon(icon)
        except Exception:
            pass

    def _build_menu(self):
        from qt.core import QWidgetAction, QLabel, QHBoxLayout, QWidget, QStyle
        menu = QMenu(self.gui)
        # Custom widget for current provider (colored icon, non-interactive)
        widget = QWidget(menu)
        layout = QHBoxLayout(widget)
        layout.setContentsMargins(8, 4, 8, 4)
        from qt.core import Qt
        from qt.core import QPushButton, QClipboard
        self.provider_icon_label = QLabel(widget)
        self.provider_text_label = QLabel(widget)
        self.provider_text_label.setTextInteractionFlags(Qt.TextSelectableByMouse)
        self.provider_text_label.setWordWrap(True)
        self.provider_text_label.setStyleSheet('font-size: 11px;')
        self.copy_button = QPushButton('📋', widget)
        self.copy_button.setToolTip(_('Copy to clipboard'))
        self.copy_button.setFixedSize(22, 22)
        self.copy_button.setStyleSheet('QPushButton { border: none; background: transparent; font-size: 13px; } QPushButton:hover { background: #e0e0e0; }')
        def copy_text():
            clipboard = QApplication.clipboard()
            clipboard.setText(self.provider_text_label.text())
        self.copy_button.clicked.connect(copy_text)
        layout.addWidget(self.provider_icon_label)
        layout.addWidget(self.provider_text_label, 1)
        layout.addWidget(self.copy_button)
        widget.setLayout(layout)
        wa = QWidgetAction(menu)
        wa.setDefaultWidget(widget)
        menu.addAction(wa)
        menu.addSeparator()
        # Reinstate orange hover effect for menu items
        menu.setStyleSheet('''
            QMenu::item:selected {
                background-color: #ffd580;
                color: #000000;
            }
        ''')
        ag = QActionGroup(menu)
        ag.setExclusive(False)
        sep_added = False
        for action in ACTIONS:
            act = menu.addAction(action['label'])
            act.setCheckable(True)
            act.setChecked(action['key'] == self.selected_action)
            try:
                act.setData(action['key'])
            except Exception:
                pass
            act.triggered.connect(lambda checked, k=action['key']: self._on_menu_action_selected(k))
            ag.addAction(act)
            if not sep_added and action['key'].startswith('ai_settings'):
                sep_added = True
        self.current_provider_action = wa  # For compatibility in update_current_provider
        return menu

    def open_ai_settings_viewer(self):
        try:
            from calibre.gui2.viewer.llm import LLMSettingsDialog as ViewerLLMSettingsDialog
            d = ViewerLLMSettingsDialog(self.gui)
            # Restore last tab index if available
            last_tab = plugin_prefs.get('last_tab_viewer', 0)
            try:
                d.tabs.setCurrentIndex(last_tab)
            except Exception:
                pass
            # Save tab index on change
            try:
                d.tabs.currentChanged.connect(lambda idx: plugin_prefs.__setitem__('last_tab_viewer', idx))
            except Exception:
                pass
            d.exec()
        except Exception:
            import traceback
            traceback.print_exc()

    def open_ai_settings_library(self):
        try:
            from calibre.gui2.dialogs.llm_book import LLMSettingsDialog as BookLLMSettingsDialog
            d = BookLLMSettingsDialog(self.gui)
            # Restore last tab index if available
            last_tab = plugin_prefs.get('last_tab_library', 0)
            try:
                d.tabs.setCurrentIndex(last_tab)
            except Exception:
                pass
            # Save tab index on change
            try:
                d.tabs.currentChanged.connect(lambda idx: plugin_prefs.__setitem__('last_tab_library', idx))
            except Exception:
                pass
            d.exec()
        except Exception:
            import traceback
            traceback.print_exc()

    def _on_menu_action_selected(self, key):
        # Set the default action
        try:
            self.set_selected_action(key)
        except Exception:
            pass
        # Do not auto-open settings dialogs when the settings entry is
        # selected. Selecting simply sets the default action for the button.

    def _get_selected(self):
        for a in ACTIONS:
            if a['key'] == self.selected_action:
                return a
        return ACTIONS[0]

    def set_selected_action(self, key):
        self.selected_action = key
        plugin_prefs['selected_action'] = key
        # update icon and tooltip
        sel = self._get_selected()
        try:
            self.qaction.setIcon(common_icons.get_icon(sel['icon']))
            # Do not set the toolbar text; keep the toolbar icon-only
            self.qaction.setToolTip(sel.get('label', ''))
        except Exception:
            pass
        # update menu checks
        for act in self.menu.actions():
            try:
                if act.isSeparator():
                    continue
            except Exception:
                pass
            k = act.data()
            if k:
                act.setChecked(k == key)

    def trigger_selected_action(self):
        """When the main plugin action is triggered (toolbar button or shortcut),
        open the dialog for the currently selected action."""
        sel = self._get_selected()
        cb = getattr(self, sel['callback'], None)
        if cb:
            try:
                cb()
            except Exception:
                import traceback
                traceback.print_exc()
        # Update menu state to reflect current selection
        for act in self.menu.actions():
            if hasattr(act, 'data'):
                try:
                    k = act.data()
                    if k:
                        act.setChecked(k == self.selected_action)
                except Exception:
                    pass

    def do_action1(self):
        # Open viewer LLM settings
        try:
            self.open_ai_settings_viewer()
        except Exception:
            import traceback
            traceback.print_exc()

    def do_action2(self):
        # Open library/book LLM settings
        try:
            self.open_ai_settings_library()
        except Exception:
            import traceback
            traceback.print_exc()

    def update_current_provider(self):
        prefs().refresh()
        p = plugin_for_purpose(AICapabilities.text_to_text)
        if p:
            name = p.name
            model = pref_for_provider(p.name, 'text_model', '')
            print(f"DEBUG: provider {name}, text_model: {model}")
            if model:
                if isinstance(model, tuple) and len(model) == 2:
                    # OpenRouter: (id, name)
                    text = f"{name}: {model[1]}"
                elif isinstance(model, dict) and 'name' in model:
                    # GitHub: {'name': name, 'id': id}
                    text = f"{name}: {model['name']}"
                elif isinstance(model, str):
                    # Ollama, LM Studio: str
                    text = f"{name}: {model}"
                else:
                    text = f"{name}: {model}"
            else:
                # Try 'model' key
                model = pref_for_provider(p.name, 'model', '')
                print(f"DEBUG: provider {name}, model: {model}")
                if model:
                    text = f"{name}: {model}"
                else:
                    text = name
            # Set provider-specific icon
            icon_map = {
                'OpenRouter': 'images/openrouter_icon.png',
                'GoogleAI': 'images/google_icon.png',
                'GitHubAI': 'images/github_icon.png',
                'OllamaAI': 'images/ollama_icon.png',
                'LMStudio': 'images/lm_studio_icon.png',
                'OpenAI': 'images/open_ai_icon.png',
            }
            icon_name = icon_map.get(name, 'images/default_provider_icon.png')
            try:
                icon = common_icons.get_icon(icon_name)
                self.provider_icon_label.setPixmap(icon.pixmap(24, 24))
            except Exception:
                self.provider_icon_label.clear()
        else:
            text = _('None')
            self.provider_icon_label.clear()
        self.provider_text_label.setText(str(text))
        self.provider_text_label.setToolTip(str(text))
