local Device = require("device")

if not Device:isPocketBook() then
    return { disabled = true, }
end

local WidgetContainer = require("ui/widget/container/widgetcontainer")
local _ = require("gettext")
local T = require("ffi/util").template
local logger = require("logger")
local util = require("util")
local SQ3 = require("lua-ljsqlite3/init")
local pocketbookDbConn = SQ3.open("/mnt/ext1/system/explorer-3/explorer-3.db")
local ffi = require("ffi")
local inkview = ffi.load("inkview")
local bookIds = {}
local UIManager = require("ui/uimanager")
local InfoMessage = require("ui/widget/infomessage")

local PocketbookSync = WidgetContainer:extend{
    name = "pocketbooksync",
    is_doc_only = false,
}

function PocketbookSync:init()
    self.ui.menu:registerToMainMenu(self)
    
    -- Register as active widget to receive events even when hidden
    table.insert(self.ui.active_widgets, self)
    
end

local collection_info_text = _([[
The function allows you to automatically remove a book from a specified collection after reading it.]])

local about_text = _([[
A KOReader plugin that syncs reading progress from KOReader to PocketBook Library, primarily to make the book progress bars on PocketBook's home screen accurate.

If you want the book to be automatically removed from a specific collection after reading, set the collection name in the settings.]])

function PocketbookSync:getCollectionMenuTable()
    local collections = self:getKOReaderCollections()
    local sub_item_table = {}
    
    -- Add "None" option
    table.insert(sub_item_table, {
        text = _("None (disable auto-removal)"),
        checked_func = function()
            return G_reader_settings:readSetting("to_read_collection_name") == nil
        end,
        callback = function()
            G_reader_settings:delSetting("to_read_collection_name")
            G_reader_settings:delSetting("to_read_collection_id")
            G_reader_settings:flush()
            logger.info("Pocketbook Sync: Collection settings cleared")
            UIManager:show(InfoMessage:new{
                text = _("Auto-removal disabled"),
                timeout = 2,
            })
        end,
        radio = true,
        separator = true,
    })
    
    -- Add all collections
    for i, coll_name in ipairs(collections) do
        table.insert(sub_item_table, {
            text = coll_name,
            checked_func = function()
                return coll_name == G_reader_settings:readSetting("to_read_collection_name")
            end,
            callback = function()
                G_reader_settings:saveSetting("to_read_collection_name", coll_name)
                G_reader_settings:flush()
                
                local collection_id = self:getCollectionIdByName(coll_name)
                if collection_id then
                    G_reader_settings:saveSetting("to_read_collection_id", collection_id)
                    G_reader_settings:flush()
                    logger.info("Pocketbook Sync: Collection '" .. coll_name .. "' (ID: " .. tostring(collection_id) .. ") saved")
                else
                    logger.info("Pocketbook Sync: Collection ID not found for '" .. coll_name .. "'")
                    G_reader_settings:delSetting("to_read_collection_id")
                    G_reader_settings:flush()
                end
                
                UIManager:show(InfoMessage:new{
                    text = T(_("Collection set: %1"), coll_name),
                    timeout = 2,
                })
            end,
            radio = true,
        })
    end
    
    return {
        text_func = function()
            local collection_name = G_reader_settings:readSetting("to_read_collection_name")
            if collection_name then
                return T(_("Collection: %1"), collection_name)
            else
                return _("Collection: None")
            end
        end,
        sub_item_table = sub_item_table,
        help_text = collection_info_text,
    }
end

function PocketbookSync:addToMainMenu(menu_items)
    menu_items.pocketbook_sync = {
		sorting_hint = "tools",
        text = _("PocketBook Sync"),
        sub_item_table = {
			{
                text = _("About Pocketbook Sync"),
                callback = function()
                    UIManager:show(InfoMessage:new{
                        text = about_text,
                    })
                end,
                keep_menu_open = true,
                separator = true,
            },
            self:getCollectionMenuTable(),
        }
    }
end

function PocketbookSync:getCollectionIdByName(name)
    if not name or name == "" then
        return nil
    end
    
    local stmt = pocketbookDbConn:prepare("SELECT id FROM bookshelfs WHERE name = ? AND is_deleted != 1 LIMIT 1")
    local row = stmt:reset():bind(name):step()
    stmt:close()
    
    if row == nil then
        logger.info("Pocketbook Sync: Collection '" .. name .. "' not found")
        return nil
    end
    
	local id_str = tostring(row[1]):gsub("LL$", "")
	local id_num = tonumber(id_str)
	if not id_num then
		logger.warn("Pocketbook Sync: Failed to convert collection ID to number, value: " .. tostring(id_str))
		return nil
	end
    logger.info("Pocketbook Sync: Collection ID found: " .. tostring(id_num))
    
    return id_num
end

-- Get list of all collections from KOReader
function PocketbookSync:getKOReaderCollections()
    local success, ReadCollection = pcall(require, "readcollection")
    if not success then
        logger.warn("Pocketbook Sync: ReadCollection module not available")
        return {}
    end
    
    local collections = {}
    for coll_name in pairs(ReadCollection.coll) do
        table.insert(collections, coll_name)
    end
    
    -- Sort collections alphabetically
    table.sort(collections)
    
    return collections
end

-- New function to remove book from KOReader collection
function PocketbookSync:removeFromKOReaderCollection(book_path, collection_name)
    if not collection_name or collection_name == "" then
        return
    end
    
    local success, ReadCollection = pcall(require, "readcollection")
    if not success then
        logger.warn("Pocketbook Sync: ReadCollection module not available")
        return
    end
    
    -- Check if the collection exists in KOReader
    local collections = ReadCollection.coll
    if not collections or not collections[collection_name] then
        logger.info("Pocketbook Sync: KOReader collection '" .. collection_name .. "' not found")
        return
    end
    
    -- Remove book from collection
    if collections[collection_name][book_path] then
        collections[collection_name][book_path] = nil
        logger.info("Pocketbook Sync: Removed book from KOReader collection '" .. collection_name .. "'")
        
        -- Save the updated collections
        ReadCollection:saveCollections()
    else
        logger.info("Pocketbook Sync: Book not found in KOReader collection '" .. collection_name .. "'")
    end
end

-- New function to check if book exists in KOReader collection
function PocketbookSync:isBookInKOReaderCollection(book_path, collection_name)
    if not collection_name or collection_name == "" then
        return false
    end
    
    local success, ReadCollection = pcall(require, "readcollection")
    if not success then
        return false
    end
    
    local collections = ReadCollection.coll
    if not collections or not collections[collection_name] then
        return false
    end
    
    return collections[collection_name][book_path] ~= nil
end

function PocketbookSync:saveCollectionSettings(name)
    local collection_id = self:getCollectionIdByName(name)
    
    G_reader_settings:saveSetting("to_read_collection_name", name)
    G_reader_settings:flush()
    
    if collection_id then
        G_reader_settings:saveSetting("to_read_collection_id", collection_id)
        logger.info("Pocketbook Sync: Collection ID saved: " .. collection_id)
        G_reader_settings:flush()
    else
        logger.info("Pocketbook Sync: Collection ID not found for '" .. name .. "'")
        G_reader_settings:delSetting("to_read_collection_id")
        G_reader_settings:flush()
    end
    
    local saved_id = G_reader_settings:readSetting("to_read_collection_id")
    logger.info("Pocketbook Sync: Verification - saved ID: " .. tostring(saved_id))
end

function PocketbookSync:deleteCollectionSettings()
    local success = pcall(function()
        G_reader_settings:delSetting("to_read_collection_name")
        G_reader_settings:delSetting("to_read_collection_id")
    end)
    
    if not success then
        logger.error("Pocketbook Sync: Failed to delete collection settings")
    end
    
    G_reader_settings:flush()
end

pocketbookDbConn:set_busy_timeout(1000)

local function GetCurrentProfileId()
    local profile_name = inkview.GetCurrentProfile()
    if profile_name == nil then
        return 1
    else
        local stmt = pocketbookDbConn:prepare("SELECT id FROM profiles WHERE name = ?")
        local profile_id = stmt:reset():bind(ffi.string(profile_name)):step()
        stmt:close()
        return profile_id[1]
    end
end

local profile_id = GetCurrentProfileId()

function PocketbookSync:clearCache()
    bookIds = {}
end

function PocketbookSync:sync()
    self:doSync(self:prepareSync())
end

function PocketbookSync:prepareSync()
    -- onFlushSettings called during koreader exit and after onCloseDocument
    -- would raise an error in some of the self.document methods and we can
    -- avoid that by checking if self.ui.document is nil
    if not self.ui.document then
        return nil
    end

    local folder, file = self:getFolderFile()
    if not folder or folder == "" or not file or file == "" then
        logger.info("Pocketbook Sync: No folder/file found for " .. self.view.document.file)
        return nil
    end

    local globalPage = self.view.state.page
    local flow = self.document:getPageFlow(globalPage)

    -- skip sync if not in the main flow
    if flow ~= 0 then
        return nil
    end

    local totalPages = self.document:getTotalPagesInFlow(flow)
    local page = self.document:getPageNumberInFlow(globalPage)

    local summary = self.ui.doc_settings:readSetting("summary")
    local status = summary and summary.status
    local completed = (status == "complete" or page == totalPages) and 1 or 0

    -- Calculate reading progress percentage
    local progress_percent = 0
    if totalPages > 0 then
        progress_percent = math.ceil((page / totalPages) * 100)
    end

    -- Save progress to doc_settings
    self.ui.doc_settings:saveSetting("pocketbook_sync_progress", {
        percent = progress_percent,
        current_page = page,
        total_pages = totalPages,
        last_sync = os.time()
    })
    self.ui.doc_settings:flush()

    -- hide the progress bar if we're on the title/cover page
    --
    -- we'll never set cpage=1 so the progress bar will seem to jump a bit at
    -- the start of a book, but there's no nice way to fix that: to use the
    -- full range, we'd need to map pages 2 to last-1 to cpages 1 to last-1,
    -- and that always skips one position; skipping the first one is the least
    -- surprising behaviour
    if page == 1 then
        page = 0
    end

    return {
        folder = folder,
        file = file,
        totalPages = totalPages,
        page = page,
        completed = completed,
        time = os.time(),
        book_path = self.view.document.file, -- Add book path for KOReader collections
    }
end

function PocketbookSync:doSync(data)
    if not data then
        return
    end

    local cacheKey = data.folder .. data.file

    if not bookIds[cacheKey] then
        local sql = [[
            SELECT book_id
            FROM files
            WHERE
                folder_id = (SELECT id FROM folders WHERE name = ? LIMIT 1)
            AND filename = ?
            LIMIT 1
        ]]
        local stmt = pocketbookDbConn:prepare(sql)
        local row = stmt:reset():bind(data.folder, data.file):step()
        stmt:close()

        if row == nil then
            logger.info("Pocketbook Sync: Book id for " .. data.folder .. "/" .. data.file .. " not found")
            return
        end
        bookIds[cacheKey] = row[1]
    end

    local book_id = bookIds[cacheKey]
    local sql = [[
            REPLACE INTO books_settings
            (bookid, profileid, cpage, npage, completed, opentime)
            VALUES (?, ?, ?, ?, ?, ?)
        ]]
		
	pocketbookDbConn:exec("BEGIN TRANSACTION")
	
    local stmt = pocketbookDbConn:prepare(sql)
    stmt:reset():bind(book_id, profile_id, data.page, data.totalPages, data.completed, data.time):step()
    stmt:close()
	
	pocketbookDbConn:exec("COMMIT")
	pocketbookDbConn:exec("PRAGMA wal_checkpoint")
	pocketbookDbConn:exec("VACUUM")

    -- If book is marked as completed, remove it from collections
    if data.completed == 1 then
        local collection_name = G_reader_settings:readSetting("to_read_collection_name")
        local collection_id = G_reader_settings:readSetting("to_read_collection_id")
        
        -- Remove from PocketBook collection
        if collection_id and collection_id ~= "" then
            collection_id = tonumber(collection_id)
			logger.info("Collection ID", collection_id)
            local check_sql = "SELECT bookshelfid FROM bookshelfs_books WHERE bookid = ? AND bookshelfid = ? LIMIT 1"
            local check_stmt = pocketbookDbConn:prepare(check_sql)
            local check_row = check_stmt:reset():bind(book_id, collection_id):step()
            check_stmt:close()
            if check_row then
                local del_sql = "DELETE FROM bookshelfs_books WHERE bookid = ? AND bookshelfid = ?"
				
				pocketbookDbConn:exec("BEGIN TRANSACTION")
				
                local del_stmt = pocketbookDbConn:prepare(del_sql)
                del_stmt:reset():bind(book_id, collection_id):step()
                del_stmt:close()
				
				pocketbookDbConn:exec("COMMIT")
				pocketbookDbConn:exec("PRAGMA wal_checkpoint(FULL)")
				pocketbookDbConn:exec("VACUUM")
            end
        end
        
        -- Remove from KOReader collection
        if collection_name and collection_name ~= "" and data.book_path then
            self:removeFromKOReaderCollection(data.book_path, collection_name)
        end
    end
end

function PocketbookSync:getFolderFile()
    local path = self.view.document.file
    local folder, file = util.splitFilePathName(path)
    local folderTrimmed = folder:match("(.*)/")
    if folderTrimmed ~= nil then
        folder = folderTrimmed
    end
    return folder, file
end

function PocketbookSync:getEventSource()
    local trace = debug.traceback("", 3)
    -- Extract the most relevant caller info
    if trace:find("beforeSuspend") then
        return "beforeSuspend"
    elseif trace:find("handleMiscEv") then
        return "handleMiscEv"
    elseif trace:find("onCloseDocument") then
        return "onCloseDocument"
    else
        return "unknown"
    end
end

function PocketbookSync:onFlushSettings()
    local source = self:getEventSource()
    
    if source == "beforeSuspend" then
        -- logger.info("PocketbookSync: onFlushSettings ignored (source: " .. source .. ")")
        return
    end
    
    logger.info("PocketbookSync: onFlushSettings event triggered (source: " .. source .. ")")
    
    -- Create page snapshot before sync
    inkview.PageSnapshot()
    
    self:sync()
end

return PocketbookSync
