from calibre.gui2.actions import InterfaceAction
from calibre.gui2 import error_dialog, info_dialog
from PyQt5.Qt import QMenu, QToolButton, Qt, QTimer

from calibre_plugins.n_pages_previews.main import PreviewDialog
from calibre_plugins.n_pages_previews.config import prefs

try:
    load_translations()
except NameError:
    pass  # load_translations() added in calibre 1.9


class NPagesPreviewsAction(InterfaceAction):
    name = 'N-Pages Previews'
    action_spec = ('N-Pages Previews', 'images/icon.png', 'Generate multi-page previews', None)
    action_type = 'current'

    def genesis(self):
        # This method is called once per plugin when Calibre initializes
        # print("DEBUG: N-Pages Previews genesis() called")

        # Load and set custom icon
        icon = get_icon()
        if icon and not icon.isNull():
            self.qaction.setIcon(icon)

        # Set the default action to toggle the preview panel
        self.qaction.setCheckable(True)
        self.qaction.setToolTip(_('Toggle preview panel; use the arrow for more options'))
        self.qaction.triggered.connect(self.toggle_preview_panel)

        # Create popup menu and build it once
        self.popup_menu = QMenu(self.gui)
        self.qaction.setMenu(self.popup_menu)
        self.build_menu()  # Build menu once during initialization

        # Ensure toolbar button shows dropdown arrow but still allows default click
        QTimer.singleShot(0, self._configure_toolbar_button)
        self.popup_menu.aboutToShow.connect(self.update_generate_action_label)

        # Create preview panel
        self.preview_panel = None

        # Connect to selection model after Calibre finishes building the main UI
        QTimer.singleShot(0, self._connect_selection_model)
        self.update_generate_action_label()

        # print("DEBUG: Menu created and connected")

    def toolbar_button_clicked(self):
        """Legacy handler retained for compatibility; now toggles preview panel."""
        self.toggle_preview_panel()

    def build_menu(self):
        """Build the popup menu"""
        # print("DEBUG: build_menu() called")

        # Toggle preview panel action
        self.create_menu_action(
            self.popup_menu,
            'toggle-panel',
            _('Toggle Preview Panel'),
            triggered=self.toggle_preview_panel
        )

        self.popup_menu.addSeparator()

        # Show preview action
        # print("DEBUG: Creating 'Show Preview' action")
        self.create_menu_action(
            self.popup_menu,
            'show-preview',
            _('Show Preview Dialog'),
            triggered=self.show_preview_dialog
        )

        # Generate preview action
        self.generate_action = self.create_menu_action(
            self.popup_menu,
            'generate-preview',
            _('Generate Preview for Selected Books'),
            triggered=self.generate_previews_for_selection
        )
        if self.generate_action is None and self.popup_menu.actions():
            self.generate_action = self.popup_menu.actions()[-1]

        self.popup_menu.addSeparator()

        # Configure action
        self.create_menu_action(
            self.popup_menu,
            'configure',
            _('Configure'),
            triggered=self.show_configuration
        )

    def show_preview_dialog(self):
        try:
            # print("DEBUG: show_preview_dialog called")

            # Check dependencies first
            if not check_dependencies(self.gui):
                # print("DEBUG: Dependencies check failed")
                return

            # print("DEBUG: Dependencies OK")

            rows = self.gui.library_view.selectionModel().selectedRows()
            if not rows:
                error_dialog(self.gui, _('No selection'),
                            _('Please select at least one book'), show=True)
                return

            if len(rows) > 1:
                error_dialog(self.gui, _('Multiple selection'),
                            _('Please select only one book to preview'), show=True)
                return

            book_id = self.gui.library_view.model().id(rows[0])
            db = self.gui.current_db.new_api

            # print(f"DEBUG: About to show preview dialog for book_id={book_id}")

            # Show preview dialog
            d = PreviewDialog(self.gui, db, book_id, prefs)
            d.exec_()

            # print("DEBUG: Preview dialog closed")

        except Exception as e:
            import traceback
            error_dialog(
                self.gui,
                'Error',
                f'An error occurred: {str(e)}\n\n{traceback.format_exc()}',
                show=True
            )

    def generate_previews_for_selection(self):
        try:
            # print("DEBUG: generate_previews_for_selection called")

            # Check dependencies first
            if not check_dependencies(self.gui):
                return

            from calibre_plugins.n_pages_previews.generator import GeneratePreviewsDialog

            rows = self.gui.library_view.selectionModel().selectedRows()
            if not rows:
                error_dialog(self.gui, _('No selection'),
                            _('Please select at least one book'), show=True)
                return

            book_ids = [self.gui.library_view.model().id(row) for row in rows]
            db = self.gui.current_db.new_api

            # print(f"DEBUG: About to show generation dialog for {len(book_ids)} books")

            # Show generation dialog
            d = GeneratePreviewsDialog(self.gui, db, book_ids, prefs, auto_start=True)
            d.exec_()

        except Exception as e:
            import traceback
            error_dialog(
                self.gui,
                'Error',
                f'An error occurred: {str(e)}\n\n{traceback.format_exc()}',
                show=True
            )

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)

    def toggle_preview_panel(self):
        """Toggle the preview panel visibility"""
        from calibre_plugins.n_pages_previews.preview_panel import PreviewPanel

        if self.preview_panel is None:
            # Create panel
            # print("DEBUG: Creating preview panel")
            self.preview_panel = PreviewPanel(self.gui)
            self.gui.addDockWidget(Qt.RightDockWidgetArea, self.preview_panel)

        if self.preview_panel.isVisible():
            # print("DEBUG: Hiding preview panel")
            self.preview_panel.hide()
            self.qaction.setChecked(False)
        else:
            # print("DEBUG: Showing preview panel")
            self.preview_panel.show()
            # Trigger initial load
            self.preview_panel.on_book_selected()
            self.qaction.setChecked(True)

    def _configure_toolbar_button(self):
        """Configure toolbar button popup mode so arrow opens the menu."""
        try:
            for button in self.gui.findChildren(QToolButton):
                if button.defaultAction() is self.qaction:
                    button.setPopupMode(QToolButton.MenuButtonPopup)
                    self.toolbar_button = button
                    return
        except Exception:
            pass

    def update_generate_action_label(self, *args):
        """Update menu label to reflect current selection count."""
        action = getattr(self, 'generate_action', None)
        if not action:
            return

        try:
            selection_model = self.gui.library_view.selectionModel()
        except Exception:
            selection_model = None

        if not selection_model:
            action.setText(_('Generate Preview'))
            action.setEnabled(False)
            return

        rows = selection_model.selectedRows()
        count = len(rows)

        if count == 0:
            action.setText(_('Generate Preview'))
            action.setEnabled(False)
        elif count == 1:
            action.setText(_('Generate Preview'))
            action.setEnabled(True)
        else:
            action.setText(f_('Generate ({count})'))
            action.setEnabled(True)

    def _connect_selection_model(self):
        """Defer selection model hookup until the library view is available."""
        library_view = getattr(self.gui, 'library_view', None)
        if library_view is None:
            # Try again shortly; Calibre wires library_view after actions initialize
            QTimer.singleShot(200, self._connect_selection_model)
            return

        try:
            selection_model = library_view.selectionModel()
            if selection_model:
                selection_model.selectionChanged.connect(self.update_generate_action_label)
        except Exception:
            pass


# Check if required dependencies are installed
def check_dependencies(parent):
    """Check if PDF rendering is available - always true since we use Calibre's built-in tools"""
    # No external dependencies needed - we use Calibre's built-in tools
    return True


# Get icon function
def get_icon():
    """Get icon for the plugin"""
    from PyQt5.Qt import QIcon, QPixmap
    import os

    # Method 1: Try to load icon from plugin resources
    try:
        from calibre_plugins.n_pages_previews import get_resources
        icon_data = get_resources('images/icon.png')
        if icon_data:
            pixmap = QPixmap()
            if pixmap.loadFromData(icon_data) and not pixmap.isNull():
                return QIcon(pixmap)
    except Exception:
        pass

    # Method 2: Try to load from filesystem (for development/testing)
    try:
        plugin_dir = os.path.dirname(os.path.abspath(__file__))
        icon_path = os.path.join(plugin_dir, 'images', 'icon.png')
        if os.path.exists(icon_path):
            return QIcon(icon_path)
    except Exception:
        pass

    # No icon available
    return None
