import os
import hashlib
import subprocess
import tempfile
from PyQt5.Qt import QImage, QPixmap

try:
    load_translations()
except NameError:
    pass  # load_translations() added in calibre 1.9

class PreviewGenerator:
    """
    Generates preview images from PDF pages with caching using Calibre's built-in tools
    """

    def generate_previews(self, pdf_path, num_pages, cache_dir,
                         image_width=800, quality=85, progress_callback=None):
        """
        Generate preview images from first N pages of PDF using Calibre's ebook-convert

        Args:
            pdf_path: Path to PDF file
            num_pages: Number of pages to preview
            cache_dir: Directory to cache images
            image_width: Target width for images
            quality: JPEG quality (50-100)
            progress_callback: Optional callback(current, total)

        Returns:
            List of paths to generated preview images
        """
        # Calculate cache key based on settings
        cache_key = self._get_cache_key(pdf_path, num_pages, image_width, quality)

        # Use Calibre's ebook-convert to extract pages as images
        previews = self._generate_with_calibre(
            pdf_path, num_pages, cache_dir, cache_key,
            image_width, quality, progress_callback
        )

        return previews

    def _get_cache_key(self, pdf_path, num_pages, image_width, quality):
        """Generate cache key from PDF path and settings"""
        # Include file modification time in key
        mtime = os.path.getmtime(pdf_path)
        key_string = f'{pdf_path}_{mtime}_{num_pages}_{image_width}_{quality}'
        return hashlib.md5(key_string.encode()).hexdigest()[:12]

    def _get_cached_path(self, cache_dir, cache_key, page_num):
        """Get path for cached preview image"""
        return os.path.join(cache_dir, f'{cache_key}_page_{page_num}.jpg')

    def _generate_with_calibre(self, pdf_path, num_pages, cache_dir, cache_key,
                               image_width, quality, progress_callback):
        """
        Generate previews by converting PDF to images using Calibre's ebook-convert tool
        """
        previews = []

        # Find ebook-convert binary
        calibre_bin = self._find_calibre_binary()
        if not calibre_bin:
            raise RuntimeError(_("Could not find Calibre's ebook-convert binary"))

        # print(f"DEBUG: Using ebook-convert at: {calibre_bin}")

        # Convert each page individually
        for page_num in range(1, num_pages + 1):
            if progress_callback:
                progress_callback(page_num, num_pages)

            # Check cache
            cache_path = self._get_cached_path(cache_dir, cache_key, page_num)
            if os.path.exists(cache_path):
                previews.append(cache_path)
                continue

            try:
                # Create temp HTML file for single page
                with tempfile.TemporaryDirectory() as tmpdir:
                    # Convert PDF page to HTML first, then to image
                    # This is a workaround since ebook-convert doesn't directly extract single pages

                    # For now, use a simpler approach: convert entire PDF to images
                    # and extract the ones we need
                    if page_num == 1:  # Only convert once
                        self._convert_pdf_to_images(pdf_path, cache_dir, cache_key,
                                                   num_pages, image_width, quality)

                    if os.path.exists(cache_path):
                        previews.append(cache_path)

            except Exception as e:
                print(f"DEBUG: Error converting page {page_num}: {e}")
                # Create a placeholder if conversion fails
                self._create_placeholder(cache_path, image_width, page_num)
                previews.append(cache_path)

        return previews

    def _find_calibre_binary(self):
        """Find Calibre's ebook-convert binary"""
        # Check common locations
        possible_paths = [
            '/Applications/calibre.app/Contents/MacOS/ebook-convert',
            '/usr/bin/ebook-convert',
            'ebook-convert'  # Try PATH
        ]

        for path in possible_paths:
            if os.path.exists(path):
                return path

        # Try which command
        try:
            result = subprocess.run(['which', 'ebook-convert'],
                                  capture_output=True, text=True)
            if result.returncode == 0:
                return result.stdout.strip()
        except:
            pass

        return None

    def _find_ghostscript(self):
        """Find Ghostscript (gs) binary"""
        possible_paths = [
            '/opt/homebrew/bin/gs',  # Homebrew on Apple Silicon
            '/usr/local/bin/gs',      # Homebrew on Intel
            '/usr/bin/gs',            # System
            'gs'                      # Try PATH
        ]

        for path in possible_paths:
            if os.path.exists(path):
                return path

        # Try which command
        try:
            result = subprocess.run(['which', 'gs'],
                                  capture_output=True, text=True)
            if result.returncode == 0:
                return result.stdout.strip()
        except:
            pass

        return None

    def _convert_pdf_to_images(self, pdf_path, cache_dir, cache_key, num_pages,
                              image_width, quality):
        """
        Convert PDF pages to individual images using Ghostscript
        """
        # Try to find gs (Ghostscript)
        gs_path = self._find_ghostscript()

        if not gs_path:
            # print("DEBUG: Ghostscript not found, using placeholders")
            # Fallback to placeholders
            for page_num in range(1, num_pages + 1):
                cache_path = self._get_cached_path(cache_dir, cache_key, page_num)
                if not os.path.exists(cache_path):
                    self._create_placeholder(cache_path, image_width, page_num)
            return

        # print(f"DEBUG: Using Ghostscript at: {gs_path}")

        # Calculate DPI to achieve target width (assuming 8.5" wide page)
        dpi = int(image_width / 8.5)

        try:
            for page_num in range(1, num_pages + 1):
                cache_path = self._get_cached_path(cache_dir, cache_key, page_num)

                if os.path.exists(cache_path):
                    continue

                # Use Ghostscript to render this page
                cmd = [
                    gs_path,
                    '-dNOPAUSE',
                    '-dBATCH',
                    '-dSAFER',
                    '-sDEVICE=jpeg',
                    f'-dJPEGQ={quality}',
                    f'-r{dpi}',
                    f'-dFirstPage={page_num}',
                    f'-dLastPage={page_num}',
                    f'-sOutputFile={cache_path}',
                    pdf_path
                ]

                result = subprocess.run(cmd, capture_output=True, text=True, timeout=30)

                if result.returncode != 0:
                    # print(f"DEBUG: gs failed for page {page_num}: {result.stderr[:200]}")
                    self._create_placeholder(cache_path, image_width, page_num)
                # else:
                    # print(f"DEBUG: Generated page {page_num}")

        except Exception as e:
            # print(f"DEBUG: Error in _convert_pdf_to_images: {e}")
            # Create placeholders for any remaining pages
            for page_num in range(1, num_pages + 1):
                cache_path = self._get_cached_path(cache_dir, cache_key, page_num)
                if not os.path.exists(cache_path):
                    self._create_placeholder(cache_path, image_width, page_num)

    def _create_placeholder(self, path, width, page_num):
        """Create a placeholder image"""
        from PyQt5.Qt import QPainter, QFont, Qt

        height = int(width * 1.4)  # Approximate page ratio
        img = QImage(width, height, QImage.Format_RGB32)
        img.fill(Qt.white)

        painter = QPainter(img)
        painter.setPen(Qt.gray)
        font = QFont()
        font.setPixelSize(48)
        painter.setFont(font)
        painter.drawText(img.rect(), Qt.AlignCenter, f"Pag. {page_num}")
        painter.end()

        img.save(path, 'JPEG', 85)
