import os
from PyQt5.Qt import (QDialog, QVBoxLayout, QLabel, QScrollArea,
                      QWidget, QPushButton, QProgressBar, QHBoxLayout,
                      QPixmap, Qt)
from PyQt5.QtCore import QThread, pyqtSignal

from calibre.gui2 import error_dialog, info_dialog
from calibre_plugins.n_pages_previews.preview_generator import PreviewGenerator

try:
    load_translations()
except NameError:
    pass  # load_translations() added in calibre 1.9


class PreviewGeneratorThread(QThread):
    """Background thread for generating previews"""
    progress_update = pyqtSignal(int, int)  # current, total
    preview_ready = pyqtSignal(int, object)  # page_num, pixmap
    finished_signal = pyqtSignal(bool, str)  # success, message

    def __init__(self, db, book_id, num_pages, image_width, quality):
        QThread.__init__(self)
        self.db = db
        self.book_id = book_id
        self.num_pages = num_pages
        self.image_width = image_width
        self.quality = quality
        self.generator = PreviewGenerator()

    def run(self):
        try:
            # Get book format path
            fmt = self.db.has_format(self.book_id, 'PDF')
            if not fmt:
                self.finished_signal.emit(False, _('Book does not have PDF format'))
                return

            pdf_path = self.db.format_abspath(self.book_id, 'PDF')
            if not pdf_path or not os.path.exists(pdf_path):
                self.finished_signal.emit(False, _('PDF file not found'))
                return

            # Get cache directory (same as book directory)
            book_dir = os.path.dirname(pdf_path)
            cache_dir = os.path.join(book_dir, '.previews')
            os.makedirs(cache_dir, exist_ok=True)

            # Generate previews
            previews = self.generator.generate_previews(
                pdf_path,
                self.num_pages,
                cache_dir,
                self.image_width,
                self.quality,
                progress_callback=lambda cur, total: self.progress_update.emit(cur, total)
            )

            # Emit pixmaps for each preview
            for page_num, image_path in enumerate(previews, 1):
                pixmap = QPixmap(image_path)
                self.preview_ready.emit(page_num, pixmap)

            self.finished_signal.emit(
               True,
               _("Generated {count} preview(s)").format(count=len(previews)))


        except Exception as e:
            self.finished_signal.emit(False, str(e))


class PreviewDialog(QDialog):
    def __init__(self, parent, db, book_id, prefs, scroll_to_page=None):
        QDialog.__init__(self, parent)
        self.db = db
        self.book_id = book_id
        self.prefs = prefs
        self.scroll_to_page = scroll_to_page

        # Get book title
        metadata = db.get_metadata(book_id)
        self.setWindowTitle(
            _("Preview: {title}").format(title=metadata.title))

        self.resize(900, 700)

        # Layout
        layout = QVBoxLayout()
        self.setLayout(layout)

        # Status label
        self.status_label = QLabel(_('Generating previews...'))
        layout.addWidget(self.status_label)

        # Progress bar
        self.progress_bar = QProgressBar()
        layout.addWidget(self.progress_bar)

        # Scroll area for previews
        self.scroll = QScrollArea()
        self.scroll.setWidgetResizable(True)
        self.scroll.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)

        self.preview_widget = QWidget()
        self.preview_layout = QVBoxLayout()
        self.preview_widget.setLayout(self.preview_layout)
        self.scroll.setWidget(self.preview_widget)

        layout.addWidget(self.scroll)

        # Close button
        button_layout = QHBoxLayout()
        close_btn = QPushButton(_('Close'))
        close_btn.clicked.connect(self.accept)
        button_layout.addStretch()
        button_layout.addWidget(close_btn)
        layout.addLayout(button_layout)

        # Start generating previews
        self.start_generation()

    def start_generation(self):
        self.thread = PreviewGeneratorThread(
            self.db,
            self.book_id,
            self.prefs['num_pages'],
            self.prefs['image_width'],
            self.prefs['image_quality']
        )
        self.thread.progress_update.connect(self.update_progress)
        self.thread.preview_ready.connect(self.add_preview)
        self.thread.finished_signal.connect(self.generation_finished)
        self.thread.start()

    def update_progress(self, current, total):
        self.progress_bar.setMaximum(total)
        self.progress_bar.setValue(current)
        self.status_label.setText(
            _("Generating preview {current} of {total}...").format(current=current, total=total))


    def add_preview(self, page_num, pixmap):
        # Add page number label
        page_label = QLabel(
            _("<b>Page {page_num}</b>").format(page_num=page_num))

        page_label.setAlignment(Qt.AlignCenter)

        # Store reference to page label for scrolling
        if page_num == self.scroll_to_page:
            self.target_widget = page_label

        self.preview_layout.addWidget(page_label)

        label = QLabel()
        label.setPixmap(pixmap.scaledToWidth(
            self.prefs['image_width'],
            Qt.SmoothTransformation
        ))
        label.setAlignment(Qt.AlignCenter)

        self.preview_layout.addWidget(label)

    def generation_finished(self, success, message):
        if success:
            self.status_label.setText(f'✓ {message}')
            self.progress_bar.hide()

            # Scroll to target page if specified
            if hasattr(self, 'target_widget') and self.target_widget:
                # Use a timer to scroll after layout is complete
                from PyQt5.QtCore import QTimer
                QTimer.singleShot(100, lambda: self.scroll.ensureWidgetVisible(self.target_widget))
        else:
            self.status_label.setText(f'✗ Error: {message}')
            error_dialog(self, _('Preview Generation Failed'), message, show=True)
