import os
from PyQt5.Qt import (QDialog, QVBoxLayout, QLabel, QProgressBar,
                      QPushButton, QHBoxLayout, QTextEdit)
from PyQt5.QtCore import QThread, pyqtSignal

from calibre.gui2 import error_dialog, info_dialog
from calibre_plugins.n_pages_previews.preview_generator import PreviewGenerator

try:
    load_translations()
except NameError:
    pass  # load_translations() added in calibre 1.9

class BatchGeneratorThread(QThread):
    """Background thread for batch generating previews"""
    progress_update = pyqtSignal(int, int, str)  # current_book, total_books, book_title
    log_message = pyqtSignal(str)
    finished_signal = pyqtSignal(int, int)  # successful, failed

    def __init__(self, db, book_ids, num_pages, image_width, quality):
        QThread.__init__(self)
        self.db = db
        self.book_ids = book_ids
        self.num_pages = num_pages
        self.image_width = image_width
        self.quality = quality
        self.generator = PreviewGenerator()

    def run(self):
        successful = 0
        failed = 0

        for idx, book_id in enumerate(self.book_ids, 1):
            try:
                # Get book metadata
                metadata = self.db.get_metadata(book_id)
                title = metadata.title

                self.progress_update.emit(idx, len(self.book_ids), title)
                self.log_message.emit(
                    _("Processing: {title}").format(title=title))


                # Check for PDF format
                fmt = self.db.has_format(book_id, 'PDF')
                if not fmt:
                    self.log_message.emit(_("  ✗ Skipped: No PDF format"))
                    failed += 1
                    continue

                pdf_path = self.db.format_abspath(book_id, 'PDF')
                if not pdf_path or not os.path.exists(pdf_path):
                    self.log_message.emit(_("  ✗ Error: PDF file not found"))
                    failed += 1
                    continue

                # Setup cache directory
                book_dir = os.path.dirname(pdf_path)
                cache_dir = os.path.join(book_dir, '.previews')
                os.makedirs(cache_dir, exist_ok=True)

                # Generate previews
                previews = self.generator.generate_previews(
                    pdf_path,
                    self.num_pages,
                    cache_dir,
                    self.image_width,
                    self.quality
                )

                self.log_message.emit(
                   _("  ✓ Generated {count} preview(s)").format(count=len(previews)))

                successful += 1

            except Exception as e:
                self.log_message.emit(f'  ✗ Error: {str(e)}')
                failed += 1

        self.finished_signal.emit(successful, failed)


class GeneratePreviewsDialog(QDialog):
    """Dialog for batch generating previews"""

    def __init__(self, parent, db, book_ids, prefs, auto_start=False):
        QDialog.__init__(self, parent)
        self.db = db
        self.book_ids = book_ids
        self.prefs = prefs
        self.auto_start = auto_start

        self.setWindowTitle(
            _("Generate Previews for {count} Book(s)").format(count=len(book_ids)))

        self.resize(600, 400)

        # Layout
        layout = QVBoxLayout()
        self.setLayout(layout)

        # Status label
        self.status_label = QLabel(
             _("Preparing to generate previews for {count} book(s)...").format(count=len(book_ids)))
        layout.addWidget(self.status_label)

        # Progress bar
        self.progress_bar = QProgressBar()
        self.progress_bar.setMaximum(len(book_ids))
        layout.addWidget(self.progress_bar)

        # Log output
        log_label = QLabel('Log:')
        layout.addWidget(log_label)

        self.log_text = QTextEdit()
        self.log_text.setReadOnly(True)
        layout.addWidget(self.log_text)

        # Buttons
        button_layout = QHBoxLayout()

        self.start_btn = QPushButton(_('Start'))
        self.start_btn.clicked.connect(self.start_generation)
        button_layout.addWidget(self.start_btn)

        button_layout.addStretch()

        self.cancel_btn = QPushButton(_('Cancel'))
        self.cancel_btn.clicked.connect(self.reject)
        button_layout.addWidget(self.cancel_btn)

        self.ok_btn = QPushButton('OK')
        self.ok_btn.clicked.connect(self.accept)
        self.ok_btn.setEnabled(False)
        self.ok_btn.setVisible(False)
        button_layout.addWidget(self.ok_btn)

        layout.addLayout(button_layout)

        # Auto-start if requested
        if self.auto_start:
            from PyQt5.QtCore import QTimer
            QTimer.singleShot(100, self.start_generation)

    def start_generation(self):
        self.start_btn.setEnabled(False)
        self.start_btn.setVisible(False)
        self.cancel_btn.setText('Cancel')

        self.thread = BatchGeneratorThread(
            self.db,
            self.book_ids,
            self.prefs['num_pages'],
            self.prefs['image_width'],
            self.prefs['image_quality']
        )
        self.thread.progress_update.connect(self.update_progress)
        self.thread.log_message.connect(self.add_log_message)
        self.thread.finished_signal.connect(self.generation_finished)
        self.thread.start()

    def update_progress(self, current, total, title):
        self.progress_bar.setValue(current)
        self.status_label.setText(_("Processing {current} of {total}: {title}").format(
        current=current, total=total, title=title)
    )

    def add_log_message(self, message):
        self.log_text.append(message)

    def generation_finished(self, successful, failed):
        self.status_label.setText(
                _("✓ Complete! Successful: {successful}, Failed: {failed}").format(
        successful=successful, failed=failed)
        )

        # Show OK button, hide Cancel
        self.cancel_btn.setVisible(False)
        self.ok_btn.setVisible(True)
        self.ok_btn.setEnabled(True)
        self.ok_btn.setDefault(True)

        if failed == 0 and not self.auto_start:
            info_dialog(
                self,
                _('Success'),
                _("Successfully generated previews for {successful} book(s)").format(successful=successful),
                show=True
            )
