import os
import hashlib
from PyQt5.Qt import QImage
from PyQt5.QtCore import Qt

from calibre.ebooks.metadata.pdf import page_images


IMAGE_FORMAT = 'jpeg'
IMAGE_EXTENSION = 'jpg'
PAGE_NUMBER_PADDING = 3


class PreviewGenerator:
    """
    Generates preview images from PDF pages with caching using Calibre's built-in tools
    """

    def generate_previews(self, pdf_path, num_pages, cache_dir,
                         image_width=800, quality=85, progress_callback=None):
        """
        Generate preview images from first N pages of PDF using Calibre's built-in renderer

        Args:
            pdf_path: Path to PDF file
            num_pages: Number of pages to preview
            cache_dir: Directory to cache images
            image_width: Target width for images
            quality: JPEG quality (50-100)
            progress_callback: Optional callback(current, total)

        Returns:
            List of paths to generated preview images
        """
        # Calculate cache key based on settings
        cache_key = self._get_cache_key(pdf_path, num_pages, image_width, quality)

        previews = self._generate_with_calibre(
            pdf_path,
            num_pages,
            cache_dir,
            cache_key,
            image_width,
            quality,
            progress_callback
        )

        return previews

    def _get_cache_key(self, pdf_path, num_pages, image_width, quality):
        """Generate cache key from PDF path and settings"""
        # Include file modification time in key
        mtime = os.path.getmtime(pdf_path)
        key_string = f'{pdf_path}_{mtime}_{num_pages}_{image_width}_{quality}'
        return hashlib.md5(key_string.encode()).hexdigest()[:12]

    def _get_cached_path(self, cache_dir, cache_key, page_num):
        """Get path for cached preview image"""
        legacy_path = os.path.join(
            cache_dir,
            f'{cache_key}_page_{page_num}.{IMAGE_EXTENSION}'
        )
        padded_path = os.path.join(
            cache_dir,
            f'{cache_key}-{str(page_num).zfill(PAGE_NUMBER_PADDING)}.{IMAGE_EXTENSION}'
        )
        hyphen_path = os.path.join(
            cache_dir,
            f'{cache_key}-{page_num}.{IMAGE_EXTENSION}'
        )

        if os.path.exists(legacy_path):
            return legacy_path
        if os.path.exists(padded_path):
            return padded_path
        if os.path.exists(hyphen_path):
            return hyphen_path

        return legacy_path

    def _generate_with_calibre(self, pdf_path, num_pages, cache_dir, cache_key,
                               image_width, quality, progress_callback):
        """
        Generate previews by rendering PDF pages using Calibre's built-in page_images helper
        """
        cached_paths = [
            self._get_cached_path(cache_dir, cache_key, page_num)
            for page_num in range(1, num_pages + 1)
        ]
        needs_render = any(not os.path.exists(path) for path in cached_paths)
        render_failed = False

        if needs_render:
            try:
                page_images(
                    pdf_path,
                    cache_dir,
                    1,
                    num_pages,
                    IMAGE_FORMAT,
                    cache_key
                )
            except Exception:
                render_failed = True

            self._finalize_generated_images(
                cache_dir,
                cache_key,
                cached_paths,
                image_width,
                quality
            )

            if render_failed:
                for page_index, cache_path in enumerate(cached_paths, start=1):
                    if not os.path.exists(cache_path):
                        self._create_placeholder(cache_path, image_width, page_index)

        previews = []
        for page_index, cache_path in enumerate(cached_paths, start=1):
            if progress_callback:
                progress_callback(page_index, num_pages)
            if not os.path.exists(cache_path):
                self._create_placeholder(cache_path, image_width, page_index)
            previews.append(cache_path)

        return previews

    def _finalize_generated_images(self, cache_dir, cache_key, cached_paths,
                                   image_width, quality):
        """Move rendered images to final locations and apply final quality settings"""
        for page_index, cache_path in enumerate(cached_paths, start=1):
            # Find the rendered image from pdftoppm
            raw_path = None
            for candidate in self._get_rendered_candidates(cache_dir, cache_key, page_index):
                if os.path.exists(candidate):
                    raw_path = candidate
                    break

            if raw_path:
                # Move the rendered image to the final cache location
                if raw_path != cache_path:
                    if os.path.exists(cache_path):
                        os.remove(cache_path)
                    os.rename(raw_path, cache_path)

            if not os.path.exists(cache_path):
                continue

            # Apply final quality settings
            image = QImage(cache_path)
            if image.isNull():
                self._create_placeholder(cache_path, image_width, page_index)
                continue

            # Ensure exact width and apply JPEG quality
            if image_width > 0 and abs(image.width() - image_width) > 1:  # Small tolerance for rounding
                scaled = image.scaledToWidth(image_width, Qt.SmoothTransformation)
            else:
                scaled = image

            # Always re-save to apply the specified JPEG quality
            scaled.save(cache_path, 'JPEG', quality)

    def _create_placeholder(self, path, width, page_num):
        """Create a placeholder image"""
        from PyQt5.Qt import QPainter, QFont, Qt

        height = int(width * 1.4)  # Approximate page ratio
        img = QImage(width, height, QImage.Format_RGB32)
        img.fill(Qt.white)

        painter = QPainter(img)
        painter.setPen(Qt.gray)
        font = QFont()
        font.setPixelSize(48)
        painter.setFont(font)
        painter.drawText(img.rect(), Qt.AlignCenter, f"Page {page_num}")
        painter.end()

        img.save(path, 'JPEG', 85)

    def _get_rendered_candidates(self, cache_dir, cache_key, page_num):
        # page_images/pdftoppm outputs files like cache_key-1.jpg or cache_key-001.jpg
        padded_path = os.path.join(
            cache_dir,
            f'{cache_key}-{str(page_num).zfill(PAGE_NUMBER_PADDING)}.{IMAGE_EXTENSION}'
        )
        simple_path = os.path.join(
            cache_dir,
            f'{cache_key}-{page_num}.{IMAGE_EXTENSION}'
        )
        return (padded_path, simple_path)

