"""
Dockable preview panel that shows book page previews
"""
import os
import shutil
from PyQt5.Qt import (QDockWidget, QWidget, QVBoxLayout, QScrollArea,
                      QLabel, QPixmap, Qt, QHBoxLayout, QPushButton,
                      QFrame, QSizePolicy, QGridLayout, QToolButton, QIcon,
                      QSlider)
from PyQt5.QtCore import QTimer

from calibre_plugins.n_pages_previews.config import prefs
from calibre_plugins.n_pages_previews.preview_generator import PreviewGenerator


class PreviewPanel(QDockWidget):
    """Dockable panel showing book previews"""

    def __init__(self, gui):
        QDockWidget.__init__(self, 'Page Previews', gui)
        self.gui = gui
        self.current_book_id = None
        self.selected_book_ids = []
        self.generator = PreviewGenerator()

        # Setup widget
        self.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea)
        self.setFeatures(QDockWidget.DockWidgetFeature.DockWidgetMovable |
                        QDockWidget.DockWidgetFeature.DockWidgetClosable)
        self.setMinimumWidth(380)

        # Create main widget
        main_widget = QWidget()
        layout = QVBoxLayout()
        main_widget.setLayout(layout)

        # Status label
        self.status_label = QLabel('Select a PDF book to see previews')
        self.status_label.setAlignment(Qt.AlignCenter)
        self.status_label.setWordWrap(True)
        layout.addWidget(self.status_label)

        # Zoom slider
        zoom_layout = QHBoxLayout()
        zoom_label = QLabel('Zoom:')
        zoom_layout.addWidget(zoom_label)

        self.zoom_slider = QSlider(Qt.Horizontal)
        self.zoom_slider.setMinimum(20)  # 20%
        self.zoom_slider.setMaximum(1000)  # 1000%
        self.zoom_slider.setValue(prefs.get('preview_zoom', 100))
        # Try to set tick marks (different PyQt versions have different constant names)
        try:
            self.zoom_slider.setTickPosition(QSlider.TicksBelow)
        except AttributeError:
            try:
                self.zoom_slider.setTickPosition(QSlider.TicksBelowSlider)
            except AttributeError:
                pass  # Tick marks not available, slider will work without them
        self.zoom_slider.setTickInterval(100)
        self.zoom_slider.valueChanged.connect(self.on_zoom_changed)
        zoom_layout.addWidget(self.zoom_slider)

        self.zoom_value_label = QLabel(f'{self.zoom_slider.value()}%')
        self.zoom_value_label.setMinimumWidth(50)
        zoom_layout.addWidget(self.zoom_value_label)

        layout.addLayout(zoom_layout)

        # Scroll area for thumbnail grid
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setVerticalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        scroll.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)

        self.preview_widget = QWidget()
        self.preview_layout = QGridLayout()
        self.preview_layout.setSpacing(5)
        self.preview_widget.setLayout(self.preview_layout)
        scroll.setWidget(self.preview_widget)

        layout.addWidget(scroll)

        # Control buttons
        button_layout = QHBoxLayout()

        self.refresh_btn = QPushButton('Refresh')
        self.refresh_btn.clicked.connect(self.refresh_previews)
        self.refresh_btn.setEnabled(False)
        button_layout.addWidget(self.refresh_btn)

        self.generate_btn = QPushButton('Generate')
        self.generate_btn.clicked.connect(self.generate_previews)
        self.generate_btn.setEnabled(False)
        button_layout.addWidget(self.generate_btn)

        layout.addLayout(button_layout)

        self.setWidget(main_widget)
        self.resize(420, 600)

        # Connect to book selection changes
        self.gui.library_view.selectionModel().selectionChanged.connect(self.on_book_selected)

        # Store cache directory for resize events
        self.current_cache_dir = None

        # Track whether current book has previews (for Re-generate button)
        self.has_previews = False

        # Throttle resize events to avoid excessive re-rendering
        self.resize_timer = QTimer()
        self.resize_timer.setSingleShot(True)
        self.resize_timer.timeout.connect(self._handle_resize)

        # Prevent multiple simultaneous display operations
        self.is_displaying = False

        # print("DEBUG: PreviewPanel created")

    def on_book_selected(self):
        """Called when book selection changes"""
        rows = self.gui.library_view.selectionModel().selectedRows()
        model = self.gui.library_view.model()
        self.selected_book_ids = [model.id(row) for row in rows]

        if not self.selected_book_ids:
            self.current_book_id = None
            self.has_previews = False
            self.clear_previews()
            self.status_label.setText('Select a book to see previews')
            self.refresh_btn.setEnabled(False)
            self.generate_btn.setEnabled(False)
            self.update_generate_button_label()
            return

        # Use the last selected book (same as book details panel behavior)
        book_id = self.selected_book_ids[-1]
        reload_needed = book_id != self.current_book_id
        self.current_book_id = book_id
        self.update_generate_button_label()

        if not reload_needed:
            # Selection count might have changed even if focused book stayed the same
            # Still update refresh button state based on current book availability
            return

        if self.current_book_id is None:
            return  # Same book, no need to refresh

        self.load_previews()

    def load_previews(self):
        """Load previews for current book"""
        if self.current_book_id is None:
            self.update_generate_button_label()
            return

        db = self.gui.current_db.new_api

        # Check if book has PDF
        fmt = db.has_format(self.current_book_id, 'PDF')
        if not fmt:
            self.has_previews = False
            self.clear_previews()
            self.status_label.setText('Selected book does not have PDF format')
            self.refresh_btn.setEnabled(False)
            self.generate_btn.setEnabled(False)
            self.update_generate_button_label()
            return

        # Get PDF path
        pdf_path = db.format_abspath(self.current_book_id, 'PDF')
        if not pdf_path or not os.path.exists(pdf_path):
            self.has_previews = False
            self.clear_previews()
            self.status_label.setText('PDF file not found')
            self.refresh_btn.setEnabled(False)
            self.generate_btn.setEnabled(False)
            self.update_generate_button_label()
            return

        # Check for cached previews
        book_dir = os.path.dirname(pdf_path)
        cache_dir = os.path.join(book_dir, '.previews')

        if os.path.exists(cache_dir) and os.listdir(cache_dir):
            # Load cached previews
            self.has_previews = True
            self.display_cached_previews(cache_dir)
            self.status_label.setText(f'Showing cached previews')
            self.refresh_btn.setEnabled(True)
            self.generate_btn.setEnabled(True)
            self.update_generate_button_label()
        else:
            # No previews yet
            self.has_previews = False
            self.clear_previews()
            self.status_label.setText('No previews generated. Click "Generate" to create them.')
            self.refresh_btn.setEnabled(False)
            self.generate_btn.setEnabled(True)
            self.update_generate_button_label()

    def display_cached_previews(self, cache_dir):
        """Display cached preview images as clickable thumbnails in a responsive grid"""
        # Prevent re-entrant calls
        if self.is_displaying:
            return

        self.is_displaying = True

        try:
            self.clear_previews()

            # Store cache directory for resize events
            self.current_cache_dir = cache_dir

            # Find all preview images and sort by page number
            # Group by page number and keep only the most recent file for each page
            files_by_page = {}

            for fname in os.listdir(cache_dir):
                # Skip macOS metadata files (AppleDouble files starting with ._ or any hidden files)
                if fname.startswith('.') or fname.startswith('._'):
                    continue

                if fname.endswith('.jpg') and '_page_' in fname:
                    # Extract page number
                    try:
                        page_num = int(fname.split('_page_')[1].replace('.jpg', ''))
                        filepath = os.path.join(cache_dir, fname)

                        # Verify file exists and is readable (and not just metadata)
                        if os.path.exists(filepath) and os.path.getsize(filepath) > 4096:
                            # Keep the most recent file for each page number
                            if page_num not in files_by_page:
                                files_by_page[page_num] = (filepath, os.path.getmtime(filepath))
                            else:
                                # Replace if this file is newer
                                existing_path, existing_time = files_by_page[page_num]
                                current_time = os.path.getmtime(filepath)
                                if current_time > existing_time:
                                    files_by_page[page_num] = (filepath, current_time)
                    except Exception as e:
                        # Skip files that can't be parsed or read
                        print(f"DEBUG: Skipping file {fname}: {e}")
                        pass

            # Convert to list of (page_num, filepath) and sort by page number
            preview_files = sorted([(page, path) for page, (path, _) in files_by_page.items()])

            # Calculate image width based on zoom level
            zoom_percent = self.zoom_slider.value()
            base_width = 140
            desired_image_width = int(base_width * zoom_percent / 100)

            # Calculate number of columns based on available width and image size
            available_width = self.preview_widget.width() - 20  # Account for margins
            num_columns = max(1, available_width // (desired_image_width + 10))  # 10px for spacing

            # Constrain image width to never exceed available space (important for high zoom levels)
            max_image_width = available_width - 10  # Leave small margin
            image_width = min(desired_image_width, max_image_width)

            # Display previews in responsive grid
            for idx, (page_num, filepath) in enumerate(preview_files):
                row = idx // num_columns
                col = idx % num_columns

                # Create clickable label instead of button for cleaner look
                page_frame = QFrame()
                page_frame.setFrameShape(QFrame.StyledPanel)
                page_frame.setStyleSheet("""
                    QFrame {
                        border: 1px solid #ccc;
                        border-radius: 4px;
                        padding: 5px;
                        background: white;
                    }
                    QFrame:hover {
                        border: 2px solid #0066cc;
                        background: #f0f0f0;
                    }
                """)

                frame_layout = QVBoxLayout()
                frame_layout.setContentsMargins(0, 0, 0, 0)
                frame_layout.setSpacing(0)
                page_frame.setLayout(frame_layout)

                # Image
                pixmap = QPixmap(filepath)
                if not pixmap.isNull():
                    # Scale to fit within the constrained image_width (respects aspect ratio)
                    scaled = pixmap.scaledToWidth(image_width, Qt.SmoothTransformation)

                    # Create container for image with overlay
                    img_container = QWidget()
                    img_container_layout = QVBoxLayout()
                    img_container_layout.setContentsMargins(0, 0, 0, 0)
                    img_container_layout.setSpacing(0)
                    img_container.setLayout(img_container_layout)

                    img_label = QLabel()
                    img_label.setPixmap(scaled)
                    img_label.setAlignment(Qt.AlignCenter)
                    img_label.setCursor(Qt.PointingHandCursor)

                    # Make the frame clickable
                    img_label.mousePressEvent = lambda event, p=page_num: self.open_preview_at_page(p)

                    img_container_layout.addWidget(img_label)

                    # Add page number as small overlay in bottom-right corner
                    page_num_label = QLabel(f'{page_num}', img_label)
                    page_num_label.setStyleSheet("""
                        background: rgba(0, 0, 0, 0.6);
                        color: white;
                        padding: 2px 6px;
                        border-radius: 3px;
                        font-size: 10px;
                        font-weight: bold;
                    """)
                    page_num_label.adjustSize()
                    # Position in bottom-right corner with small margin
                    page_num_label.move(
                        scaled.width() - page_num_label.width() - 5,
                        scaled.height() - page_num_label.height() - 5
                    )

                    frame_layout.addWidget(img_container)
                else:
                    # Show error if image failed to load
                    error_label = QLabel(f'Page {page_num}\nFailed to load')
                    error_label.setAlignment(Qt.AlignCenter)
                    error_label.setStyleSheet('color: red; padding: 20px;')
                    frame_layout.addWidget(error_label)

                self.preview_layout.addWidget(page_frame, row, col)

            # Add stretch at the end
            max_row = len(preview_files) // num_columns
            self.preview_layout.setRowStretch(max_row + 1, 1)

        finally:
            # Always reset the flag
            self.is_displaying = False

    def clear_previews(self):
        """Clear all preview widgets"""
        # Remove all widgets from layout
        while self.preview_layout.count():
            item = self.preview_layout.takeAt(0)
            if item.widget():
                widget = item.widget()
                widget.setParent(None)
                widget.deleteLater()

        # Reset row stretches
        for row in range(self.preview_layout.rowCount()):
            self.preview_layout.setRowStretch(row, 0)

    def refresh_previews(self):
        """Refresh current previews"""
        self.load_previews()

    def generate_previews(self):
        """Generate or regenerate previews for current book(s)"""
        if not self.selected_book_ids:
            return

        # If regenerating, clear cache directories for all selected books
        if self.has_previews:
            db = self.gui.current_db.new_api

            for book_id in self.selected_book_ids:
                try:
                    pdf_path = db.format_abspath(book_id, 'PDF')
                    if pdf_path and os.path.exists(pdf_path):
                        book_dir = os.path.dirname(pdf_path)
                        cache_dir = os.path.join(book_dir, '.previews')
                        if os.path.exists(cache_dir):
                            shutil.rmtree(cache_dir)
                            print(f"DEBUG: Cleared cache for book {book_id}")
                except Exception as e:
                    print(f"DEBUG: Error clearing cache for book {book_id}: {e}")

        # Show generation dialog
        from calibre_plugins.n_pages_previews.generator import GeneratePreviewsDialog

        db = self.gui.current_db.new_api
        book_ids = self.selected_book_ids if len(self.selected_book_ids) > 1 else [self.current_book_id]
        d = GeneratePreviewsDialog(self.gui, db, book_ids, prefs, auto_start=True)
        d.exec_()

        # Always reload previews after generation dialog closes
        self.load_previews()

    def open_preview_at_page(self, page_num):
        """Open full preview dialog scrolled to specific page"""
        if self.current_book_id is None:
            return

        # print(f"DEBUG: Opening preview at page {page_num}")

        from calibre_plugins.n_pages_previews.main import PreviewDialog

        db = self.gui.current_db.new_api
        d = PreviewDialog(self.gui, db, self.current_book_id, prefs, scroll_to_page=page_num)
        d.exec_()

    def on_zoom_changed(self, value):
        """Handle zoom slider changes"""
        self.zoom_value_label.setText(f'{value}%')

        # Save zoom preference
        prefs['preview_zoom'] = value

        # Reload previews if we have any
        if self.current_book_id is not None:
            db = self.gui.current_db.new_api
            fmt = db.has_format(self.current_book_id, 'PDF')
            if fmt:
                pdf_path = db.format_abspath(self.current_book_id, 'PDF')
                if pdf_path and os.path.exists(pdf_path):
                    book_dir = os.path.dirname(pdf_path)
                    cache_dir = os.path.join(book_dir, '.previews')
                    if os.path.exists(cache_dir) and os.listdir(cache_dir):
                        self.display_cached_previews(cache_dir)

    def update_generate_button_label(self):
        """Update the floating panel's generate button text based on selection count and preview state."""
        count = len(self.selected_book_ids)

        # Determine base text based on whether previews exist
        base_text = 'Re-generate' if self.has_previews else 'Generate'

        if count <= 1:
            self.generate_btn.setText(base_text)
        else:
            self.generate_btn.setText(f'{base_text} ({count})')

        if count == 0:
            self.generate_btn.setEnabled(False)

    def resizeEvent(self, event):
        """Handle panel resize to reflow columns (throttled)"""
        super().resizeEvent(event)

        # Start/restart timer to throttle resize handling
        if self.current_cache_dir:
            self.resize_timer.stop()
            self.resize_timer.start(300)  # Wait 300ms after last resize before updating

    def _handle_resize(self):
        """Actually handle the resize after throttling"""
        if self.current_cache_dir and os.path.exists(self.current_cache_dir):
            self.display_cached_previews(self.current_cache_dir)
