"""
Dockable preview panel that shows book page previews
"""
import os
from PyQt5.Qt import (QDockWidget, QWidget, QVBoxLayout, QScrollArea,
                      QLabel, QPixmap, Qt, QHBoxLayout, QPushButton,
                      QFrame, QSizePolicy, QGridLayout, QToolButton, QIcon)

from calibre_plugins.n_pages_previews.config import prefs
from calibre_plugins.n_pages_previews.preview_generator import PreviewGenerator


class PreviewPanel(QDockWidget):
    """Dockable panel showing book previews"""

    def __init__(self, gui):
        QDockWidget.__init__(self, 'Page Previews', gui)
        self.gui = gui
        self.current_book_id = None
        self.selected_book_ids = []
        self.generator = PreviewGenerator()

        # Setup widget
        self.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea)
        self.setFeatures(QDockWidget.DockWidgetFeature.DockWidgetMovable |
                        QDockWidget.DockWidgetFeature.DockWidgetClosable)
        self.setMinimumWidth(380)

        # Create main widget
        main_widget = QWidget()
        layout = QVBoxLayout()
        main_widget.setLayout(layout)

        # Status label
        self.status_label = QLabel('Select a PDF book to see previews')
        self.status_label.setAlignment(Qt.AlignCenter)
        self.status_label.setWordWrap(True)
        layout.addWidget(self.status_label)

        # Scroll area for thumbnail grid
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setVerticalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        scroll.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)

        self.preview_widget = QWidget()
        self.preview_layout = QGridLayout()
        self.preview_layout.setSpacing(5)
        self.preview_widget.setLayout(self.preview_layout)
        scroll.setWidget(self.preview_widget)

        layout.addWidget(scroll)

        # Control buttons
        button_layout = QHBoxLayout()

        self.refresh_btn = QPushButton('Refresh')
        self.refresh_btn.clicked.connect(self.refresh_previews)
        self.refresh_btn.setEnabled(False)
        button_layout.addWidget(self.refresh_btn)

        self.generate_btn = QPushButton('Generate')
        self.generate_btn.clicked.connect(self.generate_previews)
        self.generate_btn.setEnabled(False)
        button_layout.addWidget(self.generate_btn)

        layout.addLayout(button_layout)

        self.setWidget(main_widget)
        self.resize(420, 600)

        # Connect to book selection changes
        self.gui.library_view.selectionModel().selectionChanged.connect(self.on_book_selected)

        # print("DEBUG: PreviewPanel created")

    def on_book_selected(self):
        """Called when book selection changes"""
        rows = self.gui.library_view.selectionModel().selectedRows()
        model = self.gui.library_view.model()
        self.selected_book_ids = [model.id(row) for row in rows]

        if not self.selected_book_ids:
            self.current_book_id = None
            self.clear_previews()
            self.status_label.setText('Select a book to see previews')
            self.refresh_btn.setEnabled(False)
            self.generate_btn.setEnabled(False)
            self.update_generate_button_label()
            return

        # Use the last selected book (same as book details panel behavior)
        book_id = self.selected_book_ids[-1]
        reload_needed = book_id != self.current_book_id
        self.current_book_id = book_id
        self.update_generate_button_label()

        if not reload_needed:
            # Selection count might have changed even if focused book stayed the same
            # Still update refresh button state based on current book availability
            return

        if self.current_book_id is None:
            return  # Same book, no need to refresh

        self.load_previews()

    def load_previews(self):
        """Load previews for current book"""
        if self.current_book_id is None:
            self.update_generate_button_label()
            return

        db = self.gui.current_db.new_api

        # Check if book has PDF
        fmt = db.has_format(self.current_book_id, 'PDF')
        if not fmt:
            self.clear_previews()
            self.status_label.setText('Selected book does not have PDF format')
            self.refresh_btn.setEnabled(False)
            self.generate_btn.setEnabled(False)
            self.update_generate_button_label()
            return

        # Get PDF path
        pdf_path = db.format_abspath(self.current_book_id, 'PDF')
        if not pdf_path or not os.path.exists(pdf_path):
            self.clear_previews()
            self.status_label.setText('PDF file not found')
            self.refresh_btn.setEnabled(False)
            self.generate_btn.setEnabled(False)
            self.update_generate_button_label()
            return

        # Check for cached previews
        book_dir = os.path.dirname(pdf_path)
        cache_dir = os.path.join(book_dir, '.previews')

        if os.path.exists(cache_dir) and os.listdir(cache_dir):
            # Load cached previews
            self.display_cached_previews(cache_dir)
            self.status_label.setText(f'Showing cached previews')
            self.refresh_btn.setEnabled(True)
            self.generate_btn.setEnabled(True)
            self.update_generate_button_label()
        else:
            # No previews yet
            self.clear_previews()
            self.status_label.setText('No previews generated. Click "Generate" to create them.')
            self.refresh_btn.setEnabled(False)
            self.generate_btn.setEnabled(True)
            self.update_generate_button_label()

    def display_cached_previews(self, cache_dir):
        """Display cached preview images as clickable thumbnails in a grid"""
        self.clear_previews()

        # Find all preview images and sort by page number
        preview_files = []
        for fname in os.listdir(cache_dir):
            if fname.endswith('.jpg') and '_page_' in fname:
                # Extract page number
                try:
                    page_num = int(fname.split('_page_')[1].replace('.jpg', ''))
                    preview_files.append((page_num, os.path.join(cache_dir, fname)))
                except:
                    pass

        preview_files.sort()  # Sort by page number

        # Display previews in 2 columns (odd pages on left, even pages on right)
        # This mimics a book spread layout
        for page_num, filepath in preview_files:
            # Determine column: odd pages (1,3,5...) go to left (col 0), even pages (2,4,6...) go to right (col 1)
            col = 0 if page_num % 2 == 1 else 1
            # Row is based on pairs: pages 1-2 in row 0, pages 3-4 in row 1, etc.
            row = (page_num - 1) // 2

            # Create clickable label instead of button for cleaner look
            page_frame = QFrame()
            page_frame.setFrameShape(QFrame.StyledPanel)
            page_frame.setStyleSheet("""
                QFrame {
                    border: 1px solid #ccc;
                    border-radius: 4px;
                    padding: 5px;
                    background: white;
                }
                QFrame:hover {
                    border: 2px solid #0066cc;
                    background: #f0f0f0;
                }
            """)

            frame_layout = QVBoxLayout()
            page_frame.setLayout(frame_layout)

            # Image
            pixmap = QPixmap(filepath)
            if not pixmap.isNull():
                # Scale to fit panel width (~140px per column)
                scaled = pixmap.scaledToWidth(140, Qt.SmoothTransformation)

                img_label = QLabel()
                img_label.setPixmap(scaled)
                img_label.setAlignment(Qt.AlignCenter)
                img_label.setCursor(Qt.PointingHandCursor)

                # Make the frame clickable
                img_label.mousePressEvent = lambda event, p=page_num: self.open_preview_at_page(p)

                frame_layout.addWidget(img_label)

            self.preview_layout.addWidget(page_frame, row, col)

        # Add stretch at the end
        max_row = (len(preview_files) - 1) // 2 + 1
        self.preview_layout.setRowStretch(max_row, 1)

    def clear_previews(self):
        """Clear all preview widgets"""
        while self.preview_layout.count():
            item = self.preview_layout.takeAt(0)
            if item.widget():
                item.widget().deleteLater()

    def refresh_previews(self):
        """Refresh current previews"""
        self.load_previews()

    def generate_previews(self):
        """Generate previews for current book"""
        if not self.selected_book_ids:
            return

        # Show generation dialog
        from calibre_plugins.n_pages_previews.generator import GeneratePreviewsDialog

        db = self.gui.current_db.new_api
        book_ids = self.selected_book_ids if len(self.selected_book_ids) > 1 else [self.current_book_id]
        d = GeneratePreviewsDialog(self.gui, db, book_ids, prefs, auto_start=True)
        d.exec_()

        # Always reload previews after generation dialog closes
        self.load_previews()

    def open_preview_at_page(self, page_num):
        """Open full preview dialog scrolled to specific page"""
        if self.current_book_id is None:
            return

        # print(f"DEBUG: Opening preview at page {page_num}")

        from calibre_plugins.n_pages_previews.main import PreviewDialog

        db = self.gui.current_db.new_api
        d = PreviewDialog(self.gui, db, self.current_book_id, prefs, scroll_to_page=page_num)
        d.exec_()

    def update_generate_button_label(self):
        """Update the floating panel's generate button text based on selection count."""
        count = len(self.selected_book_ids)
        if count <= 1:
            self.generate_btn.setText('Generate')
        else:
            self.generate_btn.setText(f'Generate ({count})')

        if count == 0:
            self.generate_btn.setEnabled(False)
