#!/usr/bin/env python
__license__ = "GPL v3"
__copyright__ = "2025, ARR8 <>"

from calibre.ebooks.metadata.sources.base import Source, Option


class Storygraph(Source):
    name = "StoryGraph"
    author = "ARR8"
    version = (2, 0, 2)
    minimum_calibre_version = (3, 6, 0)
    description = _("Downloads metadata and covers from the StoryGraph")

    capabilities = frozenset(["identify", "cover"])
    touched_fields = frozenset(
        [
            "title",
            "authors",
            "tags",
            "pubdate",
            "comments",
            "publisher",
            "identifier:isbn",
            "identifier:storygraph",
            "rating",
            "series",
            "languages",
        ]
    )
    supports_gzip_transfer_encoding = True
    has_html_comments = True
    can_get_multiple_covers = False
    options = [
        Option(
            "usefirstpub",
            "bool",
            True,
            "Use original pub year over edition",
            "Use the original publication *year* over the edition publication *date*",
        )
    ]

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        with self:
            from .storygraph import StorygraphImpl

            self._impl = StorygraphImpl(self)

    def _get_book_url(self, guid):
        if guid:
            return "https://app.thestorygraph.com/books/" + guid

    def get_book_url(self, identifiers):
        guid = identifiers.get("storygraph", None)
        if guid:
            return "storygraph", guid, self._get_book_url(guid)

    def get_cached_cover_url(self, identifiers):
        guid = identifiers.get("storygraph", None)
        if not guid:
            isbn = identifiers.get("isbn", None)
            if isbn is not None:
                guid = self.cached_isbn_to_identifier(isbn)
        return self.cached_identifier_to_cover_url(guid)

    def identify(
        self,
        log,
        result_queue,
        abort,
        title=None,
        authors=None,
        identifiers={},
        timeout=30,
    ) -> None:
        self._impl.identify(
            log, result_queue, abort, title, authors, identifiers, timeout
        )

    def download_cover(
        self,
        log,
        result_queue,
        abort,
        title=None,
        authors=None,
        identifiers={},
        timeout=30,
        get_best_cover=False,
    ) -> None:
        cached_url = self.get_cached_cover_url(identifiers)
        if not cached_url:
            log.info("No cached cover found, running identify")
            cover_url = self._impl.get_cover_url(
                log, abort, title, authors, identifiers, timeout
            )

        if not cover_url:
            log.info("No cover found")
            return

        try:
            log("Downloading cover from:", cover_url)
            cdata = self._impl.get_cover(cover_url, timeout)
            result_queue.put((self, cdata))
        except:
            log.exception("Failed to download cover from:", cached_url)


if __name__ == "__main__":
    from calibre_plugins.storygraph.storygraph import Worker

    for desc in (
        '<h4 class="section-heading uppercase">Description</h4><div class="trix-content mt-3"><div>For twelve thousand years the Galactic Empire has ruled supreme. Now it is dying. But only Hari Seldon, creator of the revolutionary science of psychohistory, can see into the future -- to a dark age of ignorance, barbarism, and warfare that will last thirty thousand years. To preserve knowledge and save mankind, Seldon gathers the best minds in the Empire -- both scientists and scholars -- and brings them to a bleak planet at the edge of the Galaxy to serve as a beacon of hope for a future generations. He calls his sanctuary the Foundation.<br><br>But soon the fledgling Foundation finds itself at the mercy of corrupt warlords rising in the wake of the receding Empire. Mankind\\\'s last best hope is faced with an agonizing choice: submit to the barbarians and be overrun -- or fight them and be destroyed.</div></div>',
        "'\\n            $(\\'.read-more-btn\\').on(\\'click\\', function(e) {\\n              e.preventDefault()\\n              $(this).parent().parent().html(\\'<h4 class=\"section-heading uppercase\">Description</h4><div class=\"trix-content mt-3\"><div>With a new introduction by Zadie Smith<br>Into the intrigue and violence of Indo-China comes Pyle, a young idealistic American sent to promote democracy through a mysterious “Third Force.” As his naïve optimism starts to cause bloodshed, his friend Fowler, a cynical foreign correspondent, finds it hard to stand aside and watch. But even as he intervenes he wonders why: for the sake of politics, or for love?<br><br><\\\\/div></div>\\')\\n            })\\n          '",
        "'\\n            $(\\'.read-more-btn\\').on(\\'click\\', function(e) {\\n              e.preventDefault()\\n              $(this).parent().parent().html(\\'<h4 class=\"section-heading uppercase\">Description</h4><div class=\"trix-content mt-3\"><div>Here there be dragons . . . and the denizens of Ankh-Morpork wish one huge firebreather would return from whence it came. Long believed extinct, a superb specimen of <em>draco nobilis<\\\\/em> (\\\\\"noble dragon\\\\\" for those who don\\\\\\'t understand italics) has appeared in Discworld\\\\\\'s greatest city. Not only does this unwelcome visitor have a nasty habit of charbroiling everything in its path, in rather short order it is crowned King (it is a <em>noble<\\\\/em> dragon, after all . . .).<br><br>Meanwhile, back at Unseen University, an ancient and long-forgotten volume--<em>The Summoning of Dragons<\\\\/em>--is missing from the Library\\\\\\'s shelves. To the rescue come Captain Vimes, Constable Carrot, and the rest of the Night Watch who, along with other brave citizens, risk everything, including a good roasting, to dethrone the flying monarch and restore order to Ankh-Morpork (before it\\\\\\'s burned to a crisp). A rare tale, well done as only Terry Pratchett can.<\\\\/div></div>\\')\\n            })\\n          '",
        "'\\n            $(\\'.read-more-btn\\').on(\\'click\\', function(e) {\\n              e.preventDefault()\\n              $(this).parent().parent().html(\\'<h4 class=\"section-heading uppercase\">Description</h4><div class=\"trix-content mt-3\"><b><i>Octopus <\\\\/i>is a real-life thriller that tells the inside story of an audacious hedge fund fraud and the wild search, by a colorful cast of rogues and schemers, for a “secret market” beneath the financial market we all know.<\\\\/b><br/><b>\\xa0<\\\\/b><br/>Sam Israel was a man who seemed to have it all – until the hedge fund he ran, Bayou, imploded and he became the target of a nationwide manhunt.\\xa0 Born into one of America’s most illustrious trading families, Israel was determined to strike out on his own.\\xa0 So after apprenticing with one of the greatest hedge fund traders of the 1980’s, Sam founded his own fund and promised his investors <i>guaranteed <\\\\/i>profits.\\xa0 With the proprietary computer program he’d created, he claimed to be able to predict the future. <br/>\\xa0<br/>But his future was already beginning to unravel.<br/>\\xa0<br/>After suffering devastating losses and fabricating fake returns, Israel knew it was only a matter of time before his <i>real <\\\\/i>performance would be discovered, so when a former black-ops intelligence operative told him about a “secret market” run by the Fed, Israel bet his last \\\\$150 million on a chance to make billions. Thus began his year-long adventure in “the Upperworld” -- a society populated by clandestine bankers, shady European nobility, and spooks issuing cryptic warnings about a mysterious cabal known as the Octopus. <br/>\\xa0<br/>Whether the “secret market” was real or a con, Israel was all in – and as the pressures mounted and increasingly sinister violence crept into his life, he struggled to break free of the Octopus’ tentacles.</div>\\')\\n            })\\n          '",
    ):
        print("Prod version", Worker.parse_comments(desc))
        pass

    input("Press enter to run live test")

    from calibre.ebooks.metadata.sources.test import (
        authors_test,
        test_identify_plugin,
        title_test,
    )

    tests = [
        (  # A title and author search
            {"title": "The Husband's Secret", "authors": ["Liane Moriarty"]},
            [
                title_test("The Husband's Secret", exact=True),
                authors_test(["Liane Moriarty"]),
            ],
        ),
        (  # An isbn
            {
                "identifiers": {"isbn": "9780393355680"},
            },
            [
                title_test(
                    '"Surely You\'re Joking, Mr. Feynman!": Adventures of a Curious Character',
                    exact=True,
                ),
                authors_test(["Richard P. Feynman", "Ralph Leighton", "Bill Gates"]),
            ],
        ),
        (  # No isbn
            {
                "identifiers": {"storygraph": "7e968921-56de-4fd7-bc41-7403c03d94dd"},
            },
            [title_test("Unsong", exact=True), authors_test(["Scott Alexander"])],
        ),
    ]

    test_identify_plugin(
        Storygraph.name,
        tests,
        modify_plugin=lambda p: (
            setattr(p, "touched_fields", p.touched_fields - {"identifier:isbn"}),
        ),
    )
