#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2012, Pr.BarnArt, based on the Barnes work by Grant Drake'
__docformat__ = 'restructuredtext en'

import socket, re, datetime
import json
import re
import html5lib

from collections import OrderedDict
from threading import Thread
from lxml.html import fromstring, tostring, etree
from calibre.utils.icu import lower,upper

from xml.etree import ElementTree
from lxml import html


from calibre.ebooks.metadata.book.base import Metadata
from calibre.library.comments import sanitize_comments_html
from calibre.utils.cleantext import clean_ascii_chars
from calibre.utils.localization import canonicalize_lang


import calibre_plugins.BOL_NL.config as cfg

class Worker(Thread): # Get  details

    '''
    Get book details from BOL.com book page in a separate thread
    related to BOL_NL
    '''
    name                    = 'Worker'
    description             = _('Get book details from BOL.com book page in a separate thread')
    author                  = 'Pr. BarnArt'
    version                 = (5, 2, 9)
    minimum_calibre_version = (0, 8, 5)


    lm = {
            'eng': ('English', 'Englisch','Engels'),
            'fra': ('French', 'Français','Frans'),
            'ita': ('Italian', 'Italiano','Italiaans'),
            'nld': ('Dutch','Nederlands'),
            'deu': ('German', 'Deutsch','Duits','Deutsch, Schwedisch'),
            'spa': ('Spanish', 'Espa\xf1ol', 'Espaniol','Spaans'),
            'jpn': ('Japanese', u'日本語','Japans'),
            'por': ('Portuguese', 'Português','Portugees'),
            }
    lang_map = {}
    for code, names in lm.items():
        for name in names:
            lang_map[name] = code
    
    def __init__(self, combo, result_queue, browser, log, relevance, plugin, timeout=20):
        Thread.__init__(self)
        self.daemon = True
        #distract from combo
        self.url = combo[0]
        self.rating = combo[1]
        self.result_queue =  result_queue
        self.log, self.timeout = log, timeout
        self.relevance, self.plugin = relevance, plugin
        self.browser = browser.clone_browser()
        self.cover_url = self.bol_nl_id = self.isbn = None
        self.recensies_bol = None
        self.ebook = False


    def run(self):
        try:

            self.get_details()
        except:
            self.log.exception('get_details failed for url: %r'%self.url)

    def get_details(self):
        try:
            self.log.info('BOL_NL url: %r'%self.url)
            self.browser.addheaders = [
                    ('User-Agent', 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) '
                    'AppleWebKit/537.36 (KHTML, like Gecko) '
                    'Chrome/120.0.0.0 Safari/537.36')
            ]       
            raw = self.browser.open_novisit(self.url, timeout=self.timeout).read().strip()
        except Exception as e:
            if callable(getattr(e, 'getcode', None)) and \
                    e.getcode() == 404:
                self.log.error('URL malformed: %r'%self.url)
                return
            attr = getattr(e, 'args', [None])
            attr = attr if attr else [None]
            if isinstance(attr[0], socket.timeout):
                msg = 'Bol.com timed out. Try again later.'
                self.log.error(msg)
            else:
                msg = 'Failed to make details query: %r'%self.url
                self.log.exception(msg)
            return
        if isinstance(raw, bytes):
            raw = raw.decode('utf-8', errors='ignore')

            '''raw = raw.decode('utf-8', errors='replace')'''
        raw = clean_ascii_chars(raw)  
        if '<title>404 - ' in raw:
            self.log.error('URL malformed: %r'%self.url)
            return  
        try:
            # Tell html5lib to use lxml tree builder
            parser = html5lib.HTMLParser(tree=html5lib.getTreeBuilder("lxml"), namespaceHTMLElements=False)
            document = parser.parse(raw)

            # Grab the body safely
            body = document.find('.//body')
            
            root = html.fromstring(html.tostring(body)) if body is not None else html.fromstring(html.tostring(document))


        except Exception:
            msg = f'Failed to parse Bol.com details page: {self.url!r}'
            self.log.exception(msg)
            return
       
        '''
        match = re.search(r"<body[^>]*>([\s\S]*?)</body>", raw, re.IGNORECASE)
        if match:
            body_html = match.group(1)
        else:
            print("⚠️ No <body> tag found.")
            body_html = raw  # fallback
    
        parser = html.HTMLParser(encoding='utf-8', recover=True)
        lines = body_html.splitlines()
        for i, line in enumerate(lines[30:45], start=31):
            print(f"{i:03d}: {line[:200]}")
        #print(raw[:10000])
       
        f = open("result.html", "w")
        try:
            f.write(body_html.rstrip('\n') )
        finally:
            f.close()
       
        try:
            #root = html.document_fromstring(body_html, parser=parser)
            root = html.fragment_fromstring(body_html, create_parent=True, parser=parser)
            
        except:
            msg = 'Failed to parse Bol.com details page: %r'%self.url
            self.log.exception(msg)
            return
      
        
        
        
        
        '''




        default_recensies_bol = cfg.DEFAULT_STORE_VALUES[cfg.KEY_RECENSIES_BOL]
        self.recensies_bol = cfg.plugin_prefs[cfg.STORE_NAME].get(cfg.KEY_RECENSIES_BOL, default_recensies_bol)
        default_lp_covers = cfg.DEFAULT_STORE_VALUES[cfg.KEY_GET_LP_COVERS]
        self.lp_covers = cfg.plugin_prefs[cfg.STORE_NAME].get(cfg.KEY_GET_LP_COVERS, default_lp_covers)
        default_lponly_covers = cfg.DEFAULT_STORE_VALUES[cfg.KEY_GET_LPONLY_COVERS]
        self.lponly_covers = cfg.plugin_prefs[cfg.STORE_NAME].get(cfg.KEY_GET_LPONLY_COVERS, default_lponly_covers)

        self.parse_details(root,raw)



    def parse_details(self, root,raw):
        html_content = etree.tostring(root, pretty_print=True, encoding="unicode")
        # Logging the full HTML content (with tags)
        #self.log(f'root HTML: {repr(html_content)}')  # Using repr() to show raw output
        #self.log(repr(raw))
        #print(repr(raw))
        #with open("tst2.htm", "w", encoding="utf-8") as file:
        #   file.write(html_content)

        bol_nl_id = None
        try:
            bol_nl_id = self.parse_bol_nl_id(self.url)
        except:
            self.log.exception('Error parsing BOL-NL id for url: %r'%self.url)
            bol_nl_id = None
        try:
            title = self.parse_title(root)

            #series = None
            #series_index = None
        except:
            self.log.exception('Error parsing title for url: %r'%self.url)
            title = None
        if not title:
            self.log.error('Could not find title for %r'%self.url)
        try:
            authors = self.parse_authors(root)


        except:
            self.log.exception('Error parsing authors for url: %r'%self.url)
            authors = []
        if not authors:
            self.log.error('Could not find authors for %r'%self.url)

        if not title or not authors :
            self.log.error('Could not find title/authors/bol_nl_id for %r'%self.url)
            self.log.error('bol_nl_id: %r Title: %r Authors: %r'%(bol_nl_id, title,
                authors))

            return
        mi = Metadata(title, authors)

        #if series:
        #   mi.series = series
        #   mi.series_index = series_index
        self.bol_nl_id = bol_nl_id

        try:
            mi.language=self.parse_language(root)
        except:
            self.log.exception('Error parsing language for url: %r'%self.url)

        try:
            isbn = self.parse_isbn(raw,root)
            if isbn:
                self.isbn = isbn
                mi.isbn = isbn
        except:
            self.log.exception('Error parsing ISBN for url: %r'%self.url)

        #do rating coversion from txt to integer
        pattern= re.compile("^\d\d?\d?\d?")
        #print("pattern3", pattern)
        if pattern.match(self.rating):  
            rat=float(self.rating)
            rat = rat + 0.5
            rat=int(rat)
            mi.rating=rat

        try:
            mi.comments = self.parse_comments(root)
            #print('mi.comments:',mi.comments)
        except:
            self.log.exception('Error parsing comments for url: %r'%self.url)

        try:
            self.cover_url = self.parse_cover(root)

        except:
            self.log.exception('Error parsing cover for url: %r'%self.url)
        mi.has_cover = bool(self.cover_url)
        print('mi.has_cover:',mi.has_cover)
        

        try:
            tags = self.parse_tags(root)
            if tags:
                mi.tags = tags
        except:
            self.log.exception('Error parsing tags for url: %r'%self.url)

        try:
            mi.publisher = self.parse_publisher(root)
        except:
            self.log.exception('Error parsing publisher for url: %r'%self.url)

        try:
            mi.pubdate = self.parse_published_date(root)
        except:
            self.log.exception('Error parsing published date for url: %r'%self.url)

        mi.source_relevance = self.relevance

        if self.bol_nl_id:
            #print('self.bol_nl_id:',self.bol_nl_id)
            if self.isbn:
                self.plugin.cache_isbn_to_identifier(self.isbn, self.bol_nl_id)
            '''
                if self.cover_url:
                self.plugin.cache_identifier_to_cover_url(self.bol_nl_id,self.cover_url)

            '''

        #self.plugin.clean_downloaded_metadata(mi)
        print('mi:', mi)
        print('mi.has_cover1:',mi.has_cover)
        self.result_queue.put(mi)

    def parse_bol_nl_id(self, url):
        return re.search('bol.com/(.*/.*/\d+)', url).groups(0)[0]

    def parse_title(self,root):


        title_node=root.xpath('//section [@id="product_title"]/h1/span [@data-test="title"]')
        print('title_node1 :',title_node)
        title = None
        if title_node:
            print("Raw element title:", html.tostring(title_node[0], pretty_print=True).decode())

            title=title_node[0].text_content().strip()
            print('title:',title)
        if title == None:
            title_node=root.xpath('//div [@id="main_block"]/h1 [@itemprop="name"]')
            print('title_node2 :',title_node)
            if title_node:
                title=title_node[0].text_content().strip()
        if title == None:
            title_node=root.xpath('//h1 [@itemprop="name"]')
            print('title_node3 :',title_node)
            if title_node:
                title=title_node[0].text_content().strip()
        if title:
            pos=title.find('EBOOK')
            if pos>-1:
                title=title[:pos]
                title=title.rstrip()

        return title

    def parse_authors(self, root):
        #search productinfo
        auteur_node = root.xpath('//div[@class="specs__party-groups"]/div[text()[contains(.,"Auteur")]]/a')
        authors = []
        print ('auteur_node0:', auteur_node)
        if auteur_node:
            a=auteur_node[0]
            print("Raw element:", html.tostring(a, pretty_print=True).decode())
            for auteurs in auteur_node:
                auteur=auteurs.text_content().strip()
                authors.append(auteur)
                valid_contrib = None
            return authors


        auteur_node = root.xpath('//div[@class="product_detail_title boxedbottom_s fleft"]/div/p[text()[contains(.,"Auteur")]]/span[@class="po_pdp_creator2"]/a')
        #print ('auteur_node1:', auteur_node)
        if auteur_node:
            for auteurs in auteur_node:
                auteur=auteurs.text_content().strip()
                authors.append(auteur)
                valid_contrib = None
            return authors
        #search other way to show productinfo
        auteur_node = root.xpath('//div[@class="product_detail_title boxedbottom_s fleft"]/div/div/a/div/span[@class="heavy"]')
        #print ('auteur_node2:', auteur_node)
        if auteur_node:
            for auteurs in auteur_node:
                auteur=auteurs.text_content().strip()
                #print ('auteur:', auteur)
                authors.append(auteur)
                valid_contrib = None
            return authors
        #search other way to show productinfo
        auteur_node = root.xpath('//div[@class="pdp-header__meta"]/div[text()[contains(.,"Auteur")]]/a')
        #print ('auteur_node3:', auteur_node)
        if auteur_node:
            for auteurs in auteur_node:
                auteur=auteurs.text_content().strip()
                #print ('auteur:', auteur)
                authors.append(auteur)
                valid_contrib = None
            return authors




        auteur_node = root.xpath('//div[@class="specs"]/dl[@class="specs__list"]/dt[text()[contains(.,"Auteur")]]/following-sibling::dd[1]/a')
        #print ('auteur_node4:', auteur_node)
        if auteur_node:
            for auteurs in auteur_node:
                auteur=auteurs.text_content().strip()
                #print ('auteur:', auteur)
                authors.append(auteur)
                valid_contrib = None
            return authors

        #search other way to show productinfo
        auteur_node = root.xpath('//div/h3[text() [contains(.,"Productinformatie")]]/../table/tbody/tr/td[text()[contains(.,"Auteur")]]/../td [@class="specs_descr"]/a')
        #print ('auteur_node5:', auteur_node)
        if auteur_node:
            for auteurs in auteur_node:
                auteur=auteurs.text_content().strip()
                authors.append(auteur)
                valid_contrib = None
            return authors


    def parse_isbn(self, raw,root):
        isbn = None
        isbn_node = root.xpath('//dl/dt[text()[contains(.,"EAN")]]/following-sibling::dd[1]')
        #print('isbn_node0', isbn_node)
        if isbn_node:
            for node in isbn_node:
                isbn=node.text_content().strip()
                #print('isbn0', isbn)
                return isbn

        findstart = raw.find('<h2 class="h5 bottom_xxs inline_header">Productinformatie</h2>')
        if findstart<1:
            findstart = raw.find('<h2 class="h5 bottom_xxs inline_header">Productinformatie</h2>')
        prodtxt=raw[findstart:findstart+2000]
        findstart=prodtxt.find('EAN</dt>')
        if findstart>0:
            isbn=prodtxt[findstart+36:findstart+46]
            #print('isbn1', isbn)
            return isbn

        isbn_node = root.xpath('//table [@class="boxedbottom_s"]/tbody/tr')
        #print('isbn_node2', isbn_node)
        if isbn_node:
            for label in isbn_node:
                txt=label.text_content().strip()
                find=txt.find('EAN')
                if find>-1:
                    isbn=txt[find+6:].strip()
                    #print('isbn2', isbn)
                    return isbn

        isbn_node = root.xpath('//div[@class="specs"]/h3[text()[contains(.,"Productinformatie")]]/following-sibling::dl/dt[text()[contains(.,"EAN")]]/following-sibling::dd[1]')
        #print('isbn_node3', isbn_node)
        if isbn_node:
            for node in isbn_node:
                isbn=node.text_content().strip()
                #print('isbn3', isbn)
                return isbn                                              
        isbn_node = root.xpath('//div[@class="specs"]/h3[text()[contains(.,"EAN")]]/following-sibling::dl/div/dt[text()[contains(.,"EAN")]]/following-sibling::dd[1]')
        #print('isbn_node4', isbn_node)
        if isbn_node:
            for node in isbn_node:
                isbn=node.text_content().strip()
                #print('isbn3', isbn)
                return isbn
        isbn_node = root.xpath('///script[@type="application/json" and @data-payload-id="off-canvas-specifications-slot"]/text()')
        #print('isbn_node-script', isbn_node)
        if isbn_node:
            json_data = isbn_node[0]  # Extract the JSON string
            parsed_data = json.loads(json_data)  # Parse it into a Python dictionary

            # Extract "EAN" value from the parsed JSON data
            ean = None
            for group in parsed_data.get("specs", {}).get("groups", []):
                for item in group.get("items", []):
                    if item.get("label") == "EAN":
                        ean = item.get("values", [{}])[0].get("value", "")
            return ean

            # Output the extracted EAN value
            #print("EAN:", ean_value)
       

    def parse_rating(self, root):
        rating_node= root.xpath('//div [@class="wrapper sub_title_meta small_details fleft"]/div [@id="bazaarvoice_reviews_header_summary"]/a')
        #rating_node = root.xpath('//div [@id="bazaarvoice_reviews_header_summary"]/span/img/@src')
        if rating_node:
            txt= rating_node.text_content().strip()
            rating_class = rating_node[0]
            match = re.search('product-rating (.+)', rating_class)
            if match:
                rating_text = match.groups(0)[0]
                rating_parts = rating_text.split(' ')
                rating_values = ['zero','one','two','three','four','five']
                rating_value = float(rating_values.index(rating_parts[0]))
                if len(rating_parts) > 1:
                    rating_value += 0.5
                return rating_value
        else:
            rating_node = root.xpath('//span[contains(@class,"section_updateRating")]/@class')
            if rating_node:
                rating_text = rating_node[0][4:6]
                rating_value = float(rating_text[0])
                if rating_text[1] == 'h':
                    rating_value += 0.5
                return rating_value

    def parse_language(self, root):
        taal = None
        lang_node = root.xpath('//div [@class="first_tab_paragraph"]/label')
        #print('lang_node:',lang_node)
        if lang_node:
            for label in lang_node:
                txt=label.text_content().strip()
                find=txt.find('Taal:')
                if find >=0:
                    lang=txt.rpartition(':')[2].strip()
                    print('langa:', lang)
                    self.log.info("_parse_language: lang=", lang)
                    ans = self.lang_map.get(lang, None)
                    self.log.info("_parse_language: ans=", ans)
                    if ans:
                        return ans
                    ans = canonicalize_lang(lang)
                    self.log.info("_parse_language: ans=", ans)
                    if ans:
                        return ans
                    break
        lang_node = root.xpath('//table [@class="boxedbottom_s"]/tbody/tr')
        #print('lang_node2:',lang_node)
        if lang_node:
                for label in lang_node:
                    txt=label.text_content().strip()
                    find=txt.find('Taal:')
                    if find >=0:
                        lang=txt.rpartition(':')[2].strip()
                        #print('langb:', lang)
                        self.log.info("_parse_language: lang=", lang)
                        ans = self.lang_map.get(lang, None)
                        self.log.info("_parse_language: ans=", ans)
                        if ans:
                            return ans
                        ans = canonicalize_lang(lang)
                        self.log.info("_parse_language: ans=", ans)
                        if ans:
                            return ans
                        break
        lang_node= root.xpath('//script[@type="application/json" and @data-payload-id="off-canvas-specifications-slot"]/text()')
        #print('lang_node script: ',lang_node)
        if lang_node:
            json_data = lang_node[0]  # Extract the JSON string
            parsed_data = json.loads(json_data)  # Parse it into a Python dictionary

            # Extract the "Taal" value from the parsed JSON data
            lang = None
            for group in parsed_data.get("specs", {}).get("groups", []):
                for item in group.get("items", []):
                    if item.get("label") == "Taal":
                        lang = item.get("values", [{}])[0].get("value", "")

            # Output the extracted Taal value
            #print("Taal:",lang)
            ans = self.lang_map.get(lang, None)
            self.log.info("_parse_language: ans=", ans)
            if ans:
                return ans
            ans = canonicalize_lang(lang)
            self.log.info("_parse_language: ans=", ans)
            if ans:
                return ans
           
        
            
        lang_node = root.xpath('//div [@class="specs"]/dl [@class="specs__list"]/dt  [@class="specs__title" and contains(.,"Taal")]//following-sibling::dd [@class="specs__value"]')
        #print('lang_node3: ',lang_node)
        if lang_node:
                for label in lang_node:
                    found=False
                    txt=label.text_content().strip()
                    find=txt.find('Nederlands')
                    if find>=0:
                        lang=txt
                        found=True
                    find=txt.find('Engels')
                    if find>=0:
                        lang=txt 
                        found=True
                    find=txt.find('Frans')
                    if find>=0:
                        lang=txt
                        found=True
                    find=txt.find('Duits')
                    if find>=0:
                        lang=txt
                        found=True
                    if found:
                        #print('langc:', lang)
                        self.log.info("_parse_language: lang=", lang)
                        ans = self.lang_map.get(lang, None)
                        self.log.info("_parse_language: ans=", ans)
                        if ans:
                            return ans
                        ans = canonicalize_lang(lang)
                        #self.log.info("_parse_language: ans=", ans)
                        if ans:
                            return ans
                        break

    def parse_tags(self, root):
        # BOL_NL has multiple optional sections which can be used as tags depending on the user's preference.
        calibre_tags = list()
        if cfg.plugin_prefs[cfg.STORE_NAME][cfg.KEY_GET_CAT_AS_TAGS]:
            cat_list = list ()
            cat_list.append('Actie & Avontuur')
            cat_list.append('Kind & Jeugd')
            cat_list.append('Literatuur')
            cat_list.append('Romans algemeen')
            cat_list.append('Humoristische romans')
            cat_list.append('Historische romans')
            cat_list.append('Erotische romans')
            cat_list.append('Feelgood romans')
            cat_list.append('Literaire romans')
            cat_list.append('Streekromans')
            cat_list.append('Literaire thrillers')
            cat_list.append('Literaire non-fictie')
            cat_list.append('Baby- & Peuterboeken')
            cat_list.append('Prentenboeken')
            cat_list.append('Zelf lezen')
            cat_list.append('Young Adult')
            cat_list.append('Sprookjesboeken')
            cat_list.append('Fantasy> Meer fantasy')
            cat_list.append('Horror> Meer Horror')
            cat_list.append('Thrillers & Fantasy')
            cat_list.append('Thrillers & Spanning')
            cat_list.append('True Crime')
            cat_list.append('Spanning')
            cat_list.append('Thrillers')
            cat_list.append('Detectives')
            cat_list.append('Mystery & Detective')
            cat_list.append('Meer mysteries & detectives')
            cat_list.append('Gezondheid & Psychologie')
            cat_list.append('Religie & Esoterie')
            cat_list.append('Eten & Koken')
            cat_list.append('Reizen & Talen')
            cat_list.append('Geschiedenis & Politiek')
            cat_list.append('Kunst & Cultuur')
            cat_list.append('Sport & Hobby')
            cat_list.append('Wonen & Tuinieren')
            cat_list.append('Zakelijk & Studie')
            cat_list.append('Business')
            cat_list.append('Computer')
            cat_list.append('School & Examen')
            cat_list.append('Studieboeken ')
            self._append2_cat_tags(root, cat_list, calibre_tags)

        if cfg.plugin_prefs[cfg.STORE_NAME][cfg.KEY_GET_AGE_AS_TAGS]:
            self._append_tags(root, 'Leeftijd', calibre_tags)

        if len(calibre_tags) > 0:
            return calibre_tags


    def _append2_cat_tags(self, root, cat_list, calibre_tags):
        tags_nodes=root.xpath('//dl/dd/ul/li/a');
        tags_nodes=root.xpath('//dl [@class="specs__list"]/dt  [@class="specs__title" and contains(.,"Categorieën")]//following-sibling::dd [@class="specs__value"]/ul/li/a')
        #print ('tag_nodes2:',tags_nodes)
        if not tags_nodes:
            tags_nodes=root.xpath('//dl [@class="specs__list"]/div/dt  [@class="specs__title" and contains(.,"Categorieën")]//following-sibling::dd [@class="specs__value"]/ul/li/a')
            #print ('tag_nodes3:',tags_nodes)
        if tags_nodes:
            for tags_node in tags_nodes:
                txt=tags_node.text_content().strip()
                #print ('tag:',txt)
                if txt in cat_list:
                    #print('found:', txt)
                    if txt not in calibre_tags:
                        calibre_tags.append(txt)



    def _append_tags(self, root, group, calibre_tags):
        tags_nodes=root.xpath('//dt [@class="specs__title" and contains(.,"Leeftijd")]/following-sibling::dd[1]/a');
        #print('leeftijd:', tags_nodes)
        if tags_nodes:
            for tags_node in tags_nodes:
                tag=tags_node.text_content().strip()
                if tag not in calibre_tags:
                    calibre_tags.append(tag)

    def parse_publisher(self, root):
        publisher = None
        #print("root:",root.text_content().strip())
        pub_node = root.xpath('//ul [@class="bullet-list"]/li [@data-attr-key="PUBLIS"]')
        #print("publisher_node1:",pub_node )
        if pub_node:
            publisher=pub_node[0].text_content().strip()
        if publisher == None:
             bind_node=root.xpath('//div[@class="specs__party-groups"]/div[text()[contains(.,"Uitgever")]]/a')
             #print("publisher_node3:",bind_node )
             if bind_node:

                publisher=bind_node[0].text_content().strip()
                #print("publisher:",publisher )

        if publisher == None:
            bind_node=root.xpath('//span [@class="product_platform"]')
            #print("publisher_node2:",bind_node )
            pos=3
            if bind_node:
                txt=bind_node[0].text_content().strip()
                pos=txt.find('Ebook');
                if pos>1:
                    pos=2
                else:
                    pos=3
            txtpath='%s%s%s'%('//ul [@class="default_listitem"]/li[',pos,']')
            pub_node = root.xpath(txtpath)
            #print("publisher_node2a:",pub_node )
            if pub_node:
                publisher=pub_node[0].text_content().strip()
                dum=publisher.find('pagina')
                if dum>-1:
                    self.ebook=True
                    txtpath='%s%s%s'%('//ul [@class="default_listitem"]/li[',3,']')
                    pub_node = root.xpath(txtpath)
                    if pub_node:
                        publisher=pub_node[0].text_content().strip()
        if publisher == None:
            bind_node=root.xpath('//div[@data-test="specifications"]/div/div/div/div[@class="specs"]/dl/dt[text()[contains(.,"Uitgever")]]/./following-sibling::dd')
            print("publisher_node3:",bind_node )
            #print(bind_node[0].text_content().strip())
            if bind_node:
                publisher=bind_node[0].text_content().strip()
        if publisher == None:
            bind_node=root.xpath('//div[@class="specs"]/dl/div/dt[text()[contains(.,"Uitgever")]]/./following-sibling::dd')
            #print("publisher_node4:",bind_node )
            # print(bind_node[0].text_content().strip())
            if bind_node:
                publisher=bind_node[0].text_content().strip()
        if publisher == None:
            
            bind_node=root.xpath('//div[@class="specs"]/dl/div/dt[text()[contains(.,"Hoofduitgeverij")]]/./following-sibling::dd')
            #print("publisher_node5:",bind_node )
            #print(bind_node[0].text_content().strip())
            if bind_node:
                publisher=bind_node[0].text_content().strip()
        if publisher == None:
            
            bind_node=root.xpath('//script[@type="application/json" and @data-payload-id="off-canvas-specifications-slot"]/text()')
            #print("publisher_node6:",bind_node )
            if bind_node:
                json_data = bind_node[0]  # Extract the JSON string
                parsed_data = json.loads(json_data)  # Parse it into a Python dictionary

                # Extract "Vrijdag, Uitgeverij" from the parsed JSON data
                
                for group in parsed_data.get("specs", {}).get("groups", []):
                    for item in group.get("items", []):
                        if item.get("label") == "Hoofduitgeverij":
                            publisher = item.get("values", [{}])[0].get("value", "")

                # Output the extracted publisher
                #print("Hoofduitgeverij:", publisher)
            
        if publisher == None:
            publisher=""


        return(publisher)

    def parse_published_date(self, root):
        pub_date = None
        pub_node = root.xpath('//ul [@class="bullet-list"]/li [@data-attr-key="DATE"]')
        #print('pub_node1',pub_node)
        if pub_node:
            pub_date= pub_node[0].text_content().strip()
        if not(pub_date is None):
            return  self._convert_date_text(pub_date.strip())
        else:
            bind_node=root.xpath('//span [@class="product_platform"]')

            pos=4
            if bind_node:
                txt=bind_node[0].text_content().strip()
                pos=txt.find('Ebook');
                if pos>1:
                    pos=3
                else:
                    pos=4
                if self.ebook:
                    pos=4
            txtpath='%s%s%s'%('//ul [@class="default_listitem"]/li[',pos,']')
            pub_node = root.xpath(txtpath)
            #print('pub_node2',bind_node)
            if pub_node:
                pub_date=pub_node[0].text_content().strip()
                if (re.search(r'[0-9]{4,}', pub_date) is None):
                    pos+=1
                    txtpath='%s%s%s'%('//ul [@class="default_listitem"]/li[',pos,']')
                    pub_node = root.xpath(txtpath)
                    if pub_node:
                        pub_date=pub_node[0].text_content().strip()
                        p =re.search(r'[0-9]{4,}', pub_date)
                        if( p is None):
                            pub_date= None
                            
                        
                            
            if not(pub_date is None):
                return  self._convert_date_text(pub_date.strip())
        pub_node=root.xpath('//div [@class="specs"]/dl [@class="specs__list"]/dt  [@class="specs__title" and contains(.,"Verschijningsdatum")]//following-sibling::dd [@class="specs__value"]')
        #print('pub_node3',pub_node)
        if pub_node:
            pub_date= ElementTree.tostring(pub_node[0],method='text').decode();
            return self._convert_date_text(pub_date.strip())
        pub_node= root.xpath('//ul[@class="product-small-specs product-small-specs--large"]/li/meta[@itemprop="datePublished"]/@content')
        #print('pub_node4',pub_node)
        if pub_node:
            pub_date=ElementTree.tostring(pub_node[0],method='text').decode();
            #print('pub_date:', pub_date)
            return  self._convert_date_text(pub_date.strip())
        pub_node= root.xpath('//div[@class="specs"]/dl/div/dt[text()[contains(.,"releasedatum")]]/./following-sibling::dd')
        #print('pub_node5',pub_node)
        if pub_node:
            pub_date=ElementTree.tostring(pub_node[0],method='text').decode();
            #print('pub_date:', pub_date)
            return  self._convert_date_text(pub_date.strip())
        pub_node= root.xpath('//script[@type="application/json" and @data-payload-id="off-canvas-specifications-slot"]/text()')
        #print('pub_node script',pub_node)
        if pub_node:
            json_data = pub_node[0]  # Extract the JSON string
            parsed_data = json.loads(json_data)  # Parse it into a Python dictionary

            # Extract the "Oorspronkelijke releasedatum" value from the parsed JSON data
            pub_date = None
            for group in parsed_data.get("specs", {}).get("groups", []):
                for item in group.get("items", []):
                    if item.get("label") == "Oorspronkelijke releasedatum":
                        pub_date = item.get("values", [{}])[0].get("value", "")

            # Output the extracted release date
            #print("Oorspronkelijke releasedatum:", pub_date)
            return  self._convert_date_text(pub_date.strip())
                    

    def _convert_date_text(self, date_text):
        from calibre.utils.date import utc_tz
        # Note that the date text could be "2003", "december 2003" or "December 10th 2003"
        date_text=date_text.strip()
        #print('pub_date:', date_text,"lengte:" ,len(date_text))
        #year,month,day=date_text.split("-");
        ##print("datum", year,month,day)
        pattern= re.compile("^\d\d\d\d-\d?\d-\d?\d")
        #print("pattern", pattern)
        if pattern.match(date_text):           
            #print('pub_date:', date_text)
            parts=date_text.split("-");
            year= int(parts[0])
            month=int(parts[1])
            day=int(parts[2])
            #print("datum", year,month,day)
            return datetime.datetime(year, month, day, tzinfo=utc_tz) #FOUND
        pattern= re.compile("^\d?\d-\d?\d-\\d\d\d\d")
        #print("pattern2", pattern)
        if pattern.match(date_text):           
            #print('pub_date2:', date_text)
            parts=date_text.split("-");
            year= int(parts[2])
            month=int(parts[1])
            day=int(parts[0])
            #print("datum2", year,month,day)
            return datetime.datetime(year, month, day, tzinfo=utc_tz) #FOUND
        pattern= re.compile("^\d\d\d\d")
        #print("pattern3", pattern)
        if pattern.match(date_text):           
            #print('pub_date3:', date_text)
            year= int(date_text)
            month=1
            day=2
            #print("datum3", year,month,day)
            return datetime.datetime(year, month, day, tzinfo=utc_tz) #FOUND
        month_dict = {"januari":1, "februari":2, "maart":3, "april":4, "mei":5, "juni":6,
        "juli":7, "augustus":8, "september":9, "oktober":10, "november":11, "december":12}
        date_text=date_text.lower()
        #print ( date_text.islower())
        pattern= re.compile("^[a-zA_Z]+\s\d\d\d\d")
        #print("pattern4", pattern)
        if pattern.match(date_text):
            parts=date_text.split(" ");
            #print(parts, len(parts))
            year=int (parts[1])
            month_name = parts[0]
            month = month_dict.get(month_name, 1)
            day=2
            #print("datum4", year,month,day) 
            return datetime.datetime(year, month, day, tzinfo=utc_tz) #FOUND
        pattern= re.compile("^\d?\d\s[a-zA_Z]+\s\d\d\d\d")
        #print("pattern5", pattern)
        if pattern.match(date_text):
            parts=date_text.split(" ");
            year=int(parts[2])
            month_name = parts[1]
            month = int(month_dict.get(month_name, 1))
            day= int(parts[0])
            #print("datum5", year,month,day) 
            return datetime.datetime(year, month, day, tzinfo=utc_tz) #FOUND
            
        year=2000
        month = 1
        day = 2 # to prevent timezone  problems, p.e. if day =1 then in calibre it shows the predious month
        return datetime.datetime(year, month, day, tzinfo=utc_tz)



    def parse_comments(self, root):
        comments = None
        description_node = root.xpath('//div [@class="product-description"] [@data-test="description"]')
        #print('description_node1:',description_node)
        if description_node:
            #comments_org = tostring(description_node[0])
            comments_org = ElementTree.tostring(description_node[0]).decode()
            comments=comments_org
            ##print('comments:',comments)
            pos=comments.find('<p itemprop="description">')
            #print('pos:',pos)
            if pos>0:
                comments=comments[pos+26:].strip()
            comments = comments.replace('<div data-test="description" class="product-description">','').strip()
            comments = comments.replace('<h2 class="bol_header h3 bottom_s divide_bottom">Samenvatting</h2>','').strip()
            comments = comments.replace('<h2 class="bol_header h3 bottom_s divide_bottom">Productbeschrijving</h2>','').strip()
            comments = comments.replace('<div class="content content_box_edge_r big">','').strip()
            comments=comments.replace('<h2 class="h5 bottom_xxs">Beschrijving</h2>','')
            comments=comments.replace('<div class="bottom_m">','')
            comments=comments.replace('<h2>','')
            comments=comments.replace('<strong>','')
            comments=comments.replace('</strong>','<br>')
            comments=comments.replace('</h2>','<br>')
            #comments=comments.replace('</p>','<br>')
            comments=comments.replace('<br><br>','<br>')
            comments=comments.replace('<div class="clear_autoheight"></div>','')
            end=comments.find('</div>')
            sub_end=comments.find('<h3 class="h-top--s">Recensie(s)</h3>')

            if end>0:
                if sub_end>0 and sub_end<end and sub_end>-1:
                    comments=comments[0:sub_end]
                else:
                    comments=comments[0:end]

            sub_end=comments.find('<br>Ook verkrijgbaar in')
            if sub_end>0:
                comments=comments[0:sub_end]

            #print('comments2:',comments)
            if self.recensies_bol:
                #print('comments2a:',comments)
                recensies_node=description_node = root.xpath('//div [@class="content"]/div')
                if recensies_node:
                    #print('comments2b:',comments)
                    #recensies_org = tostring(recensies_node[0])
                    recensies_org = ElementTree.tostring(recensies_node[0]).decode()
                    start= recensies_org.find('<h2 class="h4 bottom_xxs">Recensie(s)</h2>')
                    if  start:
                        comments2=recensies_org[start:]
                        comments2=comments2.replace('<h2 class="bol_header h3 bottom_s divide_bottom">Samenvatting</h2>','').strip()
                        comments2=comments2.replace('<strong>','')
                        comments2=comments2.replace('</strong>','')
                        comments2=comments2.replace('<h2 class="h5 bottom_xxs">Recensie(s)</h2>','<strong>Recensie(s)</strong><br><br>')
                        comments2=comments2.replace('<h2 class="h4 bottom_xxs">Recensie(s)','<strong>Recensie(s)</strong><br><br>')
                        comments2=comments2.replace('</h2>','')
                        comments2=comments2.replace('\n',' ')

                        comments2=comments2.replace('<div class="clear_autoheigh">','')

                        end=comments2.find('</div>')
                        if end>0:
                            comments2=comments2[0:end]
                        comments=comments+comments2
                else:
                    #print('comments2c:',comments)
                    recensies_node=description_node = root.xpath('//div [@class="professional-review"][@data-test="review-content"]')
                    if recensies_node:
                        #print('comments2d:',comments)
                        recensie= ElementTree.tostring(recensies_node[0], method='xml').decode();
                        #comments2='<br><strong>Recensie(s)</strong><br><br>' +recensies_node[0].text_content().strip()
                        comments2='<br><strong>Recensie(s)</strong><br><br>' +recensie.strip()
                        comments=comments+comments2

            comments= comments + '<br>(source: Bol.com)<br>'
            #print('comments3:',comments)
        if comments:
            pos=comments.find('<div class="clear_autoheight">')
            comments = comments.replace('<div class="clear_autoheight">','').strip()
            comments = comments.replace('&lt;br/&gt;&lt;br/&gt;','<br>')
            #print('comments4:',comments)
            return comments
        description_node = root.xpath('//div [@class="js_readmore_content bottom_l" and @itemprop="description"]    /div/div[ @class="content content_box_edge_r big"]')
        #print('description_node2:',description_node)
        if description_node:
            comments_org = tostring(description_node[0])
            comments=comments_org
            comments = comments.replace('<div class="content content_box_edge_r big">','').strip()
            comments = comments.replace('<div data-test="description" class="product-description">','').strip()
            comments=comments.replace('<h2 class="h5 bottom_xxs">Beschrijving</h2>','')
            comments=comments.replace('<div class="bottom_m">','')
            comments=comments.replace('<h2>','')
            comments=comments.replace('<strong>','')
            comments=comments.replace('</strong>','')
            comments=comments.replace('<h2 class="bol_header h3 bottom_s divide_bottom">Samenvatting</h2>','').strip()
            comments=comments.replace('<h2 class="h5 bottom_xxs">Recensie(s)</h2>','<strong>Recensie(s)</strong><br><br>')
            comments=comments.replace('<h2 class="h4 bottom_xxs">Recensie(s)</h2>','<strong>Recensie(s)</strong><br><br>')
            comments=comments.replace('</h2>','')

            comments=comments.replace('\n',' ')
            end=comments.find('<h2 ')
            if end>0:
                comments=comments[0:end]
            else:
                end=comments.find('/div>')
                if end>0:
                    comments=comments[0:end]
            start=comments_org.find('<h2 class="h5 bottom_xxs">Recensie(s)</h2>')
            if start<0:
                start=comments_org.find('<h2 class="h4 bottom_xxs">Recensie(s)</h2>')

            if start>0:
                if  self.recensies_bol:
                    comments2=comments_org[start:]
                    comments2=comments2.replace('<h2 class="bol_header h3 bottom_s divide_bottom">Samenvatting</h2>','').strip()
                    comments2=comments2.replace('<h2 class="h5 bottom_xxs">Recensie(s)</h2>','<strong>Recensie(s)</strong><br><br>')
                    comments2=comments2.replace('<h2 class="h4 bottom_xxs">Recensie(s)</h2>','<strong>Recensie(s)</strong><br><br>')

                    comments2 =comments2.replace('\n',' ')
                    end=comments2.find('</div>')
                    if end>0:
                        comments2=comments2[0:end]
                    comments=comments+comments2
            else:
                if  self.recensies_bol:
                    recensies_node=description_node = root.xpath('//div [@class="professional-review"][@data-test="review-content"]')
                    if recensies_node:
                        comments2='<br><strong>Recensie(s)</strong><br><br>' +recensies_node[0].text_content().strip()
                        comments=comments+comments2
            comments= comments + '<br>(source: Bol.com)<br>'
        if comments:
            return comments

        description_node = root.xpath('//div[@itemprop="description" and @class="product-description"]')
        #print('description_node3:',description_node)
        if description_node:
            comments_org=''
            tel=1
            for d_node in  description_node:
                #print(tel, " ",tostring(d_node))
                comments_org += tostring(d_node)
                tel+=1
            comments=comments_org
            comments = comments.replace('<div data-test="description" class="product-description">','').strip()
            commentsa=re.sub('<div[^<]*>', '', comments)
            commentsb=re.sub('<\div[^<]*>', '', commentsa)
            commentsc=re.sub('<p[^<]*>', '', commentsb)
            comments=commentsc

            comments=comments.replace('\n',' ')
            end = comments.find('<h3 class="h-top--s">Recensie(s)')
            if end>0:
                comments=comments[0:end]
            start=comments_org.find('<h3 class="h-top--s">Recensie(s)</h3>')
            if start<0:
                start=comments_org.find('<h2 class="h-top--s">Recensie(s)</h2>')

            if start>0:
                if  self.recensies_bol:
                    comments2=comments_org[start:]
                    comments2=comments2.replace('<h3 class="h-top--s">Recensie(s)</h3>','<br><br><strong>Recensie(s)</strong><br><br>')
                    comments2=comments2.replace('<h2 class="h-top--s">Recensie(s)</h2>','<br><br><strong>Recensie(s)</strong><br><br>')

                    comments2 =comments2.replace('\n',' ')
                    end=comments2.find('</div>')
                    if end>0:
                        comments2=comments2[0:end]
                    comments=comments+comments2
            else:
                if  self.recensies_bol:
                    recensies_node=description_node = root.xpath('//div [@class="professional-review"][@data-test="review-content"]')
                    if recensies_node:
                        comments2='<br><strong>Recensie(s)</strong><br><br>' +recensies_node[0].text_content().strip()
                        comments=comments+comments2
            comments= comments + '<br><br>(source: Bol.com)<br>'
            comments = comments.replace('<div data-test="description" class="product-description">','').strip()
            comments=comments.replace('</div>','').strip()
            comments=comments.replace('</p>','').strip()
            comments=comments.replace('<div>','').strip()
            comments=comments.replace('<p>','').strip()
            if comments:
                #print(comments)
                return comments
        description_node = root.xpath('//div[@id="root-off-canvas-description-slot"]/button/div[2]')
        #print('description_node4:',description_node)
        if description_node:
            #comments_org = tostring(description_node[0])
            comments_org = ElementTree.tostring(description_node[0]).decode()
            comments=comments_org
            ##print('comments:',comments)
            pos=comments.find('<p itemprop="description">')
            #print('pos:',pos)
            if pos>0:
                comments=comments[pos+26:].strip()
            comments = comments.replace('<div data-test="description" class="product-description">','').strip()
            comments = comments.replace('<h2 class="bol_header h3 bottom_s divide_bottom">Samenvatting</h2>','').strip()
            comments = comments.replace('<h2 class="bol_header h3 bottom_s divide_bottom">Productbeschrijving</h2>','').strip()
            comments = comments.replace('<div class="content content_box_edge_r big">','').strip()
            comments=comments.replace('<h2 class="h5 bottom_xxs">Beschrijving</h2>','')
            comments=comments.replace('<div class="bottom_m">','')
            comments=comments.replace('<h2>','')
            comments=comments.replace('<strong>','')
            comments=comments.replace('</strong>','<br>')
            comments=comments.replace('</h2>','<br>')
            #comments=comments.replace('</p>','<br>')
            comments=comments.replace('<br><br>','<br>')
            comments=comments.replace('<div class="clear_autoheight"></div>','')
            end=comments.find('</div>')
            sub_end=comments.find('<h3 class="h-top--s">Recensie(s)</h3>')

            if end>0:
                if sub_end>0 and sub_end<end and sub_end>-1:
                    comments=comments[0:sub_end]
                else:
                    comments=comments[0:end]

            sub_end=comments.find('<br>Ook verkrijgbaar in')
            if sub_end>0:
                comments=comments[0:sub_end]

            #print('comments2:',comments)
            if self.recensies_bol:
                #print('comments2a:',comments)
                recensies_node=description_node = root.xpath('//div [@class="content"]/div')
                if recensies_node:
                    #print('comments2b:',comments)
                    #recensies_org = tostring(recensies_node[0])
                    recensies_org = ElementTree.tostring(recensies_node[0]).decode()
                    start= recensies_org.find('<h2 class="h4 bottom_xxs">Recensie(s)</h2>')
                    if  start:
                        comments2=recensies_org[start:]
                        comments2=comments2.replace('<h2 class="bol_header h3 bottom_s divide_bottom">Samenvatting</h2>','').strip()
                        comments2=comments2.replace('<strong>','')
                        comments2=comments2.replace('</strong>','')
                        comments2=comments2.replace('<h2 class="h5 bottom_xxs">Recensie(s)</h2>','<strong>Recensie(s)</strong><br><br>')
                        comments2=comments2.replace('<h2 class="h4 bottom_xxs">Recensie(s)','<strong>Recensie(s)</strong><br><br>')
                        comments2=comments2.replace('</h2>','')
                        comments2=comments2.replace('\n',' ')

                        comments2=comments2.replace('<div class="clear_autoheigh">','')

                        end=comments2.find('</div>')
                        if end>0:
                            comments2=comments2[0:end]
                        comments=comments+comments2
                else:
                    #print('comments2c:',comments)
                    recensies_node=description_node = root.xpath('//div [@class="professional-review"][@data-test="review-content"]')
                    if recensies_node:
                        #print('comments2d:',comments)
                        recensie= ElementTree.tostring(recensies_node[0], method='xml').decode();
                        #comments2='<br><strong>Recensie(s)</strong><br><br>' +recensies_node[0].text_content().strip()
                        comments2='<br><strong>Recensie(s)</strong><br><br>' +recensie.strip()
                        comments=comments+comments2

            comments= comments + '<br>(source: Bol.com)<br>'
            #print('comments3:',comments)
        if comments:
            pos=comments.find('<div class="clear_autoheight">')
            comments = comments.replace('<div class="clear_autoheight">','').strip()
            comments = comments.replace('&lt;br/&gt;&lt;br/&gt;','<br>')
            #print('comments4:',comments)
            return comments       
               
        comments=''
        if  self.recensies_bol:
                    recensies_node=description_node = root.xpath('//div [@class="professional-review"][@data-test="review-content"]')
                    if recensies_node:
                        comments2='<br><strong>Recensie(s)</strong><br><br>' +recensies_node[0].text_content().strip()
                        comments=comments+comments2
                        comments= comments + '<br><br>(source: Bol.com)<br>'      
        return comments



    def parse_cover(self, root):
        if self.lponly_covers or self.lp_covers:
            if self.isbn:
                search_url='http://www.literatuurplein.nl/zoeken-resultaat.jsp?x=38&y=8&isbn=' + self.isbn
                #print('url:',search_url)
                br1=self.browser
                timeout=30
                error = False
                try:
                    response = br1.open_novisit(search_url, timeout=timeout).read().strip()
                except Exception as e:

                    if callable(getattr(e, 'getcode', None)) and \
                            e.getcode() == 404:
                        self.log.error('URL malformed: %r'%self.url)
                        ##print('ex:',e)
                        error = True

                if not error:
                    raw = response.decode('utf-8', errors='replace')
                    if '<title>404 - ' in raw:
                        self.log.error('URL cover literaturplein malformed: %r'%self.url)
                        return
                    try:
                        rawroot = fromstring(clean_ascii_chars(raw))
                    except Exception as e:
                        return
                    result_node=rawroot.xpath('//div [@class="content content-header"]/div/img [@class="item-header-image keepAspect"]/@data-original')
                    #print('result_node cover:', result_node)
                    if result_node:
                        url="".join(result_node)
                        self.plugin.cache_identifier_to_cover_url(self.bol_nl_id,url)
                        return url
        if self.lponly_covers:
            return

        result_node=root.xpath('//div [@class="product-image--content"]/div/img/@src')
        print('result_node0',result_node)
        if result_node:
            img_url = result_node[0]
            self.plugin.cache_identifier_to_cover_url(self.bol_nl_id, img_url)
            return img_url

        result_node=root.xpath('//img [@class="js_product_img"]/@src')
        print('result_node1',result_node)
        if result_node:
            img_url = result_node[0]
            img_url = img_url.replace('regular','large')
            self.plugin.cache_identifier_to_cover_url(self.bol_nl_id, img_url)
            return img_url
        #geen  kijkexemplaar
        result_node=root.xpath('//div [@class="product_image"]/img/@src')
        print('result_node2',result_node)
        if result_node:
            img_url=result_node[0]
            pos=img_url.find('noimage')
            if pos<0:
                self.plugin.cache_identifier_to_cover_url(self.bol_nl_id, img_url)
                return img_url
        result_node=root.xpath('//div [@class="carousel_media"]/div/img/@src')
        print('result_node3',result_node)
        if result_node:

            img_url = result_node[0]
            #img_url = img_url.replace('regular','large')
            self.plugin.cache_identifier_to_cover_url(self.bol_nl_id, img_url)
            return img_url

        result_node=root.xpath('//div [@class="carousel_media"]/div/div/img/@src')
        print('result_node4',result_node)
        if result_node:
            img_url = result_node[0]
            #img_url = img_url.replace('regular','large')
            self.plugin.cache_identifier_to_cover_url(self.bol_nl_id, img_url)
            return img_url
        result_node=root.xpath('//div [@id="main_product_image_container"]/div/img/@src')
        print('result_node5',result_node)
        if result_node:
            img_url = result_node[0]
            #img_url = img_url.replace('regular','large')
            self.plugin.cache_identifier_to_cover_url(self.bol_nl_id, img_url)
            return img_url
        tempurl= self.url
        parts= tempurl.split('/')
        aantal= len(parts)
        #print (aantal,'parts:',parts)
        imgnr=parts[aantal-2]
        #print (imgnr)
        result_node=root.xpath('//meta [@property="og:image"]/@content')
        print ('result cover1:',result_node)
        
        if result_node:
            img_url = result_node[0]
            print('img_url:',img_url)
            #img_url = img_url.replace('regular','large')
            self.plugin.cache_identifier_to_cover_url(self.bol_nl_id, img_url)
            return img_url  
        result_node=root.xpath('//div[@class="book book--large"]/div[@class="book__cover"]/img [@class="book__cover-image"]/@src')
        print ('result cover2:',result_node)
        if result_node:
            img_url = result_node[0]
            img_url = img_url.replace('regular','large')
            self.plugin.cache_identifier_to_cover_url(self.bol_nl_id, img_url)
            return img_url
        result_node=root.xpath('//div[@class="book book--xl"]/div[@class="book__cover"]/img [@class="book__cover-image"]/@src')
        print ('result cover3:',result_node)
        if result_node:
            img_url = result_node[0]
            #img_url = img_url.replace('regular','large')
            self.plugin.cache_identifier_to_cover_url(self.bol_nl_id, img_url)
            return img_url
        result_node=root.xpath('//div[@class="book__cover"]/img [@class="book__cover-image"]/@src')
        print ('result cover4:',result_node)
        if result_node:
            img_url = result_node[0]
            #img_url = img_url.replace('regular','large')
            self.plugin.cache_identifier_to_cover_url(self.bol_nl_id, img_url)
            return img_url
        result_node=root.xpath('//div[@class="image-slot"]/img [@class="js_selected_image has-zoom"]/@data-zoom-image-url')
        print ('result cover5:',result_node)
        if result_node:
            img_url = result_node[0]
            print('img url:',img_url)
            #img_url = img_url.replace('regular','large')
            self.plugin.cache_identifier_to_cover_url(self.bol_nl_id, img_url)
            return img_url
        result_node=root.xpath('//div[@class="image-slot"]/img [@class="js_selected_image"]/@src')
        print ('result cover6:',result_node)
        if result_node:
            img_url = result_node[0]
            print('img url:',img_url)
            #img_url = img_url.replace('regular','large')
            self.plugin.cache_identifier_to_cover_url(self.bol_nl_id, img_url)
            return img_url
        result_node=root.xpath('//wsp-selected-item-image-zoom-modal-application[@class="h-pointer"]/button/img [@id="image-zoom-modal-selected-image"]/@src')
        print ('result cover7:',result_node)
        if result_node:
            img_url = result_node[0]
            print('img url:',img_url)
            #img_url = img_url.replace('regular','large')
            self.plugin.cache_identifier_to_cover_url(self.bol_nl_id, img_url)
            return img_url
            
        