#!/usr/bin/env python
# -*- coding: utf-8 -*-
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import, print_function)

__license__   = 'GPL v3'
__copyright__ = '2014, Roman Cupisz <roman.cupisz+calibre@gmail.com>, 2015-2025 enhancements by Becky <becky@fr.pl>'
__docformat__ = 'restructuredtext en'

PLUGINVER = (2, 3, 6)

import ctypes  # An included library with Python install.
import time, re, sys

try:
    from urllib.parse import quote
except ImportError:
    from urllib import quote

try:
    # For Python 3.0 and later
    from urllib.request import Request, urlopen
except ImportError:
    # Fall back to Python 2's urllib2
    from urllib2 import Request, urlopen

try:
    from queue import Empty, Queue
except ImportError:
    from Queue import Empty, Queue
from six import text_type as unicode

from lxml.html import fromstring

from calibre import as_unicode
from calibre import prints
from calibre.ebooks.metadata.sources.base import Source
from calibre.utils.icu import lower
from calibre.utils.cleantext import clean_ascii_chars
from calibre.utils.localization import get_udc
from calibre.constants import numeric_version

# from calibre.ebooks.metadata import check_isbn

from calibre.constants import DEBUG

# PyQt libraries
try:
    from qt.core import (QtCore, QtGui, QtWidgets, QApplication, QAction,
                        QMessageBox, QDialog, Qt,
                        QMenu, QHeaderView, QLabel, QTableWidget, QIcon,
                        QPixmap, QSize, QAbstractItemView, QTableWidgetItem,
                        QPushButton, QHBoxLayout, QVBoxLayout, QPalette,
                        QColor)
except ImportError:
    from PyQt5 import QtCore, QtGui, QtWidgets
    from PyQt5.Qt import (QApplication, QAction, QMessageBox, QDialog, Qt,
                        QMenu, QHeaderView, QLabel, QTableWidget, QIcon,
                        QPixmap, QSize, QAbstractItemView, QTableWidgetItem,
                        QPushButton, QHBoxLayout, QVBoxLayout, QPalette,
                        QColor)

from functools import total_ordering
from calibre.utils.localization import canonicalize_lang, get_lang
from polyglot.builtins import iteritems, cmp

BASE_TIME = None

BeckyTries = 0
BeckyTitle = None

def debug_print(*args):
    global BASE_TIME
    if BASE_TIME is None:
        BASE_TIME = time.time()
    if DEBUG:
        prints('DEBUG: %6.1f'%(time.time()-BASE_TIME), *args)

try:
    # debug_print("LubimyCzytac::__init__.py - loading translations")
    load_translations()
except NameError:
    # debug_print("LubimyCzytac::__init__.py - exception when loading translations")
    pass  # load_translations() added in calibre 1.9


# Comparing Metadata objects for relevance {{{
words = ("the", "a", "an", "of", "and")
prefix_pat = re.compile(r'^(%s)\s+'%("|".join(words)))
trailing_paren_pat = re.compile(r'\(.*\)$')
whitespace_pat = re.compile(r'\s+')

def cleanup_title(s):
    if not s:
        s = _('Unknown')
    s = s.strip().lower()
    s = prefix_pat.sub(' ', s)
    s = trailing_paren_pat.sub('', s)
    s = whitespace_pat.sub(' ', s)
    return s.strip()


@total_ordering
class MetadataCompareKeyGen:

    '''
    Generate a sort key for comparison of the relevance of Metadata objects,
    given a search query. This is used only to compare results from the same
    metadata source, not across different sources.

    The sort key ensures that an ascending order sort is a sort by order of
    decreasing relevance.

    The algorithm is:

        * Prefer results that have at least one identifier the same as for the query
        * Prefer results with a cached cover URL
        * Prefer results with all available fields filled in
        * Prefer results with the same language as the current user interface language
        * Prefer results that are an exact title match to the query
        * Prefer results with longer comments (greater than 10% longer)
        * Use the relevance of the result as reported by the metadata source's search
           engine
    '''

    def __init__(self, mi, source_plugin, title, authors, identifiers):
        if not mi:
            self.base = (2,2,2,2,2)
            self.comments_len = 0
            self.extra = 0
            return

        isbn = 1 if mi.isbn and identifiers.get('isbn', None) is not None \
                and mi.isbn == identifiers.get('isbn', None) else 2

        all_fields = 1 if source_plugin.test_fields(mi) is None else 2

        cl_title = cleanup_title(title)
        cl_title_mi = cleanup_title(mi.title)
        exact_title = 1 if title and \
                cl_title == cl_title_mi else 2

        contains_title = 1 if title and \
                cl_title in cl_title_mi else 2

        auths = []
        if authors:
            for a in authors:
                auths.append(a.split(" ")[-1])
        miauths = []
        for a in mi.authors:
            miauths.append(a.split(" ")[-1])

        author_segments = list(set(miauths) & set(auths)) #authors surname list compare

        has_cover = 2 if (not source_plugin.cached_cover_url_is_reliable or
                source_plugin.get_cached_cover_url(mi.identifiers) is None) else 1


        #Wersja Calibre
        #self.base = (same_identifier, has_cover, all_fields, language, exact_title)
        #self.comments_len = len((mi.comments or '').strip())
        #self.extra = getattr(mi, 'source_relevance', 0)

        #Wersja test 01
        #self.base = (isbn, has_cover, all_fields, exact_title)

        #Wersja test 02
        #self.base = (exact_title, isbn, contains_title, -len(author_segments), all_fields, has_cover)

        #Wersja Becky 20211212
        self.base = (exact_title, isbn, contains_title, -len(author_segments), all_fields, has_cover)

        self.comments_len = len(mi.comments.strip() if mi.comments else '')
        self.extra = (getattr(mi, 'source_relevance', 0), )


    def compare_to_other(self, other):
        a = cmp(self.base, other.base)
        if a != 0:
            return a
        cx, cy = self.comments_len, other.comments_len
        if cx and cy:
            t = (cx + cy) / 20
            delta = cy - cx
            if abs(delta) > t:
                return -1 if delta < 0 else 1
        return cmp(self.extra, other.extra)

    def __eq__(self, other):
        return self.compare_to_other(other) == 0

    def __ne__(self, other):
        return self.compare_to_other(other) != 0

    def __lt__(self, other):
        return self.compare_to_other(other) < 0

    def __le__(self, other):
        return self.compare_to_other(other) <= 0

    def __gt__(self, other):
        return self.compare_to_other(other) > 0

    def __ge__(self, other):
        return self.compare_to_other(other) >= 0


# }}}



class LubimyCzytac(Source):

    name                    = 'LubimyCzytac'
    description             = _('Download book metadata and covers from LubimyCzytac.pl')
    author                  = 'Roman Cupisz, enhancements and code maintenance by Becky'
    version                 = PLUGINVER   # The version number of this plugin
    minimum_calibre_version = (3, 48, 0)

    capabilities = frozenset(['identify', 'cover'])
    touched_fields = frozenset(['title', 'authors', 'identifier:lubimyczytac',
        'identifier:isbn', 'rating', 'comments', 'publisher', 'pubdate',
        'series', 'tags', 'languages'])
    has_html_comments = True
    supports_gzip_transfer_encoding = True
    prefer_results_with_isbn = False

    list_of_editions = list()
    list_of_editions_url = list()
    list_of_editions_img = list()

    current_lcid = None
    searcheditions_with_lcid = False

    PLUGIN_VERSION = ".".join([str(x) for x in version])
    CALIBRE_VERSION = ".".join([str(x) for x in numeric_version])

    ID_NAME   = 'lubimyczytac'
    BASE_URL  = 'https://lubimyczytac.pl'
    BOOK_PATH = '/ksiazka/'
    BOOK_ADD = '/ksiazka'

    config_help_message = "<p>"+_('Calibre')+": <b>"+CALIBRE_VERSION+"</b> • \
        "+_('Plugin version')+": <b>"+PLUGIN_VERSION+"</b> • "+_('Please ' \
        'report bugs through the <a href="https://www.mobileread.com/forums/showthread.php?t=282922">MobileRead</a> forum.')

    def config_widget(self):
        '''
        Overriding the default configuration screen for our own custom configuration
        '''
        from calibre_plugins.lubimyczytac.config import ConfigWidget
        return ConfigWidget(self)

    def get_book_url(self, identifiers):
        lubimy_czytac_id = identifiers.get(self.ID_NAME, None)
        if lubimy_czytac_id:
            return (self.ID_NAME, lubimy_czytac_id,
                    '%s%s%s%s'%(LubimyCzytac.BASE_URL, LubimyCzytac.BOOK_PATH, lubimy_czytac_id, LubimyCzytac.BOOK_ADD))

    def id_from_url(self, url):
        match = re.match(r'%s/(ksiazka|audiobook)/(\d+)/.*' % LubimyCzytac.BASE_URL, url)
        if match:
            return (self.ID_NAME, match.group(2))
        return None


    def identify_results_keygen(self, title=None, authors=None,
            identifiers={}):
        '''
        Return a function that is used to generate a key that can sort Metadata
        objects by their relevance given a search query (title, authors,
        identifiers).

        These keys are used to sort the results of a call to :meth:`identify`.

        For details on the default algorithm see
        :class:`InternalMetadataCompareKeyGen`. Re-implement this function in
        your plugin if the default algorithm is not suitable.
        '''

        def keygen(mi):
            return MetadataCompareKeyGen(mi, self, title, authors,
                identifiers)
        return keygen


    def becky_get_author_tokens(self, log, authors, only_first_author=True):
        '''
        Take a list of authors and return a list of tokens useful for an
        AND search query. This function tries to return tokens in
        first name middle names last name order, by assuming that if a comma is
        in the author name, the name is in lastname, other names form.
        '''

        if authors:
            # Leave ' in there for Irish names
            remove_pat = re.compile(r'[!@#$%^&*()（）「」{}`~"\[\]/]')
            replace_pat = re.compile(r'[+.:;,，。；：]')
            if only_first_author:
                authors = authors[:1]
            for au in authors:
                has_comma = ',' in au
                au = replace_pat.sub(' ', au)
                parts = au.split()
                #log.info(u'Parts1 = %s'%parts)
                if has_comma:
                    # au probably in ln, fn form
                    parts = parts[1:] + parts[:1]

                #log.info(u'Parts2 = %s'%parts)
                if len(parts) > 1:
                   if parts[-2] == "von" or parts[-2] == "van":
                      parts[-1] = parts[-2] + " " + parts[-1]
                      del parts[-2]
                #log.info(u'Parts3 = %s'%parts)
                for tok in parts:
                    #log.info(u'tok1 = %s'%tok)
                    tok = remove_pat.sub('', tok).strip()
                    #log.info(u'tok2 = %s'%tok)
                    #if len(tok) > 2 and tok.lower() not in ('von', 'van',
                    #        _('Unknown').lower()):
                    if len(tok) > 2 and tok.lower() not in (
                            _('Unknown').lower()):
                        #log.info(u'tok3 = %s'%tok)
                        yield tok
                        #log.info(u'tok4 = %s'%tok)


    def create_query(self, log, title=None, authors=None, hiddenauthors=None, identifiers={}):
        lubimy_czytac_id = identifiers.get(self.ID_NAME, None)
        if lubimy_czytac_id:
            return '%s/ksiazka/wydania/%s/ksiazka'%(LubimyCzytac.BASE_URL, lubimy_czytac_id)

        q = ''
        ta=0
        # log.info('Becky debug T0=%s'%title)

        import calibre_plugins.lubimyczytac.config as cfg
        swapnames = cfg.plugin_prefs[cfg.STORE_NAME].get(cfg.SWAP_NAMES, cfg.DEFAULT_STORE_VALUES[cfg.SWAP_NAMES])
        LCissue = cfg.plugin_prefs[cfg.STORE_NAME].get(cfg.LC_ISSUE, cfg.DEFAULT_STORE_VALUES[cfg.LC_ISSUE])
        # log.info(u'SwapNames: %s'%swapnames)

        if authors:
            # ilu_autorow = len(authors)
            # log.info(u'ilu-authorow=%s'%ilu_autorow)
            # if ilu_autorow == 1:  # Jedziemy dalej tylko dla pojedynczego autora
            author_tokens = list(self.becky_get_author_tokens(log, authors,
                    only_first_author=True))
            #log.info(u'WAZNE!\nauthor_tokens=%s'%author_tokens)
            if author_tokens:
                tokens = [quote(t.encode('utf-8') if isinstance(t, unicode) else t) for t in author_tokens]
            if not swapnames:
                #log.info(u'Tokens = %s'%tokens)
                ostatni_czlon = tokens[-1:]  # .split()[-1] # Wez tylko nazwisko (ostatni człon autora)
                # log.info(u'01ostatni-czlon=%s'%ostatni_czlon)
                # liczba_czlonow = len(ostatni_czlon)
                # log.info(u'01len-ostatni-czlon=%s'%liczba_czlonow)
                if len(ostatni_czlon) > 0:
                    ta=1
                    # q += '+'.join(tokens).split()[-1]
                    q += '+'.join(ostatni_czlon)
            else:  # Tylko dla konfiguracji, w której najpierw jest nazwisko, a potem imię
                ostatni_czlon = tokens[0]  # Wez tylko nazwisko (w tym przypadku będzie to PIERWSZY człon autora)
                # log.info(u'02ostatni-czlon=%s'%ostatni_czlon)
                # liczba_czlonow = len(ostatni_czlon)
                # log.info(u'02len-ostatni-czlon=%s'%liczba_czlonow)
                if len(ostatni_czlon) > 0:
                    ta=1
                    # q += '+'.join(tokens).split()[-1]
                    q += ''.join(ostatni_czlon)

        # Sprawdzamy, czy w konfiguracji jest włączona opcja LC_ISSUE
        if (LCissue == 1):
            q= ''

        if title:
            # log.info('Becky debug T1=%s'%title)
            title = title.replace('?','')
            # debug_print("LubimyCzytac:: title2: ", title)
            # log.info('Becky debug T2=%s'%title)
            title = title.replace('_',' ')
            # debug_print("LubimyCzytac:: title3: ", title)
            # log.info('Becky debug T3=%s'%title)

            # Tu spacjalna poprawka, która ma kolosalne znaczenie w przypadku
            # tytułów, w których występuje cudzysłów. Dla ułatwienia robimy myk
            # i bierzemy z tytułu tylko ten fragment przed cudzysłowem
            if '\"' in title or ',,' in title:
                title = title.split('"')[0].split(',,')[0]
            # debug_print("LubimyCzytac:: title4: ", title)

            # LC BUG:
            # Ponieważ w serwisie LC nie można wyszukać tytułu z ampersandem (&),
            # wyszukajmy tylko to, co jest przed tym znakiem
            if '&' in title:
                title = title.split('&')[0]

            # Usuwamy hash (#) z tytułu
            if "#" in title:
                title = title.replace('#','')

            # Usuwamy nawiasy z tytułu
            if "(" in title:
                title = title.replace('(','')
            if ")" in title:
                title = title.replace(')','')

            # Naprawiamy apostrof, licząc na to, że w LC będzie poprawnie
            # Warunek usunięty w wersji 2.2.22, bo niestety w LC nie ma konsekwencji w zapisie apostrofów
            # if "'" in title:
            #     title = title.replace('\'','’')
            # log.info('LubimyCzytac:: title_inside = %s'%title)

            # 2.2.22 START
            # wyszukajmy tylko to, co jest przed tym znakiem
            if "'" in title:
                title = title.split('\'')[0]
            # 2.2.22 END

            # 2.2.19 START
            # Zamieniamy dywiz na trzy gwiazdki, aby przetrwały split
            if " - " in title:
                title = title.replace(' - ',' *** ')
            # 2.2.19 END

            # 2.2.20 START
            # Obsługujemy ukośnik (slash)
            if "/" in title and not " / " in title:
                title_tokens = [token for token in title.lower().split(' ') if len(token)>1]
            else:
                title_tokens = list(self.get_title_tokens(title,
                               strip_joiners=False, strip_subtitle=True))
            # log.info('Becky debug T4=%s'%title_tokens)
            # log.info('LubimyCzytac:: title_tokens = %s'%title_tokens)
            # 2.2.20 END


            if title_tokens:
                # 2.2.19 START
                # Jeśli istnieją trzy gwiazdki w tytule to zamieniamy je z powrotem na dywiz
                if "***" in title_tokens:
                    title_tokens = self.replaceInList(title_tokens,"***","-")
                # 2.2.19 END
                if ta==1:
                    q += '%20'
                tokens = [quote(t.encode('utf-8') if isinstance(t, unicode) else t) for t in title_tokens]
                # log.info('Becky debug T5=%s'%tokens)
                q = q + '%20'.join(tokens)
            # debug_print("LubimyCzytac:: tokens: ", tokens)
            # debug_print("LubimyCzytac:: q: ", q)
        if not q:
            return None
        return 'https://lubimyczytac.pl/szukaj/ksiazki?phrase=' + q

    def get_cached_cover_url(self, identifiers):
        url = None
        lubimy_czytac_id = identifiers.get(self.ID_NAME, None)
        if lubimy_czytac_id is None:
            isbn = identifiers.get('isbn', None)
            if isbn is not None:
                lubimy_czytac_id = self.cached_isbn_to_identifier(isbn)
        if lubimy_czytac_id is not None:
            url = self.cached_identifier_to_cover_url(lubimy_czytac_id)
        return url

    def cached_identifier_to_cover_url(self, id_):
        with self.cache_lock:
            url = self._get_cached_identifier_to_cover_url(id_)
            if not url:
                # Try for a "small" image in the cache
                url = self._get_cached_identifier_to_cover_url('small/'+id_)
            return url

    def _get_cached_identifier_to_cover_url(self, id_):
        # This must only be called once we have the cache lock
        url = self._identifier_to_cover_url_cache.get(id_, None)
        return url

    def identify(self, log, result_queue, abort, title=None, authors=None, hiddenauthors=None,
            identifiers={}, timeout=30):

        '''
        Note this method will retry without identifiers automatically if no
        match is found with identifiers.
        '''
        global BeckyTries
        global BeckyTitle

        # Dodatkowa funkcja przydatna przy wlaczonej opcji "W polu autorów stosuję zapis Nazwisko Imię"
        def swap_names(a):
            if ',' in a:
                parts = a.split(',')
                if len(parts) <= 1:
                    return a
                surname = parts[0]
                return '%s %s' % (' '.join(parts[1:]), surname)
            else:
                parts = a.split(None)
                if len(parts) <= 1:
                    return a
                surname = parts[-1]
                return '%s %s' % (surname, ' '.join(parts[:-1]))

        log.info('--- START ---')
        matches = []
        # If we have a LubimyCzytac.pl id then we do not need to fire a "search"
        # at lubimyczytac.pl. Instead we will go straight to the URL for that book.
        lubimy_czytac_id = identifiers.get(self.ID_NAME, None)
        log.info('\nTitle: %s\nAuthors: %s\n'%(title, authors))
        # isbn = check_isbn(identifiers.get('isbn', None))
        br = self.browser

        # Odczytujemy z konfiguracji, czy użytkownik chce wybierać wśród innych wydań
        # Jeśli searcheditions = False (domyślnie) to będziemy parsowali wyniki wyszukiwania
        # Jeśli searcheditions = True to będziemy parsowali wydania
        import calibre_plugins.lubimyczytac.config as cfg

        searcheditions = cfg.plugin_prefs[cfg.STORE_NAME].get(cfg.SELECT_OTHER_EDITIONS, cfg.DEFAULT_STORE_VALUES[cfg.SELECT_OTHER_EDITIONS])
        # log.info('searcheditions: %s'%searcheditions)
        swapnames = cfg.plugin_prefs[cfg.STORE_NAME].get(cfg.SWAP_NAMES, cfg.DEFAULT_STORE_VALUES[cfg.SWAP_NAMES])
        is_dark = cfg.plugin_prefs[cfg.STORE_NAME].get(cfg.DARK_THEME, cfg.DEFAULT_STORE_VALUES[cfg.DARK_THEME])

        if lubimy_czytac_id and searcheditions:
            log.info('lubimy_czytac_id exists. I\'m looking for other editions.')
            global current_lcid
            current_lcid = '%s/ksiazka/%s/ksiazka'%(LubimyCzytac.BASE_URL, lubimy_czytac_id)
            matches.append('%s/ksiazka/%s/ksiazka'%(LubimyCzytac.BASE_URL, lubimy_czytac_id))
            log.info('current_lcid: %s'%current_lcid)
        if lubimy_czytac_id and not searcheditions:
            log.info('lubimy_czytac_id exists. Not looking for other editions.')
            matches.append('%s/ksiazka/%s/ksiazka'%(LubimyCzytac.BASE_URL, lubimy_czytac_id))
        else:
            # Added 2.0.15
            # global response
            # response = None
            if title is not None:
                title = get_udc().decode(title)
            else:
                title = ''
            if authors is not None:
                authors = [get_udc().decode(a) for a in authors]
            query = self.create_query(log, title=title, authors=authors, hiddenauthors=authors,
                    identifiers=identifiers)

            response = None
            if query is None:
                log.error('Insufficient metadata to construct query')
                # log.error('Niewystarczajace metadane do skonstruowania zapytania')
                return

            try:
                log.info('Zapytanie: %s'%query)
                response = br.open_novisit(query, timeout=timeout)

            except Exception as e:
                # BECKY: Disable ISBN - LC does not support it for search.
                # if ISBN and callable(getattr(e, 'getcode', None)) and e.getcode() == 404:
                # We did a lookup by ISBN but did not find a match
                # We will fallback to doing a lookup by title author
                #    log.info('Failed to find match for ISBN: %s'%isbn)
                if callable(getattr(e, 'getcode', None)) and e.getcode() == 404:
                    log.error('No matches for identify query')
                    if lubimy_czytac_id:
                        log.error('BECKY HINT: Czyzby identyfikator LC byl bledny?')
                        log.error('BECKY HINT: Usun go z metadanych i sprobuj jeszcze raz.')
                    return as_unicode(e)

            # Anything from this point below is for title/authors based searches.
            # if not lubimy_czytac_id and response:
            if response:
                try:
                    raw = response.read().strip()
                    # open('D:\\Tools\\WORKSPACE\\LC\\log_rc1.html', 'wb').write(raw)
                    raw = raw.decode('utf-8', errors='replace')
                    if not raw:
                        log.error('Failed to get raw result for query')
                        return
                    '''
                    f = open("D:\\Tools\\WORKSPACE\\LC\\log_becky.html", "w")
                    try:
                        f.write(raw.rstrip('\n') )
                    finally:
                        f.close()
                    '''

                    root = fromstring(clean_ascii_chars(raw))
                except:
                    msg = 'Failed to parse LubimyCzytac.pl page for query'
                    log.exception(msg)
                    return msg

                # BECKY WAS HERE!

                # Now grab the matches from the search results, provided the
                # title and authors appear to be for the same book

                searcheditions_with_lcid = False
                if searcheditions and lubimy_czytac_id is not None:
                    searcheditions_with_lcid = True

                # Tutaj zamieniamy kolejnosc ukrytych autorow jesli w konfiguracji
                # uzytkownik wybral "Nazwisko Imię" w polu autorów
                changed_hiddenauthors = None
                if hiddenauthors:
                    if swapnames:
                        changed_hiddenauthors = [swap_names(a) for a in hiddenauthors]
                    else:
                        changed_hiddenauthors = hiddenauthors

                self._parse_search_results(log, title, authors, changed_hiddenauthors, root, matches, timeout, searcheditions_with_lcid)

        if abort.is_set():
            return
        jeden_autor=''
        if authors:
            for s in authors:
                jeden_autor=s
                # log.info('Jeden autor: %s'%jeden_autor)
                break

        if lubimy_czytac_id and matches and len(matches) > 1:
            # ---------
            app  = QApplication(sys.argv)
            app.setStyle('Fusion')

            key  = 0
            key2 = 0

            edi_dialog= EditionsDialog(self.list_of_editions,key,title,jeden_autor,self.list_of_editions_img,is_dark,log)
            key2 = edi_dialog.exec_()  # key2 another way to retrieve de selectedindex
            key  = edi_dialog.getValue()

            if key == 0:
                del matches[:]  # usuwamy cala liste
                matches.insert(0, current_lcid)
                log.info('BECKY INFO: Uzytkownik wybral z listy aktualne wydanie')
                log.info('BECKY INFO: lub kliknal na X/Anuluj/Escape.')
            else:
                if key>0:
                    result=self.list_of_editions_url[key]
                    result_url = result
                    del matches[:]  # usuwamy cala liste
                    matches.insert(0, result_url)
                else:
                    log.error('BECKY INFO: Mozliwy problem z innymi wydaniami.')
                    return

        if not matches:

            # if title and authors:
            if title:
                log.info('No matches found. Now a few more tries using only'
                        ' the title and authors, then only the title.')

                # Jesli wciaz nie ma dopasowania...

                # Ponieważ wyszukiwarka w serwisie LC coraz częsciej źle działa.
                # Ponownie wprowadzam wyszukiwanie tylko po tytule [2.1.7]

                # debug_print("LubimyCzytac:: jestem przed 0: ", BeckyTries, " -> ", BeckyTitle)
                if not matches and BeckyTries == 0:
                    log.info('---\nBECKY -- test # 0: Title only (without authors)...')
                    log.info('T0=%s'%title)
                    BeckyTries = 5  # Wiem, że wartość 5 wydaje się bez sensu, ale tak to wymyśliłam :)
                                    # Dzięki temu późniejszy warunek przy tutule przed kropką bez autorów jest spełniony
                    BeckyTitle = title  # Tutaj zapamiętujemy pełny tytuł, bo jeszcze się może przydać
                    # log.info('BeckyTries=%s'%BeckyTries)
                    return self.identify(log, result_queue, abort, title=title,
                            authors=None, hiddenauthors=authors, timeout=timeout)

                # debug_print("LubimyCzytac:: jestem przed 1: ", BeckyTries, " -> ", BeckyTitle)
                # Jesli w tytule jest kropka -- sprawdzamy to co przed kropka + autorzy
                if "." in title and not matches:
                    log.info('---\nBECKY -- test # 1A: Title before dot only (with authors)...')
                    title = title.split(".")[0] # Sprawdz tylko czesc tytulu przed kropka
                    log.info('T1A=%s'%title)
                    BeckyTries = 1
                    # Przywracamy autorow
                    authors = hiddenauthors
                    # log.info('BeckyTries=%s'%BeckyTries)
                    return self.identify(log, result_queue, abort, title=title,
                            authors=authors, hiddenauthors=authors, timeout=timeout)

                # debug_print("LubimyCzytac:: jestem przed 2: ", BeckyTries, " -> ", BeckyTitle)
                # Powtarzamy operację, al etym razem bez autorów
                if not matches and BeckyTries == 1:
                    log.info('---\nBECKY INFO: test # 1B: Title before dot only (without authors)...')
                    log.info('T1B=%s'%title)
                    BeckyTries = 2
                    # log.info('BeckyTries=%s'%BeckyTries)
                    return self.identify(log, result_queue, abort, title=title,
                            authors=None, hiddenauthors=authors, timeout=timeout)

                # Jesli w tytule jest kropka -- sprawdzamy to co przed kropka
                # if "." in title and not matches:
                #    log.info('---\nBECKY INFO: test # 1C: Title before dot only (without authors)...')
                #    title = title.split(".")[0]  # Sprawdz tylko czesc tytulu przed kropka
                #    log.info('T1C=%s'%title)
                #    return self.identify(log, result_queue, abort, title=title,
                #            authors=None, timeout=timeout)

                # Jesli w tytule jest kropka -- sprawdzamy to co przed kropka i bez autora
                # if not matches:
                #    log.info('---\nBECKY -- test # 1D: Title before dot only (without authors)...')
                #    log.info('T1D=%s'%title)
                #    return self.identify(log, result_queue, abort, title=title,
                #            authors=None, timeout=timeout)

                # Usuwam pojedyncze litery z tytulu (to nie jest dobry pomysl!)
                # title_ciach = ' '.join([w for w in title.split() if len(w)>1])
                # if ((not matches) and (len(title_ciach) < len(title))):
                #    log.info('---\nBECKY -- test # 2: Title without single letters ...')
                #    log.info('T2=%s'%title_ciach)
                #    return self.identify(log, result_queue, abort, title=title_ciach,
                #            authors=None, timeout=timeout)

                # Jesli w tytule jest kropka -- sprawdzamy to co po kropce
                # if "." in title and not matches:
                #    log.info('---\nBECKY -- test # 3: Title after dot only...')
                #    title = title.split(".")[1] # Sprawdz tylko czesc tytulu po kropce
                #    log.info('T3=%s'%title)
                #    return self.identify(log, result_queue, abort, title=title,
                #            authors=None, timeout=timeout)

                # debug_print("LubimyCzytac:: jestem przed last: ", BeckyTries, " -> ", BeckyTitle)
                if not matches:
                    debug_print("LubimyCzytac:: LAST")
                    log.error('LAST')


            # debug_print("LubimyCzytac:: jestem przed error: ", BeckyTries, " -> ", BeckyTitle)
            log.error('No matches found with query: %r'%query)
            return

        # Tutaj warunki dodatkowe, gdy przetestowaliśmy przed kropką, ale warto jeszcze zajrzeć, co się dzieje po kropce
        # Istnieje tylko jeden wynik, ale pochodzi on z testowania przed kropką (BeckyTries = 1 lub 2)
        if len(matches) == 0 and BeckyTries != 1 and BeckyTries != 2 and BeckyTries != 3 and BeckyTries != 5:

            # debug_print("LubimyCzytac:: jestem przed 3: ", BeckyTries, " -> ", BeckyTitle)
            # Jesli w tytule jest kropka -- sprawdzamy to co przed kropka + autorzy

            # Ten fragment nie działa poprawnie, ale wciąż liczę na cud, że wymyślę, jak to zrobić, aby zadziałał :)
            if BeckyTitle:
                if "." in BeckyTitle:
                    log.info('---\nBECKY -- test # 2A: Title AFTER dot only (with authors)...')
                    title = BeckyTitle
                    # title = title.split(".")[1]  # Sprawdz tylko czesc tytulu po kropce
                    # log.info('T2A-01=%s'%title)
                    title = title.split(".")[1].strip() # Sprawdz tylko czesc tytulu po kropce
                    # log.info('T2A-02=%s'%title)
                    title = title.split(' ', 1)[0]  # (a dokładniej tylko pierwszy wyraz)
                    # log.info('T2A-03=%s'%title)
                    BeckyTries = 3
                    # Przywracamy autorow
                    authors = hiddenauthors
                    # log.info('BeckyTries=%s'%BeckyTries)
                    return self.identify(log, result_queue, abort, title=title,
                            authors=authors, hiddenauthors=authors, timeout=timeout)

        # Przeczyszczenie listy matches
        if matches and len(matches) > 1:
            log.info('----- BECKY START  -----')
            log.info('BECKY INFO: Sprawdzamy, co nam zostalo...')
            log.info('BECKY INFO: title: %s'%title)
            log.info('BECKY INFO: authors: %s'%authors)
            log.info('BECKY INFO: hiddenauthors: %s'%hiddenauthors)
            log.info('BECKY INFO: timeout: %s'%timeout)
            log.info('----- BECKY KONIEC -----')
            log.info('-----')
            log.info('BECKY INFO: matches BEFORE: %d'% len(matches))
            matches = self.sortAndUniqq(matches)
            log.info('BECKY INFO: matches UNIQUE: %d'% len(matches))
            # log.info('-----')
        # log.debug('Starting workers for:\n%s' % ('\n'.join(matches,)))


        from calibre_plugins.lubimyczytac.worker import Worker
        workers = [Worker(url, result_queue, br, log, i, self) for i, url in
                enumerate(matches) if url]

        log.info('-------------------------------  BEDA PARSOWANE  -------------------------------')

        for w in workers:
            w.start()
            # Don't send all requests at the same time
            time.sleep(0.1)

        while not abort.is_set():
            a_worker_is_alive = False
            for w in workers:
                w.join(0.2)
                if abort.is_set():
                    break
                if w.is_alive():
                    a_worker_is_alive = True
            if not a_worker_is_alive:
                break

        # log.info('BECKY INFO: Koniec pracy')
        return None

    def _parse_search_results(self, log, orig_title, orig_authors, hiddenauthors, root, matches, timeout, searcheditions_with_lcid):

        # Resetujemy aktualne wydanie
        current_book_title = None
        current_book_image = None

        # Resetujemy inne wydania tej ksiazki
        # Nie uzywac .clear(), bo w starej wersji Calibre 3.48 to nie zadziala!)
        del self.list_of_editions[:]
        del self.list_of_editions_url[:]
        del self.list_of_editions_img[:]

        if searcheditions_with_lcid:
            # results = root.xpath('*//div[@id="editionsList"]//div[@class="authorAllBooks__single"]//div[contains(@class,"authorAllBooks__singleText ")]')
            results = root.xpath('*//div[@id="editionsList"]//div[@class="authorAllBooks__single"]')

            # Pobieramy tytuł aktualnego wydania książki
            # Moglibyśmy użyć orig_title, ale ta zmienna jest już po przejściach i nie nadaje się do naszych celów
            # current_book_title = root.xpath('*//div[contains(@class,"title-container")]/@data-title')
            current_book_title = root.xpath('*//div[contains(@class,"title-container")]/h1[@class="book__title"]/text()')
            # Czyszczę sobie tytuł aktualnej książki
            current_book_title[0] = current_book_title[0].strip()
            log.info('current_book_title: %s'%current_book_title)

            # Pobieramy link do okładki aktualnego wydania książki
            current_book_image = root.xpath('*//div[contains(@class,"book-cover-hld")]/div[contains(@class,"book-cover")]//img[contains(@class,"img-fluid")]/@src')
            log.info('current_book_image: %s'%current_book_image)
        else:
            # results = root.xpath('*//div[@class="listSearch"]//div[@class="authorAllBooks__single"]//div[contains(@class,"authorAllBooks__singleText")]')
            results = root.xpath('*//div[@class="listSearch"]//div[@class="authorAllBooks__single"]')

        if not results:
            return

        import calibre_plugins.lubimyczytac.config as cfg

        fast_match = cfg.plugin_prefs[cfg.STORE_NAME].get(cfg.FAST_MATCH, cfg.DEFAULT_STORE_VALUES[cfg.FAST_MATCH])
        if searcheditions_with_lcid:
            max_results = cfg.plugin_prefs[cfg.STORE_NAME].get(cfg.KEY_MAX_DOWNLOADS_OTHER_EDITIONS, cfg.DEFAULT_STORE_VALUES[cfg.KEY_MAX_DOWNLOADS_OTHER_EDITIONS])
            fast_match = False  # Wyłączamy fast_match jeśli włączona jest opcja wybierania wśród innych wydań
        else:
            max_results = cfg.plugin_prefs[cfg.STORE_NAME].get(cfg.KEY_MAX_DOWNLOADS, cfg.DEFAULT_STORE_VALUES[cfg.KEY_MAX_DOWNLOADS])

        # found_title = False
        no_matches = []

        # 2.1.0 START
        if searcheditions_with_lcid:
            # Wstawiamy dane dot. aktualnego wydania książki na początek listy
            self.list_of_editions.insert(0, _("Current edition:") + "\n" + current_book_title[0])
            # self.list_of_editions_url.insert(0, current_book_url[0])
            self.list_of_editions_url.insert(0, current_lcid)
            self.list_of_editions_img.insert(0, current_book_image[0])
            # log.info("Po wstawieniu:")
            # log.info(self.list_of_editions)
            # log.info(self.list_of_editions_url)
            # log.info(self.list_of_editions_img)
        # 2.1.0 END

        # log.info('Parsing results: %s ' % results )
        i = 0
        for result in results:
            i = i + 1
            # log.info('---\nTEST [%s]' % (i))
            log.info('\nParsing result %s: %s ' % (i,result))
            if searcheditions_with_lcid:
                image_url = result.xpath('.//div[contains(@class,"authorAllBooks__singleImg ")]//img[contains(@class,"img-fluid")]/@data-src')
                log.info('BECKY INFO: image_url')
                log.info(image_url[0])
            title = result.xpath('.//div[contains(@class,"authorAllBooks__singleText ")]/div/a[contains(@class,"authorAllBooks__singleTextTitle")]//text()')
            book_url = result.xpath('.//div[contains(@class,"authorAllBooks__singleText ")]/div/a[contains(@class,"authorAllBooks__singleTextTitle")]/@href')
            authors = result.xpath('.//div[contains(@class,"authorAllBooks__singleText ")]/div/a[contains(@href,"autor")]//text()')

            if not title or not book_url or not authors:
                continue
            title = title[0]
            book_url = LubimyCzytac.BASE_URL + book_url[0]
            # author = authors[0]

            # 2.1.0 START
            if searcheditions_with_lcid:
                log.error('BECKY INFO: Dodajemy do listy wydan')
                title_for_other_editions_window = title.strip().replace('\n','').replace('  ',' ').replace(':','').replace(';','').replace('-','').replace('?','').replace('  ',' ')
                self.list_of_editions.append(title_for_other_editions_window)
                self.list_of_editions_url.append(book_url)
                self.list_of_editions_img.append(image_url[0])
                log.info(self.list_of_editions)
                log.info(self.list_of_editions_url)
                log.info(self.list_of_editions_img)
            # 2.1.0 END

            if authors is not None:
                authors = [get_udc().decode(a) for a in authors]  # usuwamy diakrytyki
                authors = [lower(a) for a in authors]  # minuskuly
                if authors[0]:
                    author_tokens = self.becky_get_author_tokens(log, authors,
                        only_first_author=True)
                    if author_tokens:
                        tokens = [quote(t.encode('utf-8') if isinstance(t, unicode) else t) for t in author_tokens]
                        nazwisko = tokens[-1:]  # .split()[-1] # Wez tylko nazwisko (ostatni człon autora)
                        log.info('BECKY INFO: Nazwisko autora=%s'%nazwisko)

            if orig_authors is not None:
                orig_authors = [lower(a) for a in orig_authors]
                if orig_authors[0]:
                    orig_author_tokens = self.becky_get_author_tokens(log, orig_authors,
                        only_first_author=True)
                    if orig_author_tokens:
                        tokens = [quote(t.encode('utf-8') if isinstance(t, unicode) else t) for t in orig_author_tokens]
                        orig_nazwisko = tokens[-1:]  # .split()[-1] # Wez tylko nazwisko (ostatni człon autora)
                        log.info('BECKY INFO: Nazwisko orig autora=%s'%orig_nazwisko)

            if hiddenauthors is not None:
                hiddenauthors = [lower(a) for a in hiddenauthors]
                if hiddenauthors[0]:
                    hidden_author_tokens = self.becky_get_author_tokens(log, hiddenauthors,
                        only_first_author=True)
                    if hidden_author_tokens:
                        tokens = [quote(t.encode('utf-8') if isinstance(t, unicode) else t) for t in hidden_author_tokens]
                        hidden_nazwisko = tokens[-1:]  # .split()[-1] # Wez tylko nazwisko (ostatni człon autora)
                        log.info('BECKY INFO: Nazwisko ukrytego autora=%s'%hidden_nazwisko)

            # 2.0.24 START - przy porownywaniu tytulow usuwamy obce diaktytyki
            if title is not None:
                title = get_udc().decode(title)
            # 2.0.24 END

            # 2.0.35g START - przy porownywaniu tytulow usuwamy troche interpunkcji i podwojne odstepy
                title = title.lower().strip()
                title = title.replace('  ',' ').replace(',','').replace('.','').replace(':','').replace(';','').replace('-','').replace('?','').replace('  ',' ')

            if orig_title is not None:
                orig_title = orig_title.lower()
                orig_title = orig_title.replace('  ',' ').replace(',','').replace('.','').replace(':','').replace(';','').replace('-','').replace('?','').replace('  ',' ')
            # 2.0.35g END

            # log.info('[%s] Original title: %s,\n    Title: %s,\n    Original authors: %s,\n    Authors:%s,\n    Book url: %s\n' % (i, orig_title, title, orig_authors, authors, book_url))
            log.info('''[%s] Original title: %s,
    Title: %s,
    Original authors: %s,
    Hidden authors: %s,
    Authors:%s,
    Book url: %s'''
    % (i, orig_title, title, orig_authors, hiddenauthors, authors, book_url,))

            if (self.match(title,orig_title) and self.contains(authors,orig_authors)):
                # 2.0.24 START - zgodnosc tytulu i autora powinna byc priorytetowa
                # Nie jest to idealne rozwiązanie, bo w przypadku wydań zbiorowych może się zdarzyć,
                # że przed szukaną pozycją wcześniej jest jakiś pakiet i to on zostanie wybrany jako zgodny
                # 2.0.32: dodano opcję w konfiguracji: [ ] Szybkie dopasowanie (pierwszy wynik z pasującym tytułem i autorem)

                # log.info('dopasowania przed: ', matches)
                if fast_match:
                    del matches[:]  # usuwamy cala liste
                    matches.insert(0, book_url)
                    log.info('[%s] matches title and authors' % (i))
                    break
                else:
                    matches.append(book_url)
                # log.info('dopasowania po: ', matches)
                # 2.0.32 END
                # 2.0.24 END

            # 2.0.24 START2
            # Zgodność odwrotna jest opcją eksperymentalną i zapewne zostanie usunięta w przyszłości
            # 2.1.1 Robimy to oczywiście tylko wtedy, gdy autor istnieje
            if authors is not None and orig_authors is not None:
                if (self.match(authors[0],orig_title) and self.match(title,orig_authors[0])):
                    del matches[:]  # usuwamy cala liste
                    matches.insert(0, book_url)
                    log.info('BECKY INFO: Jest zgodnosc odwrotna!')
                    log.info('BECKY INFO: [%s] matches inverse title and authors' % (i))
                    break
            # 2.0.24 END2

            if self.match(title,orig_title):
                # Dodajemy do matches jesli już taki wpis nie istnieje
                if book_url not in matches:
                    matches.append(book_url)
                # log.info('dopasowania [tytul]: ', matches)
                log.info('BECKY INFO: [%s] matches title (w przyblizeniu)' % (i))

                # 2.1.6-2.1.7 START
                if orig_authors and authors:
                    if self.contains(authors,orig_authors):
                        # del matches[:]  # usuwamy cala liste
                        # matches.insert(0, book_url)
                        # Przesuwamy znalezienie na początek. Nie jest do końca dobre rozwiązanie,
                        # bo zgodny tytuł wcale nie oznacza, że to prawidłowy tytuł...
                        if book_url not in matches:
                            matches.insert(0, matches)
                        else:
                            matches.insert(0, matches.pop())
                        log.info('BECKY INFO: [%s] plus original authors' % (i))
                        # break
                if not authors:
                    if hiddenauthors:
                        if self.contains(authors,hiddenauthors):
                            #del matches[:]  # usuwamy cala liste
                            #matches.insert(0, book_url)
                            #matches.insert(0, matches.pop())
                            if book_url not in matches:
                                matches.insert(0, matches)
                            else:
                                matches.insert(0, matches.pop())
                            log.info('BECKY INFO: [%s] plus hidden authors' % (i))
                            # break
                # 2.1.6-2.1.7 END
                # 2.2.17 START
                else:
                    if hiddenauthors:
                        if self.contains(authors,hiddenauthors):
                            #del matches[:]  # usuwamy cala liste
                            #matches.insert(0, book_url)
                            #matches.insert(0, matches.pop())
                            if fast_match:
                                del matches[:]  # usuwamy cala liste
                                matches.insert(0, book_url)
                                log.info('BECKY INFO: [%s] plus hidden authors' % (i))
                                break
                # 2.2.17 END
            else:
                log.info('BECKY INFO: match authors %s %s to orig_authors %s %s'%(authors,type(authors),orig_authors,type(orig_authors)))
                if self.contains(authors,orig_authors):
                    # Dodajemy do matches jesli już taki wpis nie istnieje
                    if book_url not in matches:
                        matches.append(book_url)
                    # log.info('dopasowania [autor]: ', matches)
                    log.info('BECKY INFO: [%s] matches authors' % (i))
                else:
                    no_matches.append(book_url)
                    log.info('BECKY INFO: [%s] no matches' % (i))
            if len(matches) >= max_results:
                log.info('BECKY INFO: reached max results limit: %s' % (max_results))
                break

        if no_matches and not matches:
            matches.extend(no_matches)

    def match(self, item1, item2):
        if not item2:
            return False
        if isinstance(item1, unicode) and isinstance(item2, unicode):
            return item1 == item2 or item1 in item2 or item2 in item1

    def contains(self, list1, list2):
        if not list1 or not list2:
            return False
        for el in list1:
            if any(el in s for s in list2):
                return True
        return False

    def sortAndUniqq(self,input):
        output = []
        for x in input:
            if x not in output:
                output.append(x)
        output.sort()
        return output


    def replaceInList(self, arr, find, replace):
        # fast and readable
        base=0
        for cnt in range(arr.count(find)):
            offset=arr.index(find, base)
            arr[offset]=replace
            base=offset+1
        return arr

    def download_cover(self, log, result_queue, abort,
            title=None, authors=None, identifiers={}, timeout=30):
        cached_url = self.get_cached_cover_url(identifiers)
        if cached_url is None:
            log.info('No cached cover found, running identify')
            rq = Queue()
            self.identify(log, rq, abort, title=title, authors=authors, hiddenauthors=authors,
                    identifiers=identifiers)
            if abort.is_set():
                return
            results = []
            while True:
                try:
                    results.append(rq.get_nowait())
                except Empty:
                    break
            results.sort(key=self.identify_results_keygen(
                title=title, authors=authors, identifiers=identifiers))
            for mi in results:
                cached_url = self.get_cached_cover_url(mi.identifiers)
                if cached_url is not None:
                    break
        if cached_url is None:
            log.info('No cover found')
            return

        if abort.is_set():
            return
        br = self.browser
        log.info('Downloading cover from:', cached_url)
        try:
            cdata = br.open_novisit(cached_url, timeout=timeout).read()
            result_queue.put((self, cdata))
        except:
            log.exception('Failed to download cover from:', cached_url)


class EditionsDialog(QDialog):

    icon_name = 'images/lubimyczytac.png'

    def __init__(self,listdata,key,title,authors,imgurl,is_dark,log):
        super(EditionsDialog, self).__init__(None, Qt.WindowTitleHint | Qt.WindowCloseButtonHint)
        self.data= list()
        self.data=listdata
        self.imgurl= list()
        self.imgurl=imgurl
        self.title=title
        self.author=authors

        if is_dark:
            # Becky DARK THEME START
            # Kolory okienka z wydaniami dla trybu ciemnego
            self.palette = self.palette()
            # główne okienko z wyborem wydania
            self.palette.setColor(QPalette.Window, QColor(45, 45, 45))
            self.palette.setColor(QPalette.WindowText, QColor('white'))
            # przyciski na dole okienka
            self.palette.setColor(QPalette.Button, QColor(45, 45, 45))
            self.palette.setColor(QPalette.ButtonText, QColor('white'))
            # tło tabelki z wydaniami
            self.palette.setColor(QPalette.Base, QColor(18,18,18));
            # tekst na żółtym tle
            self.palette.setColor(QPalette.Text, QColor("black"))
            # kolor podświetlenia
            self.palette.setColor(QPalette.Highlight, QColor(10, 100, 200))
            # przypisanie zmienionej palety do okienka
            self.setPalette(self.palette)
            # Becky DARK THEME END

        self.initUI(is_dark)

    def initUI(self,is_dark):
        label_select=QLabel(_("Select other edition"))

        self.lw = QTableWidget()

        self.lw.setRowCount(len(self.data))
        self.lw.setColumnCount(1)

        # self.lw.setColumnWidth(1, 300)
        self.lw.horizontalHeader().setStretchLastSection(True)
        self.lw.setColumnWidth(0, 150)
        self.lw.setIconSize(QSize(106, 150))

        # Becky: hide headers
        self.lw.horizontalHeader().setVisible(False)
        self.lw.verticalHeader().setVisible(False)

        # Becky: select rows
        self.lw.setSelectionBehavior(QAbstractItemView.SelectRows)

        self.lw.verticalHeader().setDefaultSectionSize(160)
        self.lw.setMinimumHeight(485);

        current_row = 0

        for (item_image, item_title) in zip(self.imgurl, self.data):
            url = item_image
            req = Request(url, headers={'User-Agent': 'Mozilla/5.0'})
            data = urlopen(req).read()
            pixmap = QPixmap()
            pixmap.loadFromData(data)
            icon = QIcon(pixmap)

            self.lw.setItem(current_row, 0, QTableWidgetItem(icon, item_title))
            if current_row == 0:
                # Yellow
                self.lw.item(0, 0).setBackground(QtGui.QColor(255,220,64))

            # Becky DARK THEME START
            else:
                if is_dark:
                    self.lw.item(current_row, 0).setBackground(QtGui.QColor(18,18,18))
                    self.lw.item(current_row, 0).setForeground(QtGui.QColor('white'))
            # Becky DARK THEME END

            current_row += 1

        self.lw.selectRow(0)
        myRows=self.lw.rowCount()

        self.lw.doubleClicked.connect(self.SelectedRow)
        okButton = QPushButton("OK")
        okButton.clicked.connect(self.SelectedRow)
        cancelButton = QPushButton(_("Cancel"))
        cancelButton.clicked.connect(self.Cancel)

        hbox = QHBoxLayout()
        # hbox.addStretch(1)
        hbox.addWidget(cancelButton)
        hbox.addWidget(okButton)

        layout=QVBoxLayout()
        layout.addStretch(1)
        layout.addWidget(label_select)
        layout.addWidget(self.lw)
        layout.addLayout(hbox)
        self.setLayout(layout)

        if myRows>5:
            myRows=5
        # self.setGeometry(200,200,500,120+myRows*160)
        self.setWindowTitle(_('Select edition'))
        self.setWindowIcon(get_icons(self.icon_name))

        okButton.setFocus()

    def SelectedRow(self):
        self.done(self.lw.currentRow())

    def Cancel(self):
        self.lw.selectRow(0)
        self.SelectedRow()

    def closeEvent(self, event):
        key=self.lw.currentRow()
        self.done(self.lw.currentRow())

    def getValue(self):
        return self.lw.currentRow()


if __name__ == '__main__':  # tests
    # To run these test use:
    # calibre-debug -e __init__.py
    from calibre.ebooks.metadata.sources.test import (test_identify_plugin, title_test, authors_test, series_test)

    lctests = [
        (
          {
             'identifiers':{'lubimyczytac': '4965040'},
             'title':'Krew i miód',
             'authors':['Shelby Mahurin']
          },
          [
             title_test('Krew i miód'),
             authors_test(['Shelby Mahurin']),
          ]
        ),
        (
          {
             'identifiers':{'lubimyczytac': '4954343'},
             'title':'Cień utraconego świata',
             'authors':['James Islington']
          },
          [
             title_test('Cień utraconego świata'),
             authors_test(['James Islington']),
          ]
        ),

    ]

    def do_test(lctests, start=0, stop=None):
        if stop is None:
            stop = len(lctests)
        lctests = lctests[start:stop]
        test_identify_plugin(LubimyCzytac.name, lctests)

    do_test(lctests)

# series_test('Opowieści z meekhańskiego pogranicza', 1.0),
