﻿import os
import sys
import re
from pathlib import Path
from typing import Iterable, List
from plugin_utils import Qt, QtCore, QtGui, QtWidgets, QAction
from plugin_utils import PluginApplication, iswindows
from sigil_vars import ENV_VARIABLES

_VERSION_PATTERN = re.compile(r'<version>([^<]*)</version>')
isbeta = True

# ---------------------------------------------------------
# BETA
# ---------------------------------------------------------
def showbeta():
    if isbeta:
        return " beta"
    else:
        return "" 

# ---------------------------------------------------------
# PLATFORM DETECTION
# ---------------------------------------------------------
isosx = sys.platform.startswith("darwin")
islinux = sys.platform.startswith("linux")
iswindows = sys.platform.startswith("win")

# ----------------------------------
# LANG CODES
# ----------------------------------
ISO_639_2_CODES = {
    "aa", "ab", "ae", "af", "ak", "am", "an", "ar", "ar-AE", "ar-BH", "ar-DZ",
    "ar-EG", "ar-IQ", "ar-JO", "ar-KW", "ar-LB", "ar-LY", "ar-MA", "ar-OM",
    "ar-QA", "ar-SY", "ar-TN", "ar-YE", "as", "av", "ay", "az", "az-AZ", "ba",
    "be", "bg", "bh", "bi", "bm", "bn", "bo", "br", "bs", "ca", "ca-ES", "ce",
    "ch", "co", "cr", "cs", "cu", "cv", "cy", "da", "da-DK", "de", "de-AT",
    "de-CH", "de-DE", "de-LI", "de-LU", "dv", "dz", "ee", "el", "el-GR", "en",
    "en-AU", "en-BZ", "en-CA", "en-CB", "en-GB", "en-IE", "en-IN", "en-JM",
    "en-PH", "en-TT", "en-US", "en-ZA", "eo", "es", "es-AR", "es-BO", "es-CL",
    "es-CO", "es-CR", "es-DO", "es-EC", "es-ES", "es-GT", "es-HN", "es-MX",
    "es-NI", "es-PA", "es-PE", "es-PR", "es-PY", "es-SV", "es-UY", "es-VE",
    "et", "eu", "fa", "ff", "fi", "fj", "fo", "fr", "fr-BE", "fr-CA", "fr-CH",
    "fr-FR", "fr-LU", "fy", "ga", "gd", "gd-IE", "gl", "gn", "gu", "gv", "ha",
    "he", "hi", "ho", "hr", "ht", "hu", "hu-HU", "hy", "hz", "ia", "id",
    "id-ID", "ie", "ig", "ii", "ik", "io", "is", "it", "it-CH", "it-IT", "iu",
    "ja", "jv", "ka", "kg", "ki", "kj", "kk", "kl", "km", "kn", "ko", "kr",
    "ks", "ku", "kv", "kw", "ky", "la", "lb", "lg", "li", "ln", "lo", "lt",
    "lu", "lv", "lv-LV", "mg", "mh", "mi", "mk", "ml", "mn", "mr", "ms",
    "ms-BN", "ms-MY", "mt", "my", "na", "nb", "nd", "ne", "ng", "nl", "nl-BE",
    "nl-NL", "nn", "no", "nr", "nv", "ny", "oc", "oj", "om", "or", "os", "pa",
    "pi", "pl", "ps", "pt", "pt-BR", "pt-TT", "qaa-qtz", "qu", "rm", "rn", "ro",
    "ro-MO", "ro-RO", "ru", "ru-MO", "rw", "sa", "sc", "sd", "se", "sg", "si",
    "sk", "sl", "sm", "sn", "so", "sq", "sr", "sr-RS", "ss", "st", "su", "sv",
    "sv-FI", "sv-SE", "sw", "ta", "te", "tg", "th", "ti", "tk", "tl", "tn",
    "to", "tr", "tr-TR", "ts", "tt", "tw", "ty", "ug", "uk", "uk-UA", "ur",
    "uz", "uz-UX", "ve", "vi", "vo", "wa", "wo", "xh", "yi", "yo", "za", "zh",
    "zh-CN", "zh-HK", "zh-MO", "zh-SG", "zh-TW", "zu", "ace", "ach", "ada",
    "ady", "afa", "afh", "ain", "akk", "ale", "alg", "alt", "ang", "anp", "apa",
    "arc", "arn", "arp", "art", "arw", "ast", "ath", "aus", "awa", "bad", "bai",
    "bal", "ban", "bas", "bat", "bej", "bem", "ber", "bho", "bik", "bin", "bla",
    "bnt", "bra", "btk", "bua", "bug", "byn", "cad", "cai", "car", "cau", "ceb",
    "cel", "chb", "chg", "chk", "chm", "chn", "cho", "chp", "chr", "chy", "cmc",
    "cop", "cpe", "cpf", "cpp", "crh", "crp", "csb", "cus", "dak", "dar", "day",
    "del", "den", "dgr", "din", "doi", "dra", "dsb", "dua", "dum", "dyu", "efi",
    "egy", "eka", "elx", "enm", "ewo", "fan", "fat", "fil", "fiu", "fon", "frm",
    "fro", "frr", "frs", "fur", "gaa", "gay", "gba", "gem", "gez", "gil", "gmh",
    "goh", "gon", "gor", "got", "grb", "grc", "gsw", "gug", "gwi", "hai", "haw",
    "hil", "him", "hit", "hmn", "hsb", "hup", "iba", "ijo", "ilo", "inc", "ine",
    "inh", "ira", "iro", "jbo", "jpr", "jrb", "kaa", "kab", "kac", "kam", "kar",
    "kaw", "kbd", "kha", "khi", "kho", "kmb", "kok", "kos", "kpe", "krc", "krl",
    "kro", "kru", "kum", "kut", "lad", "lah", "lam", "lez", "lol", "loz", "lua",
    "lui", "lun", "luo", "lus", "mad", "mag", "mai", "mak", "man", "map", "mas",
    "mdf", "mdr", "men", "mga", "mic", "min", "mis", "mkh", "mnc", "mni", "mno",
    "moh", "mos", "mul", "mun", "mus", "mwl", "mwr", "myn", "myv", "nah", "nai",
    "nap", "nds", "new", "nia", "nic", "niu", "nog", "non", "nqo", "nso", "nub",
    "nwc", "nym", "nyn", "nyo", "nzi", "osa", "ota", "oto", "paa", "pag", "pal",
    "pam", "pap", "pau", "peo", "phi", "phn", "pon", "pra", "pro", "raj", "rap",
    "rar", "roa", "rom", "rup", "sad", "sah", "sai", "sal", "sam", "sas", "sat",
    "scn", "sco", "sel", "sem", "sga", "sgn", "shn", "sid", "sio", "sit", "sla",
    "sma", "smi", "smj", "smn", "sms", "snk", "sog", "son", "srn", "srr", "ssa",
    "suk", "sus", "sux", "syc", "syr", "tai", "tem", "ter", "tet", "tig", "tiv",
    "tkl", "tlh", "tli", "tmh", "tog", "tpi", "tsi", "tum", "tup", "tut", "tvl",
    "tyv", "udm", "uga", "umb", "und", "vai", "vot", "wak", "wal", "war", "was",
    "wen", "xal", "yao", "yap", "ypk", "zap", "zbl", "zen", "znd", "zun", "zxx",
    "zza"
}

def is_valid_language(code):
    return code in ISO_639_2_CODES

def unique_preserve_order(seq: Iterable[str]) -> List[str]:
    seen = set()
    out = []
    for x in seq:
        if x not in seen:
            seen.add(x)
            out.append(x)
    return out

def normalize_lang_candidate(s: str) -> str:
    if s is None:
        return s
    return s.strip().replace('_', '-')

def normalize_lang(raw: str) -> str:
    if raw is None:
        return raw
    s = normalize_lang_candidate(raw)
    if s == "":
        return s

    if '-' in s:
        left, right = s.split('-', 1)
        left_l = left.lower()
        right_u = right.upper()
        right_l = right.lower()

        # variants to check
        candidates = [
            f"{left_l}-{right_u}",
            f"{left_l}-{right_l}",
            f"{left.lower()}-{right}",
            s  # original cleaned
        ]
        for cand in candidates:
            if cand in ISO_639_2_CODES:
                return cand
        return f"{left_l}-{right_u}"

    if len(s) in (2, 3):
        cand = s.lower()
        if cand in ISO_639_2_CODES:
            return cand
        return cand

    return s

def normalize_languages_list(raw_val: str, *, keep_order: bool = True) -> List[str]:
    if raw_val is None:
        return []
    # remove all whitespace inside
    compact = re.sub(r'\s+', '', raw_val)
    parts = [p for p in compact.split(',') if p != '']
    normalized = [normalize_lang(p) for p in parts]
    return unique_preserve_order(normalized) if keep_order else list(dict.fromkeys(normalized))


# ----------------------------------
# PLATFORM
# ----------------------------------

def platform_match(platform):
    current_platform = sys.platform
    if platform == "all":
        return True
    if platform == "windows" and current_platform.startswith("win"):
        return True
    if platform == "linux" and current_platform.startswith("linux"):
        return True
    if platform == "mac" and current_platform.startswith("darwin"):
        return True
    return False


# ----------------------------------
# UNIQUE
# ----------------------------------
def unique(seq):
    return list(dict.fromkeys(seq))


# ----------------------------------
# PLUGIN VERSION
# ----------------------------------
def get_current_version(bk):
    _installed_version = None
    ppath = Path(bk._w.plugin_dir) / bk._w.plugin_name / "plugin.xml"
    with open(ppath,'rb') as f:
        data = f.read()
        data = data.decode('utf-8', 'ignore')
        m = _VERSION_PATTERN.search(data)
        if m:
            _installed_version = m.group(1).strip()
    return _installed_version


# ---------------------------------------------------------
# ENV-VARS FILE HANDLING
# ---------------------------------------------------------
def get_env_vars_path(plugin_dir):
    prefs_dir = os.getenv("SIGIL_PREFS_DIR")
    if prefs_dir and Path(prefs_dir).is_dir():
        return (Path(prefs_dir) / "env-vars.txt").resolve()
    return (Path(plugin_dir) / ".." / "env-vars.txt").resolve()


def load_env_vars(file_path):
    envs = {}
    if os.path.exists(file_path):
        with open(file_path, "r", encoding="utf-8") as f:
            for line in f:
                line = line.strip()
                if not line or line.startswith("#") or "=" not in line:
                    continue
                key, val = line.split("=", 1)
                # Clean loaded lang codes
                if key == "SIGIL_ONLY_USE_LANGCODES":
                    languages = normalize_languages_list(val)
                    val = ",".join(languages)
                envs[key.strip()] = val.strip()
    return envs


def save_env_vars(file_path, new_envs):
    """
    Save updated environment variables, preserving comments and unknown entries,
    and remove known variables that were unchecked or cleared.
    """
    lines = []
    seen = set()
    known_keys = {v["name"] for v in ENV_VARIABLES}

    if os.path.exists(file_path):
        with open(file_path, "r", encoding="utf-8") as f:
            for line in f:
                orig_line = line.rstrip("\n")
                stripped = line.strip()
                if not stripped:
                    lines.append(orig_line)
                    continue

                if stripped.startswith("#"):
                    lines.append(orig_line)
                    continue

                if "=" not in stripped:
                    lines.append(orig_line)
                    continue

                key, _ = stripped.split("=", 1)
                key = key.strip()

                if key == "SIGIL_PREFS_DIR":
                    continue

                if key in known_keys:
                    if key in new_envs:
                        lines.append(f"{key}={new_envs[key]}")
                        seen.add(key)
                else:
                    # unknown variable: preserve as-is
                    lines.append(orig_line)
                    seen.add(key)

    for key, val in new_envs.items():
        if key not in seen and key in known_keys and key != "SIGIL_PREFS_DIR":
            print(f"ADD new var {key} -> {val}")
            lines.append(f"{key}={val}")

    with open(file_path, "w", encoding="utf-8") as f:
        for line in lines:
            f.write(line + "\n")


# ---------------------------------------------------------
# DIALOG
# ---------------------------------------------------------
class EnvVarDialog(QtWidgets.QDialog):
    def __init__(self, env_vars, existing, plugin_version):
        super().__init__()
        self.setWindowTitle("Sigil Environment Variables" + " " + plugin_version + showbeta())
        self.resize(570, 580)
        self.setSizePolicy(QtWidgets.QSizePolicy.Fixed, QtWidgets.QSizePolicy.Fixed)

        self.widgets = {}
        self._values_to_save = {}
        layout = QtWidgets.QVBoxLayout()
        form = QtWidgets.QFormLayout()

        self.defaults = {}

        for var in env_vars:
            if not platform_match(var["platform"]):
                continue

            name = var["name"]
            desc = var["description"]
            vtype = var["type"]
            current_val = existing.get(name, "")

            if vtype == "bool":
                w = QtWidgets.QCheckBox()
                w.setChecked(bool(current_val))
                self.defaults[name] = False
            elif vtype == "int":
                w = QtWidgets.QSpinBox()
                min_val = var.get("min", -10000)
                max_val = var.get("max", 10000)
                w.setRange(min_val, max_val)
                w.setFixedWidth(70)

                # 0 true default
                self.defaults[name] = 0

                if current_val:
                    try:
                        val = int(current_val)
                        val = max(w.minimum(), min(w.maximum(), val))
                        w.setValue(val)
                    except ValueError:
                        pass

            elif vtype == "special":
                w = QtWidgets.QLabel()
                if name == "SIGIL_PREFS_DIR":
                    prefs_dir = os.getenv("SIGIL_PREFS_DIR")
                    if prefs_dir and os.path.isdir(prefs_dir):
                        w.setText(prefs_dir)
                    else:
                        w.setText("Not set")
                self.defaults[name] = ""
            else:  # string
                w = QtWidgets.QLineEdit()
                if current_val:
                    w.setText(current_val)
                if name == "SIGIL_FOCUS_HIGHLIGHT_COLOR":
                    w.setFixedWidth(70)
                else:
                    w.setFixedWidth(260)
                self.defaults[name] = ""


            uniform_height = 24
            w.setMinimumHeight(uniform_height)
            w.setToolTip(desc)


            # --- Is variable set in system env? ---
            row_widget = QtWidgets.QWidget()
            row_layout = QtWidgets.QHBoxLayout()
            row_layout.setContentsMargins(0, 0, 0, 0)
            row_layout.addWidget(w)

            if name in os.environ and name not in existing:
                value = os.environ[name]
                addendum = ""
                if value == "1":
                    addendum = " (= enabled)"
                mark = QtWidgets.QLabel("[!]")
                ss = "QLabel {color: #D53051; font-weight: bold;} QTipLabel {color:initial; font-weight: normal;}";
                mark.setStyleSheet(ss)
                mark.setToolTip(f"Variable '{name}' is set in system.<br/>Value: {value}{addendum}")
                row_layout.addWidget(mark)

            row_widget.setLayout(row_layout)

            form.addRow(QtWidgets.QLabel(name), row_widget)

            #form.setAlignment(w, QtCore.Qt.AlignLeft | QtCore.Qt.AlignVCenter)
            form.setAlignment(w, QtCore.Qt.AlignVCenter)
            form.setAlignment(row_widget, QtCore.Qt.AlignLeft | QtCore.Qt.AlignVCenter)

            self.widgets[name] = (w, vtype)

        # --- Unknown variables (readonly labels) ---
        unknowns = {
            k: v for k, v in existing.items()
            if k not in [v["name"] for v in env_vars]
        }
        if unknowns:
            form.addRow(QtWidgets.QLabel("<b>Unknown variables (read-only):</b>"), QtWidgets.QLabel(""))
            for name, val in unknowns.items():
                lbl = QtWidgets.QLabel(val)
                lbl.setToolTip("Unknown variable from env-vars.txt")
                form.addRow(QtWidgets.QLabel(name), lbl)

        # --- Scrollable area ---
        form_widget = QtWidgets.QWidget()
        form_widget.setLayout(form)

        scroll = QtWidgets.QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        scroll.setWidget(form_widget)

        layout.addWidget(scroll)


        buttons = QtWidgets.QDialogButtonBox(
            QtWidgets.QDialogButtonBox.Save | QtWidgets.QDialogButtonBox.Cancel
        )
        buttons.accepted.connect(self.on_accept)
        buttons.rejected.connect(self.reject)

        layout.addWidget(buttons)
        self.setLayout(layout)


    def get_values(self):
        results = {}
        hex_color_re = re.compile(r"^#[0-9A-Fa-f]{6}$")

        for name, (w, vtype) in self.widgets.items():
            default_val = self.defaults[name]

            if vtype == "bool":
                if w.isChecked():
                    results[name] = "1"

            elif vtype == "int":
                if w.value() != default_val:
                    if name == "SIGIL_PREVIEW_TIMEOUT":
                        if w.value() > 0 and w.value() < 1000:
                            results[name] = "1000"
                    else:
                        results[name] = str(w.value())

            else:  # string
                val = w.text().strip()
                if val and val != default_val:
                    if name == "SIGIL_FOCUS_HIGHLIGHT_COLOR":
                        if hex_color_re.match(val):
                            results[name] = val
                        else:
                            QtWidgets.QMessageBox.warning(
                                self,
                                "Invalid color",
                                f"Value '{val}' for {name} is not a valid color in #rrggbb format."
                            )
                            w.setFocus()
                            w.selectAll()
                            return None
                    elif name == "SIGIL_ONLY_USE_LANGCODES":
                        languages = normalize_languages_list(val)
                        val_norm = ",".join(languages)
                        w.setText(val_norm)
                        invalid = [lang for lang in languages if not is_valid_language(lang)]
                        if not invalid:
                            results[name] = val_norm
                        else:
                            QtWidgets.QMessageBox.warning(
                                self,
                                "Oops!",
                                "Invalid language codes:\n" + ", ".join(invalid)
                            )
                            w.setFocus()
                            #w.selectAll()
                            return None
                    else:
                        results[name] = val

        return results


    def on_accept(self):
        """Validation before dialog close"""
        values = self.get_values()
        if values is None:
            return  # validation error, dialog still open
        self._values_to_save = values
        self.accept()


def run(bk):
    if bk.launcher_version() < 20250810:
        print(f"Error: {bk._w.plugin_name} requires a newer version of Sigil >= 2.6.2\n")
        known_sigil_envs = {v["name"] for v in ENV_VARIABLES}
        sigil_vars = {key: value for key, value in os.environ.items() if key in known_sigil_envs}
        if sigil_vars:
            print('You have defined the following Sigil environment variables:')
            for key, value in sigil_vars.items():
                print(f"{key}: {value}")
        return -1

    icon = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'plugin.svg')
    mdp = True if iswindows else False
    app = PluginApplication(sys.argv, bk, app_icon=icon, match_dark_palette=mdp,
                            dont_use_native_menubars=True)

    plugin_dir = bk._w.plugin_dir
    env_file = get_env_vars_path(plugin_dir)
    existing = load_env_vars(env_file)

    _current_version = get_current_version(bk)
    dlg = EnvVarDialog(ENV_VARIABLES, existing, _current_version)
    if dlg.exec_() == QtWidgets.QDialog.Accepted:
        new_values = dlg.get_values()
        if new_values:
            save_env_vars(env_file, new_values)
            # summary
            print(f"File: {env_file}\n")
            print("Saved variables:")
            for k, v in new_values.items():
                if k != "SIGIL_PREFS_DIR":
                    print(f"{k}={v}")

            # unknowns preserved
            unknowns = {k: v for k, v in existing.items() if k not in [v["name"] for v in ENV_VARIABLES]}
            if unknowns:
                print("\nPreserved unknown variables:")
                for k, v in unknowns.items():
                    print(f"{k}={v}")
    else:
        print("The user closed the dialog box.")

    print("\nPlease note that any changes to Sigil’s environment variables\nwill only take effect the next time Sigil is launched.")
    return 0

def main():
    print("I reached main when I should not have\n")
    return -1

if __name__ == "__main__":
    sys.exit(main())
