#!/usr/bin/env python

__license__   = 'GPL v3'
__copyright__ = '2023-2024, Thiago Oliveira <thiago.eec@gmail.com>'
__docformat__ = 'restructuredtext en'

# Load translation files (.mo) on the folder 'translations'
load_translations()

# Standard libraries
import os
from functools import partial

# PyQt libraries
from qt.core import (Qt, QtCore, QWidget, QVBoxLayout, QLabel, QLineEdit, QGroupBox, QComboBox,
                     QDialogButtonBox, QIcon, QGridLayout, QCheckBox, QApplication, QPushButton,
                     QPixmap, QDialog, QTabWidget, QIntValidator, QEvent)

# Calibre libraries
from calibre.utils.filenames import expanduser
from calibre.utils.config import JSONConfig, config_dir
from calibre.gui2 import gprefs, info_dialog, question_dialog, error_dialog, choose_dir, choose_files
from calibre.gui2.keyboard import ShortcutConfig
from calibre_plugins.Reading_Goal.__init__ import PLUGIN_VERSION, PLUGIN_NAME, PLUGIN_THREAD
from calibre.gui2.preferences.create_custom_column import CreateNewCustomColumn


# Check calibre theme
def is_dark_theme() -> bool:
    return QApplication.instance().is_dark_theme


# Custom columns map
DEFAULT_LOOKUP_READING_PROGRESS_COLUMN: str     = '#percentage'
DEFAULT_LOOKUP_STATUS_DATE_COLUMN: str          = '#read_date'
DEFAULT_LOOKUP_PAGE_COUNT_COLUMN: str           = '#pages'
DEFAULT_LOOKUP_GENRE_COLUMN: str                = '#genres'
DEFAULTS: dict[str, dict] = {
                DEFAULT_LOOKUP_READING_PROGRESS_COLUMN: {
                    'column_heading': _('% Read'),
                    'datatype': 'float',
                    'description': _('Reading progress'),
                    'number_format': '{:.0f}%',
                },
                DEFAULT_LOOKUP_STATUS_DATE_COLUMN: {
                    'column_heading': _('Status date'),
                    'datatype': 'datetime',
                    'description': _('Reading progress date'),
                },
                DEFAULT_LOOKUP_PAGE_COUNT_COLUMN: {
                    'column_heading': _('Pages'),
                    'datatype': 'int',
                    'description': _('Page count'),
                    'number_format': None,
                },
                DEFAULT_LOOKUP_GENRE_COLUMN: {
                    'column_heading': _("Genres"),
                    'datatype': 'text',
                    'description': _("Book genre"),
                },
            }

# This is where all preferences for this plugin will be stored
prefs = JSONConfig('plugins/Reading_Goal')

# Set config defaults
prefs.defaults['reading_progress_column'] = ''
prefs.defaults['status_date_column'] = ''
prefs.defaults['page_count_column'] = ''
prefs.defaults['genre_column'] = ''
prefs.defaults['sorting'] = 'Title'
prefs.defaults['main_action'] = 'Statistics'
prefs.defaults['autofill'] = True
prefs.defaults['auto_add_column'] = ''
prefs.defaults['auto_add_value'] = ''
prefs.defaults['auto_add_challenges'] = ''
prefs.defaults['auto_add_check'] = False
prefs.defaults['backup_path'] = os.path.join(expanduser('~'), 'Reading Goal')
prefs.defaults['backup_interval'] = 7
prefs.defaults['backup_date'] = {}
prefs.defaults['backup_files'] = ''
prefs.defaults['allow_other_years'] = False
prefs.defaults['rereading_shelf'] = True
prefs.defaults['use_rereading_color'] = True
prefs.defaults['use_rereading_counter'] = False
prefs.defaults['keep_rereading_color'] = False
prefs.defaults['jump_to_selected'] = False
prefs.defaults['pages_instead_of_books'] = False
prefs.defaults['auto_collapse'] = ''
prefs.defaults['shelf_colors'] = ''
prefs.defaults['state'] = {'CustomChallengesDialog': '', 'EditGoalDialog': '',
                           'LinkBookDialog': '', 'EditRecordsDialog': ''}
prefs.defaults['database_migrated'] = False
prefs.defaults['geometry_reset'] = True

# Simplified version that works on recent calibre versions
# def get_icon(icon_name):
#     icon_name = 'images/' + icon_name
#
#     # Choose the suffix
#     tc = 'dark' if is_dark_theme() else 'light'
#     sq, ext = os.path.splitext(icon_name)
#     sq = f'{sq}_{tc}{ext}'
#
#     return get_icons(sq)


def get_icon(icon_name: str) -> QIcon:
    # Check to see whether the icon exists as a Calibre resource
    # This will enable skinning if the user stores icons within a folder like:
    # ...\AppData\Roaming\calibre\resources\images\Plugin Name\

    icon_name = 'images/' + icon_name

    # First, look for the themed icon (-for-dark-theme / -for-light-theme)
    tc = 'dark' if is_dark_theme() else 'light'
    sq, ext = os.path.splitext(icon_name)
    sq = f'{sq}-for-{tc}-theme{ext}'
    icon = QIcon.ic(PLUGIN_NAME + '/' + sq.replace('images/', ''))  # (..\resources\images\Plugin Name\)
    if icon.isNull():
        icon = QIcon.ic(sq.replace('images/', ''))  # Theme package
    if icon.isNull():
        # Then, look for the regular icon (..\resources\images\Plugin Name\)
        sq, ext = os.path.splitext(icon_name)
        if 'help' not in icon_name:  # There is only one help icon
            sq = f'{sq}_{tc}{ext}'
        else:
            sq = icon_name
        icon = QIcon.ic(PLUGIN_NAME + '/' + sq.replace('images/', ''))
        if not icon.isNull():
            return icon
    else:
        return icon

    # As we did not find an icon elsewhere, look within our zip resources
    return get_icons(sq)


class ConfigWidget(QWidget):

    def __init__(self, plugin_action, tab=0):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        layout = QVBoxLayout()
        self.setLayout(layout)

        tab_widget = QTabWidget(self)
        layout.addWidget(tab_widget)

        self.main_tab = MainTab(self)
        self.columns_tab = ColumnsTab(self)
        tab_widget.addTab(self.main_tab, ' ' + _('&Main') + ' ')
        tab_widget.addTab(self.columns_tab, ' ' + _('&Columns') + ' ')
        tab_widget.setCurrentIndex(tab)

        self.must_restart = False
        self._get_create_new_custom_column_instance = None

        self.resize(self.sizeHint())

    def refresh_widget(self, opt: str = None) -> None:
        rereading_shelf_checked = self.main_tab.rereading_shelf_check.isChecked()
        use_rereading_color_checked = self.main_tab.use_rereading_color_check.isChecked()
        if opt == 'rereading_shelf':
            if rereading_shelf_checked:
                self.main_tab.use_rereading_color_check.setChecked(not rereading_shelf_checked)
        elif opt == 'use_rereading_color':
            if use_rereading_color_checked:
                self.main_tab.rereading_shelf_check.setChecked(not use_rereading_color_checked)
        if not (rereading_shelf_checked or use_rereading_color_checked):
            self.main_tab.use_rereading_counter_check.setChecked(False)
            self.main_tab.use_rereading_counter_check.setEnabled(False)
            if is_dark_theme():
                self.main_tab.use_rereading_counter_check.setStyleSheet('color: gray')
        else:
            self.main_tab.use_rereading_counter_check.setEnabled(True)
            self.main_tab.use_rereading_counter_check.setStyleSheet('')

        self.update()

    def get_directory(self, mouseEvent: QEvent) -> None:
        c = choose_dir(self, PLUGIN_NAME + 'dir_chooser', _('Select a folder to save your backups to'))
        if c:
            self.columns_tab.directory_txtBox.setReadOnly(False)
            self.columns_tab.directory_txtBox.setText(c)
            self.columns_tab.directory_txtBox.setReadOnly(True)

    def restore_backup(self) -> None:
        # Library id
        self.library_id = self.plugin_action.gui.current_db.library_id
        try:
            database_path = os.path.join(config_dir, 'plugins')
            database_file = os.path.join(database_path, 'Reading_Goal_Data_' + self.library_id + '.json')

            # Restore backup file
            ans = question_dialog(self, _('Are you sure?'),
                                  _('Notice that the existing database will be completely replaced '
                                    'with the new one. Do you want to proceed?'),
                                  show_copy_button=False, override_icon=QIcon.ic('dialog_warning.png'))
            if ans:
                import_file_path = choose_files(self, 'plugin-reading-goal-import-file',
                                                _('Select the database file you want to import'),
                                                filters=[(_('JSON file'), ['json'])],
                                                select_only_single_file=True)
                if import_file_path:
                    with open(import_file_path[0], 'r') as i_f:
                        candidate = i_f.read()
                        if 'general_info' not in candidate and 'timestamp' not in candidate:
                            return error_dialog(self.plugin_action.gui, _('Invalid database'),
                                                _('The selected file is not a valid database file'),
                                                show_copy_button=True, show=True)
                        with open(database_file, 'w') as b_f:
                            b_f.write(candidate)

                    info_dialog(self, _('Backup'), _('Backup restored'), show_copy_button=False, show=True)

            return None

        except:
            import traceback
            return error_dialog(self.plugin_action.gui, _('Restore backup'),
                                _('Restoring backup failed. Click on \'Show details\' for more info.'),
                                det_msg=traceback.format_exc(), show_copy_button=True, show=True)

    def reset_dialogs(self) -> None:
        auto_skip = gprefs['questions_to_auto_skip']
        for item in auto_skip:
            if item in ('plugin-reading-goal-remove-challenge-again', 'plugin-reading-goal-remove-goal-again'):
                gprefs['questions_to_auto_skip'].remove(item)
        info_dialog(self, _('Done'), _('Confirmation dialogs have all been reset'), show=True, show_copy_button=False)

    def edit_shortcuts(self) -> None:
        d = KeyboardConfigDialog(self.plugin_action.gui, PLUGIN_NAME)
        if d.exec_() == d.Accepted:
            self.plugin_action.gui.keyboard.finalize()

    def get_custom_columns(self, column_types: list[str] | str) -> dict:
        custom_columns = self.plugin_action.gui.library_view.model().custom_columns
        available_columns = {}
        for key, column in custom_columns.items():
            typ = column['datatype']
            if typ in column_types or column_types == 'all':
                available_columns[key] = column
        return available_columns

    def create_custom_column(self, combo_box: QComboBox, custom_columns: dict,
                             lookup_name: str = None, is_multiple: bool = False) -> None:
        display_params = {
            'description': DEFAULTS[lookup_name]['description'],
        }

        datatype = DEFAULTS[lookup_name]['datatype']

        if datatype in ['int', 'float']:
            display_params['number_format'] = DEFAULTS[lookup_name]['number_format']

        column_heading = DEFAULTS[lookup_name]['column_heading']
        new_lookup_name = lookup_name
        create_new_custom_column_instance = self.get_create_new_custom_column_instance()
        result = create_new_custom_column_instance.create_column(new_lookup_name, column_heading, datatype,
                                                                 is_multiple,
                                                                 display=display_params,
                                                                 generate_unused_lookup_name=True,
                                                                 freeze_lookup_name=False)
        if result[0] == CreateNewCustomColumn.Result.COLUMN_ADDED:
            combo_box.populate_combo(custom_columns, result[1], column_heading)
            self.must_restart = True

    def get_create_new_custom_column_instance(self) -> CreateNewCustomColumn:
        if self._get_create_new_custom_column_instance is None:
            self._get_create_new_custom_column_instance = CreateNewCustomColumn(self.plugin_action.gui)
        return self._get_create_new_custom_column_instance

    def about(self) -> None:
        text = '<hr/>' \
               '<p>' + _('Created by Thiago Oliveira')+'</p>' \
               '<p>' + _('For help using this plugin, click on the \'?\' icon on the upper right corner '
                         'of the configuration dialog. Then, hover the mouse over the options, and click '
                         'when the cursor changes.') + '</p>' \
               '<p>' + _('You can also find instructions in the plugin thread at ') + \
               '<a href="' + PLUGIN_THREAD + '">Mobileread.com</a>.</p>' + \
               '<p>' + _('If any doubt still persists, feel free to ask there. Suggestions are welcomed too.') + '</p>'

        about_text = '{0}{1}'.format(PLUGIN_NAME + ' v' + PLUGIN_VERSION, text)
        AboutDialog(self, get_icon('info.png'), about_text).exec_()

    def save_settings(self) -> None:
        # Avoid saving localized strings to prefs
        main_action_dict = {
                            _('Statistics'): 'Statistics',
                            _('Edit reading goal'): 'Edit reading goal',
                            _('Add to reading goal'): 'Add to reading goal'
                            }
        sorting_dict = {
                        _('Title'): 'Title',
                        _('Title sort'): 'Title sort',
                        _('Date (asc)'): 'Date (asc)',
                        _('Date (desc)'): 'Date (desc)'
                        }

        # Main tab
        prefs['main_action'] = main_action_dict[self.main_tab.main_action_combo.currentText()]
        prefs['sorting'] = sorting_dict[self.main_tab.sorting_combo.currentText()]
        prefs['autofill'] = self.main_tab.autofill_check.isChecked()
        prefs['allow_other_years'] = self.main_tab.allow_other_years_check.isChecked()
        prefs['rereading_shelf'] = self.main_tab.rereading_shelf_check.isChecked()
        prefs['use_rereading_color'] = self.main_tab.use_rereading_color_check.isChecked()
        prefs['use_rereading_counter'] = self.main_tab.use_rereading_counter_check.isChecked()
        prefs['keep_rereading_color'] = self.main_tab.keep_color_check.isChecked()
        prefs['jump_to_selected'] = self.main_tab.jump_to_selected.isChecked()
        prefs['pages_instead_of_books'] = self.main_tab.pages_instead_of_books.isChecked()
        prefs['auto_collapse'] = self.main_tab.auto_collapse.text()
        prefs['shelf_colors'] = self.main_tab.shelf_colors.text()

        # Columns tab
        prefs['reading_progress_column'] = self.columns_tab.reading_progress_column_combo.get_selected_column()
        prefs['status_date_column'] = self.columns_tab.status_date_column_combo.get_selected_column()
        prefs['page_count_column'] = self.columns_tab.page_count_column_combo.get_selected_column()
        prefs['genre_column'] = self.columns_tab.genre_column_combo.get_selected_column()
        if self.columns_tab.auto_add_column_combo.get_selected_column() != prefs['auto_add_column']:
            # If this field has changed, mark the database for a global check of auto added books
            prefs['auto_add_check'] = True
        prefs['auto_add_column'] = self.columns_tab.auto_add_column_combo.get_selected_column()
        if prefs['auto_add_column']:
            if self.columns_tab.auto_add_value.text() != prefs['auto_add_value']:
                # If this field has changed, mark the database for a global check of auto added books
                prefs['auto_add_check'] = True
            prefs['auto_add_value'] = self.columns_tab.auto_add_value.text()
        else:
            prefs['auto_add_value'] = ''
        if self.columns_tab.auto_add_challenges.text() != prefs['auto_add_challenges']:
            # If this field has changed, mark the database for a global check of auto added books
            prefs['auto_add_check'] = True
        prefs['auto_add_challenges'] = self.columns_tab.auto_add_challenges.text()
        prefs['backup_path'] = str(self.columns_tab.directory_txtBox.displayText())
        prefs['backup_interval'] = self.columns_tab.backup_interval_txtBox.text()
        prefs['backup_files'] = self.columns_tab.backup_files_txtBox.text()


class MainTab(QWidget):

    def __init__(self, parent_dialog):
        self.parent_dialog = parent_dialog
        QWidget.__init__(self)
        layout = QGridLayout()
        self.setLayout(layout)

        # --- Customize interface --- #
        customize_group_box = QGroupBox(_('Customize interface'))
        layout.addWidget(customize_group_box, 0, 0, 1, 1)
        customize_group_box_layout = QGridLayout()
        customize_group_box.setLayout(customize_group_box_layout)

        # Main action combobox
        self.main_action_box_label = QLabel(_('Main &button:'), self)
        tooltip = _('Select which action will be used for the main button on calibre GUI')
        self.main_action_box_label.setToolTip(tooltip)
        self.main_action_combo = QComboBox()
        self.main_action_combo.setToolTip(tooltip)
        self.main_action_combo.setMinimumWidth(200)
        self.main_action_combo.addItems({_('Add to reading goal'), _('Edit reading goal'), _('Statistics')})
        self.main_action_combo.model().sort(0)
        self.main_action_box_label.setBuddy(self.main_action_combo)
        customize_group_box_layout.addWidget(self.main_action_box_label, 0, 0)
        customize_group_box_layout.addWidget(self.main_action_combo, 0, 1)
        self.main_action_combo.setCurrentText(_(prefs['main_action']))

        # Sorting
        self.sorting_box_label = QLabel(_('&Sort by:'), self)
        tooltip = _("Choose how books should be sorted on the 'Edit reading goal' dialog")
        self.sorting_box_label.setToolTip(tooltip)
        self.sorting_combo = QComboBox()
        self.sorting_combo.setToolTip(tooltip)
        self.sorting_combo.setMinimumWidth(200)
        self.sorting_combo.addItems({_('Date (asc)'), _('Date (desc)'), _('Title'), _('Title sort')})
        self.sorting_combo.model().sort(0)
        self.sorting_box_label.setBuddy(self.sorting_combo)
        customize_group_box_layout.addWidget(self.sorting_box_label, 1, 0)
        customize_group_box_layout.addWidget(self.sorting_combo, 1, 1)
        self.sorting_combo.setCurrentText(_(prefs['sorting']))

        # --- Main options --- #
        main_options_group_box = QGroupBox(_('Main options'))
        layout.addWidget(main_options_group_box, 2, 0, 1, 1)
        main_options_group_box_layout = QGridLayout()
        main_options_group_box.setLayout(main_options_group_box_layout)

        # Autofill next year reading goal
        self.autofill_check = QCheckBox(_('Migrate unfinished books to the &next year\'s goal'), self)
        self.autofill_check.setToolTip(_('Automatically moves unfinished books to your next year\'s reading goal'))
        main_options_group_box_layout.addWidget(self.autofill_check, 0, 0)
        # Load the checkbox with the current preference setting
        self.autofill_check.setChecked(prefs['autofill'])

        # Allow other year's reading goal
        self.allow_other_years_check = QCheckBox(_('&Allow adding other year\'s reading goal'), self)
        self.allow_other_years_check.setToolTip(_('Check this option if you want the ability to add other year\'s '
                                                  'reading goal.\nAll selected books must have the same year '
                                                  'in their records.'))
        main_options_group_box_layout.addWidget(self.allow_other_years_check, 1, 0)
        # Load the checkbox with the current preference setting
        self.allow_other_years_check.setChecked(prefs['allow_other_years'])

        # Use rereading shelf
        self.rereading_shelf_check = QCheckBox(_('&Use a \'Rereading\' shelf'), self)
        self.rereading_shelf_check.setToolTip(_('Check this option to group the books you are rereading.'))
        main_options_group_box_layout.addWidget(self.rereading_shelf_check, 2, 0)
        # Load the checkbox with the current preference setting
        self.rereading_shelf_check.setChecked(prefs['rereading_shelf'])

        # Use rereading color
        self.use_rereading_color_check = QCheckBox(_('Us&e a \'Rereading\' color, instead of a shelf'), self)
        self.use_rereading_color_check.setToolTip(_('Check this option to use a different color for rereadings.'))
        main_options_group_box_layout.addWidget(self.use_rereading_color_check, 3, 0)
        # Load the checkbox with the current preference setting
        self.use_rereading_color_check.setChecked(prefs['use_rereading_color'])

        # Make sure 'rereading_shelf_check' and 'use_rereading_color_check' are never both checked
        self.rereading_shelf_check.stateChanged.connect(partial(self.parent_dialog.refresh_widget,
                                                                opt='rereading_shelf'))
        rereading_shelf_checked = self.rereading_shelf_check.isChecked()
        if rereading_shelf_checked:
            self.use_rereading_color_check.setChecked(not rereading_shelf_checked)
        self.use_rereading_color_check.stateChanged.connect(partial(self.parent_dialog.refresh_widget,
                                                                    opt='use_rereading_color'))
        use_rereading_color_checked = self.use_rereading_color_check.isChecked()
        if use_rereading_color_checked:
            self.rereading_shelf_check.setChecked(not use_rereading_color_checked)

        # Use rereading counter
        self.use_rereading_counter_check = QCheckBox(_('Use a rereading c&ounter'), self)
        self.use_rereading_counter_check.setToolTip(_('Check this option to add a counter for '
                                                      'how many times you\'ve reread the book.'))
        main_options_group_box_layout.addWidget(self.use_rereading_counter_check, 4, 0)
        # Load the checkbox with the current preference setting
        self.use_rereading_counter_check.setChecked(prefs['use_rereading_counter'])
        if not (rereading_shelf_checked or use_rereading_color_checked):
            self.use_rereading_counter_check.setChecked(False)
            self.use_rereading_counter_check.setEnabled(False)
            self.use_rereading_counter_check.setStyleSheet('color: gray')

        # Keep reareading color after 100%
        self.keep_color_check = QCheckBox(_('&Keep \'Rereading\' color for finished books'), self)
        self.keep_color_check.setToolTip(_('Check this option to retain the line color for books migrating from the '
                                           '\'Rereading\' shelf to the \'Read\' shelf.'))
        main_options_group_box_layout.addWidget(self.keep_color_check, 5, 0)
        # Load the checkbox with the current preference setting
        self.keep_color_check.setChecked(prefs['keep_rereading_color'])

        # Jump to the selected book
        self.jump_to_selected = QCheckBox(_('&Jump to the selected book'), self)
        self.jump_to_selected.setToolTip(_('When opening the \'Edit reading goal\' dialog, jump to the selected book '
                                           'in your calibre library.'))
        main_options_group_box_layout.addWidget(self.jump_to_selected, 6, 0)
        # Load the checkbox with the current preference setting
        self.jump_to_selected.setChecked(prefs['jump_to_selected'])

        # Pages instead of books
        self.pages_instead_of_books = QCheckBox(_('&Show pages instead of read books in statistics'), self)
        self.pages_instead_of_books.setToolTip(_('In the statistics dialog, show read pages instead of read books.'))
        main_options_group_box_layout.addWidget(self.pages_instead_of_books, 7, 0)
        # Load the checkbox with the current preference setting
        self.pages_instead_of_books.setChecked(prefs['pages_instead_of_books'])

        # --- Display options --- #
        display_options_group_box = QGroupBox(_('Display options'))
        layout.addWidget(display_options_group_box, 5, 0, 1, 1)
        display_options_group_box_layout = QGridLayout()
        display_options_group_box.setLayout(display_options_group_box_layout)

        # Auto collapse shelves
        self.auto_collapse_label = QLabel(_('Auto co&llapse:'), self)
        display_options_group_box_layout.addWidget(self.auto_collapse_label, 0, 0)
        self.auto_collapse = QLineEdit(self)
        tooltip = _('List the shelves you want to be collapsed when opening the \'Edit reading goal\' dialog.\n'
                    'Use a comma separated list with the shelves numbers. Example: 1, 3, 5')
        self.auto_collapse.setToolTip(tooltip)
        self.auto_collapse.setMinimumWidth(60)
        self.auto_collapse.setText(prefs['auto_collapse'])
        self.auto_collapse_label.setBuddy(self.auto_collapse)
        display_options_group_box_layout.addWidget(self.auto_collapse, 0, 1)
        # Load the checkbox with the current preference setting
        self.auto_collapse.setText(prefs['auto_collapse'])

        # Shelf colors
        self.shelf_colors_label = QLabel(_('S&helf colors:'), self)
        display_options_group_box_layout.addWidget(self.shelf_colors_label, 1, 0)
        self.shelf_colors = QLineEdit(self)
        tooltip = _('Set the color for each shelf on the \'Edit reading goal\' dialog.\n'
                    'Use a python dictionary with shelf number and RGB value.\n'
                    'Example: {1: (180, 230, 255), 3: (250, 190, 80)}')
        self.shelf_colors.setToolTip(tooltip)
        self.shelf_colors.setMinimumWidth(60)
        self.shelf_colors.setText(prefs['shelf_colors'])
        self.shelf_colors_label.setBuddy(self.shelf_colors)
        display_options_group_box_layout.addWidget(self.shelf_colors, 1, 1)
        # Load the checkbox with the current preference setting
        self.shelf_colors.setText(prefs['shelf_colors'])


class ColumnsTab(QWidget):

    def __init__(self, parent_dialog):
        self.parent_dialog = parent_dialog
        QWidget.__init__(self)
        layout = QGridLayout()
        self.setLayout(layout)

        self.add_icon = get_icon('add_column.png')
        self.add_icon_name = 'add_column_dark' if is_dark_theme() else 'add_column_light'

        # Help information
        help_ico_src = str(os.path.join(config_dir, 'plugins', PLUGIN_NAME, 'images', 'help.png'))
        help_img_src = '<img src="' + help_ico_src + '" width="25" height="25"> '
        add_column_ico_src = str(os.path.join(config_dir, 'plugins', PLUGIN_NAME, 'images', self.add_icon_name))
        add_column_img_src = '<img src="' + add_column_ico_src + \
                             '" style="vertical-align:bottom" width="20" height="20"> '
        whatsthis_add_column = \
            '<p>' + _('To create a new custom column, just click on the {0} icon. '
                      'In the opened dialog, if you want, edit the <i>Column heading</i> and <i>Description</i>, '
                      'and press OK. Before closing this dialog, repeat the process for each field you want to '
                      'create a new custom column for. You will be prompted to restart calibre after you are done '
                      'creating the custom columns.').format(add_column_img_src) + '</p>'
        custom_columns_whatsthis = \
            '<h4 style="vertical-align:top">' + help_img_src + _('Custom columns') + '</h4>' + \
            '<p>' + _('You need to choose which custom column stores the corresponding information. '
                      'All fields must be configured, or the plugin won\'t work.') + '</p>' \
            '<p>' + _('Each field has its specific data type, so it only lists the suitable columns available. '
                      'If you don\'t want to use any of the existing columns, you can create a new custom column '
                      'matching the data type for that field.') + '</p>' + whatsthis_add_column
        whatsthis_auto_add = \
            '<h4 style="vertical-align:top">' + help_img_src + _('Auto add') + '</h4>' + \
            '<p>' + _('If you use other plugins that sync your reading activity, like <i>Goodreads Sync</i> or '
                      '<i>Skoob Sync</i>, you might want to automatically add your books from a <i>Want to read</i> '
                      'shelf. To accomplish this, you need to configure your sync plugin to use the same column '
                      'selected here.') + \
            '<p>' + _('To trigger the auto add, this column just need to be set for the book, with any value. '
                      'You can also check for a specific value, but only for some columns types: bool, int, '
                      'float or text. If the selected column is of one of those types, a value field will be '
                      'available.') + '</p>' + \
            '<p>' + _('Notice that all values are converted to string before comparing.') + '</p>'

        # --- Custom Columns --- #
        custom_columns_group_box = QGroupBox(_('Custom columns'), self)
        custom_columns_group_box.setWhatsThis(custom_columns_whatsthis)
        layout.addWidget(custom_columns_group_box, 0, 0, 2, 2)
        custom_columns_group_box_layout = QGridLayout()
        custom_columns_group_box.setLayout(custom_columns_group_box_layout)

        # Reading progress combobox
        self.reading_progress_box_label = QLabel(_('&Reading progress:'), self)
        tooltip = _('Select the custom column that stores the reading progress.\n'
                    'Type: floating point numbers / integers.')
        self.reading_progress_box_label.setToolTip(tooltip)
        self.reading_progress_box = QComboBox()
        reading_progress_custom_columns = self.parent_dialog.get_custom_columns(['float', 'int', 'composite'])
        self.reading_progress_column_combo = CustomColumnComboBox(self, reading_progress_custom_columns,
                                                                  initial_items=[''])
        self.reading_progress_column_combo.setToolTip(tooltip)
        self.reading_progress_column_combo.setMinimumWidth(200)
        self.reading_progress_box_label.setBuddy(self.reading_progress_column_combo)
        custom_columns_group_box_layout.addWidget(self.reading_progress_box_label, 0, 0)
        custom_columns_group_box_layout.addWidget(self.reading_progress_column_combo, 0, 1)
        self.reading_progress_column_combo_index = self.reading_progress_column_combo\
            .findText(prefs['reading_progress_column'], QtCore.Qt.MatchStartsWith)
        if self.reading_progress_column_combo_index == -1:
            self.reading_progress_column_combo.setCurrentIndex(0)
            prefs['reading_progress_column'] = ''
        else:
            self.reading_progress_column_combo.setCurrentIndex(self.reading_progress_column_combo_index)

        # Reading progress - add custom column button
        self.reading_progress_add_button = QPushButton(self.add_icon, '', self)
        self.reading_progress_add_button.setToolTip(_('Create a new column for Reading progress'))
        self.reading_progress_add_button.setMaximumWidth(50)
        custom_columns_group_box_layout.addWidget(self.reading_progress_add_button, 0, 2)
        self.reading_progress_add_button.clicked.connect(partial(self.parent_dialog.create_custom_column,
                                                                 self.reading_progress_column_combo,
                                                                 reading_progress_custom_columns,
                                                                 DEFAULT_LOOKUP_READING_PROGRESS_COLUMN, False))

        # Status date combobox
        self.status_date_box_label = QLabel(_('&Status date:'), self)
        tooltip = _('Select the custom column that stores the reading progress date.\n'
                    'Type: date.')
        self.status_date_box_label.setToolTip(tooltip)
        self.status_date_box = QComboBox()
        status_date_custom_columns = self.parent_dialog.get_custom_columns(['datetime'])
        self.status_date_column_combo = CustomColumnComboBox(self, status_date_custom_columns, initial_items=[''])
        self.status_date_column_combo.setToolTip(tooltip)
        self.status_date_column_combo.setMinimumWidth(200)
        self.status_date_box_label.setBuddy(self.status_date_column_combo)
        custom_columns_group_box_layout.addWidget(self.status_date_box_label, 1, 0)
        custom_columns_group_box_layout.addWidget(self.status_date_column_combo, 1, 1)
        self.status_date_column_combo_index = self.status_date_column_combo.\
            findText(prefs['status_date_column'], QtCore.Qt.MatchStartsWith)
        if self.status_date_column_combo_index == -1:
            self.status_date_column_combo.setCurrentIndex(0)
            prefs['status_date_column'] = ''
        else:
            self.status_date_column_combo.setCurrentIndex(self.status_date_column_combo_index)

        # Status date - add custom column button
        self.status_date_add_button = QPushButton(self.add_icon, '', self)
        self.status_date_add_button.setToolTip(_('Create a new column for Status date'))
        self.status_date_add_button.setMaximumWidth(50)
        custom_columns_group_box_layout.addWidget(self.status_date_add_button, 1, 2)
        self.status_date_add_button.clicked.connect(partial(self.parent_dialog.create_custom_column,
                                                            self.status_date_column_combo,
                                                            status_date_custom_columns,
                                                            DEFAULT_LOOKUP_STATUS_DATE_COLUMN, False))

        # Page count combobox
        self.page_count_box_label = QLabel(_('&Page count:'), self)
        tooltip = _('Select the custom column that stores the page count.\n'
                    'Type: integers.')
        self.page_count_box_label.setToolTip(tooltip)
        self.page_count_box = QComboBox()
        page_count_custom_columns = self.parent_dialog.get_custom_columns(['int'])
        self.page_count_column_combo = CustomColumnComboBox(self, page_count_custom_columns, initial_items=[''])
        self.page_count_column_combo.setToolTip(tooltip)
        self.page_count_column_combo.setMinimumWidth(200)
        self.page_count_box_label.setBuddy(self.page_count_column_combo)
        custom_columns_group_box_layout.addWidget(self.page_count_box_label, 2, 0)
        custom_columns_group_box_layout.addWidget(self.page_count_column_combo, 2, 1)
        self.page_count_column_combo_index = self.page_count_column_combo.findText(
            prefs['page_count_column'], QtCore.Qt.MatchStartsWith)
        if self.page_count_column_combo_index == -1:
            self.page_count_column_combo.setCurrentIndex(0)
            prefs['page_count_column'] = ''
        else:
            self.page_count_column_combo.setCurrentIndex(self.page_count_column_combo_index)

        # Page count - add custom column button
        self.page_count_add_button = QPushButton(self.add_icon, '', self)
        self.page_count_add_button.setToolTip(_('Create a new column for Page count'))
        self.page_count_add_button.setMaximumWidth(50)
        custom_columns_group_box_layout.addWidget(self.page_count_add_button, 2, 2)
        self.page_count_add_button.clicked.connect(partial(self.parent_dialog.create_custom_column,
                                                           self.page_count_column_combo,
                                                           page_count_custom_columns,
                                                           DEFAULT_LOOKUP_PAGE_COUNT_COLUMN, False))

        # Genre combobox
        self.genre_box_label = QLabel(_('&Genre:'), self)
        tooltip = _('Select the custom column that stores your genres.\n'
                    'Type: text / composite.')
        self.genre_box_label.setToolTip(tooltip)
        self.genre_box = QComboBox()
        genre_custom_columns = self.parent_dialog.get_custom_columns(['text', 'composite'])
        self.genre_column_combo = CustomColumnComboBox(self, genre_custom_columns, initial_items=['', 'tags'])
        self.genre_column_combo.setToolTip(tooltip)
        self.genre_column_combo.setMinimumWidth(200)
        self.genre_box_label.setBuddy(self.genre_column_combo)
        custom_columns_group_box_layout.addWidget(self.genre_box_label, 3, 0)
        custom_columns_group_box_layout.addWidget(self.genre_column_combo, 3, 1)
        self.genre_column_combo_index = self.genre_column_combo.findText(
            prefs['genre_column'], QtCore.Qt.MatchStartsWith)
        if self.genre_column_combo_index == -1:
            self.genre_column_combo.setCurrentIndex(0)
            prefs['genre_column'] = ''
        else:
            self.genre_column_combo.setCurrentIndex(self.genre_column_combo_index)

        # Genre - add custom column button
        self.genre_add_button = QPushButton(self.add_icon, '', self)
        self.genre_add_button.setToolTip(_('Create a new column for Genre'))
        self.genre_add_button.setMaximumWidth(50)
        custom_columns_group_box_layout.addWidget(self.genre_add_button, 3, 2)
        self.genre_add_button.clicked.connect(partial(self.parent_dialog.create_custom_column,
                                                      self.genre_column_combo,
                                                      genre_custom_columns,
                                                      DEFAULT_LOOKUP_GENRE_COLUMN, False))

        # --- Auto add --- #
        auto_add_group_box = QGroupBox(_('Auto add'))
        auto_add_group_box.setWhatsThis(whatsthis_auto_add)
        layout.addWidget(auto_add_group_box, 2, 0, 1, 2)
        auto_add_group_box_layout = QGridLayout()
        auto_add_group_box.setLayout(auto_add_group_box_layout)

        # Auto add to goal (combobox)
        self.auto_add_box_label = QLabel(_('Add to goal:'), self)
        tooltip = _('Select the custom column that stores the value for auto added books.\n'
                    'Type: any.')
        self.auto_add_box_label.setToolTip(tooltip)
        self.auto_add_box = QComboBox()
        auto_add_custom_columns = self.parent_dialog.get_custom_columns('all')
        self.auto_add_column_combo = CustomColumnComboBox(self, auto_add_custom_columns, initial_items=[''])
        self.auto_add_column_combo.setToolTip(tooltip)
        self.auto_add_column_combo.setMaximumWidth(200)
        self.auto_add_box_label.setBuddy(self.auto_add_column_combo)
        auto_add_group_box_layout.addWidget(self.auto_add_box_label, 0, 0)
        auto_add_group_box_layout.addWidget(self.auto_add_column_combo, 0, 1)
        self.auto_add_column_combo_index = self.auto_add_column_combo.findText(
            prefs['auto_add_column'], QtCore.Qt.MatchStartsWith)
        if self.auto_add_column_combo_index == -1:
            self.auto_add_column_combo.setCurrentIndex(0)
            prefs['auto_add_column'] = ''
        else:
            self.auto_add_column_combo.setCurrentIndex(self.auto_add_column_combo_index)

        # Auto add to goal (value)
        self.auto_add_value = QLineEdit(self)
        tooltip = _('Choose which value should be used to trigger auto add.\n'
                    'Leave it blank with you want to accept any non-empty value.')
        self.auto_add_value.setToolTip(tooltip)
        self.auto_add_value.setMinimumWidth(120)  # 260
        self.auto_add_value.setText(prefs['auto_add_value'])
        auto_add_group_box_layout.addWidget(self.auto_add_value, 0, 2)
        if prefs['auto_add_column'] == '':
            self.auto_add_value.hide()
        else:
            self.show_value(initial=True)
        self.auto_add_column_combo.currentIndexChanged.connect(self.show_value)

        # Auto add challenges
        self.auto_add_challenges_label = QLabel(_('Add to challenges:'), self)
        self.auto_add_challenges = QLineEdit(self)
        tooltip = _('Use a python dictionary following the model:\n'
                    '{"challenge_name_1": {"#custom_column" : "column_value"},'
                    '"challenge_name_2": {"tags": "some_tag"}}')
        self.auto_add_challenges_label.setToolTip(tooltip)
        self.auto_add_challenges.setToolTip(tooltip)
        self.auto_add_challenges.setMinimumWidth(60)
        self.auto_add_challenges.setMaximumWidth(200)
        self.auto_add_challenges.setText(prefs['auto_add_challenges'])
        self.auto_add_challenges_label.setBuddy(self.auto_add_challenges)
        auto_add_group_box_layout.addWidget(self.auto_add_challenges_label, 1, 0)
        auto_add_group_box_layout.addWidget(self.auto_add_challenges, 1, 1)
        # Load the lineedit with the current preference setting
        self.auto_add_challenges.setText(prefs['auto_add_challenges'])

        # --- Auto backup ---
        directory_group_box = QGroupBox(_('Auto backup'), self)
        layout.addWidget(directory_group_box, 3, 0, 1, 2)
        directory_group_box_layout = QGridLayout()
        directory_group_box.setLayout(directory_group_box_layout)

        # Directory path Textbox
        # Load the textbox with the current preference setting
        self.directory_txtBox = QLineEdit(prefs['backup_path'], self)
        self.directory_txtBox.setToolTip(_('Backup folder'))
        directory_group_box_layout.addWidget(self.directory_txtBox, 0, 0, 1, 2)
        self.directory_txtBox.mousePressEvent = self.parent_dialog.get_directory
        self.directory_txtBox.setReadOnly(True)

        # Auto backup interval
        self.backup_interval_txtBox_label = QLabel(_('&Backup interval:'), self)
        _tooltip = _('Backup interval in days. An empty value means no backup.')
        self.backup_interval_txtBox_label.setToolTip(_tooltip)
        # Load the textbox with the current preference setting
        self.backup_interval_txtBox = QLineEdit(str(prefs['backup_interval']), self)
        self.backup_interval_txtBox.setAlignment(QtCore.Qt.AlignRight)
        self.backup_interval_txtBox.setMaximumWidth(100)
        self.backup_interval_txtBox.setToolTip(_tooltip)
        self.backup_interval_txtBox_label.setBuddy(self.backup_interval_txtBox)
        self.backup_interval_txtBox.setValidator(QIntValidator(0, 100))
        directory_group_box_layout.addWidget(self.backup_interval_txtBox_label, 1, 0)
        directory_group_box_layout.addWidget(self.backup_interval_txtBox, 1, 1)

        # Number of backup files
        self.backup_files_txtBox_label = QLabel(_('&Number of backup files:'), self)
        _tooltip = _('Number of backup files to keep. An empty value means keeping all.')
        self.backup_files_txtBox_label.setToolTip(_tooltip)
        # Load the textbox with the current preference setting
        self.backup_files_txtBox = QLineEdit(str(prefs['backup_files']), self)
        self.backup_files_txtBox.setAlignment(QtCore.Qt.AlignRight)
        self.backup_files_txtBox.setMaximumWidth(100)
        self.backup_files_txtBox.setToolTip(_tooltip)
        self.backup_files_txtBox_label.setBuddy(self.backup_files_txtBox)
        self.backup_files_txtBox.setValidator(QIntValidator(0, 100))
        directory_group_box_layout.addWidget(self.backup_files_txtBox_label, 2, 0)
        directory_group_box_layout.addWidget(self.backup_files_txtBox, 2, 1)

        # Restore backup button
        self.restore_backup_button = QPushButton(_('Res&tore backup'), self)
        self.restore_backup_button.setToolTip(_('Restore a previously saved backup'))
        self.restore_backup_button.clicked.connect(self.parent_dialog.restore_backup)
        layout.addWidget(self.restore_backup_button, 4, 0, 1, 2)

        # --- Buttons ---
        # Reset confirmations button
        self.reset_button = QPushButton(_('R&eset confirmation dialogs'), self)
        self.reset_button.setToolTip(_('Reset all show me again dialogs'))
        self.reset_button.clicked.connect(self.parent_dialog.reset_dialogs)
        layout.addWidget(self.reset_button, 5, 0, 1, 2)

        # Keyboard shortcuts button
        self.keyboard_shortcuts_button = QPushButton(_('&Keyboard shortcuts'), self)
        self.keyboard_shortcuts_button.setToolTip(_('Edit the keyboard shortcuts associated with this plugin'))
        self.keyboard_shortcuts_button.clicked.connect(self.parent_dialog.edit_shortcuts)
        layout.addWidget(self.keyboard_shortcuts_button, 6, 0, 1, 1)

        # About button
        self.about_button = QPushButton(_('&About'), self)
        self.about_button.setToolTip(_('About the plugin'))
        self.about_button.clicked.connect(self.parent_dialog.about)
        layout.addWidget(self.about_button, 6, 1, 1, 1)

    def show_value(self, initial: bool = False) -> None:
        if not initial:
            self.auto_add_value.setText('')
        selected_column = self.auto_add_column_combo.get_selected_column()
        custom_columns = self.parent_dialog.plugin_action.gui.library_view.model().custom_columns
        if selected_column and custom_columns[selected_column]['datatype'] in ('bool', 'float', 'int', 'text'):
            if selected_column == prefs['auto_add_column']:
                self.auto_add_value.setText(prefs['auto_add_value'])
            else:
                self.auto_add_value.setText('')
            self.auto_add_value.show()
        else:
            self.auto_add_value.hide()


class CustomColumnComboBox(QComboBox):

    def __init__(self, parent, custom_columns: dict = None, selected_column: str = '',
                 initial_items: list[str] = None, new_column_created: list[str] = None):
        custom_columns = custom_columns if custom_columns else {}
        initial_items = initial_items if initial_items else ['']
        new_column_created = new_column_created if new_column_created else []
        QComboBox.__init__(self, parent)
        self.populate_combo(custom_columns, selected_column, new_column_created, initial_items)

    def populate_combo(self, custom_columns: dict = None, selected_column: str = '',
                       new_column_created: list[str] = None, initial_items: list[str] = None) -> None:
        new_column_created = new_column_created if new_column_created else []
        initial_items = initial_items if initial_items else ['']
        self.clear()
        self.column_names = list(initial_items)
        if len(initial_items) > 0:
            self.addItems(initial_items)
        selected_idx = 0
        for idx, value in enumerate(initial_items):
            if value == selected_column:
                selected_idx = idx

        # Add new column created to combobox
        if new_column_created:
            custom_columns.update({selected_column: {'name': new_column_created}})

        for key in sorted(custom_columns.keys()):
            self.column_names.append(key)
            self.addItem('%s (%s)' % (key, custom_columns[key]['name']))
            if key == selected_column:
                selected_idx = len(self.column_names) - 1
        self.setCurrentIndex(selected_idx)

    def get_selected_column(self) -> str:
        return self.column_names[self.currentIndex()]


class SizePersistedDialog(QDialog):
    """
    This dialog is a base class for any dialogs that want their
    size/position restored when they are next opened.
    """
    def __init__(self, parent: QWidget, unique_pref_name: str) -> None:
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)
        self.finished.connect(self.dialog_closing)

    def resize_dialog(self) -> None:
        if self.geom is None:
            self.resize(self.sizeHint())
        else:
            self.restoreGeometry(self.geom)

    def dialog_closing(self, result) -> None:
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
        self.persist_custom_prefs()

    def persist_custom_prefs(self) -> None:
        """
        Invoked when the dialog is closing. Override this function to call
        save_custom_pref() if you have a setting you want persisted that you can
        retrieve in your __init__() using load_custom_pref() when next opened
        """
        pass

    def load_custom_pref(self, name: str, default: str = None) -> str:
        return gprefs.get(self.unique_pref_name+':'+name, default)

    def save_custom_pref(self, name: str, value: str) -> None:
        gprefs[self.unique_pref_name+':'+name] = value


class KeyboardConfigDialog(SizePersistedDialog):
    """
    This dialog is used to allow editing of keyboard shortcuts.
    """

    def __init__(self, gui: QWidget, group_name: str):
        SizePersistedDialog.__init__(self, gui, 'Keyboard shortcut dialog')
        self.gui = gui
        self.setWindowTitle(_('Keyboard shortcuts'))
        self.setWindowIcon(get_icon('keyboard.png'))
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        self.keyboard_widget = ShortcutConfig(self)
        layout.addWidget(self.keyboard_widget)
        self.group_name = group_name

        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self.commit)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)

        # Cause our dialog size to be restored from prefs or created on first usage
        self.resize_dialog()
        self.initialize()

    def initialize(self) -> None:
        self.keyboard_widget.initialize(self.gui.keyboard)
        self.keyboard_widget.highlight_group(self.group_name)

    def commit(self) -> None:
        self.keyboard_widget.commit()
        self.accept()


class AboutDialog(QDialog):

    def __init__(self, parent: QWidget, icon: QIcon, text: str):
        QDialog.__init__(self, parent)
        self.plugin_icon = get_icon('icon.png')
        self.resize(450, 350)
        self.layout = QGridLayout()
        self.setLayout(self.layout)
        self.logo = QLabel()
        self.logo.setMaximumWidth(110)
        self.logo.setPixmap(QPixmap(self.plugin_icon.pixmap(100, 100)))
        self.label = QLabel(text)
        self.label.setOpenExternalLinks(True)
        self.label.setWordWrap(True)
        self.setWindowTitle(_('About ') + PLUGIN_NAME)
        self.setWindowIcon(icon)
        self.layout.addWidget(self.logo, 0, 0)
        self.layout.addWidget(self.label, 0, 1)
        self.bb = QDialogButtonBox(self)
        b = self.bb.addButton(QDialogButtonBox.StandardButton.Ok)
        b.setDefault(True)
        self.layout.addWidget(self.bb, 2, 0, 1, -1)
        self.bb.accepted.connect(self.accept)
