#!/usr/bin/env python

__license__   = 'GPL v3'
__copyright__ = '2023-2024, Thiago Oliveira <thiago.eec@gmail.com>'
__docformat__ = 'restructuredtext en'

# Load translation files (.mo) on the folder 'translations'
load_translations()

# Standard libraries
import os
import os.path
import zipfile
import datetime
from functools import partial

# PyQt libraries
from qt.core import Qt, QMenu, QIcon, QVBoxLayout, QDialogButtonBox, QApplication, QUrl, QAction, QWidget

# Calibre libraries
from calibre.gui2 import open_url
from calibre.gui2.actions import InterfaceAction, menu_action_unique_name
from calibre.gui2.keyboard import finalize
from calibre.utils.config import JSONConfig, config_dir
from calibre_plugins.Reading_Goal.config import prefs, get_icon
from calibre_plugins.Reading_Goal.main import ReadingGoalTools, show_configuration
from calibre_plugins.Reading_Goal.__init__ import PLUGIN_NAME, PLUGIN_THREAD


class InterfacePlugin(InterfaceAction):

    name = PLUGIN_NAME

    # Declare the main action associated with this plugin
    # The keyboard shortcut can be None if you don't want to use a keyboard
    # shortcut. Remember that currently calibre has no central management for
    # keyboard shortcuts, so try to use an unusual/unused shortcut.

    action_spec = (PLUGIN_NAME, None, _('Manage your reading goal'), 'Ctrl+G')

    def genesis(self) -> None:
        # This method is called once per plugin, do initial setup here

        # Set the icon for this interface action
        # The get_icons function is a builtin function defined for all your
        # plugin code. It loads icons from the plugin zip file. It returns
        # QIcon objects, if you want the actual data, use the analogous
        # get_resources builtin function.
        #
        # Note that if you are loading more than one icon, for performance, you
        # should pass a list of names to get_icons. In this case, get_icons
        # will return a dictionary mapping names to QIcons. Names that
        # are not found in the zip file will result in null QIcons.

        self.main_button_icon = get_icon('icon.png')

        # Extract "what's this" icons based on theme chosen. Those cannot be extracted with get_icon,
        # since we need the image file, not a QIcon. That's because we are using them in a html <img> tag
        self.extract_resources()

        # Menu creation
        # Only the toolbar menu is created here. All the others are created after the main GUI initialization
        # This allows us to change menus on the fly
        self.tools = ReadingGoalTools(self.gui)
        self.m = QMenu(self.gui)
        self.qaction.setMenu(self.m)
        self.qaction.setIcon(self.main_button_icon)

        # This action is configurable by the user. It is actioned by clicking on the main plugin button
        self.main_button_action = self.qaction
        self.main_button_action.triggered.connect(self.set_main_action)

        # First build of menus
        self.rebuild_menus(genesis=True)

        # Set up a listener to catch a system theme change
        QApplication.instance().palette_changed.connect(self.rebuild_menus)

    def initialization_complete(self) -> None:
        # This method is called once per action, after the initialization of the main GUI
        # This is right place to create changing menus
        self.m.aboutToShow.connect(self.about_to_show_menu)

    def library_changed(self, db) -> None:
        ReadingGoalTools.library_changed_event(self)

    @staticmethod
    def show_help() -> None:
        open_url(QUrl(PLUGIN_THREAD))

    def about_to_show_menu(self) -> None:
        finalize(self.gui.keyboard.shortcuts)
        self.rebuild_menus()

    def list_challenges(self, cm: partial, challenges_menu: QMenu, option: str) -> None:
        y = datetime.datetime.now().astimezone().year
        goal_data = JSONConfig('plugins/Reading_Goal_Data_' + self.gui.current_db.library_id)
        try:
            challenges_dict = goal_data[str(y)]['summary']['challenges_dict']
            if len(challenges_dict) == 1:
                challenges_dict = {'Empty': {}}
        except:
            challenges_dict = {'Empty': {}}
        # List all challenges
        for challenge, values in challenges_dict.items():
            if challenge != 'Empty':
                end = values['end']
                current_date = datetime.datetime.now().astimezone()
                if challenge != 'Annual' and current_date < end:
                    cm(challenges_menu, challenge,
                       shortcut=False,
                       triggered=partial(self.tools.manage_reading_goal, option=option, challenge_name=challenge),
                       unique_name='ReadingGoal_' + option + challenge)
            else:
                self.empty = challenges_menu.addAction(_('No challenges'))
                self.empty.setEnabled(False)

    def rebuild_menus(self, genesis: bool = False) -> None:
        self.extract_resources()

        # Set main button icon based on theme chosen
        self.main_button_icon = get_icon('icon.png')
        self.main_button_action.setIcon(self.main_button_icon)

        # Clear all menu actions. This way we can rebuild them, with different icons.
        self.m.clear()

        # Main menu items - Created here, so icons may change on the fly.
        # Before, they were created at initialization_complete().
        cm = partial(self.create_menu_action_unique)

        # Add to goal
        self.add_to_goal = cm(self.m, _('&Add to reading goal'),
                              image='goal_add.png',
                              tooltip=_('Add the selected books to the current year\'s reading goal'),
                              triggered=partial(self.tools.manage_reading_goal, option='add_to_goal'),
                              unique_name='ReadingGoal_Add')

        # Remove from goal
        self.remove_from_goal = cm(self.m, _('&Remove from reading goal'),
                                   image='goal_remove.png',
                                   tooltip=_('Remove the selected books from the current year\'s reading goal'),
                                   triggered=partial(self.tools.manage_reading_goal, option='remove_from_goal'),
                                   unique_name='ReadingGoal_Remove')
        self.m.addSeparator()

        # Reading goal statistics
        self.goal_statistics = cm(self.m, _('&Statistics'),
                                  image='stats.png',
                                  tooltip=_('Show your reading goal statistics'),
                                  triggered=partial(self.tools.manage_reading_goal, option='goal_statistics'),
                                  unique_name='SkoobSync_Statistics')

        # Edit reading goal
        self.edit_goal = cm(self.m, _('&Edit reading goal'),
                            image='goal_edit.png',
                            tooltip=_('Edit your current reading goal'),
                            triggered=partial(self.tools.manage_reading_goal, option='edit_goal'),
                            unique_name='ReadingGoal_Edit')
        self.m.addSeparator()

        # Add to custom challenge
        if not genesis:  # Avoid an unharmful error msg on calibre initialization, since current_db is unavailable
            self.add_chal_m = QMenu()
            self.list_challenges(cm, self.add_chal_m, 'add_to_challenge')
            self.add_to_challenge = cm(self.m, _('Ad&d to challenge'),
                                       image='challenge_add.png',
                                       tooltip=_('Add the selected books to a custom challenge'),
                                       unique_name='ReadingGoal_Custom_Add')
            self.add_to_challenge.setMenu(self.add_chal_m)

        # Remove from custom challenge
        if not genesis:
            self.remove_chal_m = QMenu()
            self.list_challenges(cm, self.remove_chal_m, 'remove_from_challenge')
            self.remove_from_challenge = cm(self.m, _('Re&move from challenge'),
                                            image='challenge_remove.png',
                                            tooltip=_('Remove the selected books from a custom challenge'),
                                            unique_name='ReadingGoal_Custom_Remove')
            self.remove_from_challenge.setMenu(self.remove_chal_m)

        # Custom challenges
        self.custom_challenges = cm(self.m, _('&Custom challenges'),
                                    image='challenges.png',
                                    tooltip=_('Manage your custom challenges'),
                                    triggered=partial(self.tools.manage_reading_goal, option='custom_challenges'),
                                    unique_name='ReadingGoal_Custom')
        self.m.addSeparator()

        # Config
        self.config = cm(self.m, _('Customize &plugin'),
                         image='_config.png',
                         tooltip=_('Open configuration dialog'),
                         shortcut=False,
                         triggered=partial(show_configuration, self))

        # Help
        self.help = cm(self.m, _('&Help'),
                       image='help.png',
                       tooltip=_('Open the help page'),
                       shortcut=False,
                       triggered=self.show_help)

        self.gui.keyboard.finalize()

    def set_main_action(self) -> None:
        # Define main action based on the user choice
        self.main_button = prefs['main_action']
        if self.main_button == 'Add to reading goal':
            option = 'add_to_goal'
        elif self.main_button == 'Edit reading goal':
            option = 'edit_goal'
        else:
            option = 'goal_statistics'
        self.tools.manage_reading_goal(option=option)

    @staticmethod
    def extract_resources() -> None:
        # Extract resource files to use internally
        plugin_zip = str(os.path.join(config_dir, 'plugins', PLUGIN_NAME + '.zip'))
        plugin_dir = str(os.path.join(config_dir, 'plugins', PLUGIN_NAME))
        with zipfile.ZipFile(plugin_zip, 'r') as zf:
            zf.extract('images/add_column_light.png', plugin_dir)
            zf.extract('images/add_column_dark.png', plugin_dir)
            zf.extract('images/help.png', plugin_dir)
            zf.extract('images/add_link_light.png', plugin_dir)
            zf.extract('images/add_link_dark.png', plugin_dir)
            zf.extract('images/records_show_light.png', plugin_dir)
            zf.extract('images/records_show_dark.png', plugin_dir)
            zf.extract('images/goal_remove_light.png', plugin_dir)
            zf.extract('images/goal_remove_dark.png', plugin_dir)
            zf.extract('images/abandon_light.png', plugin_dir)
            zf.extract('images/abandon_dark.png', plugin_dir)
            zf.extract('images/records_edit_light.png', plugin_dir)
            zf.extract('images/records_edit_dark.png', plugin_dir)

    def create_menu_action_unique(ia, parent_menu: QWidget, menu_text: str, image: str = None, tooltip: str = None,
                                  shortcut: bool = None, triggered: QAction = None, is_checked: bool = None,
                                  shortcut_name: str = None, unique_name: str = None) -> QAction:
        """
        Create a menu action with the specified criteria and action, using the new
        InterfaceAction.create_menu_action() function which ensures that regardless of
        whether a shortcut is specified it will appear in Preferences->Keyboard
        """
        orig_shortcut = shortcut
        kb = ia.gui.keyboard
        if unique_name is None:
            unique_name = menu_text
        if not shortcut == False:
            full_unique_name = menu_action_unique_name(ia, unique_name)
            if full_unique_name in kb.shortcuts:
                shortcut = False
            else:
                if shortcut is not None and not shortcut == False:
                    if len(shortcut) == 0:
                        shortcut = None
                    else:
                        shortcut = _(shortcut)

        if shortcut_name is None:
            shortcut_name = menu_text.replace('&', '')

        ac = ia.create_menu_action(parent_menu, unique_name, menu_text, icon=None, shortcut=shortcut,
                                   description=tooltip, triggered=triggered, shortcut_name=shortcut_name)
        if shortcut == False and not orig_shortcut == False:
            if ac.calibre_shortcut_unique_name in ia.gui.keyboard.shortcuts:
                kb.replace_action(ac.calibre_shortcut_unique_name, ac)
        if image:
            ac.setIcon(get_icon(image))
        if is_checked is not None:
            ac.setCheckable(True)
            if is_checked:
                ac.setChecked(True)
        return ac
