# calibre_plugins/pdf_to_cbz/ui.py
import sys, os
import zipfile, tempfile, shutil

import re
import unicodedata

from calibre.gui2.actions import InterfaceAction
from calibre.utils.logging import default_log as log
from calibre.gui2 import error_dialog
from calibre.ptempfile import TemporaryDirectory
from calibre.gui2.threaded_jobs import ThreadedJob, ThreadedJobServer


try:
    from PyQt5.Qt import QMenu, QIcon, QPixmap
except ImportError:
    from PyQt4.Qt import QMenu, QIcon, QPixmap


def setup_local_pymupdf():
    """
    Prépare PyMuPDF (fitz) depuis libs/ même si le plugin Calibre est zippé.
    """
    # __file__ = .../PDFtoCBZ.zip/__init__.py
    plugin_path = __file__
    if ".zip" in plugin_path:
        plugin_zip = plugin_path.split(".zip")[0] + ".zip"
    else:
        plugin_zip = None

    temp_dir = os.path.join(tempfile.gettempdir(), "pymupdf_calibre")

    if plugin_zip and zipfile.is_zipfile(plugin_zip):
        log.info(f"pdftocbz:📦 Plugin zippé détecté : {plugin_zip}")

        with zipfile.ZipFile(plugin_zip, "r") as z:
            libs_found = [m for m in z.namelist() if m.startswith("libs/")]
            if not libs_found:
                raise FileNotFoundError("❌ Le répertoire 'libs/' est introuvable dans le plugin ZIP.")

            if os.path.exists(temp_dir):
                shutil.rmtree(temp_dir)

            log.info(f"pdftocbz:🧩 Extraction de {len(libs_found)} fichiers de 'libs/' vers : {temp_dir}")
            for member in libs_found:
                z.extract(member, temp_dir)

        libs_extracted = os.path.join(temp_dir, "libs")

    else:
        # Mode non zippé (dev local)
        libs_extracted = os.path.join(os.path.dirname(__file__), "libs")
        if not os.path.exists(libs_extracted):
            raise FileNotFoundError("❌ Dossier 'libs' introuvable à côté du plugin.")

    # Ajout dans sys.path
    if libs_extracted not in sys.path:
        sys.path.insert(0, libs_extracted)

    # Import du module PyMuPDF
    import fitz
    log.info(f"pdftocbz:✅ PyMuPDF chargé depuis {libs_extracted}")
    return fitz


try:
    fitz = setup_local_pymupdf()
    log.info("pdftocbz: ✅ PyMuPDF opérationnel")
except Exception as e:
    log.info("pdftocbz: ⚠️ Erreur chargement PyMuPDF :", e)

from functools import partial




class PDFtoCBZ(InterfaceAction):
    name = 'PDFtoCBZ'
    action_spec = ("PDF → CBZ", None, "Convert selected PDF books to CBZ", None)
    action_type = "global"

    def genesis(self):
        """
        Init GUI : créer l'action et la connecter.
        """
        log.info("pdftocbz: genesis() — initialisation GUI")
        base_plugin = self.interface_action_base_plugin
        self.dpi = base_plugin.options['dpi']
        self.ext = base_plugin.options['output_format']
        self.quality = base_plugin.options['compression']
        self.method = base_plugin.options['method']
        #log.info(self.dpi, self.ext)
        self.menu = QMenu(self.gui)
        # Get the icon for this interface action
        icon = self.get_icon('images/plugin.png')

        # The qaction is automatically created from the action_spec defined
        # above
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(icon)
        self.qaction.triggered.connect(self.convert_pdf_to_cbz)


        # build menu
        self.menu.clear()
        self.build_menu()
        #self.toggle_menu_items()


    def location_selected(self, loc):
        pass

    def library_changed(self, db):
        pass


    def convert_pdf_to_cbz(self):
        log.info("pdftocbz: convert_pdf_to_cbz called")
        db = self.gui.current_db.new_api
        base_plugin = self.interface_action_base_plugin
        self.dpi = base_plugin.options['dpi']
        self.ext = base_plugin.options['output_format']
        self.quality = base_plugin.options['compression']
        self.method = base_plugin.options['method']
        #log.info(f"pdftocbz: format {self.ext} avec compression {self.quality}")

        # récup IDs sélectionnés (essayer l'API standard, fallback si nécessaire)
        try:
            #log.info("pdftocbz: get selected")

            book_ids = list(self.gui.library_view.get_selected_ids())
        except Exception:
            log.info("pdftocbz: view model")

            model = self.gui.library_view.model()
            rows = self.gui.library_view.selectionModel().selectedRows()
            book_ids = [model.id(r) for r in rows]

        if not book_ids:
            log.info("pdftocbz: aucun livre sélectionné")
            from calibre.gui2 import info_dialog
            info_dialog(self.gui, "PDF → CBZ", "Aucun livre sélectionné.")
            return

        done = []
        skipped = []
        errors = []

        for bid in book_ids:
            meta = db.get_metadata(bid)
            self.titre = self.safe_filename(meta.title)
            log.info(f"pdftocbz: le livre {bid} est {meta.title}-{meta.series}")

            try:
                fmts = tuple(db.formats(bid))  # tuple of format names like ('EPUB','PDF')
                fmts_up = [f.upper() for f in fmts]
                log.info(f"pdftocbz: le livre {bid} contient les formats = {fmts_up}")

                if "PDF" not in fmts_up:
                    skipped.append((bid, "Pas de PDF"))
                    log.info(f"pdftocbz: {bid} ignoré (pas de PDF)")
                    continue
                if "CBZ" in fmts_up or "CBR" in fmts_up:
                    skipped.append((bid, "CBZ/CBR déjà présent"))
                    log.info(f"pdftocbz: {bid} ignoré (CBZ/CBR déjà présent)")
                    continue

                # 5️⃣ Récupérer le chemin absolu du fichier PDF
                pdf_path = db.format_abspath(bid, 'PDF')
                if not os.path.exists(pdf_path):
                    log.info(f"pdftocbz:❌ Fichier PDF introuvable : {pdf_path}")
                    return

                # 6️⃣ Ouvrir le fichier avec PyMuPDF et convertir
                try:
                    #doc = fitz.open(pdf_path)
                    #log.info(f"pdftocbz:📄 Le document contient {doc.page_count} pages.")

                    #self.convert_one(doc, bid, db)
                    self.do_heavy_task(pdf_path, bid, db)

                    #doc.close()

                except Exception as e:
                    log.info(f"pdftocbz:⚠️ Erreur lors de l'ouverture du PDF : {e}")


            except Exception as e:
                log.exception(f"pdftocbz: ERREUR inattendue pour {bid}: {e}\n{traceback.format_exc()}")
                errors.append((bid, str(e)))


    def convert_one (self, pdf_path, bid, db, **kwargs):
        """ converti un doc pdf en image dans un rep temporaire puis ajoute a la libririe"""
        log.info(f"pdftocbz:✅ co1 : {bid}")
        #job = kwargs.get('job')
        base_plugin = self.interface_action_base_plugin
        self.dpi = base_plugin.options['dpi']
        self.ext = base_plugin.options['output_format']
        self.quality = base_plugin.options['compression']
        # --- récupération des objets spéciaux passés par Calibre ---
        notification = kwargs.get("notification")
        abort = kwargs.get("abort")
        log2 = kwargs.get("log")

        items_kwargs = list(kwargs.items())
        doc = fitz.open(pdf_path)
        log.info(f"pdftocbz:📄 Le document contient {doc.page_count} pages.")
        #log.info(f"pdftocbz:📄 kwargs: {items_kwargs} ")

        done = []
        skipped = []
        errors = []

        #log.info(f"pdftocbz:✅ co2 : {bid}")
        if log2 : log2.info(f"[PDFtoCBZ] 🔧 Début du traitement pour le livre {bid}")
        with TemporaryDirectory("pdf2cbz") as tdir:

            for page_index in range(len(doc)):  # iterate over pdf pages
                #job.set_progress (page_index*100/len(doc))
                progress = page_index  / len(doc)

                if notification:
                    notification.put(("progress", progress))
                    self.set_progress(progress)

                page = doc[page_index]  # get the page
                image_list = page.get_images(full=True)
                if abort and abort.is_set():
                    log.info("[PDFtoCBZ] 🚫 Annulation du job demandée par l’utilisateur")
                    if notification:
                        notification.put(("abort", f"Job {bid} annulé"))
                    return
                # print the number of images found on the page
                if image_list:
                    log.info(f"pdftocbz:Found {len(image_list)} images on page {page_index} pour le livre {bid} soit {int(100*progress)}%")
                else:
                    log.info("pdftocbz:No images found on page", page_index)

                for image_index, img in enumerate(image_list, start=1):  # enumerate the image list
                    xref = img[0]  # get the XREF of the image
                    width = img[2]
                    height = img[3]
                    pix = fitz.Pixmap(doc, xref)  # create a Pixmap
                    #log.info(f"pdftocbz:dim {width} {height};{pix.height}; {pix.xres},{pix.yres}")

                    if pix.n - pix.alpha > 3:  # CMYK: convert to RGB first
                        pix = fitz.Pixmap(fitz.csRGB, pix)

                    temppix = os.path.join(tdir, "%s_pag%s-i%s.%s" % (self.titre, page_index, image_index, self.ext))
                    # Calcul de la hauteur en cm
                    hauteur_cm = (pix.height / pix.xres) * 2.54

                    # Si la hauteur est supérieure à 30cm, ajuster la résolution
                    if hauteur_cm > 30:
                        resolution = int((pix.height / 30) * 2.54)
                    pix.set_dpi(resolution, resolution)
                    #log.info(f"pdftocbz:dim {width} {height};{resolution}; {pix.xres}")

                    # --- Sauvegarde selon le format ---
                    if self.ext.lower() in ("jpg", "jpeg"):
                        #log.info("pdftocbz: jpg", self.quality)
                        if pix.alpha:
                            pix = fitz.Pixmap(pix, 0)  # convertit en RGB sans alpha
                        pix.save(temppix) #, jpg_quality = self.quality)
                    else:
                        log.info("pdftocbz: png")

                        pix.save(temppix)

                    pix.save(temppix)  # save the image
                    pix = None
            cbz_path = os.path.join(tdir, f"{bid}.cbz")
            try:
                self.make_cbz_from_files_in_dir(tdir, cbz_path)
            except Exception as e:
                log.exception(f"pdftocbz: création CBZ échouée pour {bid}: {e}")
                errors.append((bid, str(e)))
            # ajouter le CBZ au livre (ne remplace pas le PDF)
            try:
                added = db.add_format(bid, "CBZ", cbz_path, replace=False)
                if added:
                    log.info(f"pdftocbz: CBZ ajouté en base pour {bid}")
                    done.append(bid)
                else:
                    log.error(f"pdftocbz: db.add_format a retourné False pour {bid}")
                    errors.append((bid, "db.add_format a retourné False"))
            except Exception as e:
                log.exception(f"pdftocbz: erreur ajout CBZ DB pour {bid}: {e}")
                errors.append((bid, str(e)))
        #return cbz_path
        #progress.end()
        doc.close()

    def make_cbz_from_files_in_dir(self, temp_dir, cbz_path):
        """
        make_cbz_from_files_in_dir(temp_dir, cbz_path)
        ----------------------------------------------
        Crée un fichier CBZ à partir de toutes les images présentes dans un répertoire temporaire.

        Paramètres :
            temp_dir (str) : chemin du répertoire contenant les images (ex: 'pdf2cbz_temp')
            cbz_path (str) : chemin complet du fichier CBZ à créer (ex: 'output/book.cbz')

        Fonctionnement :
            - Liste toutes les images du dossier temporaire
            - Trie les fichiers par ordre alphabétique (pour garder les pages dans le bon ordre)
            - Crée un fichier .cbz (Comic Book Zip)
        """
        # Vérifie que le dossier existe
        if not os.path.isdir(temp_dir):
            raise FileNotFoundError(f"Le répertoire {temp_dir} n'existe pas.")

        # Récupère toutes les images du dossier
        image_files = [f for f in os.listdir(temp_dir)
                       if f.lower().endswith((".png", ".jpg", ".jpeg"))]

        if not image_files:
            raise ValueError("Aucune image trouvée dans le répertoire temporaire.")

        # Trie les fichiers pour conserver l’ordre des pages
        image_files.sort()

        # Création du CBZ (en réalité une archive ZIP renommée)
        with zipfile.ZipFile(cbz_path, 'w', zipfile.ZIP_DEFLATED) as zipf:
            for img_name in image_files:
                img_path = os.path.join(temp_dir, img_name)
                zipf.write(img_path, arcname=img_name)

        log.info(f"pdftocbz:✅ CBZ créé avec succès : {cbz_path}")

    def show_settings(self):
        log.info("pdftocbz: show_settings called")
        self.do_user_config()

    def build_menu(self):

        # add configuration entry
        self.menu_action("configure", "Configure",
                         partial(self.interface_action_base_plugin.do_user_config, (self.gui)))

    def menu_action(self, name, title, triggerfunc):
        action = self.create_menu_action(self.menu, name, title, icon=None,
                                         shortcut=None, description=None,
                                         triggered=triggerfunc, shortcut_name=None)
        setattr(self, name, action)

    def get_icon(self, icon_name):
        import os
        from calibre.utils.config import config_dir

        # Check to see whether the icon exists as a Calibre resource
        # This will enable skinning if the user stores icons within a folder like:
        # ...\AppData\Roaming\calibre\resources\images\Plugin Name\
        icon_path = os.path.join(config_dir, 'resources', 'images', self.name,
                                 icon_name.replace('images/', ''))
        if os.path.exists(icon_path):
            pixmap = QPixmap()
            pixmap.load(icon_path)
            return QIcon(pixmap)
        # As we did not find an icon elsewhere, look within our zip resources
        return get_icons(icon_name)


    def do_heavy_task(self, pdf, bid, db):
        """en test : Lancée depuis un bouton du plugin."""
        log.info(f"pdftocbz:✅ 1 : {bid}")
        base_plugin = self.interface_action_base_plugin

        self.method = base_plugin.options['method']
        # Créer server + job
        server = ThreadedJobServer()
        log.info(f"pdftocbz:✅ 2 : {bid}")

        job = ThreadedJob(
            f'convert one pdf in cbz {bid}',                   # Identifiant du job
            f'run conversion PDF... {bid}',  # Message dans la barre de progression
            self.convert_one,           # Fonction à exécuter en tâche de fond
            (pdf, bid, db),                           # Arguments de la fonction
            {},                         # Mêmes choses, mais nommés
            self.on_done               # Callback quand fini
            #gui=True

        )
        log.info(f"pdftocbz:✅ conversion method : {self.method}")


        # job souci KeyError: 'convert one pdf in cbz'
        #self.gui.job_manager.run_job(job, 'convert one pdf in cbz')
        if self.method == 3:
            # job en tache
            self.gui.job_manager.run_threaded_job(job)
        elif self.method == 1:
            #les fichiers les un derrieres les autres en taches principales
            self.convert_one(pdf, bid, db)
        else:
            # Lancer : add_job démarre le serveur si nécessaire et lance le job
            server.add_job(job) # processus en parallele

        log.info(f"pdftocbz:✅ 4 : {bid}")



    def on_done(self, job):

        """📩 Callback appelée quand la tâche est terminée."""
        if job.failed:
            log.info('Erreur Le traitement a échoué')

            #log.error(f"Le job {job.name} a échoué")
            log.error(f"Exception : {job.exception}")  # objet Exception
            #log.error(f"Traceback :\n{job.traceback}")  # chaîne complète
        else:
            result = job.result
            log.info("pdf2cbz: conversion réalisée :", result)
            #self.gui.status_bar.show_message(f"pdf2cbz: Tâche terminée : {result}", 3000)



    def safe_filename(self, titre: str) -> str:
        """
        Transforme une chaîne en un nom de fichier sûr pour tous les OS :
        - Supprime les caractères interdits
        - Remplace les lettres accentuées par leur équivalent non accentué
        - Conserve uniquement lettres, chiffres, espaces, tirets et underscores

        Paramètres :
            titre : str -> chaîne d'origine

        Retour :
            str -> nom de fichier sécurisé
        """
        # Normaliser pour enlever les accents
        titre_norm = unicodedata.normalize('NFKD', titre)
        titre_norm = "".join([c for c in titre_norm if not unicodedata.combining(c)])

        # Supprimer tous les caractères non autorisés (conserver lettres, chiffres, espaces, - et _)
        titre_sure = re.sub(r'[^a-zA-Z0-9 _-]', '', titre_norm)

        # Supprimer espaces/tirets/underscores en début et fin
        titre_sure = titre_sure.strip(" _-")

        # Limiter à 255 caractères
        return titre_sure[:20]

