#!/usr/bin/env python
from calibre.utils.config import JSONConfig
from calibre.gui2.metadata.config import ConfigWidget as DefaultConfigWidget
from PyQt5.Qt import QLabel, QVBoxLayout, QComboBox, QListWidget
from PyQt5 import Qt as QtGui

__license__ = "GPL v3"
__copyright__ = "2025, Computer"
__docformat__ = "restructuredtext en"

STORE_NAME = "Options"
KEY_MAX_DOWNLOADS = "maxDownloads"
KEY_COUNTRY = "country"
KEY_LANGUAGES = "languages"

DEFAULT_STORE_VALUES = {
    KEY_MAX_DOWNLOADS: 3,
    KEY_COUNTRY: 'SE',
    KEY_LANGUAGES: ['sv'],  # Default to Swedish
}

# Country to store mapping
COUNTRIES = {
    'BG': 'Bulgaria',
    'DK': 'Denmark',
    'EE': 'Estonia',
    'FI': 'Finland',
    'FR': 'France',
    'DE': 'Germany',
    'IS': 'Iceland',
    'ID': 'Indonesia',
    'IL': 'Israel',
    'IN': 'India',
    'TV': 'International',
    'IT': 'Italy',
    'AR': 'Latin America',
    'NL': 'Netherlands',
    'NO': 'Norway',
    'PL': 'Poland',
    'KR': 'South Korea',
    'ES': 'Spain',
    'SE': 'Sweden',
    'TH': 'Thailand',
    'TR': 'Turkey',
}

# Language codes and names
LANGUAGES = {
    'ar': 'Arabic',
    'bg': 'Bulgarian',
    'bn': 'Bengali',
    'bs': 'Bosnian',
    'ca': 'Catalan',
    'cs': 'Czech',
    'da': 'Danish',
    'de': 'German',
    'el': 'Greek',
    'en': 'English',
    'es': 'Spanish',
    'et': 'Estonian',
    'fi': 'Finnish',
    'fr': 'French',
    'he': 'Hebrew',
    'hi': 'Hindi',
    'hr': 'Croatian',
    'hu': 'Hungarian',
    'id': 'Indonesian',
    'is': 'Icelandic',
    'it': 'Italian',
    'ja': 'Japanese',
    'ko': 'Korean',
    'lt': 'Lithuanian',
    'lv': 'Latvian',
    'ml': 'Malayalam',
    'mr': 'Marathi',
    'ms': 'Malay',
    'nl': 'Dutch',
    'no': 'Norwegian',
    'pl': 'Polish',
    'pt': 'Portuguese',
    'ro': 'Romanian',
    'ru': 'Russian',
    'sk': 'Slovak',
    'sl': 'Slovenian',
    'sr': 'Serbian',
    'sv': 'Swedish',
    'sw': 'Swahili',
    'ta': 'Tamil',
    'th': 'Thai',
    'tr': 'Turkish',
    'zh': 'Chinese',
}

# This is where all preferences for this plugin will be stored
plugin_prefs = JSONConfig("plugins/Storytel")

# Set defaults
plugin_prefs.defaults[STORE_NAME] = DEFAULT_STORE_VALUES


class ConfigWidget(DefaultConfigWidget):
    def __init__(self, plugin):
        DefaultConfigWidget.__init__(self, plugin)
        c = plugin_prefs[STORE_NAME]

        # Create a new group box for our custom settings
        other_group_box_layout = QVBoxLayout()
        self.l.addLayout(other_group_box_layout, self.l.rowCount(), 0, 1, 2)

        # Country/Store selection
        country_label = QLabel("Storytel Country Store:", self)
        country_label.setToolTip(
            "Select which Storytel country store to search.\n"
            "This determines which books are available."
        )
        other_group_box_layout.addWidget(country_label)
        
        self.country_combo = QComboBox(self)
        for code, name in sorted(COUNTRIES.items(), key=lambda x: x[1]):
            self.country_combo.addItem(f"{name} ({code})", code)
        
        # Set current country
        current_country = c.get(KEY_COUNTRY, DEFAULT_STORE_VALUES[KEY_COUNTRY])
        index = self.country_combo.findData(current_country)
        if index >= 0:
            self.country_combo.setCurrentIndex(index)
        
        other_group_box_layout.addWidget(self.country_combo)

        # Language selection (multi-select)
        language_label = QLabel("Languages to include in search (multi-select):", self)
        language_label.setToolTip(
            "Select one or more languages to include in search results.\n"
            "Hold Ctrl (Cmd on Mac) to select multiple languages.\n"
            "Only books in the selected languages will be returned."
        )
        other_group_box_layout.addWidget(language_label)
        
        self.language_list = QListWidget(self)
        self.language_list.setSelectionMode(QListWidget.MultiSelection)
        
        # Add languages sorted by name
        for code, name in sorted(LANGUAGES.items(), key=lambda x: x[1]):
            item = QtGui.QListWidgetItem(f"{name} ({code})")
            item.setData(QtGui.Qt.UserRole, code)
            self.language_list.addItem(item)
        
        # Set current languages
        current_languages = c.get(KEY_LANGUAGES, DEFAULT_STORE_VALUES[KEY_LANGUAGES])
        for i in range(self.language_list.count()):
            item = self.language_list.item(i)
            code = item.data(QtGui.Qt.UserRole)
            if code in current_languages:
                item.setSelected(True)
        
        other_group_box_layout.addWidget(self.language_list)

        # Max downloads
        max_label = QLabel(
            "Maximum title/author search matches to evaluate (1 = fastest):", self
        )
        max_label.setToolTip(
            "Storytel may have multiple editions of the same book.\n"
            "Increasing this value will evaluate more search results\n"
            "when doing title/author searches.\n\n"
            "This will increase the potential likelihood of finding\n"
            "the correct edition, though it may slow down searches."
        )
        other_group_box_layout.addWidget(max_label)
        self.max_downloads_spin = QtGui.QSpinBox(self)
        self.max_downloads_spin.setMinimum(1)
        self.max_downloads_spin.setMaximum(10)
        self.max_downloads_spin.setProperty(
            "value", c.get(KEY_MAX_DOWNLOADS, DEFAULT_STORE_VALUES[KEY_MAX_DOWNLOADS])
        )
        other_group_box_layout.addWidget(self.max_downloads_spin)

    def commit(self):
        DefaultConfigWidget.commit(self)
        new_prefs = {}
        new_prefs[KEY_MAX_DOWNLOADS] = int(str(self.max_downloads_spin.value()))
        new_prefs[KEY_COUNTRY] = self.country_combo.currentData()
        
        # Get selected languages
        selected_languages = []
        for i in range(self.language_list.count()):
            item = self.language_list.item(i)
            if item.isSelected():
                code = item.data(QtGui.Qt.UserRole)
                selected_languages.append(code)
        
        # Ensure at least one language is selected
        if not selected_languages:
            selected_languages = DEFAULT_STORE_VALUES[KEY_LANGUAGES]
        
        new_prefs[KEY_LANGUAGES] = selected_languages
        plugin_prefs[STORE_NAME] = new_prefs