#!/usr/bin/env python
from calibre.utils.cleantext import clean_ascii_chars
from calibre.utils.icu import lower
from calibre.ebooks.metadata.sources.base import Source
from lxml.html import fromstring
import calibre_plugins.Storytel.config as cfg
import time
import re
import json

__license__ = "GPL v3"
__copyright__ = "2025, Computer"
__docformat__ = "restructuredtext en"

from urllib.parse import quote
from queue import Empty, Queue


class Storytel(Source):

    name = "Storytel"
    description = "Downloads metadata and covers from Storytel.se (ebooks only)"
    author = "Computer"
    version = (0, 1, 0)
    minimum_calibre_version = (0, 6, 0)

    ID_NAME = "isbn"
    capabilities = frozenset(["identify", "cover"])
    touched_fields = frozenset(
        [
            "title",
            "authors",
            "identifier:isbn",
            "comments",
            "publisher",
            "pubdate",
            "languages",
            "series",
            "tags",
        ]
    )
    has_html_comments = True
    supports_gzip_transfer_encoding = True

    STORYTEL_URL = "https://www.storytel.com"

    def config_widget(self):
        """
        Overriding the default configuration screen for our own custom configuration
        """
        from calibre_plugins.Storytel.config import ConfigWidget

        return ConfigWidget(self)

    def get_book_url(self, identifiers):
        isbn = identifiers.get("isbn", None)
        if isbn:
            # Note: We can't construct a valid URL without the slug
            # This will be populated by the identify method if ISBN is found
            return (self.ID_NAME, isbn, f"{self.STORYTEL_URL}/se/books/{isbn}")

    def get_cached_cover_url(self, identifiers):
        """Get cached cover URL for the given identifiers"""
        url = None
        isbn = identifiers.get("isbn", None)
        if isbn is not None:
            url = self.cached_identifier_to_cover_url(isbn)
        return url

    def create_query(self, log, title=None, authors=None, identifiers={}):
        """Create search query URL with filters"""
        from calibre_plugins.Storytel.config import plugin_prefs, STORE_NAME, KEY_COUNTRY, KEY_LANGUAGES
        prefs = plugin_prefs[STORE_NAME]
        country = prefs.get(KEY_COUNTRY, 'SE')
        languages = prefs.get(KEY_LANGUAGES, ['sv'])
        
        q = ""

        if title:
            title_tokens = list(
                self.get_title_tokens(
                    title, strip_joiners=False, strip_subtitle=True)
            )
            if title_tokens:
                tokens = [quote(t.encode('utf-8')) for t in title_tokens]
                q = "+".join(tokens)
        else:
            return None

        # NOTE: Storytel's search API doesn't work well when combining title and author
        # We only search by title and let Calibre's matching algorithm handle author matching
        
        # Replace + with %20 for proper URL encoding
        q = q.replace('+', '%20')
        
        # Build language filter parameter
        language_param = '%2C'.join(languages)  # URL-encoded comma separator
        
        # Build the complete query URL
        return f"https://api.storytel.net/search/client/web?query={q}&store=STHP-{country}&searchFor=omni&includeFormats=ebook&includeLanguages={language_param}"

    def identify(
        self,
        log,
        result_queue,
        abort,
        title=None,
        authors=None,
        identifiers={},
        timeout=30,
    ):
        matches = []
        br = self.browser

        # Search by title/author (ISBN search doesn't work with Storytel API)
        matches = self._search_by_title_author(title, authors, identifiers, br, log, timeout, abort)

        if abort.is_set():
            return

        if not matches:
            log.error("No matches found")
            return

        # Create worker threads to fetch metadata for each match
        from calibre_plugins.Storytel.worker import Worker

        workers = [
            Worker(match_url, result_queue, br, log, i, self)
            for i, match_url in enumerate(matches)
        ]

        for w in workers:
            w.start()
            time.sleep(0.1)

        while not abort.is_set():
            a_worker_is_alive = False
            for w in workers:
                w.join(0.2)
                if abort.is_set():
                    break
                if w.is_alive():
                    a_worker_is_alive = True
            if not a_worker_is_alive:
                break

        return None
    
    def _search_by_title_author(self, title, authors, identifiers, br, log, timeout, abort):
        """Search for books by title and author"""
        matches = []
        
        query = self.create_query(log, title=title, authors=authors, identifiers=identifiers)
        if query is None:
            log.error("Insufficient metadata to create query")
            return matches
        
        try:
            log.info(f"Querying: {query}")
            response = br.open_novisit(query, timeout=timeout)
            raw = response.read().strip()
            
            # Log the first 500 characters to see what we got
            log.debug(f"Response preview: {raw[:500]}")
            
            # Parse the page to extract the embedded JSON data
            book_urls = self._extract_book_urls_from_search(raw, log)
            
            if not book_urls:
                log.info("No ebook links found in search results")
                return matches
            
            # Get max downloads setting
            from calibre_plugins.Storytel.config import plugin_prefs, STORE_NAME, KEY_MAX_DOWNLOADS
            prefs = plugin_prefs[STORE_NAME]
            max_downloads = prefs.get(KEY_MAX_DOWNLOADS, 3)
            
            # Limit to max downloads
            for book_url in book_urls[:max_downloads]:
                if abort.is_set():
                    break
                matches.append(book_url)
                log.info(f"Found ebook: {book_url}")
            
            log.info(f"Found {len(matches)} ebook matches")
            
        except Exception as e:
            log.exception(f"Failed to parse query results: {e}")
        
        return matches
    
    def _extract_book_urls_from_search(self, html, log):
        """Extract book URLs from search results API response"""
        book_urls = []
        
        try:
            if isinstance(html, bytes):
                html = html.decode('utf-8')
            
            data = json.loads(html)
            
            if isinstance(data, dict):
                books = data.get('items', [])
            elif isinstance(data, list):
                books = data
            else:
                log.error(f"Unexpected response format: {type(data)}")
                return book_urls
            
            # Process books and filter for ebooks only
            for book in books:
                book_id = book.get('id')
                formats = book.get('formats', [])
                
                # Check if this book has ebook format
                has_ebook = any(fmt.get('type') == 'ebook' for fmt in formats)
                
                if book_id and has_ebook:
                    book_url = book.get('shareUrl')
                    if not book_url:
                        book_url = f"{self.STORYTEL_URL}/se/sv/books/{book_id}"
                    book_urls.append(book_url)
            
        except json.JSONDecodeError as e:
            log.error(f"Failed to parse JSON response: {e}")
        except Exception as e:
            log.exception(f"Error extracting book URLs: {e}")
        
        return book_urls
    
    def _is_ebook_page(self, html, log):
        """Check if the page is for an ebook (not audiobook)"""
        try:
            # Parse embedded JSON from the page
            root = fromstring(html)
            script_tags = root.xpath('//script[@id="__NEXT_DATA__"]/text()')
            
            if not script_tags:
                return False
            
            data = json.loads(script_tags[0])
            
            # Check the consumable data
            page_props = data.get('props', {}).get('pageProps', {})
            consumable = page_props.get('consumable', {})
            
            # Check format - should be 'ebook' not 'abook'
            book_format = consumable.get('format')
            if book_format == 'ebook':
                return True
            
            log.debug(f"Book format is: {book_format}")
            return False
            
        except Exception as e:
            log.debug(f"Error checking if ebook page: {e}")
            return False

    def download_cover(
        self,
        log,
        result_queue,
        abort,
        title=None,
        authors=None,
        identifiers={},
        timeout=30,
    ):
        """Download cover for the book"""
        cached_url = self.get_cached_cover_url(identifiers)
        if cached_url is None:
            log.info("No cached cover found, running identify")
            rq = Queue()
            self.identify(
                log, rq, abort, title=title, authors=authors, identifiers=identifiers
            )
            if abort.is_set():
                return
            results = []
            while True:
                try:
                    results.append(rq.get_nowait())
                except Empty:
                    break
            results.sort(
                key=self.identify_results_keygen(
                    title=title, authors=authors, identifiers=identifiers
                )
            )
            for mi in results:
                cached_url = self.get_cached_cover_url(mi.identifiers)
                if cached_url is not None:
                    break
        
        if cached_url is None:
            log.info("No cover found")
            return

        if abort.is_set():
            return
        
        br = self.browser
        try:
            log.info(f"Downloading cover from: {cached_url}")
            cdata = br.open_novisit(cached_url, timeout=timeout).read()
            result_queue.put((self, cdata))
        except:
            log.exception("Failed to download cover from:", cached_url)


if __name__ == "__main__":
    from calibre.ebooks.metadata.sources.test import (
        test_identify_plugin,
        title_test,
        authors_test,
    )

    test_identify_plugin(
        Storytel.name,
        [
            (
                {
                    "title": "Mördare utan ansikte",
                    "authors": ["Henning Mankell"],
                },
                [
                    title_test("Mördare utan ansikte", exact=True),
                    authors_test(["Henning Mankell"]),
                ],
            ),
        ],
        fail_missing_meta=True,
    )