# -*- coding: utf-8 -*-
from __future__ import absolute_import, unicode_literals, print_function
import re
from calibre.ebooks.oeb.polish.container import get_container
from lxml import etree
from calibre.ebooks.oeb.parse_utils import XHTML_NS
import traceback
from calibre.ebooks.metadata.book.base import Metadata

__license__ = 'GPL 3'
__copyright__ = '2014, Saulius P. <saulius@kofmarai.net>'
__docformat__ = 'restructuredtext en'

import time
try:
    from cStringIO import StringIO as DataIO
except:
    from io import BytesIO as DataIO

from threading import Thread
from six.moves.queue import Queue

from calibre.utils.ipc.job import BaseJob
from calibre.ptempfile import PersistentTemporaryFile
        
class HyphenateThisThread(Thread):
    PARSE_ERROR = _('There was an error executing HTML analysis. Please check ignored and parsed tag lists in plugin\'s settings. Allowed tags are alphanumerical in comma-separated list.')
    
    def __init__(self, gui):
        Thread.__init__(self)
        self.daemon = True
        self.jobs = Queue()
        self.gui = gui
        self.job_manager = self.gui.job_manager
        self._run = True
        self._r = re.compile(r"\w+|[^\w]", re.UNICODE)
    
    def _hyphenate(self, job):
        (idItem, fmt, mi, hyphs) = job.args
        self._updateJob(job, 0.01, _('Starting...'))
        
        src = self.gui.library_view.model().db.format(idItem, fmt, index_is_id=True, as_path=True)
        self._updateJob(job, 0.02, _('Exploding the book...'))
        ebook = get_container(src)
        h = hyphs[mi.language]
        
        from calibre_plugins.hyphenatethis.gui.config import prefs
        min_len = prefs['length']
        
        spines = []
        for spine in ebook.spine_items:
            spines.append(spine)
            
        l = len(spines)+1
        p = 1
        
        for spine in spines:
            parsed = ebook.parsed(ebook.abspath_to_name(spine))
            ilist, q = self._createQuery()
            try:
                texts = etree.XPath(q, namespaces={'h':XHTML_NS})(parsed)
            except:
                job.log_write(traceback.format_exc())
                job.description = self.PARSE_ERROR 
                raise self.PARSE_ERROR
            for t in texts:
                parent = t.getparent()
                if parent.tag in ilist: continue
                newt = ''
                wlist = self._r.findall(t)
                for w in wlist:
                    if len(w) < min_len or u'-' in w:
                        newt += w
                    else:
                        newt += h.inserted(w).replace('-', u'\u00AD')
                
                if t.is_text:
                    parent.text = newt
                elif t.is_tail:
                    parent.tail = newt
            
            with open(spine, 'w') as f:
                try:
                    f.write(etree.tostring(parsed, xml_declaration=True, pretty_print=True, encoding='UTF-8'))
                except:
                    f.write(etree.tostring(parsed, xml_declaration=True, pretty_print=True, encoding='UTF-8').decode('utf-8'))
            
            self._updateJob(job, float(p/l), _('Hyphenating...'))
            p += 1
            
        suff = '_hyph.'+fmt
        out = PersistentTemporaryFile(suffix=suff)
        ebook.commit(outpath=out.name)
        
        with open(out.name, 'rb') as f:
            self.gui.library_view.model().db.add_format(idItem, fmt, f, index_is_id=True)
        
        self._setCustomCol(prefs, 1, idItem, job)
        
        self._updateJob(job, 1.0, _('Book hyphenated.'))
        

    def _removehyphens(self, job):
        (idItem, fmt) = job.args
        self._updateJob(job, 0.01, _('Starting...'))
        
        src = self.gui.library_view.model().db.format(idItem, fmt, index_is_id=True, as_path=True)
        self._updateJob(job, 0.02, _('Exploding the book...'))
        ebook = get_container(src)
        
        from calibre_plugins.hyphenatethis.gui.config import prefs
        
        spines = []
        for spine in ebook.spine_items:
            spines.append(spine)
            
        l = len(spines)+1
        p = 1
        
        for spine in spines:
            parsed = ebook.parsed(ebook.abspath_to_name(spine))
            ilist, q = self._createQuery()
            try:
                texts = etree.XPath(q, namespaces={'h':XHTML_NS})(parsed)
            except:
                job.log_write(traceback.format_exc())
                job.description = self.PARSE_ERROR 
                raise self.PARSE_ERROR
            for t in texts:
                parent = t.getparent()
                if parent.tag in ilist: continue
                newt = t.replace(u'\u00AD', '')
                if t.is_text:
                    parent.text = newt
                elif t.is_tail:
                    parent.tail = newt
            
            with open(spine, 'w') as f:
                try:
                    f.write(etree.tostring(parsed, xml_declaration=True, pretty_print=True, encoding='UTF-8'))
                except:
                    f.write(etree.tostring(parsed, xml_declaration=True, pretty_print=True, encoding='UTF-8').decode('utf-8'))
            
            self._updateJob(job, float(p/l), _('Removing hyphens...'))
            p += 1
            
        suff = '_remhyph.'+fmt
        out = PersistentTemporaryFile(suffix=suff)
        ebook.commit(outpath=out.name)
        
        with open(out.name, 'rb') as f:
            self.gui.library_view.model().db.add_format(idItem, fmt, f, index_is_id=True)
        
        self._setCustomCol(prefs, 2, idItem, job)
        
        self._updateJob(job, float(1.0), _('Hyphens removed'))
    
    def _setCustomCol(self, prefs, valIdx, idItem, job):
        if len(prefs['custom_col'][0]) > 0:
            try:
                newmi = Metadata(_('Unknown'))
                custom_cols = self.gui.library_view.model().db.field_metadata.custom_field_metadata()
                col = custom_cols[prefs['custom_col'][0]]
                col['#value#'] = prefs['custom_col'][valIdx]
                newmi.set_user_metadata(prefs['custom_col'][0], col)
                id_map = {}
                id_map[idItem] = newmi
                edit_metadata_action = self.gui.iactions['Edit Metadata']
                edit_metadata_action.apply_metadata_changes(id_map)
            except:
                job.log_write(traceback.format_exc())
                job.log_write('Value could not be set to column %s.' % prefs['custom_col'][0])
        
    
    def _createQuery(self):
        from calibre_plugins.hyphenatethis.gui.config import prefs
        if prefs['tags_ignore'].strip() == '' and prefs['tags_consider'].strip() == '':
            return [], '//h:body//text()'
        ignore = ''
        ilist = []
        if prefs['tags_ignore'].strip() != '':
            for i in prefs['tags_ignore'].split(','):
                ignore += " and local-name() != '%s'" % i.strip()
                ilist.append(("{%s}"+i.strip()) % XHTML_NS)
            ignore = ignore[5:]
        consider = ''
        if prefs['tags_consider'].strip() != '':
            for i in prefs['tags_consider'].split(','):
                consider += " or local-name() = '%s'" % i.strip()
            consider = "%s" % consider[4:]
            if ignore != '':
                consider = ' and (%s)' % consider
        return ilist, "//h:body//*[%s%s]//text()" % (ignore, consider)
    
    def hyphenate(self, callback, idItem, fmt, mi, hyphs):
        description = _('Hyphenation of the book "%s" starting.') % mi.title
        job = HyphenateThisJob(callback, description, self.job_manager, idItem, fmt,  mi, hyphs)
        self.job_manager.add_job(job)
        self.jobs.put(job)
        
    
    def removehyphens(self, callback, idItem, fmt, mi):
        description = _('Hyphens removal of the book "%s" starting.') % mi.title
        job = RemoveHyphensJob(callback, description, self.job_manager, idItem, fmt)
        self.job_manager.add_job(job)
        self.jobs.put(job)
    
    def _updateJob(self, job, percent, message):
        job.notifications.put((percent, message))
        job.update()
        job.job_manager.changed_queue.put(job)
        
    def stop(self):
        self._run = False
        self.jobs.put(None)
        
    def run(self):
        while self._run:
            try:
                job = self.jobs.get()
            except:
                break
            if job is None or not self._run:
                break
            
            failed, exc = False, None
            job.start_work()
            if job.kill_on_start:
                self._abort_job(job)
                continue
            
            try:
                if type(job) is HyphenateThisJob:
                    self._hyphenate(job)
                elif type(job) is RemoveHyphensJob:
                    self._removehyphens(job)
            except Exception as e:
                if not self._run:
                    return
                import traceback
                failed = True
                exc = e
                job.log_write('\nHyphenation failed...\n')
                job.log_write(traceback.format_exc())

            if not self._run:
                break

            job.failed = failed
            job.exception = exc
            job.job_done()
            try:
                job.callback(job)
            except:
                import traceback
                traceback.print_exc()

    def _abort_job(self, job):
        job.log_write(_('Hyphenation aborted\n'))
        job.failed = False
        job.killed = True
        job.job_done()


class HyphenateThisJob(BaseJob):
    
    def __init__(self, callback, description, job_manager, ebook, fmt, mi, hyphs):
        BaseJob.__init__(self, description)
        self.exception = None
        self.job_manager = job_manager
        self.args = (ebook, fmt, mi, hyphs)
        self.results = None
        self.callback = callback
        self.log_path = None
        self._log_file = DataIO()
        try:
            self._log_file.write(self.description.encode('utf-8') + '\n')
        except:
            self._log_file.write(self.description.encode('utf-8') + '\n'.encode('utf-8'))

    @property
    def log_file(self):
        if self.log_path is not None:
            return open(self.log_path, 'rb')
        return DataIO(self._log_file.getvalue())

    def start_work(self):
        self.start_time = time.time()
        self.job_manager.changed_queue.put(self)

    def job_done(self):
        self.duration = time.time() - self.start_time
        self.percent = 1
        # Dump log onto disk
        lf = PersistentTemporaryFile('hyphenatethis_log')
        lf.write(self._log_file.getvalue())
        lf.close()
        self.log_path = lf.name
        self._log_file.close()
        self._log_file = None

        self.job_manager.changed_queue.put(self)

    def log_write(self, what):
        try:
            self._log_file.write(what)
        except:
            self._log_file.write(what.encode('utf-8'))
        

class RemoveHyphensJob(BaseJob):
    
    def __init__(self, callback, description, job_manager, ebook, fmt):
        BaseJob.__init__(self, description)
        self.exception = None
        self.job_manager = job_manager
        self.args = (ebook, fmt)
        self.results = None
        self.callback = callback
        self.log_path = None
        self._log_file = DataIO()
        try:
            self._log_file.write(self.description.encode('utf-8') + '\n')
        except:
            self._log_file.write(self.description.encode('utf-8') + '\n'.encode('utf-8'))

    @property
    def log_file(self):
        if self.log_path is not None:
            return open(self.log_path, 'rb')
        return DataIO(self._log_file.getvalue())

    def start_work(self):
        self.start_time = time.time()
        self.job_manager.changed_queue.put(self)

    def job_done(self):
        self.duration = time.time() - self.start_time
        self.percent = 1
        # Dump log onto disk
        lf = PersistentTemporaryFile('hyphenatethis_log')
        lf.write(self._log_file.getvalue())
        lf.close()
        self.log_path = lf.name
        self._log_file.close()
        self._log_file = None

        self.job_manager.changed_queue.put(self)

    def log_write(self, what):
        try:
            self._log_file.write(what)
        except:
            self._log_file.write(what.encode('utf-8'))

class DecodingError(Exception):
    pass
