# -*- coding: utf-8 -*-
from __future__ import (unicode_literals, division, absolute_import)
from __future__ import print_function
from calibre.ebooks.oeb.polish.container import get_container
from collections import Counter
from calibre_plugins.hyphenatethis.gui.prompt import SelectHyphenateFormats
from calibre.utils.localization import get_language
from calibre_plugins.hyphenatethis.hyphenator.hyphenator import Hyphenator
from calibre_plugins.hyphenatethis.hyphenator.hutils import get_dict_by_lang,\
    get_dict_left_righ_min
from calibre.gui2 import error_dialog, Dispatcher
from calibre_plugins.hyphenatethis.workers.hjob import HyphenateThisThread
import os
from calibre_plugins.hyphenatethis.gui.config import DICT_ZIP
import six

__license__ = 'GPL 3'
__copyright__ = '2014, Saulius P. <saulius@kofmarai.net>'
__docformat__ = 'restructuredtext en'

from calibre_plugins.hyphenatethis import ActionHyphenateThis, SUPPORTED_FORMATS

try:
    from PyQt5.Qt import QMenu, QMessageBox
except ImportError:
    from PyQt4.Qt import QMenu, QMessageBox

from calibre.gui2.actions import InterfaceAction

class HyphenateThisAction(InterfaceAction):

    name = 'Hyphenate This!'
    action_spec = (_('Hyphenate'), None, None, None)
    
    def genesis(self):
        self.hyphenators = {}
        self.mixin = HyphenateThisMixin(self.gui)
        self.qaction.setIcon(get_icons('icons/hyphenate-this.png'))
        self.qaction.triggered.connect(self.hyphenate)
        self.menu = QMenu()
        self.load_menu()
        
    def load_menu(self):
        self.menu.clear()
        self.menu.addAction(_('Hyphenate...'), self.hyphenate)
        self.menu.addAction(_('Remove soft-hyphens...'), self.removeHyphens)
        self.menu.addAction(_('Settings...'), self.show_configuration)
        self.menu.addAction(_('About...'), self.about)
        self.qaction.setMenu(self.menu)
    
    '''
    Generic method to select books. In case method is used to select books
    for soft hyphens removal, language check and hyphenators initialization
    might be skipped.
    '''
    def _select_books(self, check_lang=True):
        if not os.path.exists(DICT_ZIP):
            error_dialog(None, _('No dictionaries installed'),
                                _('Please install some dictionaries first'), show=True)
            return None
        
        self.mixin.genesis()
        ids = [self.gui.library_view.model().id(r) for r in self.gui.library_view.selectionModel().selectedRows()]
        metadata = self.gui.library_view.model().metadata_for(ids)
        imetadata = iter(metadata)
        
        books = {}
        self.hyphenators = {}
        
        c = Counter()
        for idItem in ids:
            mi = next(imetadata)
            for fmt in mi.formats:
                if fmt.lower() in SUPPORTED_FORMATS:
                    '''
                    Skip language check if books are selected for soft-hyphen cleanup.
                    '''
                    if check_lang:
                        d = get_dict_by_lang(six.text_type(mi.language))
                        if d is None:
                            error_dialog(self.gui, _('No dictionary'),
                                _('Book\'s "')+mi.title+_('" language is ')+get_language(mi.language)+
                                _('. There is no dictionary installed for this language. Book won\'t be hyphenated.'), show=True)
                            continue

                        if mi.language not in self.hyphenators:
                            from calibre_plugins.hyphenatethis.gui.config import prefs
                            if (mi.language+'.dic') in prefs['lroverride']:
                                lmin, rmin = prefs['lroverride'][mi.language+'.dic']['LEFTHYPHENMIN'], prefs['lroverride'][mi.language+'.dic']['RIGHTHYPHENMIN']
                                print('Hyphenator using LR:', lmin, rmin)
                            else:
                                lmin, rmin = get_dict_left_righ_min(d)
                                print('Hyphenator using self LR:', lmin, rmin)
                            self.hyphenators[mi.language] = Hyphenator(d, lmin, rmin)
                        
                    c[fmt.lower()] += 1
                    if fmt.lower() not in books:
                        books[fmt.lower()] = []
                    books[fmt.lower()].append((idItem, mi))
                    
        if not len(books):
            error_dialog(self.gui, _('No books selected'),
                    _('None of the selected books are of supported formats for hyphenation.'), show=True)
            return None
        
        d = SelectHyphenateFormats(c, _('Choose formats to hyphenate'), parent=self.gui)
        if d.exec_() != d.Accepted:
            return None
        
        rbooks = {}
        for fmt in d.selected_formats:
            rbooks[fmt] = books[fmt]
        
        return rbooks
    
    '''
    Hyphenate selected books.
    '''
    def hyphenate(self):
        books = self._select_books()
        if books is None: return
        for fmt in books:
            for (idItem, mi) in books[fmt]:
                self.mixin.startHyphenation(idItem, fmt, mi, self.hyphenators)
    
    '''
    Remove soft hyphens from selected books.
    '''
    def removeHyphens(self):
        books = self._select_books(check_lang=False)
        if books is None: return
        for fmt in books:
            for (idItem, mi) in books[fmt]:
                self.mixin.startRemoving(idItem, fmt, mi)
    '''
    Standard Calibre's method to show configuration window.
    '''
    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)
    
    '''
    "About" window display.
    '''
    def about(self):
        text = get_resources('additional/about.txt')
        qb = QMessageBox(self.gui)
        qb.setText(text.decode('utf-8') % ActionHyphenateThis.version)
        qb.setWindowTitle('About "Hyphenate This!"')
        qb.setIconPixmap(get_icons('icons/hyphenate-this.png').pixmap(128, 128))
        qb.show()
   
    '''
    Apply configuration when changed.
    ''' 
    def changeConfig(self):
        from calibre_plugins.hyphenatethis.gui.config import prefs

'''
"Mixin" class to perform actual work.
'''
class HyphenateThisMixin(object):

    def __init__(self, gui):
        self.gui = gui
    
    def genesis(self):
        '''
        Genesis must always be called before using an SaveToFormatMixin object.
        '''
        if not hasattr(self.gui, 'hyphenator_thread'):
            self.gui.hyphenator_thread = HyphenateThisThread(self.gui)

    def startHyphenation(self, idItem, fmt,  mi, hyphs):
        if not self.gui.hyphenator_thread.is_alive():
            self.gui.hyphenator_thread.start()
        self.gui.hyphenator_thread.hyphenate(Dispatcher(self._endHyphenation), idItem, fmt, mi, hyphs)
        self.gui.status_bar.show_message(_('Hyphenation of book "%s" started...') % mi.title, 3000)
        
    def _endHyphenation(self, job):
        if job.failed is True:
            error_dialog(self.gui, _('Hyphenation failed'), job.description, show=True)
    
    def startRemoving(self, idItem, fmt, mi):
        if not self.gui.hyphenator_thread.is_alive():
            self.gui.hyphenator_thread.start()
        self.gui.hyphenator_thread.removehyphens(Dispatcher(self._endRemoving), idItem, fmt, mi)
        self.gui.status_bar.show_message(_('Hyphens removal of book "%s" started...') % mi.title, 3000)
    
    def _endRemoving(self, job):
        if job.failed is True:
            error_dialog(self.gui, _('Removing hyphens failed'), job.description, show=True)
        