#!/usr/bin/python
# -*- coding: utf-8 -*-

# CSS Font-Size Keyword Converter Plugin for Sigil
# Converts CSS font-size keywords to em/rem units
#
# This plugin converts CSS font-size keywords (small, large, x-small, etc.)
# to em or rem units in all CSS contexts:
# - External CSS files (.css)
# - Embedded <style> blocks in HTML/XHTML files
# - Inline style="" attributes in HTML/XHTML files

import re
import sys
import os
import json
import uuid
import tempfile
from datetime import datetime
from tkinter import *
from tkinter import ttk
from tkinter import filedialog
from tkinter import messagebox

# Plugin identification
PLUGIN_NAME = "NewUnit"
PLUGIN_VERSION = "0.7.0"

# Configuration directory
CONFIG_DIR = os.path.join(os.path.expanduser('~'), '.sigil', 'NewUnit', 'configs')


class ToolTip:
    """
    Create a tooltip for a given widget
    """
    def __init__(self, widget, text):
        self.widget = widget
        self.text = text
        self.tooltip_window = None
        self.widget.bind("<Enter>", self.show_tooltip)
        self.widget.bind("<Leave>", self.hide_tooltip)

    def show_tooltip(self, event=None):
        if self.tooltip_window or not self.text:
            return
        x = self.widget.winfo_rootx() + 20
        y = self.widget.winfo_rooty() + self.widget.winfo_height() + 5
        self.tooltip_window = tw = Toplevel(self.widget)
        tw.wm_overrideredirect(True)
        tw.wm_geometry(f"+{x}+{y}")
        label = Label(tw, text=self.text, justify=LEFT,
                     background="#ffffe0", relief=SOLID, borderwidth=1,
                     font=("Arial", 9))
        label.pack()

    def hide_tooltip(self, event=None):
        if self.tooltip_window:
            self.tooltip_window.destroy()
            self.tooltip_window = None


def ensure_config_dir():
    """Create configuration directory if it doesn't exist"""
    try:
        if not os.path.exists(CONFIG_DIR):
            os.makedirs(CONFIG_DIR)
        return True
    except (OSError, IOError) as e:
        print(f'Warning: Could not create config directory: {e}')
        sys.stdout.flush()
        return False


def get_saved_configs():
    """Get list of saved configuration files"""
    if not ensure_config_dir():
        return []
    
    configs = []
    try:
        for filename in os.listdir(CONFIG_DIR):
            if filename.endswith('.json'):
                filepath = os.path.join(CONFIG_DIR, filename)
                try:
                    with open(filepath, 'r', encoding='utf-8') as f:
                        data = json.load(f)
                        configs.append({
                            'filename': filename,
                            'name': data.get('name', filename[:-5]),
                            'description': data.get('description', ''),
                            'filepath': filepath
                        })
                except (IOError, json.JSONDecodeError, KeyError) as e:
                    print(f'Warning: Could not load config {filename}: {e}')
                    sys.stdout.flush()
    except OSError as e:
        print(f'Warning: Could not read config directory: {e}')
        sys.stdout.flush()
    
    return sorted(configs, key=lambda x: x['name'])


def generate_unique_placeholder():
    """Generate a unique placeholder that won't conflict with CSS content"""
    return f"___CSS_COMMENT_{uuid.uuid4().hex}___"


def strip_css_comments(css_text):
    """
    Remove CSS comments and return text with placeholders and placeholder mapping
    Uses unique UUID-based placeholders to prevent conflicts
    Returns: (text_no_comments, placeholder_map)
    """
    placeholder_map = {}
    
    def save_comment(match):
        placeholder = generate_unique_placeholder()
        comment_text = match.group(0)
        placeholder_map[placeholder] = comment_text
        return placeholder
    
    text_no_comments = re.sub(r'/\*.*?\*/', save_comment, css_text, 
                              flags=re.DOTALL)
    return text_no_comments, placeholder_map


def restore_css_comments(css_text, placeholder_map):
    """
    Restore CSS comments from placeholders
    """
    for placeholder, comment in placeholder_map.items():
        css_text = css_text.replace(placeholder, comment)
    return css_text


def is_unsafe_shorthand(css_line):
    """
    Detect potentially unsafe font shorthand patterns
    Checks specifically within the font property value, not comments
    """
    # Remove comments for accurate checking
    line_no_comments = re.sub(r'/\*.*?\*/', '', css_line)
    
    # Extract the font property value if present
    font_match = re.search(r'\bfont\s*:\s*([^;{}"\']+)', line_no_comments, re.IGNORECASE)
    if not font_match:
        return False
    
    font_value = font_match.group(1)
    
    # Check for calc(), var(), or other complex functions in the value
    if re.search(r'\b(calc|var|min|max|clamp)\s*\(', font_value, re.IGNORECASE):
        return True
    
    # Check for multiple slashes (unusual line-height patterns)
    if font_value.count('/') > 1:
        return True
    
    # Check for very complex patterns with many values
    parts = re.split(r'\s+', font_value.strip())
    if len(parts) > 8:  # Unusually complex
        return True
    
    return False


def validate_backup_folder(folder_path):
    """
    Validate that backup folder exists and is writable
    Returns (is_valid, error_message)
    """
    if not folder_path:
        return False, "No backup folder specified"
    
    if not os.path.exists(folder_path):
        return False, f"Folder does not exist: {folder_path}"
    
    if not os.path.isdir(folder_path):
        return False, f"Path is not a directory: {folder_path}"
    
    # Test write permissions
    test_file = os.path.join(folder_path, f'.write_test_{uuid.uuid4().hex}.tmp')
    try:
        with open(test_file, 'w') as f:
            f.write('test')
        os.remove(test_file)
        return True, ""
    except (IOError, OSError) as e:
        return False, f"Folder is not writable: {e}"


def run(bk):
    """
    Main plugin entry point for Sigil
    """

    print("CSS Font-Size Keyword Converter Plugin v0.7.0")
    sys.stdout.flush()

    # Get preferences
    prefs = bk.getPrefs()

    # Get saved preferences or use defaults
    config = {
        'xxsmall': prefs.get('xxsmall', '0.6'),
        'xsmall': prefs.get('xsmall', '0.75'),
        'small': prefs.get('small', '0.89'),
        'medium': prefs.get('medium', '1'),
        'large': prefs.get('large', '1.2'),
        'xlarge': prefs.get('xlarge', '1.5'),
        'xxlarge': prefs.get('xxlarge', '2'),
        'smaller': prefs.get('smaller', '0.85'),
        'larger': prefs.get('larger', '1.15'),
        'xxsmall_unit': prefs.get('xxsmall_unit', 'em'),
        'xsmall_unit': prefs.get('xsmall_unit', 'em'),
        'small_unit': prefs.get('small_unit', 'em'),
        'medium_unit': prefs.get('medium_unit', 'em'),
        'large_unit': prefs.get('large_unit', 'em'),
        'xlarge_unit': prefs.get('xlarge_unit', 'em'),
        'xxlarge_unit': prefs.get('xxlarge_unit', 'em'),
        'smaller_unit': prefs.get('smaller_unit', 'em'),
        'larger_unit': prefs.get('larger_unit', 'em'),
        'convert_font_shorthand': prefs.get('convert_font_shorthand', 'false'),
        'create_backup': prefs.get('create_backup', 'false'),
        'backup_path': prefs.get('backup_path', '')
    }

    print("Scanning files for font-size keywords...")
    sys.stdout.flush()

    def build_patterns(cfg):
        conversions = []
        
        keywords = [
            ('xx-small', 'xxsmall', 'xxsmall_unit'),
            ('x-small', 'xsmall', 'xsmall_unit'),
            ('small', 'small', 'small_unit'),
            ('medium', 'medium', 'medium_unit'),
            ('large', 'large', 'large_unit'),
            ('x-large', 'xlarge', 'xlarge_unit'),
            ('xx-large', 'xxlarge', 'xxlarge_unit'),
            ('smaller', 'smaller', 'smaller_unit'),
            ('larger', 'larger', 'larger_unit'),
        ]
        
        # Safe font-size property patterns
        for css_keyword, cfg_key, unit_key in keywords:
            # Escape the keyword for safe regex use
            escaped_keyword = re.escape(css_keyword)
            pattern = (
                rf'(font-size\s*:\s*){escaped_keyword}'
                rf'(\s*(?=[;}}"\'\!]|$))'
            )
            
            value = cfg[cfg_key]
            unit = cfg[unit_key]
            
            def make_replacer(val, unt):
                def replacer(match):
                    return f'{match.group(1)}{val}{unt}{match.group(2)}'
                return replacer
            
            conversions.append((
                re.compile(pattern, re.IGNORECASE | re.MULTILINE),
                make_replacer(value, unit),
                'font-size'
            ))
        
        # Font shorthand property patterns (optional)
        if cfg.get('convert_font_shorthand', 'false') == 'true':
            for css_keyword, cfg_key, unit_key in keywords:
                escaped_keyword = re.escape(css_keyword)
                pattern = (
                    rf'(\bfont\s*:\s*(?:[^\s;{{}}]+\s+)*?)'
                    rf'{escaped_keyword}'
                    rf'(\s*(?:[/\s][^\s;{{}}]+)*\s*(?=[;}}"\']|$))'
                )
                
                value = cfg[cfg_key]
                unit = cfg[unit_key]
                
                def make_shorthand_replacer(val, unt):
                    def replacer(match):
                        return f'{match.group(1)}{val}{unt}{match.group(2)}'
                    return replacer
                
                conversions.append((
                    re.compile(pattern, re.IGNORECASE | re.MULTILINE),
                    make_shorthand_replacer(value, unit),
                    'font-shorthand'
                ))
        
        return conversions

    compiled_patterns = build_patterns(config)

    files_with_keywords = []

    for file_id, href in bk.css_iter():
        try:
            text = bk.readfile(file_id)
            if not isinstance(text, str):
                text = text.decode('utf-8')
            
            text_no_comments, _ = strip_css_comments(text)

            count = 0
            for pattern, _, _ in compiled_patterns:
                matches = pattern.findall(text_no_comments)
                if matches:
                    count += len(matches)

            if count > 0:
                files_with_keywords.append({
                    'id': file_id,
                    'href': href,
                    'type': 'css',
                    'count': count
                })
        except (UnicodeDecodeError, IOError) as e:
            print(f'Error scanning {href}: {e}')
            sys.stdout.flush()

    for file_id, href in bk.text_iter():
        try:
            text = bk.readfile(file_id)
            if not isinstance(text, str):
                text = text.decode('utf-8')
            
            text_no_comments, _ = strip_css_comments(text)

            count = 0
            for pattern, _, _ in compiled_patterns:
                matches = pattern.findall(text_no_comments)
                if matches:
                    count += len(matches)

            if count > 0:
                files_with_keywords.append({
                    'id': file_id,
                    'href': href,
                    'type': 'text',
                    'count': count
                })
        except (UnicodeDecodeError, IOError) as e:
            print(f'Error scanning {href}: {e}')
            sys.stdout.flush()

    if not files_with_keywords:
        print('No font-size keywords found in any files.')
        sys.stdout.flush()
        return 0

    print(f'Found {len(files_with_keywords)} file(s) with font-size keywords.')
    sys.stdout.flush()

    action, final_config = show_file_selector(
        bk, files_with_keywords, config, build_patterns)

    if action == 'quit':
        print('Operation completed.')
    else:
        print('Operation cancelled by user.')
    sys.stdout.flush()

    for key, value in final_config.items():
        prefs[key] = value
    bk.savePrefs(prefs)

    return 0


def show_config_window(parent, config):
    updated_config = config.copy()
    config_saved = [False]

    dialog = Toplevel(parent)
    dialog.title("Font-Size Keyword Converter - Configuration")
    
    # Make window size responsive to screen size
    screen_width = dialog.winfo_screenwidth()
    screen_height = dialog.winfo_screenheight()
    window_width = min(750, int(screen_width * 0.8))
    window_height = min(800, int(screen_height * 0.8))
    dialog.geometry(f"{window_width}x{window_height}")
    
    dialog.resizable(True, True)
    dialog.transient(parent)
    dialog.grab_set()

    canvas = Canvas(dialog)
    scrollbar = Scrollbar(dialog, orient="vertical", command=canvas.yview)
    scrollable_frame = Frame(canvas)

    scrollable_frame.bind(
        "<Configure>",
        lambda e: canvas.configure(scrollregion=canvas.bbox("all"))
    )

    canvas.create_window((0, 0), window=scrollable_frame, anchor="nw")
    canvas.configure(yscrollcommand=scrollbar.set)

    main_frame = Frame(scrollable_frame, padx=20, pady=20)
    main_frame.pack(fill=BOTH, expand=True)

    title_label = Label(main_frame, text="Configure Conversion Values",
                       font=('Arial', 14, 'bold'))
    title_label.pack(pady=(0, 10))

    # Configuration management frame
    config_mgmt_frame = LabelFrame(main_frame, text="Configuration Management", padx=10, pady=5)
    config_mgmt_frame.pack(fill=X, pady=(0, 10))

    def save_as_default():
        result = messagebox.askyesno(
            "Save as Defaults",
            "Save current configuration as your default settings?\n\n"
            "These will be used as the starting values when you open this plugin.",
            icon='question'
        )
        
        if not result:
            return
        
        current_config = {}
        for key, entry in entries.items():
            current_config[key] = entry.get()
        for key, unit_var in unit_vars.items():
            current_config[key + '_unit'] = unit_var.get()
        current_config['convert_font_shorthand'] = 'true' if convert_shorthand_var.get() else 'false'
        current_config['create_backup'] = 'true' if create_backup_var.get() else 'false'
        current_config['backup_path'] = backup_path_var.get()
        
        updated_config.update(current_config)
        
        messagebox.showinfo("Defaults Saved",
                          "Current configuration saved as defaults.\n"
                          "These settings will be used next time you run the plugin.")

    def save_to_named_config():
        name_dialog = Toplevel(dialog)
        name_dialog.title("Save Configuration")
        name_dialog.geometry("400x180")
        name_dialog.transient(dialog)
        name_dialog.grab_set()

        dialog_result = {}

        frame = Frame(name_dialog, padx=20, pady=20)
        frame.pack(fill=BOTH, expand=True)

        Label(frame, text="Configuration Name:", font=('Arial', 10, 'bold')).pack(
            anchor=W, pady=(0, 5))
        name_entry = Entry(frame, width=40)
        name_entry.pack(pady=(0, 10))
        name_entry.focus()

        Label(frame, text="Description (optional):", font=('Arial', 10, 'bold')).pack(
            anchor=W, pady=(0, 5))
        desc_entry = Entry(frame, width=40)
        desc_entry.pack(pady=(0, 15))

        def on_save():
            if not name_entry.get().strip():
                messagebox.showwarning("Missing Name",
                                     "Please enter a configuration name.")
                return
            dialog_result['name'] = name_entry.get().strip()
            dialog_result['description'] = desc_entry.get().strip()
            name_dialog.destroy()

        def on_cancel_save():
            name_dialog.destroy()

        btn_frame = Frame(frame)
        btn_frame.pack()
        Button(btn_frame, text="Cancel", command=on_cancel_save,
               width=10).pack(side=LEFT, padx=5)
        Button(btn_frame, text="Save", command=on_save,
               bg="#4CAF50", fg="white", width=10).pack(side=LEFT, padx=5)

        name_dialog.wait_window()

        if not dialog_result:
            return

        current_config = {}
        for key, entry in entries.items():
            current_config[key] = entry.get()
        for key, unit_var in unit_vars.items():
            current_config[key + '_unit'] = unit_var.get()
        current_config['convert_font_shorthand'] = 'true' if convert_shorthand_var.get() else 'false'
        current_config['create_backup'] = 'true' if create_backup_var.get() else 'false'
        current_config['backup_path'] = backup_path_var.get()

        save_data = {
            "plugin": "FontSizeKeywordToEm",
            "version": PLUGIN_VERSION,
            "name": dialog_result['name'],
            "description": dialog_result['description'],
            "created": datetime.now().isoformat(),
            "config": current_config
        }

        if not ensure_config_dir():
            messagebox.showerror("Config Error",
                               "Could not create configuration directory.")
            return

        safe_name = dialog_result['name'].replace(' ', '_')
        safe_name = ''.join(c for c in safe_name if c.isalnum() or c in ('_', '-'))
        filename = os.path.join(CONFIG_DIR, f"{safe_name}.json")

        try:
            with open(filename, 'w', encoding='utf-8') as f:
                json.dump(save_data, f, indent=2)
            messagebox.showinfo("Configuration Saved",
                              f"Configuration '{dialog_result['name']}' saved successfully.")
        except (IOError, OSError) as e:
            messagebox.showerror("Save Error",
                               f"Error saving configuration:\n{e}")

    def load_named_config():
        configs = get_saved_configs()
        
        if not configs:
            messagebox.showinfo("No Configurations",
                              "No saved configurations found.\n\n"
                              "Save a configuration first to load it later.")
            return

        load_dialog = Toplevel(dialog)
        load_dialog.title("Load Configuration")
        load_dialog.geometry("500x400")
        load_dialog.transient(dialog)
        load_dialog.grab_set()

        frame = Frame(load_dialog, padx=20, pady=20)
        frame.pack(fill=BOTH, expand=True)

        Label(frame, text="Select Configuration to Load:",
              font=('Arial', 12, 'bold')).pack(pady=(0, 10))

        list_frame = Frame(frame)
        list_frame.pack(fill=BOTH, expand=True, pady=(0, 10))

        scrollbar_list = Scrollbar(list_frame)
        scrollbar_list.pack(side=RIGHT, fill=Y)

        config_listbox = Listbox(list_frame, yscrollcommand=scrollbar_list.set,
                                font=('Arial', 10))
        config_listbox.pack(side=LEFT, fill=BOTH, expand=True)
        scrollbar_list.config(command=config_listbox.yview)

        config_map = {}
        for idx, cfg in enumerate(configs):
            display = cfg['name']
            if cfg['description']:
                display += f" - {cfg['description']}"
            config_listbox.insert(END, display)
            config_map[idx] = cfg

        desc_label = Label(frame, text="", wraplength=450, justify=LEFT,
                          fg="#666666", font=('Arial', 9))
        desc_label.pack(pady=(5, 10))

        def on_select(event):
            selection = config_listbox.curselection()
            if selection:
                cfg = config_map[selection[0]]
                desc_text = f"File: {cfg['filename']}"
                if cfg['description']:
                    desc_text += f"\n{cfg['description']}"
                desc_label.config(text=desc_text)

        config_listbox.bind('<<ListboxSelect>>', on_select)

        selected_config = [None]

        def on_load():
            selection = config_listbox.curselection()
            if not selection:
                messagebox.showwarning("No Selection",
                                     "Please select a configuration to load.")
                return
            selected_config[0] = config_map[selection[0]]
            load_dialog.destroy()

        def on_cancel_load():
            load_dialog.destroy()

        btn_frame = Frame(frame)
        btn_frame.pack()
        Button(btn_frame, text="Cancel", command=on_cancel_load,
               width=10).pack(side=LEFT, padx=5)
        Button(btn_frame, text="Load", command=on_load,
               bg="#4CAF50", fg="white", width=10).pack(side=LEFT, padx=5)

        load_dialog.wait_window()

        if not selected_config[0]:
            return

        try:
            with open(selected_config[0]['filepath'], 'r', encoding='utf-8') as f:
                data = json.load(f)

            loaded_config = data.get('config', {})
            
            for key, value in loaded_config.items():
                if key.endswith('_unit'):
                    base_key = key[:-5]
                    if base_key in unit_vars:
                        unit_vars[base_key].set(value)
                elif key == 'convert_font_shorthand':
                    convert_shorthand_var.set(value == 'true')
                elif key == 'create_backup':
                    create_backup_var.set(value == 'true')
                elif key == 'backup_path':
                    backup_path_var.set(value)
                else:
                    if key in entries:
                        entries[key].delete(0, END)
                        entries[key].insert(0, value)

            messagebox.showinfo("Configuration Loaded",
                              f"Configuration '{selected_config[0]['name']}' loaded successfully.")

        except (IOError, json.JSONDecodeError, KeyError) as e:
            messagebox.showerror("Load Error",
                               f"Error loading configuration:\n{e}")

    btn_row1 = Frame(config_mgmt_frame)
    btn_row1.pack(fill=X, pady=2)

    save_default_btn = Button(btn_row1, text="Save as Default", 
                              command=save_as_default, width=20)
    save_default_btn.pack(side=LEFT, padx=5)
    ToolTip(save_default_btn, "Save current settings as your default configuration")

    btn_row2 = Frame(config_mgmt_frame)
    btn_row2.pack(fill=X, pady=2)

    save_named_btn = Button(btn_row2, text="Save to Named Config", 
                           command=save_to_named_config, width=20)
    save_named_btn.pack(side=LEFT, padx=5)
    ToolTip(save_named_btn, "Save current settings with a custom name")

    load_named_btn = Button(btn_row2, text="Load Named Config", 
                           command=load_named_config, width=20)
    load_named_btn.pack(side=LEFT, padx=5)
    ToolTip(load_named_btn, "Load a previously saved configuration")

    # Create two-column layout
    columns_frame = Frame(main_frame)
    columns_frame.pack(fill=BOTH, expand=True, pady=(0, 10))

    # LEFT COLUMN - Keyword configurations
    left_column = Frame(columns_frame)
    left_column.pack(side=LEFT, fill=BOTH, expand=True, padx=(0, 10))

    info_label = Label(
        left_column,
        text="Set the numeric value and unit type for each font-size keyword:",
        wraplength=300, justify=LEFT)
    info_label.pack(pady=(0, 10))

    config_frame = Frame(left_column)
    config_frame.pack(fill=BOTH, expand=False, pady=(0, 10))

    Label(config_frame, text="Keyword", font=('Arial', 10, 'bold'),
          width=12, anchor=W).grid(row=0, column=0, sticky=W, pady=5)
    Label(config_frame, text="Value", font=('Arial', 10, 'bold'),
          width=8).grid(row=0, column=1, pady=5)

    em_header = Label(config_frame, text="em", font=('Arial', 10, 'bold'),
                     width=5)
    em_header.grid(row=0, column=2, pady=5)
    ToolTip(em_header, "em units are relative to parent element font size")

    rem_header = Label(config_frame, text="rem", font=('Arial', 10, 'bold'),
                      width=5)
    rem_header.grid(row=0, column=3, pady=5)
    ToolTip(rem_header, "rem units are relative to root element font size\n"
                       "(more predictable across the document)")

    entries = {}
    unit_vars = {}
    keywords = [
        ('xx-small', 'xxsmall'),
        ('x-small', 'xsmall'),
        ('small', 'small'),
        ('medium', 'medium'),
        ('large', 'large'),
        ('x-large', 'xlarge'),
        ('xx-large', 'xxlarge'),
        ('smaller', 'smaller'),
        ('larger', 'larger')
    ]

    for idx, (display, key) in enumerate(keywords, start=1):
        label = Label(config_frame, text=display + ":", width=12,
                     anchor=W)
        label.grid(row=idx, column=0, sticky=W, pady=3)

        entry = Entry(config_frame, width=8)
        entry.insert(0, config[key])
        entry.grid(row=idx, column=1, padx=5, pady=3)
        entries[key] = entry

        unit_var = StringVar(value=config.get(key + '_unit', 'em'))
        unit_vars[key] = unit_var

        Radiobutton(config_frame, text="", variable=unit_var,
                   value='em').grid(row=idx, column=2, pady=3)
        Radiobutton(config_frame, text="", variable=unit_var,
                   value='rem').grid(row=idx, column=3, pady=3)

    bulk_frame = Frame(left_column)
    bulk_frame.pack(fill=X, pady=(5, 10))

    def set_all_em():
        for unit_var in unit_vars.values():
            unit_var.set('em')

    def set_all_rem():
        for unit_var in unit_vars.values():
            unit_var.set('rem')

    Label(bulk_frame, text="Set all to:").pack(side=LEFT, padx=(0, 10))
    all_em_btn = Button(bulk_frame, text="All em", command=set_all_em,
                       width=10)
    all_em_btn.pack(side=LEFT, padx=5)
    ToolTip(all_em_btn, "Set all keywords to use em units")

    all_rem_btn = Button(bulk_frame, text="All rem", command=set_all_rem,
                        width=10)
    all_rem_btn.pack(side=LEFT, padx=5)
    ToolTip(all_rem_btn, "Set all keywords to use rem units")

    # RIGHT COLUMN - Font Shorthand and Backup Options
    right_column = Frame(columns_frame)
    right_column.pack(side=LEFT, fill=BOTH, expand=True)

    shorthand_frame = LabelFrame(right_column, 
                                text="Font Shorthand (Advanced)",
                                padx=10, pady=10)
    shorthand_frame.pack(fill=X, pady=(0, 10))

    convert_shorthand_var = BooleanVar(
        value=config.get('convert_font_shorthand', 'false') == 'true')

    shorthand_check = Checkbutton(
        shorthand_frame,
        text="Also convert font: shorthand property",
        variable=convert_shorthand_var)
    shorthand_check.pack(anchor=W, pady=(0, 5))
    ToolTip(shorthand_check, 
            "Enable conversion in font shorthand (e.g., font: bold small Arial)\n"
            "WARNING: You will be shown a preview before changes are applied")

    warning_label = Label(
        shorthand_frame,
        text="WARNING: Font shorthand conversion is complex. You will see a preview "
             "of all shorthand changes before they are applied.",
        fg="#FF6600", font=('Arial', 8), justify=LEFT, wraplength=280)
    warning_label.pack(anchor=W, padx=5)

    backup_frame = LabelFrame(right_column, text="Backup Options",
                             padx=10, pady=10)
    backup_frame.pack(fill=X, pady=(0, 10))

    create_backup_var = BooleanVar(
        value=config.get('create_backup', 'false') == 'true')
    
    backup_path_var = StringVar(value=config.get('backup_path', ''))

    backup_check = Checkbutton(
        backup_frame,
        text="Create backup copies before converting",
        variable=create_backup_var)
    backup_check.pack(anchor=W, pady=(0, 5))
    ToolTip(backup_check, 
            "Save original files to a backup folder before making changes")

    path_frame = Frame(backup_frame)
    path_frame.pack(fill=X)

    Label(path_frame, text="Backup folder:", font=('Arial', 9)).pack(anchor=W, pady=(0, 5))
    
    backup_path_entry = Entry(path_frame, textvariable=backup_path_var, 
                              width=25, state='readonly')
    backup_path_entry.pack(fill=X, pady=(0, 5))
    
    def choose_backup_folder():
        folder = filedialog.askdirectory(
            title="Select Backup Folder",
            initialdir=backup_path_var.get() if backup_path_var.get() else None
        )
        if folder:
            # Validate the selected folder
            is_valid, error_msg = validate_backup_folder(folder)
            if is_valid:
                backup_path_var.set(folder)
            else:
                messagebox.showerror("Invalid Backup Folder", error_msg)
    
    browse_btn = Button(path_frame, text="Browse...", 
                       command=choose_backup_folder, width=12)
    browse_btn.pack(fill=X)
    ToolTip(browse_btn, "Choose where to save backup files")

    info_label_backup = Label(
        backup_frame,
        text="Backup files will be saved with timestamp:\nfilename_YYYYMMDD_HHMMSS.ext",
        fg="#666666", font=('Arial', 8), justify=LEFT, wraplength=280)
    info_label_backup.pack(anchor=W, pady=(5, 0))

    error_label = Label(main_frame, text="", fg="red", wraplength=350)
    error_label.pack()

    button_frame = Frame(main_frame)
    button_frame.pack(fill=X, pady=(10, 0))

    def save_config():
        error_label.config(text="")
        
        # Validate backup configuration
        if create_backup_var.get():
            if not backup_path_var.get():
                error_label.config(text="Error: Please select a backup folder")
                return
            
            is_valid, error_msg = validate_backup_folder(backup_path_var.get())
            if not is_valid:
                error_label.config(text=f"Error: {error_msg}")
                return
        
        # Validate numeric values
        for key, entry in entries.items():
            value = entry.get().strip()
            try:
                float_value = float(value)
                if float_value <= 0:
                    error_label.config(
                        text="Error: All values must be positive numbers")
                    return
                updated_config[key] = value
            except ValueError:
                error_label.config(
                    text=f"Error: '{value}' is not a valid number")
                return

        for key, unit_var in unit_vars.items():
            updated_config[key + '_unit'] = unit_var.get()

        updated_config['convert_font_shorthand'] = 'true' if convert_shorthand_var.get() else 'false'
        updated_config['create_backup'] = 'true' if create_backup_var.get() else 'false'
        updated_config['backup_path'] = backup_path_var.get()

        config_saved[0] = True
        dialog.destroy()

    def on_cancel():
        config_saved[0] = False
        dialog.destroy()

    def reset_defaults():
        defaults = {
            'xxsmall': '0.6',
            'xsmall': '0.75',
            'small': '0.89',
            'medium': '1',
            'large': '1.2',
            'xlarge': '1.5',
            'xxlarge': '2',
            'smaller': '0.85',
            'larger': '1.15'
        }
        for key, entry in entries.items():
            entry.delete(0, END)
            entry.insert(0, defaults[key])
        for unit_var in unit_vars.values():
            unit_var.set('em')
        convert_shorthand_var.set(False)
        create_backup_var.set(False)
        backup_path_var.set('')

    Button(button_frame, text="Reset Defaults",
           command=reset_defaults).pack(side=LEFT)
    Button(button_frame, text="Cancel", command=on_cancel,
           width=10).pack(side=RIGHT, padx=5)
    save_btn = Button(button_frame, text="Save", command=save_config,
                     bg="#4CAF50", fg="white", width=10)
    save_btn.pack(side=RIGHT)
    ToolTip(save_btn, "Save configuration and return to file selection")

    canvas.pack(side="left", fill="both", expand=True)
    scrollbar.pack(side="right", fill="y")

    dialog.update_idletasks()
    x = parent.winfo_x() + (parent.winfo_width() - dialog.winfo_width()) // 2
    y = parent.winfo_y() + (parent.winfo_height() - dialog.winfo_height()) // 2
    dialog.geometry("+{0}+{1}".format(x, y))

    parent.wait_window(dialog)

    if config_saved[0]:
        return updated_config
    else:
        return config


def show_unsafe_css_dialog(unsafe_items, parent):
    """
    Show dialog with unsafe CSS patterns that will be skipped
    Returns True if user wants to continue, False to stop
    """
    unsafe_dialog = Toplevel(parent)
    unsafe_dialog.title("Unsafe CSS Detected")
    unsafe_dialog.geometry("700x500")
    unsafe_dialog.transient(parent)
    unsafe_dialog.grab_set()

    main_frame = Frame(unsafe_dialog, padx=20, pady=20)
    main_frame.pack(fill=BOTH, expand=True)

    title = Label(main_frame,
                 text="WARNING: Complex Font Shorthand Detected",
                 font=('Arial', 14, 'bold'),
                 fg="#FF6600")
    title.pack(pady=(0, 10))

    message = Label(
        main_frame,
        text="The following font shorthand properties are too complex to convert safely and will be SKIPPED during conversion. These items contain calc(), var(), or other complex functions that cannot be reliably converted using simple pattern matching. They will remain unchanged in your files and must be edited manually if you wish to convert them.",
        wraplength=650,
        justify=LEFT,
        font=('Arial', 10))
    message.pack(pady=(0, 10))

    text_frame = Frame(main_frame)
    text_frame.pack(fill=BOTH, expand=True)

    scrollbar = Scrollbar(text_frame)
    scrollbar.pack(side=RIGHT, fill=Y)

    unsafe_text = Text(text_frame, wrap=WORD,
                      yscrollcommand=scrollbar.set,
                      font=('Courier', 9),
                      height=15)
    unsafe_text.pack(side=LEFT, fill=BOTH, expand=True)
    scrollbar.config(command=unsafe_text.yview)

    unsafe_text.tag_config('file', font=('Arial', 10, 'bold'), foreground='#2196F3')
    unsafe_text.tag_config('line', foreground='#666666')
    unsafe_text.tag_config('css', foreground='#000000', background='#ffe6e6')

    for item in unsafe_items:
        unsafe_text.insert(END, f"\n{item['file']}\n", 'file')
        unsafe_text.insert(END, f"  Line {item['line']}: ", 'line')
        unsafe_text.insert(END, f"{item['css']}\n", 'css')

    unsafe_text.config(state=DISABLED)

    count_label = Label(main_frame,
                       text=f"Total unsafe patterns: {len(unsafe_items)}",
                       font=('Arial', 10, 'bold'))
    count_label.pack(pady=(10, 0))

    user_choice = [False]

    button_frame = Frame(main_frame)
    button_frame.pack(fill=X, pady=(15, 0))

    def on_stop():
        user_choice[0] = False
        unsafe_dialog.destroy()

    def on_continue():
        user_choice[0] = True
        unsafe_dialog.destroy()

    Button(button_frame, text="Stop - Don't Make Any Changes",
           command=on_stop, width=25).pack(side=LEFT, padx=5)
    Button(button_frame, text="Continue - Skip Unsafe Items",
           command=on_continue, bg="#4CAF50", fg="white",
           width=25).pack(side=RIGHT, padx=5)

    parent.wait_window(unsafe_dialog)
    return user_choice[0]


def show_shorthand_preview_dialog(shorthand_changes, safe_count, parent):
    """
    Show preview of shorthand changes that will be made
    Returns True if user wants to proceed, False to cancel
    """
    preview_dialog = Toplevel(parent)
    preview_dialog.title("Review Safe Shorthand Changes")
    preview_dialog.geometry("800x600")
    preview_dialog.transient(parent)
    preview_dialog.grab_set()

    main_frame = Frame(preview_dialog, padx=20, pady=20)
    main_frame.pack(fill=BOTH, expand=True)

    title = Label(main_frame,
                 text="Review Safe Shorthand Changes",
                 font=('Arial', 14, 'bold'))
    title.pack(pady=(0, 10))

    summary = Label(
        main_frame,
        text=f"Safe font-size conversions: {safe_count} (will be applied)\n"
             f"Safe shorthand conversions: {len(shorthand_changes)} (review below)\n\n"
             "These shorthand conversions have been validated as safe.\n"
             "Unsafe patterns have already been filtered out.",
        wraplength=750,
        justify=LEFT,
        font=('Arial', 10))
    summary.pack(pady=(0, 10))

    text_frame = Frame(main_frame)
    text_frame.pack(fill=BOTH, expand=True)

    scrollbar = Scrollbar(text_frame)
    scrollbar.pack(side=RIGHT, fill=Y)

    preview_text = Text(text_frame, wrap=WORD,
                       yscrollcommand=scrollbar.set,
                       font=('Courier', 9))
    preview_text.pack(side=LEFT, fill=BOTH, expand=True)
    scrollbar.config(command=preview_text.yview)

    preview_text.tag_config('before',
                           background='#ffd6d6',
                           foreground='#000000')
    preview_text.tag_config('after',
                           background='#d6ffd6',
                           foreground='#000000')
    preview_text.tag_config('file_header',
                           font=('Arial', 10, 'bold'),
                           foreground='#2196F3')
    preview_text.tag_config('line_num',
                           foreground='#666666')

    for change in shorthand_changes:
        preview_text.insert(END, f"\n{change['file']}\n", 'file_header')
        preview_text.insert(END, f"  Line {change['line']}:\n", 'line_num')
        preview_text.insert(END, f"    Before: {change['before']}\n", 'before')
        preview_text.insert(END, f"    After:  {change['after']}\n", 'after')

    preview_text.config(state=DISABLED)

    user_choice = [None]

    button_frame = Frame(main_frame)
    button_frame.pack(fill=X, pady=(15, 0))

    def on_cancel():
        user_choice[0] = 'cancel'
        preview_dialog.destroy()

    def on_proceed():
        user_choice[0] = 'proceed'
        preview_dialog.destroy()

    Button(button_frame, text="Skip Shorthand - Apply Font-Size Only",
           command=on_cancel, width=35).pack(side=LEFT, padx=5)
    Button(button_frame, text="Apply All (Font-Size + Shorthand)",
           command=on_proceed, bg="#4CAF50", fg="white",
           width=35).pack(side=RIGHT, padx=5)

    parent.wait_window(preview_dialog)
    return user_choice[0]


def show_preview_window(bk, selected_files, current_config,
                       build_patterns_func, parent):
    preview_window = Toplevel(parent)
    preview_window.title("Preview Changes")
    preview_window.geometry("800x600")
    preview_window.transient(parent)
    preview_window.grab_set()

    main_frame = Frame(preview_window, padx=20, pady=20)
    main_frame.pack(fill=BOTH, expand=True)

    title = Label(main_frame,
                 text=f"Preview Changes - {len(selected_files)} file(s) selected",
                 font=('Arial', 12, 'bold'))
    title.pack(pady=(0, 10))

    text_frame = Frame(main_frame)
    text_frame.pack(fill=BOTH, expand=True)

    scrollbar = Scrollbar(text_frame)
    scrollbar.pack(side=RIGHT, fill=Y)

    preview_text = Text(text_frame, wrap=WORD,
                       yscrollcommand=scrollbar.set,
                       font=('Courier', 10))
    preview_text.pack(side=LEFT, fill=BOTH, expand=True)
    scrollbar.config(command=preview_text.yview)

    preview_text.tag_config('before',
                           background='#ffd6d6',
                           foreground='#000000')
    preview_text.tag_config('before_shorthand',
                           background='#cce5ff',
                           foreground='#000000')
    preview_text.tag_config('shorthand_label',
                           foreground='#FF0000',
                           font=('Courier', 10, 'bold'))
    preview_text.tag_config('after',
                           background='#d6ffd6',
                           foreground='#000000')
    preview_text.tag_config('file_header',
                           font=('Arial', 11, 'bold'),
                           foreground='#2196F3')
    preview_text.tag_config('separator',
                           foreground='#999999')
    preview_text.tag_config('line_num',
                           foreground='#666666')

    total_changes = 0
    compiled_patterns = build_patterns_func(current_config)

    for file_info in selected_files:
        try:
            original_text = bk.readfile(file_info['id'])
            if not isinstance(original_text, str):
                original_text = original_text.decode('utf-8')

            text_no_comments, placeholder_map = strip_css_comments(original_text)

            changes = []
            lines = text_no_comments.split('\n')

            for line_num, line in enumerate(lines, 1):
                for pattern, replacement, change_type in compiled_patterns:
                    match = pattern.search(line)
                    if match:
                        new_line = pattern.sub(replacement, line)
                        changes.append({
                            'line': line_num,
                            'before': line.strip(),
                            'after': new_line.strip(),
                            'type': change_type
                        })
                        break

            if changes:
                preview_text.insert(END, f"\n{file_info['href']} ",
                                  'file_header')
                preview_text.insert(END, f"({len(changes)} changes)\n",
                                  'file_header')
                preview_text.insert(END, "-" * 60 + "\n", 'separator')
                preview_text.insert(END, "\n")

                for change in changes:
                    preview_text.insert(END, f"Line {change['line']}:\n",
                                      'line_num')
                    
                    # Use different tag and label for shorthand
                    if change.get('type') == 'font-shorthand':
                        preview_text.insert(END, "  ", 'before_shorthand')
                        preview_text.insert(END, "[SHORTHAND] ", ('shorthand_label', 'before_shorthand'))
                        preview_text.insert(END, f"{change['before']}\n", 'before_shorthand')
                    else:
                        preview_text.insert(END, f"  {change['before']}\n",
                                          'before')
                    
                    preview_text.insert(END, f"  {change['after']}\n",
                                      'after')
                    preview_text.insert(END, "\n")

                total_changes += len(changes)
        except (UnicodeDecodeError, IOError) as e:
            preview_text.insert(END, f"\nError processing {file_info['href']}: {e}\n",
                              'separator')

    preview_text.config(state=DISABLED)

    status_frame = Frame(main_frame)
    status_frame.pack(fill=X, pady=(10, 0))

    status_label = Label(status_frame,
                        text=f"Total: {total_changes} changes across "
                             f"{len(selected_files)} file(s)",
                        font=('Arial', 10, 'bold'))
    status_label.pack()

    button_frame = Frame(main_frame)
    button_frame.pack(fill=X, pady=(10, 0))

    def copy_to_clipboard():
        preview_window.clipboard_clear()
        preview_window.clipboard_append(preview_text.get('1.0', END))
        copy_btn.config(text="Copied!")
        preview_window.after(1500,
                           lambda: copy_btn.config(text="Copy to Clipboard"))

    copy_btn = Button(button_frame, text="Copy to Clipboard",
                     command=copy_to_clipboard)
    copy_btn.pack(side=LEFT, padx=5)

    Button(button_frame, text="Close",
           command=preview_window.destroy,
           width=10).pack(side=RIGHT, padx=5)


def show_file_selector(bk, files_with_keywords, initial_config,
                       build_patterns_func):
    action = 'cancel'
    current_config = initial_config.copy()
    backup_files = {}

    root = Tk()
    root.title(f"CSS Font-Size Keyword Converter - Select Files v{PLUGIN_VERSION}")
    
    # Make window size responsive
    screen_width = root.winfo_screenwidth()
    screen_height = root.winfo_screenheight()
    window_width = min(700, int(screen_width * 0.6))
    window_height = min(550, int(screen_height * 0.7))
    root.geometry(f"{window_width}x{window_height}")
    root.resizable(True, True)

    show_all = BooleanVar(value=True)
    show_text = BooleanVar(value=False)
    show_css = BooleanVar(value=False)

    changes_applied = [False]

    main_frame = Frame(root, padx=10, pady=10)
    main_frame.pack(fill=BOTH, expand=True)

    top_frame = Frame(main_frame)
    top_frame.pack(fill=X, pady=(0, 10))

    def open_config():
        nonlocal current_config
        current_config = show_config_window(root, current_config)

    config_button = Button(top_frame, text="Configuration",
                          command=open_config, bg="#2196F3", fg="white")
    config_button.pack(side=LEFT)
    ToolTip(config_button, "Customize conversion values and choose em or rem "
                          "units for each keyword")

    filter_frame = LabelFrame(main_frame, text="Filter", padx=10, pady=5)
    filter_frame.pack(fill=X, pady=(0, 10))

    def update_filter():
        populate_checkboxes()

    cb_all = Checkbutton(
        filter_frame, text="Show All", variable=show_all,
        command=lambda: [show_text.set(False), show_css.set(False),
                        update_filter()])
    cb_all.pack(side=LEFT, padx=5)
    ToolTip(cb_all, "Display all files containing font-size keywords")

    cb_text = Checkbutton(
        filter_frame, text="Show HTML/XHTML Files", variable=show_text,
        command=lambda: [show_all.set(False), show_css.set(False),
                        update_filter()])
    cb_text.pack(side=LEFT, padx=5)
    ToolTip(cb_text, "Show only HTML/XHTML text files")

    cb_css = Checkbutton(
        filter_frame, text="Show CSS Files", variable=show_css,
        command=lambda: [show_all.set(False), show_text.set(False),
                        update_filter()])
    cb_css.pack(side=LEFT, padx=5)
    ToolTip(cb_css, "Show only CSS stylesheet files")

    instructions_label = Label(
        main_frame,
        text="Select files to convert:")
    instructions_label.pack(anchor=W, pady=(0, 5))

    list_frame = Frame(main_frame)
    list_frame.pack(fill=BOTH, expand=True, pady=(0, 10))

    scrollbar = Scrollbar(list_frame)
    scrollbar.pack(side=RIGHT, fill=Y)

    # Canvas for scrollable checkbox list
    canvas_files = Canvas(list_frame, yscrollcommand=scrollbar.set)
    canvas_files.pack(side=LEFT, fill=BOTH, expand=True)
    scrollbar.config(command=canvas_files.yview)

    checkbox_frame = Frame(canvas_files)
    canvas_files.create_window((0, 0), window=checkbox_frame, anchor="nw")

    file_checkboxes = {}
    file_map = {}

    def populate_checkboxes():
        # Clear existing checkboxes
        for widget in checkbox_frame.winfo_children():
            widget.destroy()
        file_checkboxes.clear()
        file_map.clear()

        filtered_files = files_with_keywords
        if show_text.get():
            filtered_files = [f for f in files_with_keywords
                            if f['type'] == 'text']
        elif show_css.get():
            filtered_files = [f for f in files_with_keywords
                            if f['type'] == 'css']

        for idx, file_info in enumerate(filtered_files):
            var = BooleanVar(value=False)
            display_text = "{0} ({1}) - {2} match(es)".format(
                file_info['href'],
                file_info['type'].upper(),
                file_info['count']
            )
            cb = Checkbutton(checkbox_frame, text=display_text, variable=var,
                           anchor=W, justify=LEFT)
            cb.pack(fill=X, padx=5, pady=2)
            file_checkboxes[idx] = var
            file_map[idx] = file_info

        # Update scroll region
        checkbox_frame.update_idletasks()
        canvas_files.config(scrollregion=canvas_files.bbox("all"))

    populate_checkboxes()

    status_label = Label(main_frame, text="", fg="blue", wraplength=600)
    status_label.pack(fill=X, pady=(0, 10))

    button_frame = Frame(main_frame)
    button_frame.pack(fill=X)

    def on_select_all():
        for var in file_checkboxes.values():
            var.set(True)

    def on_clear_selection():
        for var in file_checkboxes.values():
            var.set(False)

    select_all_btn = Button(button_frame, text="Select All",
                           command=on_select_all)
    select_all_btn.pack(side=LEFT, padx=5)

    clear_sel_btn = Button(button_frame, text="Clear Selection",
                          command=on_clear_selection)
    clear_sel_btn.pack(side=LEFT, padx=5)

    def on_preview():
        selected_files = [file_map[idx] for idx, var in file_checkboxes.items() if var.get()]
        if not selected_files:
            status_label.config(text="Please select at least one file.",
                              fg="red")
            return

        show_preview_window(bk, selected_files, current_config,
                          build_patterns_func, root)

    def on_apply():
        selected_files = [file_map[idx] for idx, var in file_checkboxes.items() if var.get()]
        if not selected_files:
            status_label.config(text="Please select at least one file.",
                              fg="red")
            return

        # Create external backups if enabled
        backup_created = False
        if current_config.get('create_backup') == 'true':
            backup_path = current_config.get('backup_path', '')
            
            # Validate backup path
            is_valid, error_msg = validate_backup_folder(backup_path)
            if not is_valid:
                messagebox.showerror("Backup Error", error_msg)
                return
            
            try:
                timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
                
                for file_info in selected_files:
                    original_text = bk.readfile(file_info['id'])
                    if not isinstance(original_text, str):
                        original_text = original_text.decode('utf-8')
                    
                    base_name = os.path.basename(file_info['href'])
                    name, ext = os.path.splitext(base_name)
                    backup_filename = f"{name}_{timestamp}{ext}"
                    backup_filepath = os.path.join(backup_path, backup_filename)
                    
                    with open(backup_filepath, 'w', encoding='utf-8') as f:
                        f.write(original_text)
                
                backup_created = True
                print(f'\nBackup files created in: {backup_path}')
                sys.stdout.flush()
                
            except (IOError, OSError) as e:
                error_msg = f'Warning: Could not create backup files: {e}'
                print(f'\n{error_msg}')
                sys.stdout.flush()
                
                result = messagebox.askyesno(
                    "Backup Failed",
                    f"{error_msg}\n\nContinue with conversion anyway?",
                    icon='warning'
                )
                if not result:
                    return

        # Backup files in memory (for undo functionality)
        backup_files.clear()
        for file_info in selected_files:
            try:
                original_text = bk.readfile(file_info['id'])
                if not isinstance(original_text, str):
                    original_text = original_text.decode('utf-8')
                backup_files[file_info['id']] = {
                    'content': original_text,
                    'href': file_info['href']
                }
            except (UnicodeDecodeError, IOError) as e:
                print(f'Error backing up {file_info["href"]}: {e}')
                sys.stdout.flush()

        # Build patterns with current config
        compiled_patterns = build_patterns_func(current_config)

        # Check if shorthand conversion is enabled
        shorthand_enabled = current_config.get('convert_font_shorthand') == 'true'

        # Categorize changes
        safe_changes = []
        shorthand_changes = []
        unsafe_items = []

        for file_info in selected_files:
            if file_info['id'] not in backup_files:
                continue  # Skip if backup failed
                
            original_text = backup_files[file_info['id']]['content']
            text_no_comments, placeholder_map = strip_css_comments(original_text)
            lines = text_no_comments.split('\n')

            for line_num, line in enumerate(lines, 1):
                for pattern, replacement, change_type in compiled_patterns:
                    match = pattern.search(line)
                    if match:
                        new_line = pattern.sub(replacement, line)
                        
                        change_data = {
                            'file': file_info['href'],
                            'file_id': file_info['id'],
                            'line': line_num,
                            'before': line.strip(),
                            'after': new_line.strip()
                        }

                        if change_type == 'font-shorthand':
                            # Check if unsafe
                            if is_unsafe_shorthand(line):
                                unsafe_items.append({
                                    'file': file_info['href'],
                                    'line': line_num,
                                    'css': line.strip()
                                })
                            else:
                                shorthand_changes.append(change_data)
                        else:
                            safe_changes.append(change_data)
                        
                        break

        # Handle shorthand conversion workflow
        apply_shorthand = False
        
        if shorthand_enabled and (shorthand_changes or unsafe_items):
            # Step 1: Show unsafe items if any
            if unsafe_items:
                continue_after_unsafe = show_unsafe_css_dialog(unsafe_items, root)
                if not continue_after_unsafe:
                    status_label.config(text="Operation cancelled by user.",
                                      fg="orange")
                    return

            # Step 2: Show shorthand preview
            if shorthand_changes:
                decision = show_shorthand_preview_dialog(
                    shorthand_changes, len(safe_changes), root)
                
                if decision == 'proceed':
                    apply_shorthand = True
                elif decision == 'cancel':
                    apply_shorthand = False
                else:
                    # User closed dialog
                    status_label.config(text="Operation cancelled by user.",
                                      fg="orange")
                    return

        # Apply changes
        modified_files = []
        errors = []

        for file_info in selected_files:
            if file_info['id'] not in backup_files:
                continue  # Skip if backup failed
                
            try:
                original_text = backup_files[file_info['id']]['content']
                text_no_comments, placeholder_map = strip_css_comments(original_text)
                modified_text = text_no_comments
                change_count = 0

                # Build list of patterns to apply
                patterns_to_apply = []
                
                # Add safe patterns
                for pattern, replacement, change_type in compiled_patterns:
                    if change_type == 'font-size':
                        patterns_to_apply.append((pattern, replacement))
                
                # Add shorthand patterns if approved
                if apply_shorthand:
                    for pattern, replacement, change_type in compiled_patterns:
                        if change_type == 'font-shorthand':
                            patterns_to_apply.append((pattern, replacement))
                
                # Apply all patterns - pattern.sub handles both callables and strings
                for pattern, replacement in patterns_to_apply:
                    matches = pattern.findall(modified_text)
                    if matches:
                        modified_text = pattern.sub(replacement, modified_text)
                        change_count += len(matches)

                modified_text = restore_css_comments(modified_text, placeholder_map)

                if modified_text != original_text:
                    bk.writefile(file_info['id'], modified_text)
                    modified_files.append((file_info['href'], change_count))
                    if len(modified_files) == 1:
                        print('')
                        sys.stdout.flush()
                    print(f"Converted: {file_info['href']} ({change_count} change(s))")
                    sys.stdout.flush()
            except (UnicodeDecodeError, IOError) as e:
                errors.append(f"{file_info['href']}: {e}")
                print(f'Error processing {file_info["href"]}: {e}')
                sys.stdout.flush()

        if modified_files:
            total_changes = sum(count for _, count in modified_files)
            status_msg = (
                f'Successfully converted {total_changes} match(es) in {len(modified_files)} file(s).')
            if backup_created:
                status_msg += ' Backups created.'
            if not apply_shorthand and shorthand_changes:
                status_msg += ' Shorthand conversions skipped.'
            if errors:
                status_msg += f' {len(errors)} error(s) occurred.'
            status_label.config(text=status_msg, fg="green")
            print(f'\n{status_msg}')
            sys.stdout.flush()

            changes_applied[0] = True
            apply_btn.config(state=DISABLED)
            preview_btn.config(state=DISABLED)
            for widget in checkbox_frame.winfo_children():
                widget.config(state=DISABLED)
            select_all_btn.config(state=DISABLED)
            clear_sel_btn.config(state=DISABLED)
            cb_all.config(state=DISABLED)
            cb_text.config(state=DISABLED)
            cb_css.config(state=DISABLED)
            undo_btn.config(state=NORMAL)
        else:
            status_label.config(text="No changes were made.", fg="orange")

        if errors:
            print('\nErrors encountered:')
            sys.stdout.flush()
            for e in errors:
                print(f'  {e}')
                sys.stdout.flush()

    def on_undo():
        restored_count = 0
        errors = []
        print('')
        sys.stdout.flush()
        for file_id, backup_info in backup_files.items():
            try:
                bk.writefile(file_id, backup_info['content'])
                restored_count += 1
                print(f"Restored: {backup_info['href']}")
                sys.stdout.flush()
            except (IOError, OSError) as e:
                errors.append(f"{backup_info['href']}: {e}")
                print(f"Error restoring {backup_info['href']}: {e}")
                sys.stdout.flush()

        if restored_count > 0:
            status_msg = f'Undone. Restored {restored_count} file(s) to original state.'
            if errors:
                status_msg += f' {len(errors)} error(s) occurred.'
            status_label.config(text=status_msg, fg="blue")
            print(f'\n{status_msg}')
            sys.stdout.flush()
        else:
            status_label.config(text="Undo failed - no files restored.", fg="red")

        changes_applied[0] = False
        backup_files.clear()
        apply_btn.config(state=NORMAL)
        preview_btn.config(state=NORMAL)
        for widget in checkbox_frame.winfo_children():
            widget.config(state=NORMAL)
        select_all_btn.config(state=NORMAL)
        clear_sel_btn.config(state=NORMAL)
        cb_all.config(state=NORMAL)
        cb_text.config(state=NORMAL)
        cb_css.config(state=NORMAL)
        undo_btn.config(state=DISABLED)

    def on_quit():
        nonlocal action
        if changes_applied[0]:
            action = 'quit'
        else:
            action = 'cancel'
        root.quit()
        root.destroy()

    quit_btn = Button(button_frame, text="Quit", command=on_quit,
                     width=10)
    quit_btn.pack(side=RIGHT, padx=5)
    ToolTip(quit_btn, "Close plugin and save configuration")

    undo_btn = Button(button_frame, text="Undo", command=on_undo,
                     bg="#FF9800", fg="white", width=10, state=DISABLED)
    undo_btn.pack(side=RIGHT, padx=5)
    ToolTip(undo_btn, "Restore all modified files to their original state")

    apply_btn = Button(button_frame, text="Apply Changes", command=on_apply,
                      bg="#4CAF50", fg="white", width=12)
    apply_btn.pack(side=RIGHT, padx=5)
    ToolTip(apply_btn, "Convert font-size keywords in selected files\n"
                      "(changes can be undone)")

    preview_btn = Button(button_frame, text="Preview", command=on_preview,
                        bg="#2196F3", fg="white", width=10)
    preview_btn.pack(side=RIGHT, padx=5)
    ToolTip(preview_btn, "Show all changes before applying")

    root.mainloop()

    return action, current_config


def main():
    print('This plugin must be run from within Sigil.')
    return -1


if __name__ == "__main__":
    sys.exit(main())